/*

 @ author, GLSL & OpenGL code author Athanasios Theocharidis, Michael Kargas, 2012-2015

*/

#if GPU_SHADER4_COMPATIBILITY_CONDITION
    #extension GL_EXT_gpu_shader4: enable
#endif

#if GPU_SHADER_FP64_COMPATIBILITY_CONDITION
    #extension GL_ARB_gpu_shader_fp64: enable
#endif

uniform sampler3D volume3DTexture;

uniform bool oldLCDStyleTransparency;

uniform float screenRatio;
uniform vec3 cameraRight;
uniform vec3 cameraUp;
uniform vec3 cameraDirection;
uniform vec3 cameraPosition;

uniform vec2 screenSize;
uniform vec3 voxelDimension;
uniform bool voxelDensityRendering;
uniform float transparencyLevel;
uniform float transparencyFactor;
uniform bool useVoxelLighting;
uniform bool jitterStartingRays;
uniform bool useBackStepForRays;
uniform float raymarcherSkipLength;

uniform bvec4 enableIsoSurfaceRGBA;
uniform vec4 isoSurfaceRGBA;
uniform bvec4 enableDensityRGBA;
uniform vec4 densityRGBA;

uniform bool useTransferFunction;
uniform int transferFunctionType;
uniform int transferFunctionBlendingType;
uniform float transferFunctionBlendingValue;

uniform bool useImageProcessing = false;
uniform int imageProcessingType = 1;
uniform int imageProcessingBlendingType = 1;
uniform float imageProcessingBlendingValue = 1.0;

const vec3 ZERO_POINT = vec3(0.0);
const vec3 CENTER_POINT = vec3(0.5);
const float DEFAULT_RESOLUTION = 128.0;
const float DEPTH_RATIO_FACTOR = 100.0;
const vec2 ZERO_ONE_AXIS = vec2(0.0, 1.0);
const vec4 FULL_COLOR_INTENSITY = vec4(1.0);
const vec4 ZERO_COLOR_INTENSITY = vec4(0.0);

float voxelStepSize = 1.0 / voxelDimension.z;
float forwardStepSize = 0.4 * raymarcherSkipLength * voxelStepSize;
vec3 lowerBounds = vec3( (voxelDimension / DEFAULT_RESOLUTION) / (DEPTH_RATIO_FACTOR * voxelDimension) );
vec3 upperBounds = vec3(1.0 - lowerBounds);

// extra look-up/colorization related functions
vec3 rand3(in vec2 seed);
void applyOldStyleTransparency();
vec4 applyTextureFiltering(in vec3 texCoords);
vec4 applyLighting(in vec3, in vec4);
vec3 solidityAdaptation(in vec4, in vec3, in vec3, in bvec4, in vec4);
vec4 applyTransferFunction(in int, in float, in vec4, in int, in float);
vec4 applyImageProcessing(in int, in vec4, in int, in float);

// voxel rendering related functions
vec2 axisAlignedBoundingBox(in vec3, in vec3);
vec3 backStep(in vec3, in vec3, in int);
vec4 forwardSampleStep(in vec3, in vec3, in int);

vec2 axisAlignedBoundingBox(in vec3 origin, in vec3 direction)
{
  bvec3 nonInvertedDirectionAxis = greaterThanEqual(direction, ZERO_POINT);

  // Slab X
  vec2 tNearFarPlaneX = ( (nonInvertedDirectionAxis.x ? ZERO_ONE_AXIS.xy : ZERO_ONE_AXIS.yx) - origin.x ) / direction.x;

  // Slab Y
  vec2 tNearFarPlaneY = ( (nonInvertedDirectionAxis.y ? ZERO_ONE_AXIS.xy : ZERO_ONE_AXIS.yx) - origin.y ) / direction.y;

  // Slab Z
  vec2 tNearFarPlaneZ = ( (nonInvertedDirectionAxis.z ? ZERO_ONE_AXIS.xy : ZERO_ONE_AXIS.yx) - origin.z ) / direction.z;

  // We have to find now the the largest of the Near and the the smallest of the Far
  float nearDistance = (tNearFarPlaneX.x > tNearFarPlaneY.x) ? tNearFarPlaneX.x : tNearFarPlaneY.x;
  if (tNearFarPlaneZ.x > nearDistance)
    nearDistance = tNearFarPlaneZ.x;

  float farDistance = (tNearFarPlaneX.y < tNearFarPlaneY.y) ? tNearFarPlaneX.y : tNearFarPlaneY.y;
  if (tNearFarPlaneZ.y < farDistance)
    farDistance = tNearFarPlaneZ.y;

  return vec2(nearDistance, farDistance);
}

vec3 backStep(in vec3 position, in vec3 direction, in int numberOfSteps)
{
  vec4 color = ZERO_COLOR_INTENSITY;
  for (int iteration = 0; iteration <= numberOfSteps; iteration++) // has to add at least one iteration with <= check
  {
    position += direction;
    color = applyTextureFiltering(position);
    if (useVoxelLighting)
      color = applyLighting(position, color);

    #if GPU_VENDOR_NVIDIA
      if ( any( enableIsoSurfaceRGBA && lessThanEqual(color, isoSurfaceRGBA) ) ) // works only on NVidia hardware
    #else
      if ( any( equal( ivec4(enableIsoSurfaceRGBA) + ivec4( lessThanEqual(color, isoSurfaceRGBA) ), ivec4(2) ) ) ) // equivalent to above with a TWO_INTEGER_VECTOR
    #endif
      break;
  }

  return position;
}

vec4 forwardSampleStep(in vec3 position, in vec3 direction, in int numberOfSteps)
{
  vec4 color = ZERO_COLOR_INTENSITY;
  vec4 sumColor = ZERO_COLOR_INTENSITY;
  for (int iteration = 0; iteration <= numberOfSteps; iteration++) // has to add at least one iteration with <= check
  {
    color = applyTextureFiltering(position);
    if (useVoxelLighting)
      color = applyLighting(position, color);

    sumColor += transparencyFactor * transparencyLevel * (1.0 - sumColor.a) * color;
    position -= direction;
    if ( any( greaterThanEqual(sumColor, FULL_COLOR_INTENSITY) ) )
      break;
  }

  return sumColor;
}

void main()
{
  if (oldLCDStyleTransparency)
    applyOldStyleTransparency();

  vec2 texCoords = vec2(gl_FragCoord.xy / screenSize) - CENTER_POINT.xy; 
  vec3 direction = normalize(screenRatio * texCoords.x * cameraRight + texCoords.y * cameraUp + cameraDirection);
  vec3 directionStep = direction * forwardStepSize;
  #if GPU_VENDOR_NVIDIA
    if (useBackStepForRays)
      directionStep *= 3.0;
  #endif
  vec3 position = cameraPosition + CENTER_POINT;

  vec2 nearFarDistance = axisAlignedBoundingBox(position, directionStep);
  if (nearFarDistance.x >= nearFarDistance.y)
    discard;

  if (nearFarDistance.x > 0.0)
  {
    if (jitterStartingRays)
      position += (nearFarDistance.x * directionStep) + rand3(texCoords) * directionStep; // jitter starting ray to hide "wood grain" artifacts
    else
      position += (nearFarDistance.x * directionStep);
  }
  else
    nearFarDistance.x = 0.0;

  int numberOfSteps = int(nearFarDistance.y - nearFarDistance.x);
  vec3 smallBackStep = -direction * voxelStepSize;
  bvec4 disableIsoSurfaceRGBA = not(enableIsoSurfaceRGBA);
  bvec4 disableDensityRGBA = not(enableDensityRGBA);
  vec4 sumColor = ZERO_COLOR_INTENSITY;
  float sumAlpha = 0.0;
  vec4 color = ZERO_COLOR_INTENSITY;
  float alpha = 0.0;

  if (useBackStepForRays) // Michael Kargas's original backStep() implementation optimized & isosurfaces added by Thanos Theo
  {
    for (int iteration = 0; iteration <= numberOfSteps; iteration++) // has to add at least one iteration with <= check
    {
      color = applyTextureFiltering(position);

      #if GPU_VENDOR_NVIDIA
        if ( all( disableIsoSurfaceRGBA || greaterThan(color, isoSurfaceRGBA) ) ) // works only on NVidia hardware
      #else
        if ( all( greaterThan( ivec4(disableIsoSurfaceRGBA) + ivec4( greaterThan(color, isoSurfaceRGBA) ), ivec4(0) ) ) ) // equivalent to above with a ZERO_INTEGER_VECTOR
      #endif
      {
        #if GPU_VENDOR_NVIDIA
          position = backStep( position, smallBackStep, int(raymarcherSkipLength) );
          sumColor = forwardSampleStep(position, smallBackStep, numberOfSteps - iteration);
          if (voxelDensityRendering && useTransferFunction)
            sumColor = applyTransferFunction(transferFunctionType, sumColor.r, sumColor, transferFunctionBlendingType, transferFunctionBlendingValue);
        #else
          if (useVoxelLighting)
            color = applyLighting(position, color);
          if (voxelDensityRendering && useTransferFunction)
            color = applyTransferFunction(transferFunctionType, color.r, color, transferFunctionBlendingType, transferFunctionBlendingValue);

          sumColor.rgb += color.rgb;    
          sumColor.a = transparencyFactor * transparencyLevel;
        #endif

        break;
      }

      position += directionStep;

      if ( any( lessThanEqual(position, lowerBounds) ) || any( greaterThanEqual(position, upperBounds) ) )
        discard;
    }
  }
  else // normal raymarcher
  {
    int iteration = 0;
    if ( all(disableIsoSurfaceRGBA.rgb) ) // doesn't check the A isosurface because when none of the Iso Depth Channel is enabled, fail-safe mode applied on alpha channel
    {
      // original raymarcher without isosurfaces support
      while ( (iteration <= numberOfSteps) && (sumAlpha < 1.0) ) // has to add at least one iteration with <= check
      {
        color = applyTextureFiltering(position);
        position = solidityAdaptation(color, directionStep, position, disableDensityRGBA, densityRGBA);

        if (useVoxelLighting)
          color = applyLighting(position, color);        
        if (voxelDensityRendering && useTransferFunction)
          color = applyTransferFunction(transferFunctionType, color.r, color, transferFunctionBlendingType, transferFunctionBlendingValue);

        alpha = 12.0 * transparencyFactor * transparencyLevel * color.a * forwardStepSize;
        sumColor += (1.0 - sumAlpha) * color * alpha * 3.0 * transparencyFactor;
        sumAlpha += alpha;

        position += directionStep;
        iteration++;
      }
    }
    else
    {
      // adapted raymarcher with isosurfaces support
      while ( (iteration <= numberOfSteps) && (sumAlpha < 1.0) ) // has to add at least one iteration with <= check
      {
        color = applyTextureFiltering(position);
        position = solidityAdaptation(color, directionStep, position, disableDensityRGBA, densityRGBA);

        #if GPU_VENDOR_NVIDIA
          if ( all( disableIsoSurfaceRGBA || greaterThan(color, isoSurfaceRGBA) ) ) // works only on NVidia hardware
        #else
          if ( all( greaterThan( ivec4(disableIsoSurfaceRGBA) + ivec4( greaterThan(color, isoSurfaceRGBA) ), ivec4(0) ) ) ) // equivalent to above with a ZERO_INTEGER_VECTOR
        #endif
        {
          if (useVoxelLighting)
            color = applyLighting(position, color);
          if (voxelDensityRendering && useTransferFunction)
            color = applyTransferFunction(transferFunctionType, color.r, color, transferFunctionBlendingType, transferFunctionBlendingValue);                

          alpha = 12.0 * transparencyFactor * transparencyLevel * color.a * forwardStepSize;    
          sumColor += (1.0 - sumAlpha) * color * alpha * 3.0 * transparencyFactor;
          sumAlpha += alpha;
        }

        position += directionStep;
        iteration++;
      }
    }
  }

  if ( all ( equal(sumColor, ZERO_COLOR_INTENSITY) ) )
    discard;

  sumColor = clamp(sumColor, 0.0, 1.0);
  if (!voxelDensityRendering && useTransferFunction)
    sumColor = applyTransferFunction(transferFunctionType, position.z, sumColor, transferFunctionBlendingType, transferFunctionBlendingValue); // can we do it in the eyeview direction so as to create a proper ChromaDepth 3D illusion?
  if (useImageProcessing)
    sumColor = applyImageProcessing(imageProcessingType, sumColor, imageProcessingBlendingType, imageProcessingBlendingValue);
  gl_FragColor = sumColor;
}