/*

 @ author, GLSL & OpenGL code author Athanasios Theocharidis, 2012-2015

*/

#if GPU_SHADER4_COMPATIBILITY_CONDITION
    #extension GL_EXT_gpu_shader4: enable
#endif

#if GPU_SHADER_FP64_COMPATIBILITY_CONDITION
    #extension GL_ARB_gpu_shader_fp64: enable
#endif

FS_VARYING vec4 fsColor;

uniform sampler3D volume3DTexture;
uniform sampler2D cubeFBO2DTexture;

uniform bool oldLCDStyleTransparency;

uniform vec2 screenSize;
uniform vec3 voxelDimension;
uniform bool voxelDensityRendering;
uniform float transparencyLevel;
uniform float transparencyFactor;
uniform bool useVoxelLighting;
uniform bool jitterStartingRays;
uniform float raymarcherSkipLength;

uniform bvec4 enableIsoSurfaceRGBA;
uniform vec4 isoSurfaceRGBA;
uniform bvec4 enableDensityRGBA;
uniform vec4 densityRGBA;

uniform bool useTransferFunction;
uniform int transferFunctionType;
uniform int transferFunctionBlendingType;
uniform float transferFunctionBlendingValue;

uniform bool useImageProcessing = false;
uniform int imageProcessingType = 1;
uniform int imageProcessingBlendingType = 1;
uniform float imageProcessingBlendingValue = 1.0;

const vec4 FULL_COLOR_INTENSITY = vec4(1.0);
const vec4 ZERO_COLOR_INTENSITY = vec4(0.0);

float voxelStepSize = 1.0 / voxelDimension.z;
float forwardStepSize = 0.4 * raymarcherSkipLength * voxelStepSize;

// extra look-up/colorization related functions
vec3 rand3(in vec2 seed);
void applyOldStyleTransparency();
vec4 applyTextureFiltering(in vec3 texCoords);
vec4 applyLighting(in vec3, in vec4);
vec3 solidityAdaptation(in vec4, in vec3, in vec3, in bvec4, in vec4);
vec4 applyTransferFunction(in int, in float, in vec4, in int, in float);
vec4 applyImageProcessing(in int, in vec4, in int, in float);

void main()
{
  if (oldLCDStyleTransparency)
    applyOldStyleTransparency();

  vec2 texCoords = vec2(gl_FragCoord.xy / screenSize); // find the right place to lookup in the backside buffer for the back (end) position of ray
  vec4 startRayPosition = texture2D(cubeFBO2DTexture, texCoords);
  vec4 endRayPosition = fsColor; // the endRayPosition position of the ray is stored in the current cube color
  vec3 direction = vec3(endRayPosition - startRayPosition);
  float directionLength = length(direction); // the length from front to back is calculated and used to terminate the ray
  vec3 normalizedDirection = normalize(direction);
  vec3 deltaDirection = normalizedDirection * forwardStepSize;
  float deltaDirectionLength = length(deltaDirection);
  vec3 position = (jitterStartingRays) ? startRayPosition.xyz + rand3(texCoords.xy) * deltaDirection : startRayPosition.xyz; // jitter starting ray to hide "wood grain" artifacts

  vec4 sumColor = ZERO_COLOR_INTENSITY;
  float sumAlpha = 0.0;
  float sumLength = 0.0;
  vec4 color = ZERO_COLOR_INTENSITY;
  float alpha = 0.0;
  bvec4 disableIsoSurfaceRGBA = not(enableIsoSurfaceRGBA);
  bvec4 disableDensityRGBA = not(enableDensityRGBA);

  if ( all(disableIsoSurfaceRGBA.rgb) ) // doesn't check the A isosurface because when none of the Iso Depth Channel is enabled, fail-safe mode applied on alpha channel
  {
    // original raymarcher without isosurfaces support
    while ( (sumLength < directionLength) && (sumAlpha < 1.0) ) // terminate if opacity > 1 or the ray is outside the volume
    {
      color = applyTextureFiltering(position);
      position = solidityAdaptation(color, deltaDirection, position, disableDensityRGBA, densityRGBA);

      if (useVoxelLighting)
        color = applyLighting(position, color);
      if (voxelDensityRendering && useTransferFunction)
        color = applyTransferFunction(transferFunctionType, color.r, color, transferFunctionBlendingType, transferFunctionBlendingValue);

      alpha = 12.0 * transparencyFactor * transparencyLevel * color.a * forwardStepSize;
      sumColor += (1.0 - sumAlpha) * color * alpha * 3.0 * transparencyFactor;
      sumAlpha += alpha;

      position += deltaDirection;
      sumLength += deltaDirectionLength;
    }
  }
  else
  {
    // adapted raymarcher with isosurfaces support
    while ( (sumLength < directionLength) && (sumAlpha < 1.0) ) // terminate if opacity > 1 or the ray is outside the volume
    {
      color = applyTextureFiltering(position);
      position = solidityAdaptation(color, deltaDirection, position, disableDensityRGBA, densityRGBA);

      #if GPU_VENDOR_NVIDIA
        if ( all( disableIsoSurfaceRGBA || greaterThan(color, isoSurfaceRGBA) ) ) // works only on NVidia hardware
      #else
        if ( all( greaterThan( ivec4(disableIsoSurfaceRGBA) + ivec4( greaterThan(color, isoSurfaceRGBA) ), ivec4(0) ) ) ) // equivalent to above with a ZERO_INTEGER_VECTOR
      #endif
      {
        if (useVoxelLighting)
          color = applyLighting(position, color);
        if (voxelDensityRendering && useTransferFunction)
          color = applyTransferFunction(transferFunctionType, color.r, color, transferFunctionBlendingType, transferFunctionBlendingValue);

        alpha = 12.0 * transparencyFactor * transparencyLevel * color.a * forwardStepSize;
        sumColor += (1.0 - sumAlpha) * color * alpha * 3.0 * transparencyFactor;
        sumAlpha += alpha;
      }

      position += deltaDirection;
      sumLength += deltaDirectionLength;
    }
  }

  if ( all ( equal(sumColor, ZERO_COLOR_INTENSITY) ) )
    discard;

  sumColor = clamp(sumColor, 0.0, 1.0);
  if (!voxelDensityRendering && useTransferFunction)
    sumColor = applyTransferFunction(transferFunctionType, position.z, sumColor, transferFunctionBlendingType, transferFunctionBlendingValue); // can we do it in the eyeview direction so as to create a proper ChromaDepth 3D illusion?
  if (useImageProcessing)
    sumColor = applyImageProcessing(imageProcessingType, sumColor, imageProcessingBlendingType, imageProcessingBlendingValue);
  gl_FragColor = sumColor;
}