/*

 @ author, GLSL & OpenGL code author Athanasios Theocharidis, 2009-2015

*/

#if GPU_SHADER4_COMPATIBILITY_CONDITION
    #extension GL_EXT_gpu_shader4: enable
#endif

#if GPU_SHADER_FP64_COMPATIBILITY_CONDITION
    #extension GL_ARB_gpu_shader_fp64: enable
#endif

uniform sampler3D volume3DTexture;
uniform int textureFilteringMode = 1;
uniform int illuminationModel = 2;

// Machine float epsilon
const float FLT_EPSILON = 0.000001;

const float DELTA    = 0.01;
const float AMBIENT  = 0.10;
const float DIFFUSE  = 0.65;
const float SPECULAR = 0.35;
const float SPECULAR_FACTOR  = 16.0; // used for specular calculations
const float SHININESS_FACTOR = 30.0; // used for specular calculations
const float COOK_TORRENCE_MEAN   = 0.7; // used for the Cook-Torrence lighting model
const float COOK_TORRENCE_SCALE  = 0.2; // used for the Cook-Torrence lighting model
const vec3  LIGHT_VECTOR = vec3(0.0, -1.0, 1.0);

const float PI = 3.14159265;
const vec3 WEIGHT_LUMINANCE_VECTOR = vec3(0.2125, 0.7154, 0.0721);

const vec4 FULL_COLOR_INTENSITY = vec4(1.0);
const vec4 HALF_COLOR_INTENSITY = vec4(0.5);
const vec4 ZERO_COLOR_INTENSITY = vec4(0.0);

const float BORDER_PIXEL_FACTOR     = 0.05; // 5% of the minimum pixel dimension
const float MAX_BORDER_PIXEL_FACTOR = 0.50; // up to 50% of the minimum pixel dimension should be allowed

vec2 randSeed = vec2(0.0);
void initRand(in vec2 seed);
float rand1();
float rand1(in vec2 seed);
vec2 rand2();
vec2 rand2(in vec2 seed);
vec3 rand3();
vec3 rand3(in vec2 seed);
vec3 cosHemiRandom();
vec3 cosHemiRandom(in vec2 seed);
vec3 cosPowHemiRandom(float k);
vec3 cosPowHemiRandom(in vec2 seed, float k);
float smootherstep(in float edge0, in float edge1, in float x);
vec2 smootherstep(in vec2 edge0, in vec2 edge1, in vec2 x);
vec3 smootherstep(in vec3 edge0, in vec3 edge1, in vec3 x);
vec4 smootherstep(in vec4 edge0, in vec4 edge1, in vec4 x);
void applyOldStyleTransparency();
vec4 smootherTextureFiltering(in vec3 texCoords);
vec4 tricubicTextureFiltering(in vec3 texCoords);
vec4 applyTextureFiltering(in vec3 texCoords);
mat3 calculateAxisNormals(in vec3);
float applyHalfVectorNxH(in vec3 E, in vec3 N, in vec3 L);
float applyGaussianSpecularModel(in vec3 E, in vec3 N, in vec3 L, in float shininess, in float factor);
vec4 applyLambertianIlluminationModel(in vec3 position, in vec4 color);
vec4 applyGaussianIlluminationModel(in vec3 position, in vec4 color);
vec4 applyCookTorranceIlluminationModel(in vec3 position, in vec4 color);
vec4 applyLighting(in vec3, in vec4);
vec3 solidityAdaptation(in vec4, in vec3, in vec3, in bvec4, in vec4);
vec3 rainbowTransferFunction(in float);
vec3 heatedObject(in float);
vec3 chromaDepth1TransferFunction(in float);
vec3 chromaDepth2TransferFunction(in float);
vec3 applyInterpolation(in int, in vec3, in vec3, in float, in float);
vec4 applySmoothBorderEdges(in vec2, in vec4, in int);
vec4 applyTransferFunction(in int, in float, in vec4, in int, in float);
vec4 applyImageProcessing(in int, in vec4, in int, in float);

void initRand(in vec2 seed)
{
  randSeed = seed;
}

// This function returns uniformly distributed values in the range [0, 1]
float rand1()
{
  float dotProduct = dot(randSeed, vec2(12.9898,78.233));
  float x = fract(sin(dotProduct) * 43758.5453);
  randSeed.x = x;
  return x;
}

// This function returns uniformly distributed values in the range [0, 1]
float rand1(in vec2 seed)
{
  float dotProduct = dot(seed, vec2(12.9898,78.233));
  float x = fract(sin(dotProduct) * 43758.5453);
  return x;
}

// This function returns uniformly distributed values in the range [0, 1]
vec2 rand2()
{
  float dotProduct = dot(randSeed, vec2(12.9898,78.233));
  vec2 r = vec2(fract(sin(dotProduct ) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453));
  randSeed = r;
  return r;
}

// This function returns uniformly distributed values in the range [0, 1]
vec2 rand2(in vec2 seed)
{
  float dotProduct = dot(seed, vec2(12.9898,78.233));
  vec2 r = vec2(fract(sin(dotProduct ) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453));
  return r;
}

// This function returns uniformly distributed values in the range [0, 1]
vec3 rand3()
{
  float dotProduct = dot(randSeed, vec2(12.9898,78.233));
  vec3 r = vec3(fract(sin(dotProduct ) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453), fract(sin(3.0 * dotProduct) * 43758.5453));
  randSeed = r.xy;
  return r;
}

// This function returns uniformly distributed values in the range [0, 1]
vec3 rand3(in vec2 seed)
{
  float dotProduct = dot(seed, vec2(12.9898,78.233));
  vec3 r = vec3(fract(sin(dotProduct ) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453), fract(sin(3.0 * dotProduct) * 43758.5453));
  return r;
}

// Returns 3D unit vectors distributed according to a
// cosine distribution around the Z axis.
vec3 cosHemiRandom()
{
  vec2 e = rand2();

  // Jensen's method
  float sinTheta = sqrt(1.0 - e.x);
  float cosTheta = sqrt(e.x);
  float phi = 6.28318531 * e.y;

  // We could also use Malley's method (pbrt p.657), since they have the same cost:
  // r = sqrt(e.x);
  // t = 2*pi*e.y;
  // x = cos(t)*r;
  // y = sin(t)*r;
  // z = sqrt(1.0 - x*x + y*y);

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

// Returns 3D unit vectors distributed according to a
// cosine distribution around the Z axis.
vec3 cosHemiRandom(in vec2 seed)
{
  vec2 e = rand2(seed);

  // Jensen's method
  float sinTheta = sqrt(1.0 - e.x);
  float cosTheta = sqrt(e.x);
  float phi = 6.28318531 * e.y;

  // We could also use Malley's method (pbrt p.657), since they have the same cost:
  // r = sqrt(e.x);
  // t = 2*pi*e.y;
  // x = cos(t)*r;
  // y = sin(t)*r;
  // z = sqrt(1.0 - x*x + y*y);

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

// Returns 3D unit vectors distributed according to a cosine
// power distribution (pow(cos(theta), k) around the Z axis.
// The method comes from the G3D engine (http://g3d.sourceforge.net)
// The returned vector is closer to vec3(0.0, 0.0, 1.0) as k becomes big.
vec3 cosPowHemiRandom(float k)
{
  vec2 e = rand2();

  float cosTheta = pow(e.x, 1.0 / (k + 1.0));
  float sinTheta = sqrt(1.0 - cosTheta * cosTheta);
  float phi = 6.28318531 * e.y;

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

// Returns 3D unit vectors distributed according to a cosine
// power distribution (pow(cos(theta), k) around the Z axis.
// The method comes from the G3D engine (http://g3d.sourceforge.net)
// The returned vector is closer to vec3(0.0, 0.0, 1.0) as k becomes big.
vec3 cosPowHemiRandom(in vec2 seed, float k)
{
  vec2 e = rand2(seed);

  float cosTheta = pow(e.x, 1.0 / (k + 1.0));
  float sinTheta = sqrt(1.0 - cosTheta * cosTheta);
  float phi = 6.28318531 * e.y;

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

// Scale, and clamp x to 0...1 (first line) range & evaluate polynomial (second line)
#define smootherstepDefinition(edge0, edge1, x)               \
        x = clamp( (x - edge0) / (edge1 - edge0), 0.0, 1.0 ); \
        return x * x * x * (x * (x * 6.0 - 15.0) + 10.0);

// Prof. Ken Perlin suggests an improved version of the smoothstep function which has zero 1st and 2nd order derivatives at t=0 and t=1
float smootherstep(in float edge0, in float edge1, in float x)
{
  smootherstepDefinition(edge0, edge1, x);
}

// vec2 variaty of the smootherstep() function
vec2 smootherstep(in vec2 edge0, in vec2 edge1, in vec2 x)
{
  smootherstepDefinition(edge0, edge1, x);
}

// vec3 variaty of the smootherstep() function
vec3 smootherstep(in vec3 edge0, in vec3 edge1, in vec3 x)
{
  smootherstepDefinition(edge0, edge1, x);
}

// vec4 variaty of the smootherstep() function
vec4 smootherstep(in vec4 edge0, in vec4 edge1, in vec4 x)
{
  smootherstepDefinition(edge0, edge1, x);
}

void applyOldStyleTransparency()
{
  if ( mod(gl_FragCoord.x, 2.0) == mod(gl_FragCoord.y, 2.0) )
    discard; // screen-door transparency effect
}

vec4 smootherTextureFiltering(in vec3 texCoords)
{
  vec3 textureDimensions = textureSize(volume3DTexture, 0); // GL3.0+ function  
  vec3 texCoordsScaled   = texCoords * textureDimensions + 0.5;
  vec3 texCoordsInt      = floor(texCoordsScaled);
  vec3 texCoordsFrac     = smootherstep(vec3(0.0), vec3(1.0), fract(texCoordsScaled));
  vec3 smootherTexCoords = (texCoordsInt + texCoordsFrac - 0.5) / textureDimensions;

  return texture3D(volume3DTexture, smootherTexCoords);
}

vec4 tricubicTextureFiltering(in vec3 texCoords)
{
  vec3 textureDimensions = vec3(textureSize3D(volume3DTexture, 0));
  texCoords = texCoords * textureDimensions - 0.5; // shift the coordinate from [0,1] to [-0.5, textureDimensions - 0.5]

  vec3 index = floor(texCoords);
  vec3 fraction = texCoords - index;
  vec3 oneMinusFrac = 1.0 - fraction;

  vec3 w0 = 1.0 / 6.0 * oneMinusFrac * oneMinusFrac * oneMinusFrac;
  vec3 w1 = 2.0 / 3.0 - 0.5 * fraction * fraction * (2.0 - fraction);
  vec3 w2 = 2.0 / 3.0 - 0.5 * oneMinusFrac * oneMinusFrac * (2.0 - oneMinusFrac);
  vec3 w3 = 1.0 / 6.0 * fraction * fraction * fraction;

  vec3 g0 = w0 + w1;
  vec3 g1 = w2 + w3;
  vec3 mult = 1.0 / textureDimensions;
  vec3 h0 = mult * ((w1 / g0) - 0.5 + index);  // h0 = w1 / g0 - 1.0, move from [-0.5, textureDimensions - 0.5] to [0,1]
  vec3 h1 = mult * ((w3 / g1) + 1.5 + index);  // h1 = w3 / g1 + 1.0, move from [-0.5, textureDimensions - 0.5] to [0,1]

  // fetch the eight linear interpolations
  // weighting and fetching is interleaved for performance and stability reasons
  vec4 tex000 = texture3D(volume3DTexture, h0);
  vec4 tex100 = texture3D(volume3DTexture, vec3(h1.x, h0.y, h0.z));
  tex000 = mix(tex100, tex000, g0.x);  // weigh along the x-direction
  vec4 tex010 = texture3D(volume3DTexture, vec3(h0.x, h1.y, h0.z));
  vec4 tex110 = texture3D(volume3DTexture, vec3(h1.x, h1.y, h0.z));
  tex010 = mix(tex110, tex010, g0.x);  // weigh along the x-direction
  tex000 = mix(tex010, tex000, g0.y);  // weigh along the y-direction
  vec4 tex001 = texture3D(volume3DTexture, vec3(h0.x, h0.y, h1.z));
  vec4 tex101 = texture3D(volume3DTexture, vec3(h1.x, h0.y, h1.z));
  tex001 = mix(tex101, tex001, g0.x);  // weigh along the x-direction
  vec4 tex011 = texture3D(volume3DTexture, vec3(h0.x, h1.y, h1.z));
  vec4 tex111 = texture3D(volume3DTexture, h1);
  tex011 = mix(tex111, tex011, g0.x);  // weigh along the x-direction
  tex001 = mix(tex011, tex001, g0.y);  // weigh along the y-direction

  return mix(tex001, tex000, g0.z);  // weigh along the z-direction
}

vec4 applyTextureFiltering(in vec3 texCoords)
{
  if (textureFilteringMode == 1) // (standard hardware) trilinear texture filtering
  {
    return texture3D(volume3DTexture, texCoords);
  }
  else if (textureFilteringMode == 2) // smoother texture filtering
  {
    return smootherTextureFiltering(texCoords);
  }
  else if (textureFilteringMode == 3) // tricubic texture filtering
  {
    return tricubicTextureFiltering(texCoords);
  }  
}

mat3 calculateAxisNormals(in vec3 position)
{
  // vec4 normalX = applyTextureFiltering(position + DELTA_S) - applyTextureFiltering(position - DELTA_S);
  // vec4 normalY = applyTextureFiltering(position + DELTA_T) - applyTextureFiltering(position - DELTA_T);
  // vec4 normalZ = applyTextureFiltering(position + DELTA_P) - applyTextureFiltering(position - DELTA_P);

  // vec3 normalR = normalize( vec3(normalX.r, normalY.r, normalZ.r) );
  // vec3 normalG = normalize( vec3(normalX.g, normalY.g, normalZ.g) );
  // vec3 normalB = normalize( vec3(normalX.b, normalY.b, normalZ.b) );
  // vec3 normalA = normalize( vec3(normalX.a, normalY.a, normalZ.a) );

  // below re-written the above with a mat3 for execution efficiency
  vec3 deltaPosition = DELTA * position;
  vec3 deltaS = vec3(deltaPosition.x, 0.0,             0.0);
  vec3 deltaT = vec3(0.0,             deltaPosition.y, 0.0);
  vec3 deltaP = vec3(0.0,             0.0,             deltaPosition.z);
  mat3 matrix = mat3(applyTextureFiltering(position + deltaS).rgb - applyTextureFiltering(position - deltaS).rgb,
                     applyTextureFiltering(position + deltaT).rgb - applyTextureFiltering(position - deltaT).rgb,
                     applyTextureFiltering(position + deltaP).rgb - applyTextureFiltering(position - deltaP).rgb);

  return transpose(matrix); // transpose mat3 to fast emulate the normal RGB gathering above
}

float applyHalfVectorNxH(in vec3 E, in vec3 N, in vec3 L)
{
  vec3 H = normalize(L + E);

  return max(dot(N, H), FLT_EPSILON); // failsafe GLSL Shader value for specular light computation
}

float applyGaussianSpecularModel(in vec3 E, in vec3 N, in vec3 L, in float shininess, in float factor)
{
  float NxH = applyHalfVectorNxH(E, N, L);
  float angleH = acos(clamp(NxH, -1.0 + FLT_EPSILON, 1.0 - FLT_EPSILON)); // failsafe GLSL Shader value for specular light computation
  float exponent = angleH * shininess / factor;

  return exp(-exponent * exponent);
}

vec4 applyLambertianIlluminationModel(in vec3 position, in vec4 color)
{
  mat3 axisNormals = calculateAxisNormals(position);
  vec3 axisX_N = normalize(axisNormals[0]);
  vec3 axisY_N = normalize(axisNormals[1]);
  vec3 axisZ_N = normalize(axisNormals[2]);
  vec3 L = normalize(LIGHT_VECTOR - position); // -position.xyz, positional rather than directional light 
  vec3 NDotLTerms = max(vec3(dot(axisX_N, L),
                             dot(axisY_N, L),
                             dot(axisZ_N, L)), vec3(FLT_EPSILON)); // failsafe GLSL Shader value for diffuse light computations

  return clamp(AMBIENT + DIFFUSE * vec4(NDotLTerms, color.a) * color, 0.0, 1.0);
}

vec4 applyGaussianIlluminationModel(in vec3 position, in vec4 color)
{
  mat3 axisNormals = calculateAxisNormals(position);
  vec3 axisX_N = normalize(axisNormals[0]);
  vec3 axisY_N = normalize(axisNormals[1]);
  vec3 axisZ_N = normalize(axisNormals[2]);
  vec3 L = normalize(LIGHT_VECTOR - position); // -position.xyz, positional rather than directional light
  vec3 E = normalize(-position);
  vec3 NDotLTerms = max(vec3(dot(axisX_N, L),
                             dot(axisY_N, L),
                             dot(axisZ_N, L)), vec3(FLT_EPSILON)); // failsafe GLSL Shader value for diffuse light computations
  vec3 specularPowers = vec3(applyGaussianSpecularModel(E, axisX_N, L, SHININESS_FACTOR, SPECULAR_FACTOR),
                             applyGaussianSpecularModel(E, axisY_N, L, SHININESS_FACTOR, SPECULAR_FACTOR),
                             applyGaussianSpecularModel(E, axisZ_N, L, SHININESS_FACTOR, SPECULAR_FACTOR));

  return clamp(AMBIENT + DIFFUSE * vec4(NDotLTerms, color.a) * color + SPECULAR * vec4(specularPowers, color.a) * color, 0.0, 1.0);
}

vec4 applyCookTorranceIlluminationModel(in vec3 position, in vec4 color)
{
  mat3 axisNormals = calculateAxisNormals(position);
  vec3 axisX_N = normalize(axisNormals[0]);
  vec3 axisY_N = normalize(axisNormals[1]);
  vec3 axisZ_N = normalize(axisNormals[2]);
  vec3 L = normalize(LIGHT_VECTOR - position); // -position.xyz, positional rather than directional light
  vec3 E = normalize(-position);
  vec3 H = normalize(L + E);
  float EDotH = dot(E, H);
  vec3 NDotHTerms = vec3(dot(axisX_N, H),
                         dot(axisY_N, H),
                         dot(axisZ_N, H));
  vec3 NDotETerms = vec3(dot(axisX_N, E),
                         dot(axisY_N, E),
                         dot(axisZ_N, E));                         
  vec3 NDotLTerms = max(vec3(dot(axisX_N, L),
                             dot(axisY_N, L),
                             dot(axisZ_N, L)), vec3(FLT_EPSILON)); // failsafe GLSL Shader value for diffuse light computations
  // approximate Fresnel term
  float t1 = 1.0 + EDotH;
  float t1_2 = t1 * t1;
  float fresnelTerm = t1_2 * t1_2; // faster equivalent of: pow(1.0 + EDotH, 4.0)
  // approximate microfacet distribution
  vec3 deltas = acos(clamp(NDotHTerms, -1.0 + FLT_EPSILON, 1.0 - FLT_EPSILON)); // failsafe GLSL Shader value for specular light computation
  vec3 exponents = -pow(deltas / COOK_TORRENCE_MEAN, vec3(2.0));
  vec3 microfacets = COOK_TORRENCE_SCALE * exp(exponents);
  // calculate self-shadowing terms
  vec3 terms1 = 2.0 * NDotHTerms * NDotETerms / EDotH;
  vec3 terms2 = 2.0 * NDotHTerms * NDotLTerms / EDotH;
  vec3 selfShadow = min(vec3(1.0), min(terms1, terms2));
  // calculate Cook-Torrance specular term
  vec3 specularPowers = fresnelTerm * microfacets * selfShadow / NDotETerms;
  
  return clamp(DIFFUSE * vec4(NDotLTerms, color.a) * color + SPECULAR * vec4(specularPowers, color.a) * color, 0.0, 1.0);
}

vec4 applyLighting(in vec3 position, in vec4 color)
{
  if (color.a > FLT_EPSILON)
  {
    if (illuminationModel == 1) // Lambertian Lighting Model
    {
      return applyLambertianIlluminationModel(position, color);
    }
    else if (illuminationModel == 2) // Gaussian Lighting Model
    {
      return applyGaussianIlluminationModel(position, color);
    }
    else if (illuminationModel == 3) // Cook-Torrance Physically-Based Microfacet Lighting Model
    {
      return applyCookTorranceIlluminationModel(position, color);
    }
  }
  else
  {
    return color;
  }
}

vec3 solidityAdaptation(in vec4 color, in vec3 directionStep, in vec3 position, in bvec4 disableDensityRGBA, in vec4 densityRGBA)
{
  #if GPU_VENDOR_NVIDIA
    if ( all( disableDensityRGBA || greaterThan(color, 1.0 - densityRGBA) ) ) // works only on NVidia hardware
  #else
    if ( all( greaterThan( ivec4(disableDensityRGBA) + ivec4( greaterThan(color, densityRGBA) ), ivec4(0) ) ) ) // equivalent to above with a ZERO_INTEGER_VECTOR
  #endif
  {
    vec3 delta = -directionStep * 0.5;
    position += delta; 
    color = applyTextureFiltering(position);
    for (int steps = 0; steps < 2; steps++)
    {
      if ( any( greaterThan(color, 1.0 - densityRGBA) ) )
        delta *=  0.5; 
      else 
        delta *= -0.5;
      position += delta;
      color = applyTextureFiltering(position);
    }
  }

  return position;
}

vec3 rainbowTransferFunction(in float depth)
{
  if (depth < 0.0) // black if below bounds
  {
    return vec3(0.0);
  }
  else if ( (depth >= 0.0) && (depth < 0.2) ) // purple to blue ramp
  {
    vec3 color;
    color.r = 0.5 * (1.0 - depth / 0.2);
    color.g = 0.0;
    color.b = 0.5 + (0.5 * depth / 0.2);

    return clamp(color, 0.0, 1.0);
  }
  else if ( (depth >= 0.2) && (depth < 0.4) ) // blue to cyan ramp
  {
    vec3 color;
    color.r = 0.0;
    color.g = (depth - 0.2) * 5.0;
    color.b = 1.0;

    return clamp(color, 0.0, 1.0);
  }
  else if ( (depth >= 0.4) && (depth < 0.6) ) // cyan to green ramp
  {
    vec3 color;
    color.r = 0.0;
    color.g = 1.0;
    color.b = (0.6 - depth) * 5.0;
    
    return clamp(color, 0.0, 1.0);
  }
  else if ( (depth >= 0.6) && (depth < 0.8) ) // green to yellow ramp
  {
    vec3 color;
    color.r = (depth - 0.6) * 5.0;
    color.g = 1.0;
    color.b = 0.0;

    return clamp(color, 0.0, 1.0);
  }
  else if ( (depth >= 0.8) && (depth <= 1.0) ) // yellow to red ramp
  {
    vec3 color;
    color.r = 1.0;
    color.g = (1.0 - depth) * 5.0;
    color.b = 0.0;

    return clamp(color, 0.0, 1.0);
  }   
  else // if (depth > 1.0) // white if above bound
  {
    return vec3(1.0);
  }
}

vec3 heatedObject(in float depth)
{
  vec3 rgb = vec3(0.0);
  rgb.r = 3. * ( depth - (0.0 / 6.0) );

  if( depth >= (1.0 / 3.0) )
  {
    rgb.r = 1.0;
    rgb.g = 3.0 * ( depth - (1.0 / 3.0) );
  }

  if( depth >= (2.0 / 3.0) )
  {
    rgb.g = 1.0;
    rgb.b = 3.0 * ( depth - (2.0 / 3.0) );
  }

  return clamp(rgb, 0.0, 1.0);
}

vec3 chromaDepth1TransferFunction(in float depth)
{
  float r = 1.0;
  float g = 0.0;
  float b = 1.0 - 6.0 * ( depth - (5.0 / 6.0) );
  
  if ( depth <= (5.0 / 6.0) )
  {
    r = 6.0 * ( depth - (4.0 / 6.0 ) );
    g = 0.0;
    b = 1.0;
  }

  if ( depth <= (4.0 / 6.0) )
  {
    r = 0.0;
    g = 1.0 - 6.0 * ( depth - (3.0 / 6.0 ) );
    b = 1.0;
  }

  if ( depth <= (3.0 / 6.0) )
  {
    r = 0.0;
    g = 1.0;
    b = 6.0 * ( depth - (2.0 / 6.0 ) );
  }

  if ( depth <= (2.0 / 6.0) )
  {
    r = 1.0 - 6.0 * ( depth - (1.0 / 6.0 ) );
    g = 1.0;
    b = 0.0;
  }

  if ( depth <= (1.0 / 6.0) )
  {
    r = 1.0;
    g = 6.0 * depth;
  }

  return clamp(vec3(r, g, b), 0.0, 1.0);
}

vec3 chromaDepth2TransferFunction(in float depth)
{
  // these formulas are based on code from American Paper Optics at:
  // http://www.chromatek.com/Image_Design/Color_Lookup_Functions/color_lookup_functions.shtml */
  float depth2 = depth * depth;
  vec3 rgb = vec3(0.0);
  if (depth < 0.5) 
  {
    rgb.g = 1.6 * depth2 + 1.2 * depth;
  } 
  else 
  {
    rgb.g = 3.2 * depth2 - 6.8 * depth + 3.6;
    rgb.b = depth2 * -4.8 + 9.2 * depth - 3.4;
  }
  depth  = depth  / 0.9;
  depth2 = depth2 / 0.81;    
  rgb.r = -2.14 * depth2 * depth2 -1.07 * depth2 * depth + 0.133 * depth2 + 0.0667 * depth + 1.0;

  return clamp(rgb, 0.0, 1.0);
}

vec3 applyInterpolation(in int interpolationType, in vec3 color1, in vec3 color2, in float interpolationValue, in float intensity)
{
  if (interpolationType == 1) // Linear interpolation
    return mix(color1, color2, intensity * interpolationValue);
  // else if (interpolationType == 2) // Cosine interpolation
  // {
  //   float s = 0.5 * ( 1.0 - cos(PI * interpolationValue) );
  //   return vec3( color1 + s * (color2 - color1) );
  // }
  else if (interpolationType == 2) // Cubic interpolation
    return mix( color1, color2, intensity * smoothstep(0.0, 1.0, interpolationValue) );
  else // if (interpolationType == 3) // Quintic interpolation
    return mix( color1, color2, intensity * smootherstep(0.0, 1.0, interpolationValue) );
}

vec4 applySmoothBorderEdges(in vec2 screenSize, in vec4 color, in int smoothBorderEdgesBlendingType)
{
  float borderPixelSize       = min(MAX_BORDER_PIXEL_FACTOR, BORDER_PIXEL_FACTOR) * min(screenSize.x, screenSize.y);
  bvec2  lowPixelBorderBounds =    lessThanEqual(gl_FragCoord.xy,              vec2(borderPixelSize));
  bvec2 highPixelBorderBounds = greaterThanEqual(gl_FragCoord.xy, screenSize - vec2(borderPixelSize));
  if (any(lowPixelBorderBounds) || any(highPixelBorderBounds))
  {
    float pixelFactor = 0.0;
    // Left border
    if (lowPixelBorderBounds.x && !lowPixelBorderBounds.y && !highPixelBorderBounds.x && !highPixelBorderBounds.y)
    {
      pixelFactor = gl_FragCoord.x / borderPixelSize;
    }
    // Bottom border
    else if (lowPixelBorderBounds.y && !lowPixelBorderBounds.x && !highPixelBorderBounds.x && !highPixelBorderBounds.y)
    {
      pixelFactor = gl_FragCoord.y / borderPixelSize;
    }
    // Top-Left corner
    else if (lowPixelBorderBounds.x && highPixelBorderBounds.y)
    {
      pixelFactor = 1.0 - distance(gl_FragCoord.xy, vec2(0.0, screenSize.y) - vec2(-borderPixelSize, borderPixelSize)) / borderPixelSize;
    }
    // Bottom-Left corner
    else if (lowPixelBorderBounds.x && lowPixelBorderBounds.y)
    {
      pixelFactor = 1.0 - distance(gl_FragCoord.xy, vec2(borderPixelSize)) / borderPixelSize;
    }
    // Right border
    else if (highPixelBorderBounds.x && !highPixelBorderBounds.y && !lowPixelBorderBounds.x && !lowPixelBorderBounds.y)
    {
      pixelFactor = (screenSize.x - gl_FragCoord.x) / borderPixelSize;
    }
    // Top border
    else if (highPixelBorderBounds.y && !highPixelBorderBounds.x && !lowPixelBorderBounds.x && !lowPixelBorderBounds.y)
    {
      pixelFactor = (screenSize.y - gl_FragCoord.y) / borderPixelSize;
    }
    // Top-Right corner
    else if (highPixelBorderBounds.x && highPixelBorderBounds.y)
    {
      pixelFactor = 1.0 - distance(gl_FragCoord.xy, screenSize - vec2(borderPixelSize)) / borderPixelSize;
    }
    // Bottom-Right corner
    else if (highPixelBorderBounds.x && lowPixelBorderBounds.y)
    {
      pixelFactor = 1.0 - distance(gl_FragCoord.xy, vec2(screenSize.x, 0.0) - vec2(borderPixelSize, -borderPixelSize)) / borderPixelSize;
    }

    color.rgb = applyInterpolation(smoothBorderEdgesBlendingType, color.rgb, ZERO_COLOR_INTENSITY.rgb, 1.0 - pixelFactor, 1.0);
  }

  return color;
}

vec4 applyTransferFunction(in int transferFunctionType, in float depth, in vec4 color, in int transferFunctionBlendingType, in float transferFunctionBlendingValue)
{
  depth = clamp(depth, 0.0, 1.0);

  vec3 transferFunctionColor;
  if (transferFunctionType == 1) // Rainbow transfer function
  {
    transferFunctionColor = vec3(1.0) - rainbowTransferFunction(depth);
  }
  else if (transferFunctionType == 2) // Heated Object transfer function
  {
    transferFunctionColor = heatedObject(depth);
  }
  else if (transferFunctionType == 3) // Chroma Depth 1 transfer function
  {
    transferFunctionColor = chromaDepth1TransferFunction(depth);
  }
  else // if (transferFunctionType == 4) // Chroma Depth 2 transfer function
  {
    transferFunctionColor = chromaDepth2TransferFunction(depth);
  }

  return vec4(applyInterpolation(transferFunctionBlendingType, color.rgb, transferFunctionColor, transferFunctionBlendingValue, 1.0), 1.0);
}  

vec4 applyImageProcessing(in int imageProcessingType, in vec4 color, in int imageProcessingBlendingType, in float imageProcessingBlendingValue)
{
  vec3 imageProcessingColor;
  if (imageProcessingType == 1) // Color Negative
  {
    imageProcessingColor = FULL_COLOR_INTENSITY.rgb - color.rgb;
  }
  else if (imageProcessingType == 2) // Brightness
  {
    imageProcessingColor = ZERO_COLOR_INTENSITY.rgb;
  }
  else if (imageProcessingType == 3) // Contrast
  {
    imageProcessingColor = HALF_COLOR_INTENSITY.rgb;
  }
  else // if (imageProcessingType == 4) // Saturation
  {
    float luminance = dot(color.rgb, WEIGHT_LUMINANCE_VECTOR);
    imageProcessingColor = vec3(luminance);
  }

  if (imageProcessingType == 1) // For Color Negative
    return vec4(applyInterpolation(imageProcessingBlendingType, color.rgb, imageProcessingColor, imageProcessingBlendingValue, 1.0), 1.0);
  else // all the rest of the image processing effects    
    return vec4(applyInterpolation(imageProcessingBlendingType, imageProcessingColor, color.rgb, imageProcessingBlendingValue, 5.0), 1.0);
}