/*

Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
Released Under a Simplified BSD (FreeBSD) License
for academic, personal & non-commercial use.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the author and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.

A Commercial License is also available for commercial use with
special restrictions and obligations at a one-off fee. See links at:
1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

#pragma once

#ifndef __ASTAR_H__
#define __ASTAR_H__

#include "ModuleDLL.h"
#include "FunctionView.h"
#include <cstddef>
#include <cstdint>
#include <memory>

/** @brief Namespace Utils contains utility classes with mainly static CPU related methods.
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/
namespace Utils
{
  /** @brief Namespace Pathfinding contains path finding algorithms like A*.
  * @author Thanos Theo, 2018
  * @version 14.0.0.0
  */
  namespace Pathfinding
  {
    /** @brief This class encapsulates usage of the A* algorithm.
    *
    *  This class encapsulates usage of the A* algorithm.\n
  *    Path finding algorithms  conceptual ideas were based on: http://qiao.github.io/PathFinding.js/visual/.\n
  *    Heuristics for grid maps conceptual ideas were based on: http://theory.stanford.edu/~amitp/GameProgramming/Heuristics.html.\n
  *    A* prototype C/C++ implementation was based on:          https://github.com/hjweide/a-star/.\n
    *
    * @author Thanos Theo, 2018
    * @version 14.0.0.0
    */
    class UTILS_MODULE_API Astar final
    {
    public:

      /**
      * Diagonal mode enumeration.
      */
      enum class DiagonalMode : std::size_t
      {
        NO_DIADONALS   = 0,
        WITH_DIADONALS = 1
      };

      /**
      * A function to set the A* start and goal positions.
      */
      void setStartAndGoal(std::size_t start, std::size_t goal);

      /**
      * A function for setting the state.
      */
      void setStrategy(const FunctionView<float(std::size_t)>& strategy) const;

      /**
      * An implementation of the A* path finding algorithm (not taking diagonals into account).
      */
      bool execute() const;

      /**
      * A function for unraveling and marking the optimal path after after the execution of the A* algorithm.
      */
      void unravelAndMarkOptimalPath(std::uint8_t* __restrict optimalPath, std::uint8_t markValue) const;

      /**
      * The non-default explicit constructor.
      */
      explicit Astar(std::size_t width, std::size_t height, DiagonalMode diagonalMode = DiagonalMode::NO_DIADONALS) noexcept;

      Astar()  = default;
      ~Astar() = default;
      Astar(const Astar&) = delete; // copy-constructor deleted
      Astar(Astar&&)      = delete; // move-constructor deleted
      Astar& operator=(const Astar&) = delete; //      assignment operator deleted
      Astar& operator=(Astar&&)      = delete; // move-assignment operator deleted

    private:

      std::size_t width_         = 0;
      std::size_t height_        = 0;
      std::size_t arraySize_     = 0;
      std::size_t start_         = 0;
      std::size_t goal_          = 0;
      DiagonalMode diagonalMode_ = DiagonalMode::NO_DIADONALS;

      std::unique_ptr<float[]>     weights_ = nullptr;
      std::unique_ptr<float[]>      costsH_ = nullptr;
      std::unique_ptr<float[]>  totalCosts_ = nullptr;
      std::unique_ptr<std::size_t[]> paths_ = nullptr;

      /**
      * A function for calculating the H cost. Internally it uses a Manhattan distance metric.
      */
      float calculateCostH(std::size_t index) const;

      /**
      * A function for resetting the state.
      */
      void resetState() const;
    };
  } // namespace Pathfinding
} // namespace Utils

#endif // __ASTAR_H__