#include "SuperQuadricShapesProducer.h"
#include "SuperQuadricShape.h"
#include "SuperQuadricSettings.h"
#include "ModelSettings.h"
#include "MathConstants.h"
#include "CPUParallelism/CPUParallelismNCP.h"
#include "VectorTypes.h"
#include "UtilityFunctions.h"
#include <set>
#include <limits>

using namespace std;
using namespace SuperQuadrics;
using namespace Utils;
using namespace Utils::CPUParallelism;
using namespace Utils::VectorTypes;
using namespace Utils::UtilityFunctions;

namespace // anonymous namespace used instead of deprecated 'static' keyword used for cpp variable locality
{
  SuperQuadricSettings createSuperQuadricSettings(SuperQuadricShapesProducer::SuperQuadricShapeObjects superQuadricShapeObject, size_t slices, size_t segments)
  {
    SuperQuadricSettings superQuadricSettings;
    switch (superQuadricShapeObject)
    {
      case SuperQuadricShapesProducer::SPHERE:
      {
        superQuadricSettings.a1 = superQuadricSettings.a2 = superQuadricSettings.a3 = 1.0;
        superQuadricSettings.n  = 1.0;
        superQuadricSettings.e  = 1.0;
        superQuadricSettings.u1 = -PI_DBL / 2.0;
        superQuadricSettings.u2 =  PI_DBL / 2.0;
        superQuadricSettings.v1 = -PI_DBL;
        superQuadricSettings.v2 =  PI_DBL;
        superQuadricSettings.uSegments = slices;
        superQuadricSettings.vSegments = segments;
        superQuadricSettings.s1 = 0.0;
        superQuadricSettings.t1 = 0.0;
        superQuadricSettings.s2 = 1.0;
        superQuadricSettings.t2 = 1.0;
        superQuadricSettings.superQuadricShapeType = superQuadricSettings.ELLIPSOID;

        break;
      }
      case SuperQuadricShapesProducer::CYLINDER:
      {
        superQuadricSettings.a1 = superQuadricSettings.a2 = superQuadricSettings.a3 = 1.0;
        superQuadricSettings.n  = 0.0;
        superQuadricSettings.e  = 1.0;
        superQuadricSettings.u1 = -PI_DBL / 2.0;
        superQuadricSettings.u2 =  PI_DBL / 2.0;
        superQuadricSettings.v1 = -PI_DBL;
        superQuadricSettings.v2 =  PI_DBL;
        superQuadricSettings.uSegments = slices;
        superQuadricSettings.vSegments = segments;
        superQuadricSettings.s1 = 0.0;
        superQuadricSettings.t1 = 0.0;
        superQuadricSettings.s2 = 1.0;
        superQuadricSettings.t2 = 1.0;
        superQuadricSettings.superQuadricShapeType = superQuadricSettings.ELLIPSOID;

        break;
      }
      case SuperQuadricShapesProducer::STAR:
      {
        superQuadricSettings.a1 = superQuadricSettings.a2 = superQuadricSettings.a3 = 1.0;
        superQuadricSettings.n  = 4.0;
        superQuadricSettings.e  = 4.0;
        superQuadricSettings.u1 = -PI_DBL / 2.0;
        superQuadricSettings.u2 =  PI_DBL / 2.0;
        superQuadricSettings.v1 = -PI_DBL;
        superQuadricSettings.v2 =  PI_DBL;
        superQuadricSettings.uSegments = slices;
        superQuadricSettings.vSegments = segments;
        superQuadricSettings.s1 = 0.0;
        superQuadricSettings.t1 = 0.0;
        superQuadricSettings.s2 = 1.0;
        superQuadricSettings.t2 = 1.0;
        superQuadricSettings.superQuadricShapeType = superQuadricSettings.ELLIPSOID;

        break;
      }
      case SuperQuadricShapesProducer::DOUBLE_PYRAMID:
      {
        superQuadricSettings.a1 = superQuadricSettings.a2 = superQuadricSettings.a3 = 1.0;
        superQuadricSettings.n  = 2.0;
        superQuadricSettings.e  = 2.0;
        superQuadricSettings.u1 = -PI_DBL / 2.0;
        superQuadricSettings.u2 =  PI_DBL / 2.0;
        superQuadricSettings.v1 = -PI_DBL;
        superQuadricSettings.v2 =  PI_DBL;
        superQuadricSettings.uSegments = slices;
        superQuadricSettings.vSegments = segments;
        superQuadricSettings.s1 = 0.0;
        superQuadricSettings.t1 = 0.0;
        superQuadricSettings.s2 = 1.0;
        superQuadricSettings.t2 = 1.0;
        superQuadricSettings.superQuadricShapeType = superQuadricSettings.ELLIPSOID;

        break;
      }
      case SuperQuadricShapesProducer::TORUS:
      {
        superQuadricSettings.a1 = superQuadricSettings.a2 = superQuadricSettings.a3 = (0.25 + 0.25) / 2.0; // radius1 & radius2 equals to 1.0
        superQuadricSettings.alpha = 1.5;
        superQuadricSettings.n  = 1.0;
        superQuadricSettings.e  = 1.0;
        superQuadricSettings.u1 = -PI_DBL;
        superQuadricSettings.u2 =  PI_DBL;
        superQuadricSettings.v1 = -PI_DBL;
        superQuadricSettings.v2 =  PI_DBL;
        superQuadricSettings.uSegments = slices;
        superQuadricSettings.vSegments = segments;
        superQuadricSettings.s1 = 0.0;
        superQuadricSettings.t1 = 0.0;
        superQuadricSettings.s2 = 1.0;
        superQuadricSettings.t2 = 1.0;
        superQuadricSettings.superQuadricShapeType = superQuadricSettings.TOROID;

        break;
      }
      case SuperQuadricShapesProducer::PINEAPPLE_SLICE:
      {
        superQuadricSettings.a1 = superQuadricSettings.a2 = superQuadricSettings.a3 = (1.0 + 1.0) / 2.0; // radius1 & radius2 equals to 1.0
        superQuadricSettings.alpha = 1.0;
        superQuadricSettings.n  = 0.0;
        superQuadricSettings.e  = 1.0;
        superQuadricSettings.u1 = -PI_DBL;
        superQuadricSettings.u2 =  PI_DBL;
        superQuadricSettings.v1 = -PI_DBL;
        superQuadricSettings.v2 =  PI_DBL;
        superQuadricSettings.uSegments = slices;
        superQuadricSettings.vSegments = segments;
        superQuadricSettings.s1 = 0.0;
        superQuadricSettings.t1 = 0.0;
        superQuadricSettings.s2 = 1.0;
        superQuadricSettings.t2 = 1.0;
        superQuadricSettings.superQuadricShapeType = superQuadricSettings.TOROID;

        break;
      }
      case SuperQuadricShapesProducer::PILLOW:
      {
        superQuadricSettings.a1 = superQuadricSettings.a2 = superQuadricSettings.a3 = 1.0;
        superQuadricSettings.n  = 1.0;
        superQuadricSettings.e  = 0.0;
        superQuadricSettings.u1 = -PI_DBL / 2.0;
        superQuadricSettings.u2 =  PI_DBL / 2.0;
        superQuadricSettings.v1 = -PI_DBL;
        superQuadricSettings.v2 =  PI_DBL;
        superQuadricSettings.uSegments = slices;
        superQuadricSettings.vSegments = segments;
        superQuadricSettings.s1 = 0.0;
        superQuadricSettings.t1 = 0.0;
        superQuadricSettings.s2 = 1.0;
        superQuadricSettings.t2 = 1.0;
        superQuadricSettings.superQuadricShapeType = superQuadricSettings.ELLIPSOID;

        break;
      }
      case SuperQuadricShapesProducer::SQUARE_TORUS:
      {
        superQuadricSettings.a1 = superQuadricSettings.a2 = superQuadricSettings.a3 = (1.0 + 1.0) / 2.0; // radius1 & radius2 equals to 1.0
        superQuadricSettings.alpha = 1.0;
        superQuadricSettings.n  = 0.2;
        superQuadricSettings.e  = 0.2;
        superQuadricSettings.u1 = -PI_DBL;
        superQuadricSettings.u2 =  PI_DBL;
        superQuadricSettings.v1 = -PI_DBL;
        superQuadricSettings.v2 =  PI_DBL;
        superQuadricSettings.uSegments = slices;
        superQuadricSettings.vSegments = segments;
        superQuadricSettings.s1 = 0.0;
        superQuadricSettings.t1 = 0.0;
        superQuadricSettings.s2 = 1.0;
        superQuadricSettings.t2 = 1.0;
        superQuadricSettings.superQuadricShapeType = superQuadricSettings.TOROID;

        break;
      }
      case SuperQuadricShapesProducer::PINCHED_TORUS:
      {
        superQuadricSettings.a1 = superQuadricSettings.a2 = superQuadricSettings.a3 = (1.0 + 1.0) / 2.0; // radius1 & radius2 equals to 1.0
        superQuadricSettings.alpha = 1.0;
        superQuadricSettings.n  = 1.0;
        superQuadricSettings.e  = 4.0;
        superQuadricSettings.u1 = -PI_DBL;
        superQuadricSettings.u2 =  PI_DBL;
        superQuadricSettings.v1 = -PI_DBL;
        superQuadricSettings.v2 =  PI_DBL;
        superQuadricSettings.uSegments = slices;
        superQuadricSettings.vSegments = segments;
        superQuadricSettings.s1 = 0.0;
        superQuadricSettings.t1 = 0.0;
        superQuadricSettings.s2 = 1.0;
        superQuadricSettings.t2 = 1.0;
        superQuadricSettings.superQuadricShapeType = superQuadricSettings.TOROID;

        break;
      }
      case SuperQuadricShapesProducer::ROUND_CUBE:
      {
        superQuadricSettings.a1 = superQuadricSettings.a2 = superQuadricSettings.a3 = 1.0;
        // low curvature for testing purposes, adjust n and e to alter round cube curvature:
        superQuadricSettings.n  = 0.0;
        superQuadricSettings.e  = 0.0;
        superQuadricSettings.u1 = -PI_DBL / 2.0;
        superQuadricSettings.u2 =  PI_DBL / 2.0;
        superQuadricSettings.v1 = -PI_DBL;
        superQuadricSettings.v2 =  PI_DBL;
        superQuadricSettings.uSegments = slices;
        superQuadricSettings.vSegments = segments;
        superQuadricSettings.s1 = 0.0;
        superQuadricSettings.t1 = 0.0;
        superQuadricSettings.s2 = 1.0;
        superQuadricSettings.t2 = 1.0;
        superQuadricSettings.superQuadricShapeType = superQuadricSettings.ELLIPSOID;

        break;
      }
      case SuperQuadricShapesProducer::HYPERBOLOID_ONE_SHEET:
      {
        superQuadricSettings.a1 = superQuadricSettings.a2 = superQuadricSettings.a3 = 0.45;
        superQuadricSettings.n  = 1.0;
        superQuadricSettings.e  = 1.0;
        superQuadricSettings.u1 = -PI_DBL / 2.0;
        superQuadricSettings.u2 =  PI_DBL / 2.0;
        superQuadricSettings.v1 = -PI_DBL;
        superQuadricSettings.v2 =  PI_DBL;
        superQuadricSettings.uSegments = slices;
        superQuadricSettings.vSegments = segments;
        superQuadricSettings.s1 = 0.0;
        superQuadricSettings.t1 = 0.0;
        superQuadricSettings.s2 = 1.0;
        superQuadricSettings.t2 = 1.0;
        superQuadricSettings.superQuadricShapeType = superQuadricSettings.HYPERBOLOID_ONE_SHEET;

        break;
      }
      case SuperQuadricShapesProducer::HYPERBOLOID_TWO_SHEETS:
      {
        superQuadricSettings.a1 = superQuadricSettings.a2 = superQuadricSettings.a3 = 0.25;
        superQuadricSettings.n  = 1.0;
        superQuadricSettings.e  = 1.0;
        superQuadricSettings.u1 = -PI_DBL / 2.0;
        superQuadricSettings.u2 =  PI_DBL / 2.0;
        superQuadricSettings.v1 = -PI_DBL / 2.0;
        superQuadricSettings.v2 =  PI_DBL / 2.0;
        superQuadricSettings.uSegments = slices;
        superQuadricSettings.vSegments = segments;
        superQuadricSettings.s1 = 0.0;
        superQuadricSettings.t1 = 0.0;
        superQuadricSettings.s2 = 1.0;
        superQuadricSettings.t2 = 1.0;
        superQuadricSettings.superQuadricShapeType = superQuadricSettings.HYPERBOLOID_TWO_SHEETS;

        break;
      }
      default: // default case create the Sphere
      {
        superQuadricSettings.a1 = superQuadricSettings.a2 = superQuadricSettings.a3 = 1.0;
        superQuadricSettings.n  = 1.0;
        superQuadricSettings.e  = 1.0;
        superQuadricSettings.u1 = -PI_DBL / 2.0;
        superQuadricSettings.u2 =  PI_DBL / 2.0;
        superQuadricSettings.v1 = -PI_DBL;
        superQuadricSettings.v2 =  PI_DBL;
        superQuadricSettings.uSegments = slices;
        superQuadricSettings.vSegments = segments;
        superQuadricSettings.s1 = 0.0;
        superQuadricSettings.t1 = 0.0;
        superQuadricSettings.s2 = 1.0;
        superQuadricSettings.t2 = 1.0;
        superQuadricSettings.superQuadricShapeType = superQuadricSettings.ELLIPSOID;

        break;
       }
    }

    return superQuadricSettings;
  }

  // type definitions - the modern C++11 way
  using VertexTuple = tuple<double3, uint32_t>;

  // struct comparing vertices in the set -> only compares vertex position, not the index
  struct VerticesSetComparator
  {
    inline bool operator()(const VertexTuple& vertexTuple1, const VertexTuple& vertexTuple2) const
    {
      const double3 vertex1 = get<0>(vertexTuple1);
      const double3 vertex2 = get<0>(vertexTuple2);
      if (!MathFunctions::equal(vertex1.x, vertex2.x)) return (vertex1.x < vertex2.x);
      if (!MathFunctions::equal(vertex1.y, vertex2.y)) return (vertex1.y < vertex2.y);
      if (!MathFunctions::equal(vertex1.z, vertex2.z)) return (vertex1.z < vertex2.z);

      return false;
    }
  };
}

void SuperQuadricShapesProducer::createSuperQuadricShape(size_t slices, size_t segments, double shrinkGeometryFactor, bool usingGeometryExplosionFactor, size_t startingIndex, SuperQuadricShapesProducer::SuperQuadricShapeObjects superQuadricShapeObject, vector<double>& vertices)
{
  // send empty normals & texCoords (no default parameters can be used for non-const vectors as
  // C++ does not allow temporaries to be bound to non-const references
  vector<double> normals;
  vector<double> texCoords;
  createSuperQuadricShape(slices, segments, shrinkGeometryFactor, usingGeometryExplosionFactor, startingIndex, superQuadricShapeObject, vertices, normals, texCoords);
}

void SuperQuadricShapesProducer::createSuperQuadricShape(size_t slices, size_t segments, double shrinkGeometryFactor, bool usingGeometryExplosionFactor, size_t startingIndex, SuperQuadricShapesProducer::SuperQuadricShapeObjects superQuadricShapeObject, vector<double>& vertices, vector<double>& normals)
{
  // send empty normals & texCoords (no default parameters can be used for non-const vectors as
  // C++ does not allow temporaries to be bound to non-const references
  vector<double> texCoords;
  createSuperQuadricShape(slices, segments, shrinkGeometryFactor, usingGeometryExplosionFactor, startingIndex, superQuadricShapeObject, vertices, normals, texCoords);
}

void SuperQuadricShapesProducer::createSuperQuadricShape(size_t slices, size_t segments, double shrinkGeometryFactor, bool usingGeometryExplosionFactor, size_t startingIndex, SuperQuadricShapesProducer::SuperQuadricShapeObjects superQuadricShapeObject, vector<double>& vertices, vector<double>& normals, vector<double>& texCoords)
{
  const bool usingNormals            = !normals.empty();
  const bool usingTexCoords          = !texCoords.empty();
  const bool usingManipulateGeometry = !MathFunctions::equal(shrinkGeometryFactor, 1.0) || usingGeometryExplosionFactor;
  const ModelSettings modelSettings(true, usingNormals, usingTexCoords, usingManipulateGeometry, shrinkGeometryFactor, usingGeometryExplosionFactor);
  const SuperQuadricSettings superQuadricSettings = createSuperQuadricSettings(superQuadricShapeObject, slices, segments);
  SuperQuadricShape superQuadricShape(superQuadricSettings, superQuadricSettings.superQuadricShapeType, modelSettings, &vertices.data()[startingIndex], usingNormals ? &normals.data()[startingIndex] : nullptr, usingTexCoords ? &texCoords.data()[2 * (startingIndex) / 3] : nullptr);

  // Calculate delta variables (add an epsilon to avoid rendering errors with the Super Quadric shapes)
  double dU = ((superQuadricSettings.u2 - superQuadricSettings.u1) / superQuadricSettings.uSegments) + numeric_limits<double>::epsilon();
  double dV = ((superQuadricSettings.v2 - superQuadricSettings.v1) / superQuadricSettings.vSegments) + numeric_limits<double>::epsilon();
  double dS = ((superQuadricSettings.s2 - superQuadricSettings.s1) / superQuadricSettings.uSegments) + numeric_limits<double>::epsilon();
  double dT = ((superQuadricSettings.t2 - superQuadricSettings.t1) / superQuadricSettings.vSegments) + numeric_limits<double>::epsilon();

  parallelFor(superQuadricSettings.uSegments, [&](size_t i)
  {
    superQuadricShape.createGeometry(dU, dV, dS, dT, i);
  });
}

tuple<vector<double>, vector<uint32_t>> SuperQuadricShapesProducer::calculateVertexIndices(const vector<double>& vertices)
{
  set<VertexTuple, VerticesSetComparator> verticesSet;
  vector<double> verticesResult;
  vector<uint32_t> vertexIndices;

  uint32_t vertexSetIndex = 0;
  for (uint32_t vertexIndex = 0; vertexIndex < vertices.size() / 3; ++vertexIndex)
  {
    const uint32_t vertex3Index = 3 * vertexIndex;
    const double x = vertices[vertex3Index + 0];
    const double y = vertices[vertex3Index + 1];
    const double z = vertices[vertex3Index + 2];
    double3 vertex{ x, y, z };

    const auto& result = verticesSet.emplace(make_tuple(vertex, vertexSetIndex));
    if (!result.second) // if vertexTuple not inserted, element was already in
    {
      vertexIndices.emplace_back(get<1>(*result.first));
    }
    else
    {
      vertexIndices.emplace_back(vertexSetIndex);
      ++vertexSetIndex;
    }
  }

  // note that the vertices in the set are not sorted by the index, thus we have to rearrange them for proper rendering
  verticesResult.resize(3 * verticesSet.size());
  for (const auto& vertexTuple : verticesSet)
  {
    const double3 vertex        = get<0>(vertexTuple);
    const uint32_t vertexIndex  = get<1>(vertexTuple);
    const uint32_t vertex3Index = 3 * vertexIndex;
    verticesResult[vertex3Index + 0] = vertex.x;
    verticesResult[vertex3Index + 1] = vertex.y;
    verticesResult[vertex3Index + 2] = vertex.z;
  }

  return make_tuple(verticesResult, vertexIndices);
}