#include "SuperQuadricShape.h"
#include "UtilityFunctions.h"
#include <cmath>

using namespace std;
using namespace SuperQuadrics;
using namespace Utils::UtilityFunctions;

namespace // anonymous namespace used instead of deprecated 'static' keyword used for cpp variable locality
{
  /*
  *  This function implements the Cos(v,n) utility function:
  *  Cos(v,n) = sign(cos(v)) * |cos(v)|^n
  */
  inline double sqCos(double v, double n)
  {
    const double cosValue = cos(v);
    return MathFunctions::sign(cosValue) * pow(abs(cosValue), n);
  }

  /*
  *  This function implements the CosPlusAlpha(v,n,alpha) utility function:
  *  CosPlusAlpha(v,n,alpha) = alpha + Cos(v,n)
  */
  inline double sqCosPlusAlpha(double v, double n, double alpha)
  {
    return alpha + sqCos(v, n);
  }

  /*
  *  This function implements the Sin(v,n) utility function:
  *  Sin(v,n) = sign(sin(v)) * |sin(v)|^n
  */
  inline double sqSin(double v, double n)
  {
    const double sinValue = sin(v);
    return MathFunctions::sign(sinValue) * pow(abs(sinValue), n);
  }

  /*
  *  This function implements the CosH(v,n) (hyperbolic cos) utility function:
  *  CosH(v,n) = sign(CosH(v)) * |CosH(v)|^n
  */
  inline double sqCosH(double v, double n)
  {
    const double coshValue = cosh(v);
    return MathFunctions::sign(coshValue) * pow(abs(coshValue), n);
  }

  /*
  *  This function implements the SinH(v,n) (hyperbolic sin) utility function:
  *  SinH(v,n) = sign(SinH(v)) * |SinH(v)|^n
  */
  inline double sqSinH(double v, double n)
  {
    const double sinhValue = sinh(v);
    return MathFunctions::sign(sinhValue) * pow(abs(sinhValue), n);
  }
}

void SuperQuadricShape::createGeometry(double dU, double dV, double dS, double dT, size_t position)
{
  size_t verticesIndex =  18 * position * superQuadricSettings_.vSegments;
  size_t texCoordsIndex = 12 * position * superQuadricSettings_.vSegments; // 2 / 3 of all vertices

  // Initialize variables
  const double U = superQuadricSettings_.u1 + position * dU;
  const double S = superQuadricSettings_.s1 + position * dS;

  // Initialize variables for loop
  double V = superQuadricSettings_.v1;
  double T = superQuadricSettings_.t1;
  for (size_t X = 0; X < superQuadricSettings_.vSegments; ++X)
  {
    // Note: inverted the triangle creation order to get proper OpenGL CCW triangle vertex ordering for glCullFace()

    // VERTEX #1 for triangle 1
    if (modelSettings_.usingCCW)
    {
      createSuperQuadric(U + dU, V + dV, verticesIndex, S + dS, T + dT, texCoordsIndex);
    }
    else
    {
      createSuperQuadric(U, V, verticesIndex, S, T, texCoordsIndex);
    }
    verticesIndex  += 3;
    texCoordsIndex += 2;

    // VERTEX #2 for triangle 1
    createSuperQuadric(U + dU, V,      verticesIndex, S + dS, T     , texCoordsIndex);
    verticesIndex  += 3;
    texCoordsIndex += 2;

    // VERTEX #3 for triangle 1
    if (modelSettings_.usingCCW)
    {
      createSuperQuadric(U, V, verticesIndex, S, T, texCoordsIndex);
    }
    else
    {
      createSuperQuadric(U + dU, V + dV, verticesIndex, S + dS, T + dT, texCoordsIndex);
    }
    verticesIndex  += 3;
    texCoordsIndex += 2;

    if (modelSettings_.usingManipulateGeometry)
    {
      applyTriangleGeometryShrinking(verticesIndex - 9, verticesIndex - 6, verticesIndex - 3);
    }

    // VERTEX #1 for triangle 2
    if (modelSettings_.usingCCW)
    {
      createSuperQuadric(U, V + dV, verticesIndex, S, T + dT, texCoordsIndex);
    }
    else
    {
      createSuperQuadric(U, V, verticesIndex, S, T, texCoordsIndex);
    }
    verticesIndex  += 3;
    texCoordsIndex += 2;

    // VERTEX #2 for triangle 2
    createSuperQuadric(U + dU, V + dV, verticesIndex, S + dS, T + dT, texCoordsIndex);
    verticesIndex  += 3;
    texCoordsIndex += 2;

    // VERTEX #3 for triangle 2
    if (modelSettings_.usingCCW)
    {
      createSuperQuadric(U, V, verticesIndex, S, T, texCoordsIndex);
    }
    else
    {
      createSuperQuadric(U, V + dV, verticesIndex, S, T + dT, texCoordsIndex);
    }
    verticesIndex  += 3;
    texCoordsIndex += 2;

    if (modelSettings_.usingManipulateGeometry)
    {
      applyTriangleGeometryShrinking(verticesIndex - 9, verticesIndex - 6, verticesIndex - 3);
    }

    // Update variables for next loop
    V += dV;
    T += dT;
  }
}

void SuperQuadricShape::sqEllipsoid(double a1, double a2, double a3, double u, double v, double n, double e, size_t verticesIndex) const
{
  double value = sqCos(u, n);
  vertices_[verticesIndex    ] = a1 * value * sqCos(v, e);
  vertices_[verticesIndex + 1] = a2 * value * sqSin(v, e);
  vertices_[verticesIndex + 2] = a3 * sqSin(u, n);

  if (modelSettings_.usingNormals)
  {
    value = sqCos(u, 2.0 - n);
    normals_[verticesIndex    ] = value * sqCos(v, 2.0 - e) / a1;
    normals_[verticesIndex + 1] = value * sqSin(v, 2.0 - e) / a2;
    normals_[verticesIndex + 2] = sqSin(u, 2.0 - n) / a3;
  }
}

void SuperQuadricShape::sqHyperboloidOneSheet(double a1, double a2, double a3, double u, double v, double n, double e, size_t verticesIndex) const
{
  double value = sqCosH(u, n);
  vertices_[verticesIndex    ] = a1 * value * sqCos(v, e);
  vertices_[verticesIndex + 1] = a2 * value * sqSin(v, e);
  vertices_[verticesIndex + 2] = a3 * sqSinH(u, n);

  if (modelSettings_.usingNormals)
  {
    value = sqCosH(u, 2.0 - n);
    normals_[verticesIndex    ] = value * sqCos(v, 2.0 - e) / a1;
    normals_[verticesIndex + 1] = value * sqSin(v, 2.0 - e) / a2;
    normals_[verticesIndex + 2] = sqSinH(u, 2.0 - n) / a3;
  }
}

void SuperQuadricShape::sqHyperboloidTwoSheets(double a1, double a2, double a3, double u, double v, double n, double e, size_t verticesIndex) const
{
  double value1 = sqCosH(u, n);
  vertices_[verticesIndex    ] = a1 * value1 * sqCosH(v, e);
  vertices_[verticesIndex + 1] = a2 * value1 * sqSinH(v, e);
  vertices_[verticesIndex + 2] = a3 * sqSinH(u, n);

  if (modelSettings_.usingNormals)
  {
    value1 = sqCosH(u, 2.0 - n);
    normals_[verticesIndex    ] = value1 * sqCosH(v, 2.0 - e) / a1;
    normals_[verticesIndex + 1] = value1 * sqSinH(v, 2.0 - e) / a2;
    normals_[verticesIndex + 2] = sqSinH(u, 2.0 - n) / a3;
  }
}

void SuperQuadricShape::sqToroid(double a1, double a2, double a3, double u, double v, double n, double e, double alpha, size_t verticesIndex) const
{
  const double A1 = 1.0 / (a1 + alpha);
  const double A2 = 1.0 / (a2 + alpha);
  const double A3 = 1.0 / (a3 + alpha);

  double value = sqCosPlusAlpha(u, n, alpha);
  vertices_[verticesIndex    ] = A1 * value * sqCos(v, e);
  vertices_[verticesIndex + 1] = A2 * value * sqSin(v, e);
  vertices_[verticesIndex + 2] = A3 * sqSin(u, n);

  if (modelSettings_.usingNormals)
  {
    value = sqCos(u, 2.0 - n);
    normals_[verticesIndex    ] = value * sqCos(v, 2.0 - e) / A1;
    normals_[verticesIndex + 1] = value * sqSin(v, 2.0 - e) / A2;
    normals_[verticesIndex + 2] = sqSin(u, 2.0 - n) / A3;
  }
}

/*
*  Creates the SuperQuadric shape.
*/
void SuperQuadricShape::createSuperQuadric(double u, double v, size_t verticesIndex, double s, double t, size_t texCoordsIndex) const
{
  if (superQuadricShapeType_ == 0)
  {
    sqEllipsoid(superQuadricSettings_.a1, superQuadricSettings_.a2, superQuadricSettings_.a3, u, v, superQuadricSettings_.n, superQuadricSettings_.e, verticesIndex);
  }
  else if (superQuadricShapeType_ == 1)
  {
    sqHyperboloidOneSheet(superQuadricSettings_.a1, superQuadricSettings_.a2, superQuadricSettings_.a3, u, v, superQuadricSettings_.n, superQuadricSettings_.e, verticesIndex);
  }
  else if (superQuadricShapeType_ == 2)
  {
    sqHyperboloidTwoSheets(superQuadricSettings_.a1, superQuadricSettings_.a2, superQuadricSettings_.a3, u, v, superQuadricSettings_.n, superQuadricSettings_.e, verticesIndex);
  }
  else // if (superQuadricShapeType == 3)
  {
    sqToroid(superQuadricSettings_.a1, superQuadricSettings_.a2, superQuadricSettings_.a3, u, v, superQuadricSettings_.n, superQuadricSettings_.e, superQuadricSettings_.alpha, verticesIndex);
  }

  if (modelSettings_.usingTexCoords)
  {
    texCoords_[texCoordsIndex    ] = t;      // was originally s, had to reverse s with t
    texCoords_[texCoordsIndex + 1] = 1.0- s; // was originally t, had to reverse t with 1.0 - s
  }
}

void SuperQuadricShape::applyTriangleGeometryShrinking(size_t verticesTriangle1Index, size_t verticesTriangle2Index, size_t verticesTriangle3Index)
{
  const double verticesTriangle1X = vertices_[verticesTriangle1Index    ];
  const double verticesTriangle1Y = vertices_[verticesTriangle1Index + 1];
  const double verticesTriangle1Z = vertices_[verticesTriangle1Index + 2];

  const double verticesTriangle2X = vertices_[verticesTriangle2Index    ];
  const double verticesTriangle2Y = vertices_[verticesTriangle2Index + 1];
  const double verticesTriangle2Z = vertices_[verticesTriangle2Index + 2];

  const double verticesTriangle3X = vertices_[verticesTriangle3Index    ];
  const double verticesTriangle3Y = vertices_[verticesTriangle3Index + 1];
  const double verticesTriangle3Z = vertices_[verticesTriangle3Index + 2];

  const double denominator       = modelSettings_.usingGeometryExplosionFactor ? 2.0 : 3.0;
  const double triangleCentroidX = (verticesTriangle1X + verticesTriangle2X + verticesTriangle3X) / denominator;
  const double triangleCentroidY = (verticesTriangle1Y + verticesTriangle2Y + verticesTriangle3Y) / denominator;
  const double triangleCentroidZ = (verticesTriangle1Z + verticesTriangle2Z + verticesTriangle3Z) / denominator;

  vertices_[verticesTriangle1Index    ] = triangleCentroidX + modelSettings_.shrinkGeometryFactor * (verticesTriangle1X - triangleCentroidX);
  vertices_[verticesTriangle1Index + 1] = triangleCentroidY + modelSettings_.shrinkGeometryFactor * (verticesTriangle1Y - triangleCentroidY);
  vertices_[verticesTriangle1Index + 2] = triangleCentroidZ + modelSettings_.shrinkGeometryFactor * (verticesTriangle1Z - triangleCentroidZ);

  vertices_[verticesTriangle2Index    ] = triangleCentroidX + modelSettings_.shrinkGeometryFactor * (verticesTriangle2X - triangleCentroidX);
  vertices_[verticesTriangle2Index + 1] = triangleCentroidY + modelSettings_.shrinkGeometryFactor * (verticesTriangle2Y - triangleCentroidY);
  vertices_[verticesTriangle2Index + 2] = triangleCentroidZ + modelSettings_.shrinkGeometryFactor * (verticesTriangle2Z - triangleCentroidZ);

  vertices_[verticesTriangle3Index    ] = triangleCentroidX + modelSettings_.shrinkGeometryFactor * (verticesTriangle3X - triangleCentroidX);
  vertices_[verticesTriangle3Index + 1] = triangleCentroidY + modelSettings_.shrinkGeometryFactor * (verticesTriangle3Y - triangleCentroidY);
  vertices_[verticesTriangle3Index + 2] = triangleCentroidZ + modelSettings_.shrinkGeometryFactor * (verticesTriangle3Z - triangleCentroidZ);

  if (modelSettings_.usingGeometryExplosionFactor)
  {
    vertices_[verticesTriangle1Index    ] += 0.1 * random_();
    vertices_[verticesTriangle1Index + 1] += 0.1 * random_();
    vertices_[verticesTriangle1Index + 2] += 0.1 * random_();

    vertices_[verticesTriangle2Index    ] += 0.1 * random_();
    vertices_[verticesTriangle2Index + 1] += 0.1 * random_();
    vertices_[verticesTriangle2Index + 2] += 0.1 * random_();

    vertices_[verticesTriangle3Index    ] += 0.1 * random_();
    vertices_[verticesTriangle3Index + 1] += 0.1 * random_();
    vertices_[verticesTriangle3Index + 2] += 0.1 * random_();
  }
}

double SuperQuadricShape::sqEllipsoidInsideOut(double x, double y, double z) const
{
  return pow(pow(x / superQuadricSettings_.a1, 2.0 / superQuadricSettings_.e) + pow(y / superQuadricSettings_.a2, 2.0 / superQuadricSettings_.e), superQuadricSettings_.e / superQuadricSettings_.n) + pow(z / superQuadricSettings_.a3, 2.0 / superQuadricSettings_.n);
}

double SuperQuadricShape::sqHyperboloidOneSheetInsideOut(double x, double y, double z) const
{
  return pow(pow(x / superQuadricSettings_.a1, 2.0 / superQuadricSettings_.e) + pow(y / superQuadricSettings_.a2, 2.0 / superQuadricSettings_.e), superQuadricSettings_.e / superQuadricSettings_.n) - pow(z / superQuadricSettings_.a3, 2.0 / superQuadricSettings_.n);
}

double SuperQuadricShape::sqHyperboloidTwoSheetsInsideOut(double x, double y, double z) const
{
  return pow(pow(x / superQuadricSettings_.a1, 2.0 / superQuadricSettings_.e) - pow(y / superQuadricSettings_.a2, 2.0 / superQuadricSettings_.e), superQuadricSettings_.e / superQuadricSettings_.n) - pow(z / superQuadricSettings_.a3, 2.0 / superQuadricSettings_.n);
}

double SuperQuadricShape::sqToroidInsideOut(double x, double y, double z) const
{
  return pow(pow(pow(x / superQuadricSettings_.a1, 2.0 / superQuadricSettings_.e) + pow(y / superQuadricSettings_.a2, 2.0 / superQuadricSettings_.e), superQuadricSettings_.e / 2.0) - superQuadricSettings_.alpha, 2.0 / superQuadricSettings_.n) + pow(z / superQuadricSettings_.a3, 2.0 / superQuadricSettings_.n);
}

double SuperQuadricShape::insideOut(double x, double y, double z) const
{
  if (superQuadricShapeType_ == 0)
  {
    return sqEllipsoidInsideOut(x, y, z);
  }
  else if (superQuadricShapeType_ == 1)
  {
    return sqHyperboloidOneSheetInsideOut(x, y, z);
  }
  else if (superQuadricShapeType_ == 2)
  {
    return sqHyperboloidTwoSheetsInsideOut(x, y, z);
  }
  else // if (superQuadricShapeType == 3)
  {
    return sqToroidInsideOut(x, y, z);
  }
}