/*

Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
Released Under a Simplified BSD (FreeBSD) License
for academic, personal & non-commercial use.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the author and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.

A Commercial License is also available for commercial use with
special restrictions and obligations at a one-off fee. See links at:
1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

#pragma once

#ifndef __SuperQuadricShapesProducer_h
#define __SuperQuadricShapesProducer_h

#include "ModuleDLL.h"
#include <vector>
#include <tuple>
#include <cstdint>

/** @brief Namespace SuperQuadrics for the Super Quadric algorithmic shapes.
* @author Thanos Theo, 2018
* @version 14.0.0.0
*/
namespace SuperQuadrics
{
  /** @brief Many examples using the SuperQuadricShapesProducer for producing various SuperQuadric-based shapes. Using threads for N-CP processing.
  * @author Thanos Theo, 2009-2018
  * @version 14.0.0.0
  */
  struct SUPER_QUADRICS_MODULE_API SuperQuadricShapesProducer final
  {
    static constexpr std::size_t NUMBER_OF_SHAPES = 12;
    enum SuperQuadricShapeObjects : std::size_t { SPHERE                 = 0,
                                                  CYLINDER               = 1,
                                                  STAR                   = 2,
                                                  DOUBLE_PYRAMID         = 3,
                                                  TORUS                  = 4,
                                                  PINEAPPLE_SLICE        = 5,
                                                  PILLOW                 = 6,
                                                  SQUARE_TORUS           = 7,
                                                  PINCHED_TORUS          = 8,
                                                  ROUND_CUBE             = 9,
                                                  HYPERBOLOID_ONE_SHEET  = 10,
                                                  HYPERBOLOID_TWO_SHEETS = 11
                                                };

    /**
    *  Creates a given Super Quadric shape with vertices only.
    */
    static void createSuperQuadricShape(std::size_t slices, std::size_t segments, double shrinkGeometryFactor, bool usingGeometryExplosionFactor, std::size_t startingIndex, SuperQuadricShapesProducer::SuperQuadricShapeObjects superQuadricShapeObject, std::vector<double>& vertices);

    /**
    *  Creates a given Super Quadric shape with vertices & normals.
    */
    static void createSuperQuadricShape(std::size_t slices, std::size_t segments, double shrinkGeometryFactor, bool usingGeometryExplosionFactor, std::size_t startingIndex, SuperQuadricShapesProducer::SuperQuadricShapeObjects superQuadricShapeObject, std::vector<double>& vertices, std::vector<double>& normals);

    /**
    *  Creates a given Super Quadric shape with vertices, normals & texture coordinates.
    */
    static void createSuperQuadricShape(std::size_t slices, std::size_t segments, double shrinkGeometryFactor, bool usingGeometryExplosionFactor, std::size_t startingIndex, SuperQuadricShapesProducer::SuperQuadricShapeObjects superQuadricShapeObject, std::vector<double>& vertices, std::vector<double>& normals, std::vector<double>& texCoords);

    /**
    *  Calculates a vector of unique vertices along with its matching vertex indices.
    *  @return tuple with element one as vertex positions and element two as vertex indices.
    */
    static std::tuple<std::vector<double>, std::vector<std::uint32_t>> calculateVertexIndices(const std::vector<double>& vertices);

    SuperQuadricShapesProducer()  = delete;
    ~SuperQuadricShapesProducer() = delete;
    SuperQuadricShapesProducer(const SuperQuadricShapesProducer&) = delete;
    SuperQuadricShapesProducer(SuperQuadricShapesProducer&&)      = delete;
    SuperQuadricShapesProducer& operator=(const SuperQuadricShapesProducer&) = delete;
    SuperQuadricShapesProducer& operator=(SuperQuadricShapesProducer&&)      = delete;
  };
} // namespace SuperQuadrics

#endif // __SuperQuadricShapesProducer_h