/*

Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
Released Under a Simplified BSD (FreeBSD) License
for academic, personal & non-commercial use.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the author and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.

A Commercial License is also available for commercial use with
special restrictions and obligations at a one-off fee. See links at:
1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

#pragma once

#ifndef __SuperQuadricShape_h
#define __SuperQuadricShape_h

#include "ModuleDLL.h"
#include "ModelSettings.h"
#include "SuperQuadricSettings.h"
#include "Randomizers.h"

/** @brief Namespace SuperQuadrics for the Super Quadric algorithmic shapes.
* @author Thanos Theo, 2018
* @version 14.0.0.0
*/
namespace SuperQuadrics
{
  /** @brief A SuperQuadricShape is the class that creates SuperQuadric-based shapes using parametric polar coordinate equations.
  * @author Thanos Theo, 2009-2018
  * @version 14.0.0.0
  */
  class SUPER_QUADRICS_MODULE_API SuperQuadricShape final
  {
  public:

    /*
    *  Generates a solid SuperQuadrics using the parameters from superQuadricSettings and optionally generates normals & texture coordinates.
    */
    void createGeometry(double dU, double dV, double dS, double dT, std::size_t position);

    /*
    *  Tests to see if point P<x,y,z> is inside the SuperQuadric superQuadricSettings.
    *  Returns 1 if on the surface, > 1 if outside the surface, or
    *  < 1 if inside the surface
    */
    double insideOut(double x, double y, double z) const;

    /*
    *  Normal constructor to be used.
    */
    SuperQuadricShape(const SuperQuadricSettings& superQuadricSettings, std::size_t superQuadricShapeType, const ModelSettings& modelSettings, double* vertices, double* normals = nullptr, double* texCoords = nullptr) noexcept
      : superQuadricSettings_(superQuadricSettings)
      , superQuadricShapeType_(superQuadricShapeType)
      , modelSettings_(modelSettings)
      , vertices_(vertices)
      , normals_(normals)
      , texCoords_(texCoords)
    {}

    SuperQuadricShape()  = default;
    ~SuperQuadricShape() = default;
    SuperQuadricShape(const SuperQuadricShape&) = delete;
    SuperQuadricShape(SuperQuadricShape&&)      = delete;
    SuperQuadricShape& operator=(const SuperQuadricShape&) = delete;
    SuperQuadricShape& operator=(SuperQuadricShape&&)      = delete;

  private:

    /**
    *  Variable used for choosing between the  SuperQuadric shape types.
    */
    SuperQuadricSettings superQuadricSettings_;

    /**
    *  The SuperQuadric reference stores all SuperQuadric related variables.
    */
    std::size_t superQuadricShapeType_ = 0;

    /**
    *  The SuperQuadric reference stores all SuperQuadric related variables.
    */
    ModelSettings modelSettings_;

    /**
    *  The vertices memory pointer.
    */
    double* vertices_  = nullptr;

    /**
    *  The normals memory pointer.
    */
    double* normals_   = nullptr;

    /**
    *  The texCoords memory pointer.
    */
    double* texCoords_ = nullptr;

    /*
    *  The randomizer used for geometry explosions.
    */
    Utils::Randomizers::RandomRNGWELL512 random_;

    /*
    * a1, a2, and a3 are the x, y, and z scaling factors, respectfully.
    * For proper generation of the solid, u should be >= -PI / 2 and <= PI / 2.
    * Similarly, v should be >= -PI and <= PI.
    */
    void sqEllipsoid(double a1, double a2, double a3, double u, double v, double n, double e, std::size_t verticesIndex) const;

    /*
    * a1, a2, and a3 are the x, y, and z scaling factors, respectfully.
    * For proper generation of the solid, u should be >= -PI / 2 and <= PI / 2.
    * Similarly, v should be >= -PI and <= PI.
    */
    void sqHyperboloidOneSheet(double a1, double a2, double a3, double u, double v, double n, double e, std::size_t verticesIndex) const;

    /*
    * a1, a2, and a3 are the x, y, and z scaling factors, respectfully.
    * For proper generation of the solid, u should be >= -PI / 2 and <= PI / 2.
    * Similarly, v should be >= -PI / 2 and <= PI / 2.
    */
    void sqHyperboloidTwoSheets(double a1, double a2, double a3, double u, double v, double n, double e, std::size_t verticesIndex) const;

    /*
    *  a1, a2, and a3 are the x, y, and z scaling factors, respectfully.
    *  For proper generation of the solid, u should be >= -PI and <= PI.
    *  Similarly, v should be >= -PI and <= PI.
    *  Also, alpha should be > 1.
    */
    void sqToroid(double a1, double a2, double a3, double u, double v, double n, double e, double alpha, std::size_t verticesIndex) const;

    /*
    *  Creates the SuperQuadric shape.
    */
    void createSuperQuadric(double u, double v, std::size_t verticesIndex, double s, double t, std::size_t texCoordsIndex) const;

    /*
    *  Applies triangle geometry shrinking.
    */
    void applyTriangleGeometryShrinking(std::size_t verticesTriangle1Index, std::size_t verticesTriangle2Index, std::size_t verticesTriangle3Index);

    /*
    *  Check to be used in function insideOut().
    */
    double sqEllipsoidInsideOut(double x, double y, double z) const;

    /*
    *  Check to be used in function insideOut().
    */
    double sqHyperboloidOneSheetInsideOut(double x, double y, double z) const;

    /*
    *  Check to be used in function insideOut().
    */
    double sqHyperboloidTwoSheetsInsideOut(double x, double y, double z) const;

    /*
    *  Check to be used in function insideOut().
    */
    double sqToroidInsideOut(double x, double y, double z) const;
  };
} // namespace SuperQuadrics

#endif // __SuperQuadricShape_h