/*

Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
Released Under a Simplified BSD (FreeBSD) License
for academic, personal & non-commercial use.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the author and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.

A Commercial License is also available for commercial use with
special restrictions and obligations at a one-off fee. See links at:
1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

#pragma once

#ifndef __SuperQuadricSettings_h
#define __SuperQuadricSettings_h

#include "ModuleDLL.h"
#include "MathConstants.h"
#include <cstddef>

/** @brief Namespace SuperQuadrics for the Super Quadric algorithmic shapes.
* @author Thanos Theo, 2018
* @version 14.0.0.0
*/
namespace SuperQuadrics
{
  /** @brief SuperQuadricSettings is the class that acts as a placeholder of the SuperQuadricShape variables needed for the parametric polar coordinate equations.
  * @author Thanos Theo, 2009-2018
  * @version 14.0.0.0
  */
  struct SUPER_QUADRICS_MODULE_API SuperQuadricSettings final
  {
    /**
    * SuperQuadricShapeTypes enumeration class which holds information for the shape type of a SuperQuadricShape.
    */
    enum SuperQuadricShapeTypes : std::size_t { ELLIPSOID = 0, HYPERBOLOID_ONE_SHEET = 1, HYPERBOLOID_TWO_SHEETS = 2, TOROID = 3 };
    SuperQuadricShapeTypes superQuadricShapeType = ELLIPSOID;

    /**
    *  Scaling factor for x.
    */
    double a1 = 1.0;
  
    /**
    *  Scaling factor for y.
    */
    double a2 = 1.0;
  
    /**
    *  Scaling factor for z.
    */
    double a3 = 1.0;

    /**
    *  For generating toroids. This is the inner radius.
    */
    double alpha = 2.0;

    /**
    *  North-South Roundness factor.
    */
    double n = 1.0;

    /**
    *  East-West Squareness factor.
    */
    double e = 1.0;

    /**
    *  Initial U value.
    */
    double u1 = -Utils::PI_DBL / 2.0;
  
    /**
    *  Final U value.
    */
    double u2 =  Utils::PI_DBL / 2.0;

    /**
    *  Initial V value.
    */
    double v1 = -Utils::PI_DBL;
  
    /**
    *  Final V value.
    */
    double v2 =  Utils::PI_DBL;

    /**
    *  Number of segments for U.
    */
    std::size_t uSegments = 8;
  
    /**
    *  Number of segments for V.
    */
    std::size_t vSegments = 8;

    /**
    *  Initial S value.
    */
    double s1 = 0.0;
  
    /**
    *  Final S value.
    */
    double s2 = 1.0;

    /**
    *  Initial T value.
    */
    double t1 = 0.0;
  
    /**
    *  Final T value.
    */
    double t2 = 1.0;

    SuperQuadricSettings()  = default;
    ~SuperQuadricSettings() = default;
    SuperQuadricSettings(const SuperQuadricSettings&) = default;
    SuperQuadricSettings(SuperQuadricSettings&&)      = default;
    SuperQuadricSettings& operator=(const SuperQuadricSettings&) = default;
    SuperQuadricSettings& operator=(SuperQuadricSettings&&)      = default;
  };
} // namespace SuperQuadrics

#endif // __SuperQuadricSettings_h