/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  ShaderFilesGenerator.cpp:
*  ========================
*  This class includes shader files header/implementation generator related functionality.
*
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/

#include "ShaderFilesGenerator.h"
#include "AccurateTimers.h"
#include "UtilityFunctions.h"
#include <unordered_map>
#include <iostream>
#include <fstream>
#include <sstream>
#include <algorithm>
#include <experimental/filesystem>

using namespace std;
using namespace std::experimental::filesystem; // provisional namespace until the <filesystem> namespace becomes standard in C++17
using namespace Utils;
using namespace Utils::UtilityFunctions;
using namespace OpenGLRenderingEngine;
using AccurateTimer = Utils::AccurateTimers::AccurateCPUTimer;

namespace // anonymous namespace used instead of deprecated 'static' keyword used for cpp variable locality
{
  using BitsetType = bitset<OpenGLAssetManager::NUMBER_OF_TOTAL_SHADER_TYPES>;

  const string OPENGL_RENDERING_ENGINE_NAMESPACE     = "OpenGLRenderingEngine";
  const string OPENGL_RENDERING_ENGINE_DIRECTORY     = OPENGL_RENDERING_ENGINE_NAMESPACE + "/";
  const string SHADER_FILES_GENERATOR_DIRECTORY      = "ShaderFilesGenerator/";
  const string CMAKE_LISTS_FILE_NAME                 = "CMakeLists.txt";
  const string GLSL_SHADER_FILES_NAMESPACE           = "GLSLShaderFiles";
  const string GLSL_SHADER_FILES_MAIN_CLASS          = "AllGLSLShaderFiles";
  const string GLSL_SHADER_FILES_MAIN_HEADER_MAP     = "allGLSLShaderFiles_";
  const string GLSL_SHADER_APPEND_STRING_TO_FILENAME = "_scrambled";
  const string GLSL_SHADER_FILES_EXTENSION           = ".h";
  const string GLSL_SHADER_DIRECTORY_FILE_SEPARATOR  = "_/_";
  const string GLSL_SHADER_AUTHOR                    = "* @author Thanos Theo, 2009-2018";
  const string GLSL_SHADER_VERSION                   = "* @version 14.0.0.0";

  /*
  *  Displays the program usage information.
  */
  inline void giveUsage(const char* cmd, int exitCode = EXIT_FAILURE)
  {
    cout << "\nUsage: " <<     cmd      << " [-h][-help][-path #]\n";
    cout << "----------------------------------------------------\n";
    cout << "   -h or -help              this help text\n";
    cout << "   -path  #                 use given path #\n";

    exit(exitCode);
  }

  inline unordered_map<string, int> autoParseAllShaderFiles(const string& absolutePath)
  {
    unordered_map<string, int> allShaderFilesInfo;
    string currentShaderDirectoryName;
    for (const auto& pathIterator : recursive_directory_iterator(absolutePath + OPENGL_RENDERING_ENGINE_DIRECTORY + OpenGLAssetManager::getGLSLInternalDirectory()))
    {
      const path& currentPath = pathIterator.path();
      if (is_directory(currentPath))
      {
        currentShaderDirectoryName = currentPath.filename().string();
      }
      else
      {
        if (currentPath.filename().string() == CMAKE_LISTS_FILE_NAME)
        {
          // skip the CMAKE_LISTS_FILE_NAME
          continue;
        }

        const string shaderFileName      = currentPath.filename().string();
        const string shaderFilePrefix    = shaderFileName.substr(0, shaderFileName.find('.'));
        const string shaderFileExtension = currentPath.extension().string();
        const string mapKey              = currentShaderDirectoryName + GLSL_SHADER_DIRECTORY_FILE_SEPARATOR + shaderFilePrefix;

        const auto& position = allShaderFilesInfo.find(mapKey);
        if (position == allShaderFilesInfo.end())
        {
          int shaderType = 0;
          if (shaderFileExtension == OpenGLAssetManager::getVertexShadersFileNameExtension())
          {
            shaderType = OpenGLAssetManager::VS;
          }
          else if (shaderFileExtension == OpenGLAssetManager::getTessellationControlShadersFileNameExtension())
          {
            shaderType = OpenGLAssetManager::TCS;
          }
          else if (shaderFileExtension == OpenGLAssetManager::getTessellationEvaluationShadersFileNameExtension())
          {
            shaderType = OpenGLAssetManager::TES;
          }
          else if (shaderFileExtension == OpenGLAssetManager::getGeometryShadersFileNameExtension())
          {
            shaderType = OpenGLAssetManager::GS;
          }
          else if (shaderFileExtension == OpenGLAssetManager::getFragmentShadersFileNameExtension())
          {
            shaderType = OpenGLAssetManager::FS;
          }
          else // if (shaderFileExtension == OpenGLAssetManager::getComputeShadersFileNameExtension())
          {
            shaderType = OpenGLAssetManager::CS;
          }
          allShaderFilesInfo[mapKey] = shaderType;
        }
        else
        {
          int shaderType = allShaderFilesInfo[mapKey];
          if (shaderFileExtension == OpenGLAssetManager::getVertexShadersFileNameExtension())
          {
            shaderType |= OpenGLAssetManager::VS;
          }
          else if (shaderFileExtension == OpenGLAssetManager::getTessellationControlShadersFileNameExtension())
          {
            shaderType |= OpenGLAssetManager::TCS;
          }
          else if (shaderFileExtension == OpenGLAssetManager::getTessellationEvaluationShadersFileNameExtension())
          {
            shaderType |= OpenGLAssetManager::TES;
          }
          else if (shaderFileExtension == OpenGLAssetManager::getGeometryShadersFileNameExtension())
          {
            shaderType |= OpenGLAssetManager::GS;
          }
          else if (shaderFileExtension == OpenGLAssetManager::getFragmentShadersFileNameExtension())
          {
            shaderType |= OpenGLAssetManager::FS;
          }
          else // if (shaderFileExtension == OpenGLAssetManager::getComputeShadersFileNameExtension())
          {
            shaderType |= OpenGLAssetManager::CS;
          }
          allShaderFilesInfo[mapKey] = shaderType;
        }
      }
    }

    return allShaderFilesInfo;
  }

  // core shader generator function
  inline void readShaderFile(const string& absolutePath, const string& pathName, const string& shaderName, const BitsetType& shaderTypes, int shaderTypeEnum, const string& shaderFileNameExtension, vector<list<string>>& allShaderFilesForPathName)
  {
    if (shaderTypes[BitManipulationFunctions::getLowestBitPositionOfPowerOfTwoNumber(shaderTypeEnum)])
    {
      list<string> shaderString = StdReadWriteFileFunctions::readTextFile(absolutePath + OPENGL_RENDERING_ENGINE_DIRECTORY + OpenGLAssetManager::getGLSLInternalDirectory() + pathName + "/" + shaderName + shaderFileNameExtension);
      if (!shaderString.empty()) allShaderFilesForPathName.emplace_back(shaderString);
    }
  }

  inline string encodeLine(const string& line)
  {
    switch (OpenGLAssetManager::CHARACTER_ENCODING_METHOD)
    {
      case OpenGLAssetManager::NONE:
        return line;

      case OpenGLAssetManager::BASE64:
        return Base64CompressorScrambler::encodeBase64String(line);

      case OpenGLAssetManager::FLIP_BITS:
        return Base64CompressorScrambler::flipString(line);

      case OpenGLAssetManager::FLIP_XOR_SWAP_BITS:
        return Base64CompressorScrambler::flipString(Base64CompressorScrambler::xorSwapString(line));

      case OpenGLAssetManager::BASE64_FLIP_XOR_SWAP_BITS:
        return Base64CompressorScrambler::encodeBase64String(Base64CompressorScrambler::flipString(Base64CompressorScrambler::xorSwapString(line)));

      case OpenGLAssetManager::BASE64_COMPRESSION:
        return Base64CompressorScrambler::encodeBase64String(Base64CompressorScrambler::compressString(line));

      case OpenGLAssetManager::BASE64_FLIP_XOR_SWAP_BITS_COMPRESSION:
        return Base64CompressorScrambler::encodeBase64String(Base64CompressorScrambler::flipString(Base64CompressorScrambler::xorSwapString(Base64CompressorScrambler::compressString(line))));

      default:
        return line;
    }
  }

  inline void writeGLSLShaderToHeaderFile(ostringstream& ss, const string& shaderName, const string& shaderVariableName, list<string>& shaderLines)
  {
    ss << "    const std::array<const char*, " << shaderLines.size() << "> " << shaderName << "_" << shaderVariableName << " =" << endl;
    ss << "    { {" << endl;

    bool isInFirstLine = true;
    for (auto& line : shaderLines)
    {
      if (!isInFirstLine)
      {
        ss << "," << endl;
      }

      replace(line.begin(), line.end(), '"', '\''); // replacement for special characters

      if (OpenGLAssetManager::CHARACTER_ENCODING_METHOD == OpenGLAssetManager::NONE)
      {
        if (line[line.size() - 1] == '\\') // replacement for macro line expansion
        {
          line += '\\';
        }
      }

      if (string::npos != line.find("//"))
      {
        replace(line.begin(), line.end(), '\\', '/'); // replacement for special characters only if it's a comment, else it can be used for the GLSL preprocessor
      }

      ss << "        \"" << encodeLine(line) << "\"";
      isInFirstLine = false;
    }

    ss << endl;
    ss << "    } };" << endl;
  }
}

void ShaderFilesGenerator::readAllShaderFiles(const string& absolutePath, const string& pathName, const string& shaderName, int shaderType)
{
  vector<list<string>> allShaderFilesForPathName;
  const BitsetType shaderTypes(shaderType);

  // Read Vertex Shader from file
  readShaderFile(absolutePath, pathName, shaderName, shaderTypes, OpenGLAssetManager::VS, OpenGLAssetManager::getVertexShadersFileNameExtension(), allShaderFilesForPathName);

  // Read Tessellation Control Shader from file
  readShaderFile(absolutePath, pathName, shaderName, shaderTypes, OpenGLAssetManager::TCS, OpenGLAssetManager::getTessellationControlShadersFileNameExtension(), allShaderFilesForPathName);

  // Read Tessellation Evaluation Shader from file
  readShaderFile(absolutePath, pathName, shaderName, shaderTypes, OpenGLAssetManager::TES, OpenGLAssetManager::getTessellationEvaluationShadersFileNameExtension(), allShaderFilesForPathName);

  // Read Geometry Shader from file
  readShaderFile(absolutePath, pathName, shaderName, shaderTypes, OpenGLAssetManager::GS, OpenGLAssetManager::getGeometryShadersFileNameExtension(), allShaderFilesForPathName);

  // Read Fragment Shader from file
  readShaderFile(absolutePath, pathName, shaderName, shaderTypes, OpenGLAssetManager::FS, OpenGLAssetManager::getFragmentShadersFileNameExtension(), allShaderFilesForPathName);

  // Read Compute Shader from file
  readShaderFile(absolutePath, pathName, shaderName, shaderTypes, OpenGLAssetManager::CS, OpenGLAssetManager::getComputeShadersFileNameExtension(), allShaderFilesForPathName);

  allShaderFiles_[Key(pathName + "_" + shaderName, shaderTypes)] = allShaderFilesForPathName;
}

void ShaderFilesGenerator::writeAllGLSLHeaderFilesForShaders(const string& absolutePath)
{
  if (allShaderFiles_.empty())
  {
    cout << "ShaderFilesGenerator error: No shader files parsed!" << endl;
    return;
  }

  const string fullPath = absolutePath + SHADER_FILES_GENERATOR_DIRECTORY + "include/" + OpenGLAssetManager::getGLSLInternalDirectory();
  // check first & (re-)create the shader files directory
  if (StdReadWriteFileFunctions::pathExists(fullPath))
  {
    StdReadWriteFileFunctions::removeDirectory(fullPath);
  }
  // check if (re-)creation went ok
  if (!StdReadWriteFileFunctions::createDirectory(fullPath))
  {
    cout << "ShaderFilesGenerator error: '" << fullPath << "' failed to be created!" << endl;
    return;
  }

  int index = 0;
  for (const auto& mapEntry : allShaderFiles_)
  {
    Key key                      = mapEntry.first;
    vector<list<string>> shaders = mapEntry.second;
    const size_t shadersSize     = shaders.size();

    ostringstream ss;
    ss << "/**" << endl;
    ss << "* PLEASE DO NOT EDIT." << endl;
    ss << "*" << endl;
    ss << "* Automatically generated by the ShaderFilesGenerator." << endl;
    ss << "*" << endl;
    ss << "* Defines the scrambled " << key.getFirst() << " header file." << endl;
    ss << "*" << endl;
    ss << GLSL_SHADER_AUTHOR << endl;
    ss << GLSL_SHADER_VERSION << endl;
    ss << "*/" << endl;
    ss << endl;
    ss << "#include <array>" << endl;
    ss << endl;
    ss << "namespace " << OPENGL_RENDERING_ENGINE_NAMESPACE << endl;
    ss << "{" << endl;
    ss << "  namespace " << GLSL_SHADER_FILES_NAMESPACE << endl;
    ss << "  {" << endl;
    index = 0;
    if (shadersSize)
    {
      if (key.getSecond().test(BitManipulationFunctions::getLowestBitPositionOfPowerOfTwoNumber(OpenGLAssetManager::VS)))
      {
        writeGLSLShaderToHeaderFile(ss, key.getFirst(), OpenGLAssetManager::getVertexShadersFileName(), shaders[index++]);
      }

      if (key.getSecond().test(BitManipulationFunctions::getLowestBitPositionOfPowerOfTwoNumber(OpenGLAssetManager::TCS)))
      {
        writeGLSLShaderToHeaderFile(ss, key.getFirst(), OpenGLAssetManager::getTessellationControlShadersFileName(), shaders[index++]);
      }

      if (key.getSecond().test(BitManipulationFunctions::getLowestBitPositionOfPowerOfTwoNumber(OpenGLAssetManager::TES)))
      {
        writeGLSLShaderToHeaderFile(ss, key.getFirst(), OpenGLAssetManager::getTessellationEvaluationShadersFileName(), shaders[index++]);
      }

      if (key.getSecond().test(BitManipulationFunctions::getLowestBitPositionOfPowerOfTwoNumber(OpenGLAssetManager::GS)))
      {
        writeGLSLShaderToHeaderFile(ss, key.getFirst(), OpenGLAssetManager::getGeometryShadersFileName(), shaders[index++]);
      }

      if (key.getSecond().test(BitManipulationFunctions::getLowestBitPositionOfPowerOfTwoNumber(OpenGLAssetManager::FS)))
      {
        writeGLSLShaderToHeaderFile(ss, key.getFirst(), OpenGLAssetManager::getFragmentShadersFileName(), shaders[index++]);
      }

      if (key.getSecond().test(BitManipulationFunctions::getLowestBitPositionOfPowerOfTwoNumber(OpenGLAssetManager::CS)))
      {
        writeGLSLShaderToHeaderFile(ss, key.getFirst(), OpenGLAssetManager::getComputeShadersFileName(), shaders[index++]);
      }
    }
    ss << "  }" << endl;
    ss << "}";

    // create the .glsl file
    StdReadWriteFileFunctions::writeTextFile(fullPath + key.getFirst() + GLSL_SHADER_APPEND_STRING_TO_FILENAME + GLSL_SHADER_FILES_EXTENSION, ss.str());
  }
}

void ShaderFilesGenerator::writeMainGLSLClass(const string& absolutePath)
{
  ostringstream ss;
  ss << "/**" << endl;
  ss << "* PLEASE DO NOT EDIT." << endl;
  ss << "*" << endl;
  ss << "* Automatically generated by the ShaderFilesGenerator." << endl;
  ss << "*" << endl;
  ss << "* Defines the " << GLSL_SHADER_FILES_MAIN_CLASS << ".h." << endl;
  ss << "*" << endl;
  ss << GLSL_SHADER_AUTHOR << endl;
  ss << GLSL_SHADER_VERSION << endl;
  ss << "*/" << endl;
  ss << endl;
  ss << "#pragma once" << endl;
  ss << endl;
  ss << "#ifndef __" << GLSL_SHADER_FILES_MAIN_CLASS << "_h" << endl;
  ss << "#define __" << GLSL_SHADER_FILES_MAIN_CLASS << "_h" << endl;
  ss << endl;
  ss << "#include <tuple>" << endl;
  ss << "#include <string>" << endl;
  ss << "#include <unordered_map>" << endl;
  ss << endl;
  ss << "namespace " << OPENGL_RENDERING_ENGINE_NAMESPACE << endl;
  ss << "{" << endl;
  ss << "  namespace " << GLSL_SHADER_FILES_NAMESPACE << endl;
  ss << "  {" << endl;
  ss << "    class " << GLSL_SHADER_FILES_MAIN_CLASS << " final" << endl;
  ss << "    {" << endl;
  ss << "    public:" << endl;
  ss << endl;
  ss << "      static " << GLSL_SHADER_FILES_MAIN_CLASS << "& getSingleton();" << endl;
  ss << "      std::tuple<const char* const*, std::size_t> getShader(const std::string& name);" << endl;
  ss << endl;
  ss << "      ~" << GLSL_SHADER_FILES_MAIN_CLASS << "() = default;" << endl;
  ss << "      "  << GLSL_SHADER_FILES_MAIN_CLASS << "(const " << GLSL_SHADER_FILES_MAIN_CLASS << "&) = delete;" << endl;
  ss << "      "  << GLSL_SHADER_FILES_MAIN_CLASS << "(" << GLSL_SHADER_FILES_MAIN_CLASS << "&&)      = delete;" << endl;
  ss << "      "  << GLSL_SHADER_FILES_MAIN_CLASS << "& operator=(const " << GLSL_SHADER_FILES_MAIN_CLASS << "&) = delete;" << endl;
  ss << "      "  << GLSL_SHADER_FILES_MAIN_CLASS << "& operator=(" << GLSL_SHADER_FILES_MAIN_CLASS << "&&)      = delete;" << endl;
  ss << endl;
  ss << "    private:" << endl;
  ss << endl;
  ss << "      " << GLSL_SHADER_FILES_MAIN_CLASS << "(); // not deleted for Singleton Design Pattern" << endl;
  ss << endl;
  ss << "      std::unordered_map<std::string, std::tuple<const char* const*, std::size_t>> " << GLSL_SHADER_FILES_MAIN_HEADER_MAP << ";" << endl;
  ss << "    };" << endl;
  ss << "  } // namespace GLSLShaderFiles" << endl;
  ss << "} // namespace OpenGLRenderingEngine" << endl;
  ss << endl;
  ss << "#endif // __" << GLSL_SHADER_FILES_MAIN_CLASS << "_h";

  // create the header file
  StdReadWriteFileFunctions::writeTextFile(absolutePath + SHADER_FILES_GENERATOR_DIRECTORY + "include/" + GLSL_SHADER_FILES_MAIN_CLASS + ".h", ss.str());

  ss.str(""); // clear buffer
  ss << "/**" << endl;
  ss << "* PLEASE DO NOT EDIT." << endl;
  ss << "*" << endl;
  ss << "* Automatically generated by the ShaderFilesGenerator." << endl;
  ss << "*" << endl;
  ss << "* Defines the " << GLSL_SHADER_FILES_MAIN_CLASS << ".cpp." << endl;
  ss << "*" << endl;
  ss << GLSL_SHADER_AUTHOR << endl;
  ss << GLSL_SHADER_VERSION << endl;
  ss << "*/" << endl;
  ss << endl;
  ss << "#include \"" << GLSL_SHADER_FILES_MAIN_CLASS << ".h\"" << endl;
  ss << "#include \"UtilityFunctions.h\"" << endl;
  Key key;
  for (const auto& mapEntry : allShaderFiles_)
  {
    key = mapEntry.first;
    ss << "#include \"" + OpenGLAssetManager::getGLSLInternalDirectory() << key.getFirst() << GLSL_SHADER_APPEND_STRING_TO_FILENAME + GLSL_SHADER_FILES_EXTENSION << "\"" << endl;
  }
  ss << endl;
  ss << "using namespace std;" << endl;
  ss << "using namespace " << OPENGL_RENDERING_ENGINE_NAMESPACE << "::" << GLSL_SHADER_FILES_NAMESPACE << ";" << endl;
  ss << endl;
  ss << GLSL_SHADER_FILES_MAIN_CLASS << "& " << GLSL_SHADER_FILES_MAIN_CLASS << "::getSingleton()" << endl;
  ss << "{" << endl;
  ss << "  static AllGLSLShaderFiles instance;" << endl;
  ss << "  return instance;" << endl;
  ss << "}" << endl;
  ss << endl;
  ss << "tuple<const char* const*, size_t> " << GLSL_SHADER_FILES_MAIN_CLASS << "::getShader(const string& name)" << endl;
  ss << "{" << endl;
  ss << "  const auto& position = " << GLSL_SHADER_FILES_MAIN_HEADER_MAP << ".find(name);" << endl;
  ss << "  if (position != " << GLSL_SHADER_FILES_MAIN_HEADER_MAP << ".end())" << endl;
  ss << "  {" << endl;
  ss << "    return " << GLSL_SHADER_FILES_MAIN_HEADER_MAP << "[name];"<< endl;
  ss << "  }" << endl;
  ss << endl;
  ss << "  DebugConsole_consoleOutLine(\"Could not find shader name:\\n\", name);" << endl;
  ss << endl;
  ss << "  return make_tuple(nullptr, 0);"<< endl;
  ss << "}" << endl;
  ss << endl;
  ss << GLSL_SHADER_FILES_MAIN_CLASS << "::" << GLSL_SHADER_FILES_MAIN_CLASS << "()" << endl;
  ss << "{" << endl;
  for (const auto& mapEntry : allShaderFiles_)
  {
    key = mapEntry.first;

    if (key.getSecond().test(BitManipulationFunctions::getLowestBitPositionOfPowerOfTwoNumber(OpenGLAssetManager::VS)))
    {
      const string vertexShaderName = key.getFirst() + "_" + OpenGLAssetManager::getVertexShadersFileName();
      ss << "  " << GLSL_SHADER_FILES_MAIN_HEADER_MAP << ".emplace(\"" << key.getFirst() << "::" << OpenGLAssetManager::getVertexShadersFileName() << "\", make_tuple(" << vertexShaderName << ".data(), " << vertexShaderName << ".size()));" << endl;
    }

    if (key.getSecond().test(BitManipulationFunctions::getLowestBitPositionOfPowerOfTwoNumber(OpenGLAssetManager::TCS)))
    {
      const string tessellationControlShaderName = key.getFirst() + "_" + OpenGLAssetManager::getTessellationControlShadersFileName();
      ss << "  " << GLSL_SHADER_FILES_MAIN_HEADER_MAP << ".emplace(\"" << key.getFirst() << "::" << OpenGLAssetManager::getTessellationControlShadersFileName() << "\", make_tuple(" << tessellationControlShaderName << ".data(), " << tessellationControlShaderName << ".size()));" << endl;
    }

    if (key.getSecond().test(BitManipulationFunctions::getLowestBitPositionOfPowerOfTwoNumber(OpenGLAssetManager::TES)))
    {
      const string tessellationEvaluationShaderName = key.getFirst() + "_" + OpenGLAssetManager::getTessellationEvaluationShadersFileName();
      ss << "  " << GLSL_SHADER_FILES_MAIN_HEADER_MAP << ".emplace(\"" << key.getFirst() << "::" << OpenGLAssetManager::getTessellationEvaluationShadersFileName() << "\", make_tuple(" << tessellationEvaluationShaderName << ".data(), " << tessellationEvaluationShaderName << ".size()));" << endl;
    }

    if (key.getSecond().test(BitManipulationFunctions::getLowestBitPositionOfPowerOfTwoNumber(OpenGLAssetManager::GS)))
    {
      const string geometryShaderName = key.getFirst() + "_" + OpenGLAssetManager::getGeometryShadersFileName();
      ss << "  " << GLSL_SHADER_FILES_MAIN_HEADER_MAP << ".emplace(\"" << key.getFirst() << "::" << OpenGLAssetManager::getGeometryShadersFileName() << "\", make_tuple(" << geometryShaderName << ".data(), " << geometryShaderName << ".size()));" << endl;
    }

    if (key.getSecond().test(BitManipulationFunctions::getLowestBitPositionOfPowerOfTwoNumber(OpenGLAssetManager::FS)))
    {
      const string fragmentShaderName = key.getFirst() + "_" + OpenGLAssetManager::getFragmentShadersFileName();
      ss << "  " << GLSL_SHADER_FILES_MAIN_HEADER_MAP << ".emplace(\"" << key.getFirst() << "::" << OpenGLAssetManager::getFragmentShadersFileName() << "\", make_tuple(" << fragmentShaderName << ".data(), " << fragmentShaderName << ".size()));" << endl;
    }

    if (key.getSecond().test(BitManipulationFunctions::getLowestBitPositionOfPowerOfTwoNumber(OpenGLAssetManager::CS)))
    {
      const string computeShaderName = key.getFirst() + "_" + OpenGLAssetManager::getComputeShadersFileName();
      ss << "  " << GLSL_SHADER_FILES_MAIN_HEADER_MAP << ".emplace(\"" << key.getFirst() << "::" << OpenGLAssetManager::getComputeShadersFileName() << "\", make_tuple(" << computeShaderName << ".data(), " << computeShaderName << ".size()));" << endl;
    }
  }
  ss << "}";

  // create the cpp file
  StdReadWriteFileFunctions::writeTextFile(absolutePath + SHADER_FILES_GENERATOR_DIRECTORY + "src/" + GLSL_SHADER_FILES_MAIN_CLASS + ".cpp", ss.str());
}

void ShaderFilesGenerator::generateAllShaderFilesCode(const string& absolutePath)
{
  unordered_map<string, int> allShaderFilesInfo = autoParseAllShaderFiles(absolutePath);
  for (const auto& mapEntry : allShaderFilesInfo)
  {
    const string shaderDirectoryAndFileName = mapEntry.first;
    const string shaderDirectoryName        = shaderDirectoryAndFileName.substr(0, shaderDirectoryAndFileName.find(GLSL_SHADER_DIRECTORY_FILE_SEPARATOR));
    const string shaderFileName             = shaderDirectoryAndFileName.substr(shaderDirectoryAndFileName.find(GLSL_SHADER_DIRECTORY_FILE_SEPARATOR) + GLSL_SHADER_DIRECTORY_FILE_SEPARATOR.length(), string::npos);

    readAllShaderFiles(absolutePath, shaderDirectoryName, shaderFileName, mapEntry.second);
  }

  // Write all shader files header generation
  writeAllGLSLHeaderFilesForShaders(absolutePath);
  writeMainGLSLClass(absolutePath);
}

int main(int argc, char* argv[])
{
  cout << endl;
  string path = "";
  for (int i = 1; i < argc; ++i)
  {
    const string currentOption = string(argv[i]);
    if ((currentOption == "-h") || (currentOption == "-help"))
    {
      giveUsage(argv[0], EXIT_SUCCESS);
    }
    else if (currentOption == "-path")
    {
      if ((i + 1) >= argc) giveUsage(argv[0]);
      path = argv[i + 1];
      if (!StringAuxiliaryFunctions::endsWith(path, "/"))
      {
        path += "/";
      }
      cout << "ShaderFilesGenerator starting with path:\n'" << path << "'" << endl;
      ++i; // increment counter to skip the i + 1 option
    }
    else
    {
      giveUsage(argv[0]);
    }
  }

  AccurateTimer timer;
  timer.startTimer();

  ShaderFilesGenerator shaderGenerator;
  shaderGenerator.generateAllShaderFilesCode(path);

  timer.stopTimer();
  cout << "\nShaderFilesGenerator processing time: " << timer.getElapsedTimeInMilliSecs() << " ms." << endl;

  return EXIT_SUCCESS;
}