/*

Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
Released Under a Simplified BSD (FreeBSD) License
for academic, personal & non-commercial use.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the author and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.

A Commercial License is also available for commercial use with
special restrictions and obligations at a one-off fee. See links at:
1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

#pragma once

#ifndef __ShaderFilesHeaderGenerator_h
#define __ShaderFilesHeaderGenerator_h

#include "ModuleDLL.h"
#include "OpenGLAssetManager.h"
#include <string>
#include <map>
#include <bitset>
#include <vector>
#include <list>

/** @brief Namespace OpenGLRenderingEngine for the OpenGL rendering.
* @author Thanos Theo, 2018
* @version 14.0.0.0
*/
namespace OpenGLRenderingEngine
{
  /** @brief This class includes shader files header/implementation generator related functionality.
  * @author Thanos Theo, 2009-2018
  * @version 14.0.0.0
  */
  class SHADER_FILES_GENERATOR_MODULE_API ShaderFilesGenerator final // prevent inheritance with C++11 keyword
  {
  public:

    ShaderFilesGenerator()  = default;
    ~ShaderFilesGenerator() = default;
    ShaderFilesGenerator(const ShaderFilesGenerator&) = delete;
    ShaderFilesGenerator(ShaderFilesGenerator&&)      = delete;
    ShaderFilesGenerator& operator=(const ShaderFilesGenerator&) = delete;
    ShaderFilesGenerator& operator=(ShaderFilesGenerator&&)      = delete;

    void generateAllShaderFilesCode(const std::string& absolutePath);

  private:

    /// core shader generator function
    void readAllShaderFiles(const std::string& absolutePath, const std::string& pathName, const std::string& shaderName, int shaderType);
    /// core shader generator function
    void writeAllGLSLHeaderFilesForShaders(const std::string& absolutePath);
    /// core shader generator function
    void writeMainGLSLClass(const std::string& absolutePath);

    using BitsetType = std::bitset<OpenGLAssetManager::NUMBER_OF_TOTAL_SHADER_TYPES>;
    class Key : public std::unary_function<Key, bool>
    {
      public:

        Key()  = default;
        Key(const std::string& first, const BitsetType& second) noexcept : first_(first) , second_(second) {}
        ~Key() = default;
        Key(const Key&) = default;
        Key(Key&&)      = default;
        Key& operator=(const Key&) = default;
        Key& operator=(Key&&)      = default;

        bool operator<(const Key& other) const { return first_ < other.first_; }
        const std::string& getFirst()    const { return first_;                }
        const BitsetType& getSecond()    const { return second_;               }

      private:

        std::string first_ = "";
        BitsetType second_ = 0;
    };

    /// standard map for saving shader file names sorted
    std::map<Key, std::vector<std::list<std::string>>> allShaderFiles_;
  };
} // namespace OpenGLRenderingEngine

#endif // __ShaderFilesHeaderGenerator_h