#include "TestAbstractBase.h"
#include "OpenGLAssetManager.h"
#include "OpenGLDriverInfo.h"
#include "OpenGLEulerCamera.h"
#include "OpenGLUtilityFunctions.h"
#include "OpenGLILTexture.h"
#include "UtilityFunctions.h"
#include "lodepng.h"
#include <cstdint>
#include <vector>
#include <sstream>

using namespace std;
using namespace OpenGLRenderingEngine;
using namespace OpenGLRenderingEngineTests;
using namespace OpenGLRenderingEngine::OpenGLUtilityFunctions;
using namespace Utils::UtilityFunctions;

TestAbstractBase::TestAbstractBase()
{
  useUIInformationDisplayList = glGenLists(1);

  performAllGLInitializations();
}

TestAbstractBase::TestAbstractBase(int thisScreenWidth, int thisScreenHeight, const string& thisModelFileName, const string& thisModelLoaderDescriptorFileName, bool thisMultisample)
  : screenWidth(thisScreenWidth)
  , screenHeight(thisScreenHeight)
  , modelFileName(thisModelFileName)
  , modelLoaderDescriptorFileName(thisModelLoaderDescriptorFileName)
  , multisample(thisMultisample)
{
  useUIInformationDisplayList = glGenLists(1);

  performAllGLInitializations();
}

TestAbstractBase::TestAbstractBase(int thisScreenWidth, int thisScreenHeight, const string& thisTextureFileName, const string& thisModelFileName, const string& thisModelLoaderDescriptorFileName, bool thisMultisample)
  : screenWidth(thisScreenWidth)
  , screenHeight(thisScreenHeight)
  , textureFileName(thisTextureFileName)
  , modelFileName(thisModelFileName)
  , modelLoaderDescriptorFileName(thisModelLoaderDescriptorFileName)
  , multisample(thisMultisample)
{
  useUIInformationDisplayList = glGenLists(1);

  performAllGLInitializations();
}

void TestAbstractBase::writeScreenshotToFile(bool useDevIL) const
{
  const size_t arraySize = 3 * screenWidth * screenHeight; // GL_RGB pixels for framebuffer
  vector<GLubyte> imagePixels;
  imagePixels.resize(arraySize);

  // glReadBuffer(GL_BACK); // mode is initially GL_FRONT in single-buffered configurations and GL_BACK in double-buffered configurations
  glPixelStorei(GL_PACK_ALIGNMENT, 1);
  glReadPixels(0, 0, screenWidth, screenHeight, GL_RGB, GL_UNSIGNED_BYTE, imagePixels.data());

  time_t currentTime   = time(nullptr);
  tm* currentLocalTime = localtime(&currentTime);
  ostringstream filenameStream;
  filenameStream << "Screenshot Date " << (1900 + currentLocalTime->tm_year) << "-" << (currentLocalTime->tm_mon + 1) << "-" << currentLocalTime->tm_mday << " Time " << currentLocalTime->tm_hour << " " << currentLocalTime->tm_min << " " << currentLocalTime->tm_sec;

  if (useDevIL)
  {
    OpenGLILTexture::saveILImage(OpenGLAssetManager::getImagesDefaultDirectory(), filenameStream.str(), OpenGLAssetManager::getDefaultScreenshotFormat(), 3, screenWidth, screenHeight, imagePixels.data());
  }
  else
  {
    GLAuxiliaryFunctions::flipPixels(3, screenWidth, screenHeight, imagePixels.data());
    if (!StdReadWriteFileFunctions::pathExists(OpenGLAssetManager::getImagesDefaultDirectory()))
    {
      StdReadWriteFileFunctions::createDirectory(OpenGLAssetManager::getImagesDefaultDirectory());
    }
    const string screenshotPngFilename = string(OpenGLAssetManager::getImagesDefaultDirectory() + "/" + filenameStream.str() + "." + OpenGLAssetManager::getDefaultScreenshotFormat());
    uint32_t lodepngError = lodepng::encode(screenshotPngFilename, imagePixels, screenWidth, screenHeight, LodePNGColorType::LCT_RGB);
    if (lodepngError) // check if the image was saved successfully (ie: saving succeeded)
    {
      DebugConsole_consoleOutLine("Lodepng encoder error ", lodepngError, ": ", lodepng_error_text(lodepngError), " for file: ", screenshotPngFilename, ".\nNow aborting encoding the file.");
    }
    else
    {
      DebugConsole_consoleOutLine("writeScreenshotToFile() saved image with name '", screenshotPngFilename, "'.");
    }
  }
}

void TestAbstractBase::releaseAllGLResources()
{
  // just to clear VRAM memory
  if (glIsList(useUIInformationDisplayList))
  {
    glDeleteLists(useUIInformationDisplayList, 1);
  }

  if (openGLDriverInfo)
  {
    delete openGLDriverInfo;
    openGLDriverInfo = nullptr;
  }

  if (openGLEulerCamera)
  {
    delete openGLEulerCamera;
    openGLEulerCamera = nullptr;
  }
}

void TestAbstractBase::performAllGLInitializations()
{
  // initialization of the OpenGL info
  openGLDriverInfo = new OpenGLDriverInfo();
  openGLDriverInfo->printGLInfo();

  openGLEulerCamera = new OpenGLEulerCamera(45.0);

  // enable/disable features
  glClearDepth(1.0);
  glEnable(GL_DEPTH_TEST);

  if (multisample)
  {
    glEnable(GL_MULTISAMPLE);
    if (openGLDriverInfo->isNvidia())
    {
      glHint(GL_MULTISAMPLE_FILTER_HINT_NV, GL_NICEST);
    }
  }
  
  GLAuxiliaryFunctions::prepareHighQualityRendering(openGLDriverInfo->isNvidia());

  // for line anti-aliasing and blending options usage only
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  glEnable(GL_ALPHA_TEST);
  glAlphaFunc(GL_GREATER, 0); // only render if alpha > 0
  // glDepthMask(false);
}