#include "SuperQuadricsTest.h"
#include "SuperQuadricShapesProducer.h"
#include "OpenGLAssetManager.h"
#include "OpenGLDriverInfo.h"
#include "OpenGLEulerCamera.h"
#include "OpenGLUtilityFunctions.h"
#include "OpenGLModelAmbientLight.h"
#include "OpenGLLight.h"
#include "OpenGLMaterial.h"
#include "OpenGLShaderSurfaceLightingModels.h"
#include "OpenGLShaderSurfaceLightingLODModels.h"
#include "OpenGLFrameBufferObject.h"
#include "OpenGLShaderCompileAndLink.h"
#include "OpenGLShaderGLSLPreProcessorCommands.h"
#include "OpenGLILTexture.h"
#include "MathConstants.h"
#include "UtilityFunctions.h"
#include <cmath>
#include <array>
#include <iomanip>

using namespace std;
using namespace SuperQuadrics;
using namespace OpenGLRenderingEngine;
using namespace OpenGLRenderingEngineTests;
using namespace OpenGLRenderingEngine::OpenGLUtilityFunctions;
using namespace Utils::UtilityFunctions;

namespace // anonymous namespace used instead of deprecated 'static' keyword used for cpp variable locality
{
  const int DEFAULT_TESSELLATION                  = 38;
  const int UPPER_LIMIT_TESSELLATION              = 50;
  const int DEFAULT_GEOMETRY_REPETITION           = 1;
  const int UPPER_LIMIT_GEOMETRY_REPETITION       = 250;
  const array<GLfloat, 4> BLACK_BACKGROUND_COLOR  = { { 0.0f, 0.0f, 0.0f, 1.0f } };                // double braces because we initialize an array inside an std::array object
  const array<GLfloat, 4> WHITE_BACKGROUND_COLOR  = { { 1.0f, 1.0f, 1.0f, 1.0f } };                // double braces because we initialize an array inside an std::array object
  const array<GLfloat, 4> STENCIL_COLOR           = { { 0.929524f, 0.796542f, 0.178823f, 1.0f } }; // double braces because we initialize an array inside an std::array object
  const array<GLfloat, 4> DEFAULT_SHAPE_COLOR     = { { 0.25f,  0.25f, 0.25f, 1.0f } };            // double braces because we initialize an array inside an std::array object
  const GLfloat FOG_DENSITY                       = 0.015f;
  const array<GLfloat, 4> MODEL_AMBIENT_ARRAY     = { { 0.4f, 0.4f, 0.4f, 1.0f } };                // double braces because we initialize an array inside an std::array object
  const array<GLfloat, 4> LIGHT_AMBIENT_ARRAY     = { { 0.0f, 0.0f, 0.0f, 1.0f } };                // double braces because we initialize an array inside an std::array object
  const array<GLfloat, 4> LIGHT_DIFFUSE_ARRAY     = { { 1.0f, 1.0f, 1.0f, 1.0f } };                // double braces because we initialize an array inside an std::array object
  const array<GLfloat, 4> LIGHT_SPECULAR_ARRAY    = { { 1.0f, 1.0f, 1.0f, 1.0f } };                // double braces because we initialize an array inside an std::array object
  const array<GLfloat, 4> NO_LIGHT_SPECULAR_ARRAY = { { 0.0f, 0.0f, 0.0f, 1.0f } };                // double braces because we initialize an array inside an std::array object
  const GLfloat LIGHT_SPECULAR_SHININESS          = 25.0f;
  const float PI_4                                = Utils::PI_FLT / 4.0f;
}

SuperQuadricsTest::SuperQuadricsTest(int screenWidth, int screenHeight, const string& textureFileName, bool multisample) noexcept
  : TestAbstractBase(screenWidth, screenHeight, textureFileName, "", "", multisample)
  , slices(DEFAULT_TESSELLATION)
  , segments(DEFAULT_TESSELLATION)
  , geometryRepetition(DEFAULT_GEOMETRY_REPETITION)
{
  allSuperQuadricsDataDisplayList = glGenLists(1);
  prepareFog();
  prepareLighting();
  prepareShaders();
  if (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader())
  {
    prepareLODShaders();
  }
  if (openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders())
  {
    prepareGBufferEffectsShaders();
  }
  prepareFXAA_AntialiasShaders();
  prepareTexture();
  if (openGLDriverInfo->supports_GL_EXT_framebuffer_object())
  {
    if (openGLDriverInfo->supports_GL_EXT_packed_depth_stencil())
    {
      prepareDepthStencilFBO();
    }
    if (openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders())
    {
      prepareGBufferEffectsFBOs();
      prepareHSSAOData();
    }
    prepareFXAA_AntialiasFBO();
  }
  clearScreen();
}

void SuperQuadricsTest::OpenGLShaderGBufferEffectsBlurXY::initializeShaderProgram()
{
  openGLShaderGLSLPreProcessorCommands_->addHighestGLVersionDefinition();
  openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("GPU_SHADER4_COMPATIBILITY_CONDITION", openGLDriverInfo_->supports_GL_EXT_gpu_shader4());
  openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("GPU_SHADER5_COMPATIBILITY_CONDITION", openGLDriverInfo_->supports_GL_ARB_gpu_shader5());
  openGLShaderGLSLPreProcessorCommands_->addDefinitionForStartingLine();

  openGLShaderCompileAndLink_->addShaderLibraryToProgram("SurfaceLightingModels", "SurfaceLightingModels", OpenGLAssetManager::FS);
  openGLShaderCompileAndLink_->addShaderLibraryToProgram("GBufferEffects", "ApplyFullScreenQuad",          OpenGLAssetManager::VS);
  openGLShaderCompileAndLink_->addShaderLibraryToProgram("GBufferEffects", "ApplyGBufferEffectsBlurXY",    OpenGLAssetManager::FS);
  openGLShaderCompileAndLink_->linkShaderProgram();

  GLAuxiliaryFunctions_checkGLError();
}

void SuperQuadricsTest::OpenGLShaderGBufferEffectsHSSAO::initializeShaderProgram()
{
  openGLShaderGLSLPreProcessorCommands_->addHighestGLVersionDefinition();
  openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("GPU_SHADER4_COMPATIBILITY_CONDITION", openGLDriverInfo_->supports_GL_EXT_gpu_shader4());
  openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("GPU_SHADER5_COMPATIBILITY_CONDITION", openGLDriverInfo_->supports_GL_ARB_gpu_shader5());
  openGLShaderGLSLPreProcessorCommands_->addDefinitionForStartingLine();

  openGLShaderCompileAndLink_->addShaderLibraryToProgram("SurfaceLightingModels", "SurfaceLightingModels", OpenGLAssetManager::FS);
  openGLShaderCompileAndLink_->addShaderLibraryToProgram("GBufferEffects", "ApplyFullScreenQuad",          OpenGLAssetManager::VS);
  openGLShaderCompileAndLink_->addShaderLibraryToProgram("GBufferEffects", "ApplyGBufferEffectsHSSAO",     OpenGLAssetManager::FS);
  openGLShaderCompileAndLink_->linkShaderProgram();

  GLAuxiliaryFunctions_checkGLError();
}

void SuperQuadricsTest::OpenGLShaderGBufferEffects::initializeShaderProgram()
{
  openGLShaderGLSLPreProcessorCommands_->addHighestGLVersionDefinition();
  openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("GPU_SHADER4_COMPATIBILITY_CONDITION", openGLDriverInfo_->supports_GL_EXT_gpu_shader4());
  openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("GPU_SHADER5_COMPATIBILITY_CONDITION", openGLDriverInfo_->supports_GL_ARB_gpu_shader5());
  openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("DEPTH_OF_FIELD_DEBUG_MODE", DEPTH_OF_FIELD_DEBUG_MODE);
  openGLShaderGLSLPreProcessorCommands_->addDefinitionForStartingLine();

  openGLShaderCompileAndLink_->addShaderLibraryToProgram("CommonFunctions", "CommonFunctions",             OpenGLAssetManager::FS);
  openGLShaderCompileAndLink_->addShaderLibraryToProgram("SurfaceLightingModels", "SurfaceLightingModels", OpenGLAssetManager::FS);
  openGLShaderCompileAndLink_->addShaderLibraryToProgram("GBufferEffects", "ApplyFullScreenQuad",          OpenGLAssetManager::VS);
  openGLShaderCompileAndLink_->addShaderLibraryToProgram("GBufferEffects", "ApplyGBufferEffects",          OpenGLAssetManager::FS);
  openGLShaderCompileAndLink_->linkShaderProgram();

  GLAuxiliaryFunctions_checkGLError();
}

void SuperQuadricsTest::OpenGLShaderFXAA_Antialias::initializeShaderProgram()
{
  openGLShaderGLSLPreProcessorCommands_->addHighestGLVersionDefinition();
  openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("OPENGL_330_SUPPORTED", openGLDriverInfo_->supports330Shaders());
  openGLShaderGLSLPreProcessorCommands_->addDefinitionForStartingLine();

  openGLShaderCompileAndLink_->addShaderLibraryToProgram("FXAA_Antialias", "ApplyFXAA_Antialias", OpenGLAssetManager::VS | OpenGLAssetManager::FS);
  openGLShaderCompileAndLink_->linkShaderProgram();

  GLAuxiliaryFunctions_checkGLError();
}

void SuperQuadricsTest::prepareFog() const
{
  glFogi(GL_FOG_MODE, GL_EXP2);        // exponential fog mode
  glFogf(GL_FOG_DENSITY, FOG_DENSITY); // fog density value
}

void SuperQuadricsTest::prepareLighting()
{
  // enable light sources
  glEnable(GL_LIGHTING);
  glEnable(GL_VERTEX_PROGRAM_TWO_SIDE); // must manually populate the gl_BackColor state variable in the vertex shader for Intel GPUs
  glEnable(GL_NORMALIZE);

  glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_TRUE); // GL_TRUE = infinite distance from scene, optimization trick for OpenGL lighting calculations
  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);     // Enable two-sided lighting
  glShadeModel(useGouraudShading ? GL_SMOOTH : GL_FLAT);

  openGLModelAmbientLight = new OpenGLModelAmbientLight();
  openGLModelAmbientLight->setModelAmbientColor(MODEL_AMBIENT_ARRAY.data());
  openGLModelAmbientLight->enable();

  // range -2  to -+2 for all lights positions
  for (size_t i = 0; i < NUMBER_OF_LIGHTS; ++i)
  {
    allLightsPositions[i][0] = (i & 1) ?  2.0f : -2.0f; // light position x
    allLightsPositions[i][1] = (i & 2) ? -2.0f :  2.0f; // light position y
    allLightsPositions[i][2] = (i & 3) ?  2.0f : -2.0f; // light position z

    openGLLights[i] = new OpenGLLight();
    openGLLights[i]->setAmbientColor(LIGHT_AMBIENT_ARRAY.data());
    openGLLights[i]->setDiffuseColor(LIGHT_DIFFUSE_ARRAY.data());
    openGLLights[i]->setSpecularColor(LIGHT_SPECULAR_ARRAY.data());
    if (usePositionalLights)
    {
      openGLLights[i]->setPosition(allLightsPositions[i]);
    }
    else
    {
      openGLLights[i]->setDirection(allLightsPositions[i]);
    }
    openGLLights[i]->setConstantAttenuation(1.0f);
    openGLLights[i]->setLinearAttenuation(0.5f);
    openGLLights[i]->setQuadraticAttenuation(0.250f);
    openGLLights[i]->enable(GL_LIGHT0 + int(i), usePositionalLights != 0);
  }

  openGLMaterial = new OpenGLMaterial(USE_COLOR_MATERIAL);
  if (useColoredObjects)
  {
    openGLMaterial->setSpecularColor(LIGHT_SPECULAR_ARRAY.data());
    openGLMaterial->setShininess(LIGHT_SPECULAR_SHININESS);
  }
  else
  {
    openGLMaterial->setSpecularColor(NO_LIGHT_SPECULAR_ARRAY.data());
    openGLMaterial->setShininess(0.0f);
  }
  openGLMaterial->enable();
}

void SuperQuadricsTest::updateLighting()
{
  glShadeModel(useGouraudShading ? GL_SMOOTH : GL_FLAT);

  if (useColoredObjects)
  {
    openGLMaterial->setSpecularColor(LIGHT_SPECULAR_ARRAY.data());
    openGLMaterial->setShininess(LIGHT_SPECULAR_SHININESS);
  }
  else
  {
    openGLMaterial->setSpecularColor(NO_LIGHT_SPECULAR_ARRAY.data());
    openGLMaterial->setShininess(0.0f);
  }
  openGLMaterial->enable();

  sinLightAngle += (PI_4 / 75.0f);
  cosLightAngle += (PI_4 / 75.0f);
  float sinLightAngleValue = sin(sinLightAngle);
  float cosLightAngleValue = cos(cosLightAngle);

  if (usePositionalLights)
  {
    for (size_t i = 0; i < NUMBER_OF_LIGHTS; ++i)
    {
      openGLLights[i]->setPosition(allLightsPositions[i]);
      openGLLights[i]->update(GL_LIGHT0 + int(i), true, (i & 1) ? sinLightAngleValue : cosLightAngleValue,
                                                        (i & 2) ? cosLightAngleValue : sinLightAngleValue,
                                                        (i & 3) ? sinLightAngleValue : cosLightAngleValue);
    }
  }
  else
  {
    for (size_t i = 0; i < NUMBER_OF_LIGHTS; ++i)
    {
      openGLLights[i]->setDirection(allLightsPositions[i]);
      openGLLights[i]->update(GL_LIGHT0 + int(i), false, (i & 1) ? sinLightAngleValue : cosLightAngleValue,
                                                         (i & 2) ? cosLightAngleValue : sinLightAngleValue,
                                                         (i & 3) ? sinLightAngleValue : cosLightAngleValue);
    }
  }
}

void SuperQuadricsTest::prepareShaders()
{
  openGLShaderSurfaceLightingModels                       = new OpenGLShaderSurfaceLightingModels(openGLDriverInfo, false, false);
  openGLShaderSurfaceLightingModelsWithGeometry           = new OpenGLShaderSurfaceLightingModels(openGLDriverInfo,  true, false);
  openGLShaderSurfaceLightingModelsWithGeometryAndNormals = new OpenGLShaderSurfaceLightingModels(openGLDriverInfo,  true,  true);
}

void SuperQuadricsTest::prepareLODShaders()
{
  openGLShaderSurfaceLightingLODModels                       = new OpenGLShaderSurfaceLightingLODModels(openGLDriverInfo, false, false);
  openGLShaderSurfaceLightingLODModelsWithGeometry           = new OpenGLShaderSurfaceLightingLODModels(openGLDriverInfo,  true, false);
  openGLShaderSurfaceLightingLODModelsWithGeometryAndNormals = new OpenGLShaderSurfaceLightingLODModels(openGLDriverInfo,  true,  true);
}

void SuperQuadricsTest::prepareGBufferEffectsShaders()
{
  openGLShaderGBufferEffectsBlurXY = new OpenGLShaderGBufferEffectsBlurXY(openGLDriverInfo);
  openGLShaderGBufferEffectsHSSAO = new OpenGLShaderGBufferEffectsHSSAO(openGLDriverInfo);
  openGLShaderGBufferEffects = new OpenGLShaderGBufferEffects(openGLDriverInfo);
}

void SuperQuadricsTest::prepareFXAA_AntialiasShaders()
{
  openGLShaderFXAA_Antialias = new OpenGLShaderFXAA_Antialias(openGLDriverInfo);
}

void SuperQuadricsTest::prepareTexture()
{
  openGLILTexture = new OpenGLILTexture(openGLDriverInfo->supports330Shaders(), openGLDriverInfo->supports_GL_EXT_texture_filter_anisotropic(), openGLDriverInfo->getMaxTextureMaxAnisotropy());
  openGLILTexture->createILTexture(OpenGLAssetManager::getTexturesDefaultDirectory(), textureFileName);
}

void SuperQuadricsTest::prepareDepthStencilFBO()
{
  openGLFrameBufferObjectForHighQualityOutline = new OpenGLFrameBufferObject(openGLDriverInfo);
  openGLFrameBufferObjectForHighQualityOutline->setUseTexture(true);
  openGLFrameBufferObjectForHighQualityOutline->setDepthStencilRenderBuffer(true);
}

void SuperQuadricsTest::initDepthStencilFBO() const
{
  DebugConsole_consoleOutLine("Depth Stencil FBO Render-to-texture ratioFactor:\t  ", DEPTH_STENCIL_RATIO_FACTOR);
  DebugConsole_consoleOutLine("Depth Stencil FBO Render-to-texture multiplicationFactor: ", DEPTH_STENCIL_MULTIPLICATION_FACTOR);
  openGLFrameBufferObjectForHighQualityOutline->disposeFrameBufferObjectResources();
  openGLFrameBufferObjectForHighQualityOutline->initFrameBufferObjectResources(DEPTH_STENCIL_MULTIPLICATION_FACTOR * screenWidth / DEPTH_STENCIL_RATIO_FACTOR, DEPTH_STENCIL_MULTIPLICATION_FACTOR * screenHeight / DEPTH_STENCIL_RATIO_FACTOR, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
}

void SuperQuadricsTest::initSuperQuadricsShapes()
{
  const size_t arrayLength      = 6 * 3 * slices * segments; // 6 -> 2 * 3 triangles vertices, 3 for 3D vertex points
  const size_t totalArrayLength = geometryRepetition * SuperQuadricShapesProducer::NUMBER_OF_SHAPES * arrayLength;
  vertices.clear();
  vertices.resize(totalArrayLength);
  normals.clear();
  normals.resize(totalArrayLength);
  texCoords.clear();
  texCoords.resize(2 * (totalArrayLength) / 3); // 2 / 3 for 2D texture coords

  size_t arrayLengthIndex     = 0;
  double shrinkGeometryFactor = 0.0;
  for (int i = 0; i < geometryRepetition; ++i)
  {
    for (size_t j = 0; j < SuperQuadricShapesProducer::NUMBER_OF_SHAPES; ++j)
    {
      shrinkGeometryFactor = double(i + 1) / double(geometryRepetition);
      SuperQuadricShapesProducer::createSuperQuadricShape(slices, segments, shrinkGeometryFactor, (useGeometryExplosion != 0), arrayLengthIndex, SuperQuadricShapesProducer::SuperQuadricShapeObjects(j), vertices, normals, texCoords);
      arrayLengthIndex += arrayLength;
    }
  }
}

void SuperQuadricsTest::prepareGBufferEffectsFBOs()
{
  // prepare GBufferEffects FBO for pass 1
  openGLFrameBufferObjectForGBufferPass1 = new OpenGLFrameBufferObject(openGLDriverInfo);
  openGLFrameBufferObjectForGBufferPass1->setUseTexture(true);
  openGLFrameBufferObjectForGBufferPass1->setDepthRenderBuffer(true);
  // Warning: has to be an unsignet integer FBO texture for the GBuffer to work ok!
  openGLFrameBufferObjectForGBufferPass1->setTextureFormat1(GL_RGB32UI);
  openGLFrameBufferObjectForGBufferPass1->setTextureFormat2(GL_RGB_INTEGER);
  openGLFrameBufferObjectForGBufferPass1->setTextureFormatType(GL_UNSIGNED_INT);

  // prepare GBufferEffects FBO for pass 2
  openGLFrameBufferObjectRenderPass2 = new OpenGLFrameBufferObject(openGLDriverInfo);
  openGLFrameBufferObjectRenderPass2->setUseTexture(true);

  // prepare GBufferEffects FBO for pass 3
  openGLFrameBufferObjectRenderPass3 = new OpenGLFrameBufferObject(openGLDriverInfo);
  openGLFrameBufferObjectRenderPass3->setUseTexture(true);
}

void SuperQuadricsTest::prepareHSSAOData()
{
  // prepare rotationTextureData
  const int rotationTextureDataSize = 64 * 64;
  rotationTextureData.resize(rotationTextureDataSize << 2);
  float randomAngle = float(random.getRandomFloat()) * 4.0f * PI_4 * 2.0f;

  int index = 0;
  for (int i = 0; i < rotationTextureDataSize; ++i)
  {
    rotationTextureData[index++] = cos(randomAngle) * 0.5f + 0.5f;
    rotationTextureData[index++] = sin(randomAngle) * 0.5f + 0.5f;
    rotationTextureData[index++] = -sin(randomAngle) * 0.5f + 0.5f;
    rotationTextureData[index++] = cos(randomAngle) * 0.5f + 0.5f;

    randomAngle += float(random.getRandomFloat()) * 4.0f * PI_4 * 2.0f;
  }

  if (glIsTexture(rotationTextureID))
  {
    glDeleteTextures(1, &rotationTextureID);
    rotationTextureID = 0;
  }
  glGenTextures(1, &rotationTextureID);
  glActiveTexture(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_PERLIN_NOISE_3D_TEXTURE);
  glBindTexture(GL_TEXTURE_2D, rotationTextureID);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, 64, 64, 0, GL_RGBA, GL_FLOAT, rotationTextureData.data());
  glActiveTexture(GL_TEXTURE0);
  glBindTexture(GL_TEXTURE_2D, 0);

  // prepare depthSamplesData
  const int depthSamplesDataSize = 16;
  depthSamplesData.resize(depthSamplesDataSize << 1);
  randomAngle = PI_4;
  const float radius = 0.416666f;

  index = 0;
  for (int i = 0; i < depthSamplesDataSize; ++i)
  {
    depthSamplesData[index++] = cos(randomAngle) * float(i + 1) / 16.0f * radius;
    depthSamplesData[index++] = sin(randomAngle) * float(i + 1) / 16.0f * radius;

    randomAngle += (2.0f * PI_4);

    if (((i + 1) & 3) == 0) randomAngle += (2.0f * PI_4);
  }
}

void SuperQuadricsTest::initGBufferEffectsFBOs() const
{
  // init GBufferEffects FBO for pass 1
  DebugConsole_consoleOutLine("FBO GBufferEffects Pass 1 ratioFactor: ", FULLSCREEN_MAPPING_RATIO_FACTOR);
  openGLFrameBufferObjectForGBufferPass1->disposeFrameBufferObjectResources();
  openGLFrameBufferObjectForGBufferPass1->initFrameBufferObjectResources(screenWidth / FULLSCREEN_MAPPING_RATIO_FACTOR, screenHeight / FULLSCREEN_MAPPING_RATIO_FACTOR, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);

  // init GBufferEffects FBO for pass 2
  DebugConsole_consoleOutLine("FBO GBufferEffects Pass 2 ratioFactor: ", FULLSCREEN_MAPPING_RATIO_FACTOR);
  openGLFrameBufferObjectRenderPass2->disposeFrameBufferObjectResources();
  openGLFrameBufferObjectRenderPass2->initFrameBufferObjectResources(screenWidth / FULLSCREEN_MAPPING_RATIO_FACTOR, screenHeight / FULLSCREEN_MAPPING_RATIO_FACTOR, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);

  // init GBufferEffects FBO for pass 3
  DebugConsole_consoleOutLine("FBO GBufferEffects Pass 3 ratioFactor: ", FULLSCREEN_MAPPING_RATIO_FACTOR);
  openGLFrameBufferObjectRenderPass3->disposeFrameBufferObjectResources();
  openGLFrameBufferObjectRenderPass3->initFrameBufferObjectResources(screenWidth / FULLSCREEN_MAPPING_RATIO_FACTOR, screenHeight / FULLSCREEN_MAPPING_RATIO_FACTOR, ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
}

void SuperQuadricsTest::processGBufferEffectsFBOs()
{
  const float gbufferResolution[2] = { float(screenWidth), float(screenHeight) };

  // draw GBufferEffects FBO for pass 1
  openGLFrameBufferObjectForGBufferPass1->startRender();
  renderSuperQuadricsScene(true);
  openGLFrameBufferObjectForGBufferPass1->finishRender();

  if (currentGBUfferEffectType == DEPTH_OF_FIELD)
  {
    // draw GBufferEffects FBO for scene creation pass
    openGLFrameBufferObjectRenderPass3->startRender();
    clearScreen();
    openGLShaderGBufferEffectsBlurXY->enableShaderProgram();
    openGLShaderGBufferEffectsBlurXY->setUniform1i("gbufferSampler", ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLShaderGBufferEffectsBlurXY->setUniform2fv("gbufferResolution", 1, gbufferResolution);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("numberOfSamplesRange", depthOfFieldBlurXYNumberOfSamplesRange);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderPlainScene", true);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderBlurXYScene", false);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("blurX", false);
    openGLFrameBufferObjectForGBufferPass1->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, openGLDriverInfo->isNvidia());
    openGLShaderGBufferEffectsBlurXY->disableShaderProgram();
    openGLFrameBufferObjectRenderPass3->finishRender();

    // draw GBufferEffects FBO for pass 2
    openGLFrameBufferObjectRenderPass2->startRender();
    openGLShaderGBufferEffectsBlurXY->enableShaderProgram();
    openGLShaderGBufferEffectsBlurXY->setUniform1i("sceneSampler", ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
    openGLShaderGBufferEffectsBlurXY->setUniform2fv("gbufferResolution", 1, gbufferResolution);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("numberOfSamplesRange", depthOfFieldBlurXYNumberOfSamplesRange);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderPlainScene", false);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderBlurXYScene", true);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("blurX", true);
    openGLFrameBufferObjectRenderPass3->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE, openGLDriverInfo->isNvidia());
    openGLShaderGBufferEffectsBlurXY->disableShaderProgram();
    openGLFrameBufferObjectRenderPass2->finishRender();

    // draw GBufferEffects FBO for pass 3
    openGLFrameBufferObjectRenderPass3->startRender();
    openGLShaderGBufferEffectsBlurXY->enableShaderProgram();
    openGLShaderGBufferEffectsBlurXY->setUniform1i("sceneSampler", ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
    openGLShaderGBufferEffectsBlurXY->setUniform2fv("gbufferResolution", 1, gbufferResolution);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("numberOfSamplesRange", depthOfFieldBlurXYNumberOfSamplesRange);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderPlainScene", false);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderBlurXYScene", true);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("blurX", false);
    openGLFrameBufferObjectRenderPass2->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE, openGLDriverInfo->isNvidia());
    openGLShaderGBufferEffectsBlurXY->disableShaderProgram();
    openGLFrameBufferObjectRenderPass3->finishRender();
  }
  else if ((currentGBUfferEffectType == HSSAO) || (currentGBUfferEffectType == HSSAO_PLUS))
  {
    // main HSSAO pass
    openGLFrameBufferObjectRenderPass3->startRender();
    clearScreen();
    openGLShaderGBufferEffects->enableShaderProgram();
    openGLShaderGBufferEffects->setUniform1i("gbufferSampler", ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLShaderGBufferEffects->setUniform1i("gbufferEffectType", currentGBUfferEffectType);
    openGLShaderGBufferEffects->setUniform1i("rotationTextureSampler", ACTIVE_TEXTURE_UNIT_FOR_PERLIN_NOISE_3D_TEXTURE);
    openGLShaderGBufferEffects->setUniform2fv("depthSamples", 16, depthSamplesData.data());
    openGLShaderGBufferEffects->setUniform1i("useHSSAO", true);
    const float rotationTextureResolution[2] = { gbufferResolution[0] / 64.0f, gbufferResolution[1] / 64.0f };
    openGLShaderGBufferEffects->setUniform2fv("xyHSSAO", 1, rotationTextureResolution);
    openGLFrameBufferObjectForGBufferPass1->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, openGLDriverInfo->isNvidia());
    openGLShaderGBufferEffects->disableShaderProgram();
    openGLFrameBufferObjectRenderPass3->finishRender();

    const int HSSAONumberOfSamplesRange  = 4;
    const bool renderBlurXYScene         = (currentGBUfferEffectType == HSSAO);
    const bool renderWeightedBlurXYScene = (currentGBUfferEffectType == HSSAO_PLUS);

    // blur X for after HSSAO pass 1
    openGLFrameBufferObjectRenderPass2->startRender();
    openGLFrameBufferObjectForGBufferPass1->enable();
    openGLFrameBufferObjectForGBufferPass1->bindTexture(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLShaderGBufferEffectsBlurXY->enableShaderProgram();
    openGLShaderGBufferEffectsBlurXY->setUniform1i("sceneSampler", ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
    openGLShaderGBufferEffectsBlurXY->setUniform2fv("gbufferResolution", 1, gbufferResolution);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderPlainScene", false);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderBlurXYScene", renderBlurXYScene);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderWeightedBlurXYScene", renderWeightedBlurXYScene);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("blurX", true);
    if (renderWeightedBlurXYScene)
    {
      const float sxValues[3] = { 1.0f / gbufferResolution[0], 2.0f / gbufferResolution[0], 3.0f / gbufferResolution[0] };
      openGLShaderGBufferEffectsBlurXY->setUniform3fv("sxValues", 1, sxValues);
    }
    else
    {
      openGLShaderGBufferEffectsBlurXY->setUniform1i("numberOfSamplesRange", HSSAONumberOfSamplesRange);
    }
    openGLFrameBufferObjectRenderPass3->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE, openGLDriverInfo->isNvidia());
    openGLShaderGBufferEffectsBlurXY->disableShaderProgram();
    openGLFrameBufferObjectForGBufferPass1->unbind(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLFrameBufferObjectForGBufferPass1->disable();
    openGLFrameBufferObjectRenderPass2->finishRender();

    // blur Y for after HSSAO pass 2
    openGLFrameBufferObjectRenderPass3->startRender();
    openGLFrameBufferObjectForGBufferPass1->enable();
    openGLFrameBufferObjectForGBufferPass1->bindTexture(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLShaderGBufferEffectsBlurXY->enableShaderProgram();
    openGLShaderGBufferEffectsBlurXY->setUniform1i("sceneSampler", ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
    openGLShaderGBufferEffectsBlurXY->setUniform2fv("gbufferResolution", 1, gbufferResolution);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderPlainScene", false);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderBlurXYScene", renderBlurXYScene);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderWeightedBlurXYScene", renderWeightedBlurXYScene);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("blurX", false);
    if (renderWeightedBlurXYScene)
    {
      const float syValues[3] = { 1.0f / gbufferResolution[1], 2.0f / gbufferResolution[1], 3.0f / gbufferResolution[1] };
      openGLShaderGBufferEffectsBlurXY->setUniform3fv("syValues", 1, syValues);
    }
    else
    {
      openGLShaderGBufferEffectsBlurXY->setUniform1i("numberOfSamplesRange", HSSAONumberOfSamplesRange);
    }
    openGLFrameBufferObjectRenderPass2->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE, openGLDriverInfo->isNvidia());
    openGLShaderGBufferEffectsBlurXY->disableShaderProgram();
    openGLFrameBufferObjectForGBufferPass1->unbind(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLFrameBufferObjectForGBufferPass1->disable();
    openGLFrameBufferObjectRenderPass3->finishRender();
  }
}

void SuperQuadricsTest::drawGBufferEffectsFBOs()
{
  // main GBuffer render code
  if ((currentGBUfferEffectType == HSSAO) || (currentGBUfferEffectType == HSSAO_PLUS))
  {
    clearScreen();
    disableLights(false);
    openGLFrameBufferObjectForGBufferPass1->enable();
    openGLFrameBufferObjectForGBufferPass1->bindTexture(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLFrameBufferObjectRenderPass3->enable();
    openGLFrameBufferObjectRenderPass3->bindTexture(ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
    openGLShaderGBufferEffectsHSSAO->enableShaderProgram();
    openGLShaderGBufferEffectsHSSAO->setUniform1i("gbufferSampler", ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLShaderGBufferEffectsHSSAO->setUniform1i("HSSAOBlurredXYSampler", ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
    openGLFrameBufferObjectForGBufferPass1->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, openGLDriverInfo->isNvidia());
    openGLShaderGBufferEffectsHSSAO->disableShaderProgram();
    openGLFrameBufferObjectRenderPass3->unbind(ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
    openGLFrameBufferObjectRenderPass3->disable();
    openGLFrameBufferObjectForGBufferPass1->unbind(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLFrameBufferObjectForGBufferPass1->disable();
    enableLights(false);
  }
  else // for Depth-Of-Field & Edge Enhancement effects
  {
    clearScreen();
    disableLights(false);
    openGLFrameBufferObjectForGBufferPass1->enable();
    openGLFrameBufferObjectForGBufferPass1->bindTexture(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    if (currentGBUfferEffectType == DEPTH_OF_FIELD)
    {
      openGLFrameBufferObjectRenderPass3->enable();
      openGLFrameBufferObjectRenderPass3->bindTexture(ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
    }
    openGLShaderGBufferEffects->enableShaderProgram();
    openGLShaderGBufferEffects->setUniform1i("gbufferSampler", ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLShaderGBufferEffects->setUniform1i("gbufferEffectType", currentGBUfferEffectType);
    if (currentGBUfferEffectType == DEPTH_OF_FIELD)
    {
      openGLShaderGBufferEffects->setUniform1i("sceneBlurredXYSampler", ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
      openGLShaderGBufferEffects->setUniform1f("zNear", openGLEulerCamera->getViewNear());
      openGLShaderGBufferEffects->setUniform1f("zFar",  openGLEulerCamera->getViewFar());
      openGLShaderGBufferEffects->setUniform1f("range", depthOfFieldRange);
      openGLShaderGBufferEffects->setUniform1f("zFocus", depthOfFieldZFocus);
    }
    else if (currentGBUfferEffectType == EDGE_ENHANCEMENT)
    {
      const float gbufferResolution[2] = { float(screenWidth), float(screenHeight) };
      openGLShaderGBufferEffects->setUniform2fv("gbufferResolution", 1, gbufferResolution);
      double ratio = (cameraDistanceZ > 1.0) ? cameraDistanceZ / 10.0 : 1.0;
      if (ratio > 8.0) ratio = 8.0;
      const float edgeEnhancementSize[2] = { float((12.0 / 3.0) / ratio), float((12.0 / 3.0) / ratio) };
      openGLShaderGBufferEffects->setUniform2fv("edgeEnhancementSize", 1, edgeEnhancementSize);
    }
    openGLFrameBufferObjectForGBufferPass1->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, openGLDriverInfo->isNvidia());
    openGLShaderGBufferEffects->disableShaderProgram();
    if (currentGBUfferEffectType == DEPTH_OF_FIELD)
    {
      openGLFrameBufferObjectRenderPass3->unbind(ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
      openGLFrameBufferObjectRenderPass3->disable();
    }
    openGLFrameBufferObjectForGBufferPass1->unbind(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLFrameBufferObjectForGBufferPass1->disable();
    enableLights(false);
  }
}

void SuperQuadricsTest::prepareFXAA_AntialiasFBO()
{
  openGLFrameBufferObjectForFXAA_Antialias = new OpenGLFrameBufferObject(openGLDriverInfo);
  openGLFrameBufferObjectForFXAA_Antialias->setUseTexture(true);
  openGLFrameBufferObjectForFXAA_Antialias->setDepthStencilRenderBuffer(true);
  if (!openGLDriverInfo->isIntel() && openGLDriverInfo->supports_GL_EXT_framebuffer_multisample() && openGLDriverInfo->supports_GL_EXT_framebuffer_blit())
  {
    openGLFrameBufferObjectForFXAA_Antialias->setMultisampleFBO(true, 4);
  }
}

void SuperQuadricsTest::initFXAA_AntialiasFBO() const
{
  DebugConsole_consoleOutLine("FBO FXAA Antialias ratioFactor: ", FULLSCREEN_MAPPING_RATIO_FACTOR);
  openGLFrameBufferObjectForFXAA_Antialias->disposeFrameBufferObjectResources();
  openGLFrameBufferObjectForFXAA_Antialias->initFrameBufferObjectResources(screenWidth / FULLSCREEN_MAPPING_RATIO_FACTOR, screenHeight / FULLSCREEN_MAPPING_RATIO_FACTOR, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
}

void SuperQuadricsTest::drawFXAA_AntialiasFBO()
{
  // draw FXAA Antialias FBO for pass 1
  openGLFrameBufferObjectForFXAA_Antialias->startRender();
  if (openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders() &&
      !(useOutline || useHighQualityOutline) && (currentLightingModel != FIXED_PIPELINE) && (currentGBUfferEffectType != NO_GBUFFER_EFFECT))
  {
    drawGBufferEffectsFBOs();
  }
  else
  {
    renderSuperQuadricsScene();
  }
  openGLFrameBufferObjectForFXAA_Antialias->finishRender();

  // draw FXAA Antialias FBO for pass 2
  clearScreen();
  disableLights(false);
  openGLShaderFXAA_Antialias->enableShaderProgram();
  openGLShaderFXAA_Antialias->setUniform1i("frameBufferSampler", ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
  const float frameBufferResolution[2] = { float(screenWidth), float(screenHeight) };
  openGLShaderFXAA_Antialias->setUniform2fv("frameBufferResolution", 1, frameBufferResolution);
  openGLFrameBufferObjectForFXAA_Antialias->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, openGLDriverInfo->isNvidia());
  openGLShaderFXAA_Antialias->disableShaderProgram();
  enableLights(false);
}

void SuperQuadricsTest::prepareVBOs()
{
  glGenBuffers(1, &VBOVerticesID);
  glBindBuffer(GL_ARRAY_BUFFER, VBOVerticesID);
  glBufferData(GL_ARRAY_BUFFER,  vertices.size() * sizeof(GLdouble),  vertices.data(), GL_STATIC_DRAW);

  glGenBuffers(1, &VBONormalsID);
  glBindBuffer(GL_ARRAY_BUFFER, VBONormalsID);
  glBufferData(GL_ARRAY_BUFFER,   normals.size() * sizeof(GLdouble),   normals.data(), GL_STATIC_DRAW);

  glGenBuffers(1, &VBOTexCoordsID);
  glBindBuffer(GL_ARRAY_BUFFER, VBOTexCoordsID);
  glBufferData(GL_ARRAY_BUFFER, texCoords.size() * sizeof(GLdouble), texCoords.data(), GL_STATIC_DRAW);

  glBindBuffer(GL_ARRAY_BUFFER, 0);
}

void SuperQuadricsTest::deleteVBOs() const
{
  if (glIsBuffer(VBOTexCoordsID))
  {
    glDeleteBuffers(1, &VBOTexCoordsID);
  }
  if (glIsBuffer(VBONormalsID))
  {
    glDeleteBuffers(1, &VBONormalsID);
  }
  if (glIsBuffer(VBOVerticesID))
  {
    glDeleteBuffers(1, &VBOVerticesID);
  }
}

void SuperQuadricsTest::renderNoCaching(bool skipReInitSuperQuadricsData, bool enableShapeColors, bool useGBuffer)
{
  if (reInitSuperQuadricsDataForRendering && !skipReInitSuperQuadricsData)
  {
    previousAvailableMemory = howMuchCurrentlyAvailableVRAMMemory();
    // just to clear VRAM memory
    if (glIsList(allSuperQuadricsDataDisplayList))
    {
      glDeleteLists(allSuperQuadricsDataDisplayList, 1);
    }
    deleteVBOs();
    initSuperQuadricsShapes();
    reInitSuperQuadricsDataForRendering = false;
    printGLMemoryInfoOnSecondUpdate = 0;
  }

  int shapeObjectIndex = 0;
  int vertexIndex = 0;
  int texCoordIndex = 0;
  const int numberOfTrianglesPerShapeObject = 6 * slices * segments; // 6 -> 2 * 3 triangles
  int superQuadricShapeType = 0;
  bool isSuperQuadricShapeTypeWithProblematicLOD = false;
  for (int m = 0; m < geometryRepetition; ++m)
  {
    vertexIndex = 0;
    texCoordIndex = 0;
    superQuadricShapeType = 0;
    for (size_t i = 0; i < SuperQuadricShapesProducer::NUMBER_OF_SHAPES / 3; ++i)
    {
      for (size_t j = 0; j < SuperQuadricShapesProducer::NUMBER_OF_SHAPES / 4; ++j)
      {
        superQuadricShapeType++;
        isSuperQuadricShapeTypeWithProblematicLOD = false;
        glPushMatrix();
        glTranslatef(3.0f * (i - 1.5f), 3.0f * (1.0f - j), 0.0f);
        if (enableShapeColors)
        {
          if (useColoredObjects)
          {
            glColor4f(i / float(SuperQuadricShapesProducer::NUMBER_OF_SHAPES / 3.0f),
                      j / float(SuperQuadricShapesProducer::NUMBER_OF_SHAPES / 4.0f), 0.5f, 1.0f);
          }
          else
          {
            glColor4fv(DEFAULT_SHAPE_COLOR.data());
          }
        }
        vertexIndex = 3 * shapeObjectIndex++ * numberOfTrianglesPerShapeObject;
        texCoordIndex = 2 * vertexIndex / 3;
        if (isUsingLOD())
        {
          isSuperQuadricShapeTypeWithProblematicLOD = ((superQuadricShapeType == 3) || (superQuadricShapeType == 9) || (superQuadricShapeType == 12));
          if (isSuperQuadricShapeTypeWithProblematicLOD)
          {
            if (currentGeometryShader == NO_USE)
            {
              openGLShaderSurfaceLightingLODModels->disableShaderProgram();
              openGLShaderSurfaceLightingModels->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, false, false,
                                                                       useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                       !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                       useGBuffer);
            }
            else if (currentGeometryShader == PASS_THRU)
            {
              openGLShaderSurfaceLightingLODModelsWithGeometry->disableShaderProgram();
              openGLShaderSurfaceLightingModelsWithGeometry->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, true, false,
                                                                                   useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                                   !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                                   useGBuffer);
            }
            else if (currentGeometryShader == CREATE_NORMAL_GEOMETRY)
            {
              openGLShaderSurfaceLightingLODModelsWithGeometryAndNormals->disableShaderProgram();
              openGLShaderSurfaceLightingModelsWithGeometryAndNormals->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, false, true,
                                                                                             useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                                             !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                                             useGBuffer);
            }
            glBegin(GL_TRIANGLES);
          }
          else
          {
            glBegin(GL_PATCHES);
          }
        }
        else
        {
          glBegin(GL_TRIANGLES);
        }
        for (int k = 0; k < numberOfTrianglesPerShapeObject; ++k)
        {
          if (currentCachedRenderingTest == USE_NO_CACHING_METHOD_1)
          {
            glNormal3d(normals[vertexIndex], normals[vertexIndex + 1], normals[vertexIndex + 2]);
            glMultiTexCoord2d(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, texCoords[texCoordIndex], texCoords[texCoordIndex + 1]);
            glVertex3d(vertices[vertexIndex], vertices[vertexIndex + 1], vertices[vertexIndex + 2]);
          }
          else // if (currentCachedRenderingTest == USE_NO_CACHING_METHOD_2)
          {
            glNormal3dv(&normals[vertexIndex]);
            glMultiTexCoord2dv(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, &texCoords[texCoordIndex]);
            glVertex3dv(&vertices[vertexIndex]);
          }
          vertexIndex += 3;
          texCoordIndex += 2;
        }
        glEnd();
        if (isSuperQuadricShapeTypeWithProblematicLOD)
        {
          if (currentGeometryShader == NO_USE)
          {
            openGLShaderSurfaceLightingModels->disableShaderProgram();
            openGLShaderSurfaceLightingLODModels->useProgramAndUniforms(currentLODModel, tessellationAlpha, tessellationLevel, useAdaptiveTessellation,
                                                                        useTrianglePositionAdaptiveTessellationMetric, useLightAdaptiveTessellationMetric, useContourAdaptiveTessellationMetric,
                                                                        NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, false, false,
                                                                        useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                        !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                        useGBuffer);
          }
          else if (currentGeometryShader == PASS_THRU)
          {
            openGLShaderSurfaceLightingModelsWithGeometry->disableShaderProgram();
            openGLShaderSurfaceLightingLODModelsWithGeometry->useProgramAndUniforms(currentLODModel, tessellationAlpha, tessellationLevel, 
                                                                                    useAdaptiveTessellation, useTrianglePositionAdaptiveTessellationMetric, useLightAdaptiveTessellationMetric, useContourAdaptiveTessellationMetric,
                                                                                    NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, true, false,
                                                                                    useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                                    !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                                    useGBuffer);
          }
          else if (currentGeometryShader == CREATE_NORMAL_GEOMETRY)
          {
            openGLShaderSurfaceLightingModelsWithGeometryAndNormals->disableShaderProgram();
            openGLShaderSurfaceLightingLODModelsWithGeometryAndNormals->useProgramAndUniforms(currentLODModel, tessellationAlpha, tessellationLevel,
                                                                                              useAdaptiveTessellation, useTrianglePositionAdaptiveTessellationMetric, useLightAdaptiveTessellationMetric, useContourAdaptiveTessellationMetric,
                                                                                              NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, false, true,
                                                                                              useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                                              !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                                              useGBuffer);
          }
        }
        glPopMatrix();
      }
    }
  }
}

void SuperQuadricsTest::renderVertexArrays(bool enableShapeColors, bool useGBuffer)
{
  if (reInitSuperQuadricsDataForRendering)
  {
    previousAvailableMemory = howMuchCurrentlyAvailableVRAMMemory();
    // just to clear VRAM memory
    if (glIsList(allSuperQuadricsDataDisplayList))
    {
      glDeleteLists(allSuperQuadricsDataDisplayList, 1);
    }
    deleteVBOs();
    initSuperQuadricsShapes();
    reInitSuperQuadricsDataForRendering = false;
    printGLMemoryInfoOnSecondUpdate = 0;
  }

  // Enable our client state for Vertex Arrays (only for ensuring further state-based changes)
  glClientActiveTexture(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
  glEnableClientState(GL_TEXTURE_COORD_ARRAY);
  glEnableClientState(GL_NORMAL_ARRAY);
  glEnableClientState(GL_VERTEX_ARRAY);

  glTexCoordPointer(2, GL_DOUBLE, 0, texCoords.data());
  glNormalPointer(GL_DOUBLE, 0, normals.data());
  glVertexPointer(3, GL_DOUBLE, 0, vertices.data());

  const int numberOfTrianglesPerShapeObject = 6 * slices * segments; // 6 -> 2 * 3 triangles
  int arrayLengthIndex      = 0;
  int superQuadricShapeType = 0;
  bool isSuperQuadricShapeTypeWithProblematicLOD = false;
  GLuint primitiveType = 0;
  for (int k = 0; k < geometryRepetition; ++k)
  {
    for (size_t i = 0; i < SuperQuadricShapesProducer::NUMBER_OF_SHAPES / 3; ++i)
    {
      for (size_t j = 0; j < SuperQuadricShapesProducer::NUMBER_OF_SHAPES / 4; ++j)
      {
        superQuadricShapeType++;
        isSuperQuadricShapeTypeWithProblematicLOD = false;
        glPushMatrix();
        glTranslatef(3.0f * (i - 1.5f), 3.0f * (1.0f - j), 0.0f);
        if (enableShapeColors)
        {
          if (useColoredObjects)
          {
            glColor4f(i / float(SuperQuadricShapesProducer::NUMBER_OF_SHAPES / 3.0f),
                      j / float(SuperQuadricShapesProducer::NUMBER_OF_SHAPES / 4.0f), 0.5f, 1.0f);
          }
          else
          {
            glColor4fv(DEFAULT_SHAPE_COLOR.data());
          }
        }
        if (isUsingLOD())
        {
          isSuperQuadricShapeTypeWithProblematicLOD = ((superQuadricShapeType == 3) || (superQuadricShapeType == 9) || (superQuadricShapeType == 12));
          if (isSuperQuadricShapeTypeWithProblematicLOD)
          {
            if (currentGeometryShader == NO_USE)
            {
              openGLShaderSurfaceLightingLODModels->disableShaderProgram();
              openGLShaderSurfaceLightingModels->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, false, false,
                                                                       useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                       !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                       useGBuffer);
            }
            else if (currentGeometryShader == PASS_THRU)
            {
              openGLShaderSurfaceLightingLODModelsWithGeometry->disableShaderProgram();
              openGLShaderSurfaceLightingModelsWithGeometry->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, true, false,
                                                                                   useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                                   !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                                   useGBuffer);
            }
            else if (currentGeometryShader == CREATE_NORMAL_GEOMETRY)
            {
              openGLShaderSurfaceLightingLODModelsWithGeometryAndNormals->disableShaderProgram();
              openGLShaderSurfaceLightingModelsWithGeometryAndNormals->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, false, true,
                                                                                             useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                                             !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                                             useGBuffer);
            }
            primitiveType = GL_TRIANGLES;
          }
          else
          {
            primitiveType = GL_PATCHES;
          }
        }
        else
        {
          primitiveType = GL_TRIANGLES;
        }
        glDrawArrays(primitiveType, arrayLengthIndex, numberOfTrianglesPerShapeObject);
        if (isSuperQuadricShapeTypeWithProblematicLOD)
        {
          if (currentGeometryShader == NO_USE)
          {
            openGLShaderSurfaceLightingModels->disableShaderProgram();
            openGLShaderSurfaceLightingLODModels->useProgramAndUniforms(currentLODModel, tessellationAlpha, tessellationLevel,
                                                                        useAdaptiveTessellation, useTrianglePositionAdaptiveTessellationMetric, useLightAdaptiveTessellationMetric, useContourAdaptiveTessellationMetric,
                                                                        NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, false, false,
                                                                        useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                        !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                        useGBuffer);
          }
          else if (currentGeometryShader == PASS_THRU)
          {
            openGLShaderSurfaceLightingModelsWithGeometry->disableShaderProgram();
            openGLShaderSurfaceLightingLODModelsWithGeometry->useProgramAndUniforms(currentLODModel, tessellationAlpha, tessellationLevel,
                                                                                    useAdaptiveTessellation, useTrianglePositionAdaptiveTessellationMetric, useLightAdaptiveTessellationMetric, useContourAdaptiveTessellationMetric,
                                                                                    NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, true, false,
                                                                                    useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                                    !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                                    useGBuffer);
          }
          else if (currentGeometryShader == CREATE_NORMAL_GEOMETRY)
          {
            openGLShaderSurfaceLightingModelsWithGeometryAndNormals->disableShaderProgram();
            openGLShaderSurfaceLightingLODModelsWithGeometryAndNormals->useProgramAndUniforms(currentLODModel, tessellationAlpha, tessellationLevel,
                                                                                              useAdaptiveTessellation, useTrianglePositionAdaptiveTessellationMetric, useLightAdaptiveTessellationMetric, useContourAdaptiveTessellationMetric,
                                                                                              NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, false, true,
                                                                                              useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                                              !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                                              useGBuffer);
          }
        }
        glPopMatrix();
        arrayLengthIndex += numberOfTrianglesPerShapeObject;
      }
    }
  }

  // Disable our client state back to normal drawing (only for ensuring further state-based changes)
  glDisableClientState(GL_TEXTURE_COORD_ARRAY);
  glDisableClientState(GL_NORMAL_ARRAY);
  glDisableClientState(GL_VERTEX_ARRAY);
}

void SuperQuadricsTest::renderDisplayLists(bool enableShapeColors, bool useGBuffer)
{
  if (reInitSuperQuadricsDataForRendering)
  {
    previousAvailableMemory = howMuchCurrentlyAvailableVRAMMemory();
    // just to clear VRAM memory
    if (glIsList(allSuperQuadricsDataDisplayList))
    {
      glDeleteLists(allSuperQuadricsDataDisplayList, 1);
    }
    deleteVBOs();
    initSuperQuadricsShapes();
    glNewList(allSuperQuadricsDataDisplayList, GL_COMPILE);
    renderNoCaching(true, enableShapeColors, useGBuffer);
    glEndList();
    reInitSuperQuadricsDataForRendering = false;
    printGLMemoryInfoOnSecondUpdate = 0;
  }

  glCallList(allSuperQuadricsDataDisplayList);
}

void SuperQuadricsTest::renderVBOs(bool enableShapeColors, bool useGBuffer)
{
  if (reInitSuperQuadricsDataForRendering)
  {
    previousAvailableMemory = howMuchCurrentlyAvailableVRAMMemory();
    // just to clear VRAM memory
    if (glIsList(allSuperQuadricsDataDisplayList))
    {
      glDeleteLists(allSuperQuadricsDataDisplayList, 1);
    }
    deleteVBOs();
    initSuperQuadricsShapes();
    prepareVBOs();
    reInitSuperQuadricsDataForRendering = false;
    printGLMemoryInfoOnSecondUpdate = 0;
  }

  // Enable our client state for VBOs (only for ensuring further state-based changes)
  glClientActiveTexture(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
  glEnableClientState(GL_TEXTURE_COORD_ARRAY);
  glEnableClientState(GL_NORMAL_ARRAY);
  glEnableClientState(GL_VERTEX_ARRAY);

  // Bind our buffers much like we would for texturing
  glBindBuffer(GL_ARRAY_BUFFER, VBOTexCoordsID);
  glTexCoordPointer(2, GL_DOUBLE, 0, nullptr);
  glBindBuffer(GL_ARRAY_BUFFER, VBONormalsID);
  glNormalPointer(GL_DOUBLE, 0, nullptr);
  glBindBuffer(GL_ARRAY_BUFFER, VBOVerticesID);
  glVertexPointer(3, GL_DOUBLE, 0, nullptr);

  const int numberOfTrianglesPerShapeObject = 6 * slices * segments; // 6 -> 2 * 3 triangles
  int arrayLengthIndex = 0;
  int superQuadricShapeType = 0;
  bool isSuperQuadricShapeTypeWithProblematicLOD = false;
  GLuint primitiveType = 0;
  for (int k = 0; k < geometryRepetition; ++k)
  {
    for (size_t i = 0; i < SuperQuadricShapesProducer::NUMBER_OF_SHAPES / 3; ++i)
    {
      for (size_t j = 0; j < SuperQuadricShapesProducer::NUMBER_OF_SHAPES / 4; ++j)
      {
        superQuadricShapeType++;
        isSuperQuadricShapeTypeWithProblematicLOD = false;
        glPushMatrix();
        glTranslatef(3.0f * (i - 1.5f), 3.0f * (1.0f - j), 0.0f);
        if (enableShapeColors)
        {
          if (useColoredObjects)
          {
            glColor4f(i / float(SuperQuadricShapesProducer::NUMBER_OF_SHAPES / 3.0f),
                      j / float(SuperQuadricShapesProducer::NUMBER_OF_SHAPES / 4.0f), 0.5f, 1.0f);
          }
          else
          {
            glColor4fv(DEFAULT_SHAPE_COLOR.data());
          }
        }
        if (isUsingLOD())
        {
          isSuperQuadricShapeTypeWithProblematicLOD = ((superQuadricShapeType == 3) || (superQuadricShapeType == 9) || (superQuadricShapeType == 12));
          if (isSuperQuadricShapeTypeWithProblematicLOD)
          {
            if (currentGeometryShader == NO_USE)
            {
              openGLShaderSurfaceLightingLODModels->disableShaderProgram();
              openGLShaderSurfaceLightingModels->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, false, false,
                                                                       useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                       !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                       useGBuffer);
            }
            else if (currentGeometryShader == PASS_THRU)
            {
              openGLShaderSurfaceLightingLODModelsWithGeometry->disableShaderProgram();
              openGLShaderSurfaceLightingModelsWithGeometry->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, true, false,
                                                                                   useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                                   !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                                   useGBuffer);
            }
            else if (currentGeometryShader == CREATE_NORMAL_GEOMETRY)
            {
              openGLShaderSurfaceLightingLODModelsWithGeometryAndNormals->disableShaderProgram();
              openGLShaderSurfaceLightingModelsWithGeometryAndNormals->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, false, true,
                                                                                             useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                                             !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                                             useGBuffer);
            }
            primitiveType = GL_TRIANGLES;
          }
          else
          {
            primitiveType = GL_PATCHES;
          }
        }
        else
        {
          primitiveType = GL_TRIANGLES;
        }
        glDrawArrays(primitiveType, arrayLengthIndex, numberOfTrianglesPerShapeObject);
        if (isSuperQuadricShapeTypeWithProblematicLOD)
        {
          if (currentGeometryShader == NO_USE)
          {
            openGLShaderSurfaceLightingModels->disableShaderProgram();
            openGLShaderSurfaceLightingLODModels->useProgramAndUniforms(currentLODModel, tessellationAlpha, tessellationLevel,
                                                                        useAdaptiveTessellation, useTrianglePositionAdaptiveTessellationMetric, useLightAdaptiveTessellationMetric, useContourAdaptiveTessellationMetric,
                                                                        NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, false, false,
                                                                        useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                        !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                        useGBuffer);
          }
          else if (currentGeometryShader == PASS_THRU)
          {
            openGLShaderSurfaceLightingModelsWithGeometry->disableShaderProgram();
            openGLShaderSurfaceLightingLODModelsWithGeometry->useProgramAndUniforms(currentLODModel, tessellationAlpha, tessellationLevel,
                                                                                    useAdaptiveTessellation, useTrianglePositionAdaptiveTessellationMetric, useLightAdaptiveTessellationMetric, useContourAdaptiveTessellationMetric,
                                                                                    NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, true, false,
                                                                                    useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                                    !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                                    useGBuffer);
          }
          else if (currentGeometryShader == CREATE_NORMAL_GEOMETRY)
          {
            openGLShaderSurfaceLightingModelsWithGeometryAndNormals->disableShaderProgram();
            openGLShaderSurfaceLightingLODModelsWithGeometryAndNormals->useProgramAndUniforms(currentLODModel, tessellationAlpha, tessellationLevel,
                                                                                              useAdaptiveTessellation, useTrianglePositionAdaptiveTessellationMetric, useLightAdaptiveTessellationMetric, useContourAdaptiveTessellationMetric,
                                                                                              NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, false, true,
                                                                                              useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                                              !enableShapeColors, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                                              useGBuffer);
          }
        }
        glPopMatrix();
        arrayLengthIndex += numberOfTrianglesPerShapeObject;
      }
    }
  }

  glBindBuffer(GL_ARRAY_BUFFER, 0);

  // Disable our client state back to normal drawing (only for ensuring further state-based changes)
  glDisableClientState(GL_TEXTURE_COORD_ARRAY);
  glDisableClientState(GL_NORMAL_ARRAY);
  glDisableClientState(GL_VERTEX_ARRAY);
}

void SuperQuadricsTest::clearScreen(bool useGBuffer) const
{
  if (useGBuffer || blackOrWhiteBackground)
  {
    glClearColor(WHITE_BACKGROUND_COLOR[0], WHITE_BACKGROUND_COLOR[1], WHITE_BACKGROUND_COLOR[2], WHITE_BACKGROUND_COLOR[3]);
  }
  else
  {
    glClearColor(BLACK_BACKGROUND_COLOR[0], BLACK_BACKGROUND_COLOR[1], BLACK_BACKGROUND_COLOR[2], BLACK_BACKGROUND_COLOR[3]);
  }
  glClearStencil(0); // clear stencil buffer
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
}

void SuperQuadricsTest::disableLights(bool disableAllLightSources)
{
  glDisable(GL_LIGHTING);
  glDisable(GL_VERTEX_PROGRAM_TWO_SIDE); // Intel GPUs may not honor this command, must manually populate the gl_BackColor state variable in the vertex shader
  glDisable(GL_NORMALIZE);

  if (disableAllLightSources)
  {
    openGLModelAmbientLight->disable();
    for (const auto& openGLLight : openGLLights)
    {
      openGLLight->disable();
    }
    openGLMaterial->disable();
  }
}

void SuperQuadricsTest::enableLights(bool enableAllLightSources)
{
  glEnable(GL_LIGHTING);
  glEnable(GL_VERTEX_PROGRAM_TWO_SIDE); // Intel GPUs may not honor this command, must manually populate the gl_BackColor state variable in the vertex shader
  glEnable(GL_NORMALIZE);

  if (enableAllLightSources)
  {
    openGLModelAmbientLight->enable();
    for (auto& openGLLight : openGLLights)
    {
      openGLLight->enable(GL_LIGHT0, usePositionalLights != 0);
    }
    openGLMaterial->enable();
  }
}

void SuperQuadricsTest::renderSuperQuadricsScene(bool useGBuffer)
{
  if (openGLDriverInfo->supports_GL_EXT_packed_depth_stencil() && useHighQualityOutline)
  {
    openGLFrameBufferObjectForHighQualityOutline->bindTexture(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLFrameBufferObjectForHighQualityOutline->startRender();
  }

  clearScreen(useGBuffer);

  glDisable(GL_CULL_FACE); // don't cull faces to avoid problems with Hyperboloid Super Quadric shape

  //if (wireframe) // don't cull faces in wireframe mode
  //{
  //  glDisable(GL_CULL_FACE);
  //}
  //else
  //{
  //  glFrontFace(GL_CCW); // implied
  //  glEnable(GL_CULL_FACE);
  //  glCullFace(GL_BACK);
  //}

  updateLighting();

  glPushMatrix();

  openGLEulerCamera->setEulerCamera(cameraDistanceX, cameraDistanceY, cameraDistanceZ, cameraAngleX, cameraAngleY);

  if (useFog)
  {
    glFogfv(GL_FOG_COLOR, blackOrWhiteBackground ? WHITE_BACKGROUND_COLOR.data() : BLACK_BACKGROUND_COLOR.data());
    glEnable(GL_FOG);
  }
  else
  {
    glDisable(GL_FOG);
  }

  if (useOutline || useHighQualityOutline)
  {
    // enable stencil test
    glEnable(GL_STENCIL_TEST);

    // PASS1: render the Super Quadrics geometry on both color & stencil buffers
    // The reference color value will be written to the stencil buffer if the test is passed.
    glStencilFunc(GL_ALWAYS, 0x1, 0x1);
    glStencilOp(GL_KEEP, GL_KEEP, GL_REPLACE);
    glColor4fv(STENCIL_COLOR.data());
    renderSuperQuadricsGeometry(useGBuffer);

    // PASS 2: render the Super Quadrics geometry with wireframe only where the stencil value is NOT set to 1.
    glStencilFunc(GL_NOTEQUAL, 0x1, 0x1);
    glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
    if (!openGLDriverInfo->isAMDATI())
    {
      disableLights(false); // do not try to optimize away lighting for AMD-ATI hardware
    }
    glColor4fv(blackOrWhiteBackground ? BLACK_BACKGROUND_COLOR.data() : WHITE_BACKGROUND_COLOR.data()); // black/white outline color
    glLineWidth(useHighQualityOutline ? DEPTH_STENCIL_MULTIPLICATION_FACTOR * 5.0f : 5.0f);
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE); // wireframe
    renderSuperQuadricsGeometry(useGBuffer, true);
    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL); // return to fill mode
    glLineWidth(1.0f);
    if (!openGLDriverInfo->isAMDATI())
    {
      enableLights(false); // do not try to optimize away lighting for AMD-ATI hardware
    }

    // disable stencil test
    glDisable(GL_STENCIL_TEST);
  }
  else
  {
    renderSuperQuadricsGeometry(useGBuffer);
  }

  glPopMatrix();

  if (openGLDriverInfo->supports_GL_EXT_packed_depth_stencil() && useHighQualityOutline)
  {
    openGLFrameBufferObjectForHighQualityOutline->finishRender();

    clearScreen();
    disableLights(false);
    openGLFrameBufferObjectForHighQualityOutline->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, openGLDriverInfo->isNvidia(), false); // don't use a dummyVAO for quad rendering as no shader is being used here
    enableLights(false);
  }
}

void SuperQuadricsTest::renderSuperQuadricsGeometry(bool useGBuffer, bool enableStencilBufferColor)
{
  if (multisample && (currentGBUfferEffectType != NO_GBUFFER_EFFECT))
  {
    glDisable(GL_MULTISAMPLE);
  }

  if (useTexturing)
  {
    // Use the GL_DECAL texture function to effectively blend between the vertex color and the texture color
    // based on the textures alpha channel, but leave the alpha channel set to the vertex color
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);

    openGLILTexture->bindTexture(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);

    if (useSphericalMapping && (currentLightingModel == FIXED_PIPELINE))
    {
      glEnable(GL_TEXTURE_GEN_S);
      glEnable(GL_TEXTURE_GEN_T);
    }
    glEnable(GL_TEXTURE_2D);
  }
  else
  {
    glDisable(GL_TEXTURE_2D);
  }

  if (openGLDriverInfo->isAMDATI())
  {
    enableLights(); // make sure lights are enabled for AMD-ATI hardware, also note that the AMD-ATI hardware will only register the last color used for rendering the SuperQuadric shapes (yellow) to use for the shaders
  }
  if (currentLightingModel != FIXED_PIPELINE)
  {
    shaderTimer += 0.01f;
    if (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader() && useLODModels)
    {
      glPatchParameteri(GL_PATCH_VERTICES, 3);

      if (currentGeometryShader == NO_USE)
      {
        openGLShaderSurfaceLightingLODModels->useProgramAndUniforms(currentLODModel, tessellationAlpha, tessellationLevel,
                                                                    useAdaptiveTessellation, useTrianglePositionAdaptiveTessellationMetric, useLightAdaptiveTessellationMetric, useContourAdaptiveTessellationMetric,
                                                                    NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, false, false,
                                                                    useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                    enableStencilBufferColor, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                    useGBuffer);
      }
      else if (currentGeometryShader == PASS_THRU)
      {
        openGLShaderSurfaceLightingLODModelsWithGeometry->useProgramAndUniforms(currentLODModel, tessellationAlpha, tessellationLevel,
                                                                                useAdaptiveTessellation, useTrianglePositionAdaptiveTessellationMetric, useLightAdaptiveTessellationMetric, useContourAdaptiveTessellationMetric,
                                                                                NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, true, false,
                                                                                useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                                enableStencilBufferColor, blackOrWhiteBackground,useFog, false, 1.0f, shaderTimer,
                                                                                useGBuffer);
      }
      else if (currentGeometryShader == CREATE_NORMAL_GEOMETRY)
      {
        openGLShaderSurfaceLightingLODModelsWithGeometryAndNormals->useProgramAndUniforms(currentLODModel, tessellationAlpha, tessellationLevel,
                                                                                          useAdaptiveTessellation, useTrianglePositionAdaptiveTessellationMetric, useLightAdaptiveTessellationMetric, useContourAdaptiveTessellationMetric,
                                                                                          NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, false, true,
                                                                                          useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                                          enableStencilBufferColor, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                                          useGBuffer);
      }
    }
    else
    {
      if (currentGeometryShader == NO_USE)
      {
        openGLShaderSurfaceLightingModels->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, false, false,
                                                                 useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                 enableStencilBufferColor, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                 useGBuffer);
      }
      else if (currentGeometryShader == PASS_THRU)
      {
        openGLShaderSurfaceLightingModelsWithGeometry->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, true, false,
                                                                             useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                             enableStencilBufferColor, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                             useGBuffer);
      }
      else if (currentGeometryShader == CREATE_NORMAL_GEOMETRY)
      {
        openGLShaderSurfaceLightingModelsWithGeometryAndNormals->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, useFresnelFactorSchlickApproximationSpecularModel, USE_COLOR_MATERIAL, 1.0f, false, true,
                                                                                       useTexturing, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, false,
                                                                                       enableStencilBufferColor, blackOrWhiteBackground, useFog, false, 1.0f, shaderTimer,
                                                                                       useGBuffer);
      }
    }
  }

  if (wireframe)
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE); // set wireframe

  if ((currentCachedRenderingTest == USE_NO_CACHING_METHOD_1) || (currentCachedRenderingTest == USE_NO_CACHING_METHOD_2))
  {
    renderNoCaching(false, !enableStencilBufferColor, useGBuffer);
  }
  else if (currentCachedRenderingTest == USE_VERTEX_ARRAYS)
  {
    renderVertexArrays(!enableStencilBufferColor, useGBuffer);
  }
  else if (currentCachedRenderingTest == USE_DISPLAY_LISTS)
  {
    renderDisplayLists(!enableStencilBufferColor, useGBuffer);
  }
  else // if (currentCachedRenderingTest == USE_VBOS)
  {
    renderVBOs(!enableStencilBufferColor, useGBuffer);
  }

  if (wireframe)
  {
    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL); // reset wireframe state
  }

  if (currentLightingModel != FIXED_PIPELINE)
  {
    if (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader() && useLODModels)
    {
      if (currentGeometryShader == NO_USE)
      {
        openGLShaderSurfaceLightingLODModels->disableShaderProgram();
      }
      else if (currentGeometryShader == PASS_THRU)
      {
        openGLShaderSurfaceLightingLODModelsWithGeometry->disableShaderProgram();
      }
      else if (currentGeometryShader == CREATE_NORMAL_GEOMETRY)
      {
        openGLShaderSurfaceLightingLODModelsWithGeometryAndNormals->disableShaderProgram();
      }
    }
    else
    {
      if (currentGeometryShader == NO_USE)
      {
        openGLShaderSurfaceLightingModels->disableShaderProgram();
      }
      else if (currentGeometryShader == PASS_THRU)
      {
        openGLShaderSurfaceLightingModelsWithGeometry->disableShaderProgram();
      }
      else if (currentGeometryShader == CREATE_NORMAL_GEOMETRY)
      {
        openGLShaderSurfaceLightingModelsWithGeometryAndNormals->disableShaderProgram();
      }
    }
  }
  if (openGLDriverInfo->isAMDATI())
  {
    disableLights(); // make sure lights are disabled for AMD-ATI hardware
  }

  if (useTexturing)
  {
    if (useSphericalMapping && (currentLightingModel == FIXED_PIPELINE))
    {
      glDisable(GL_TEXTURE_GEN_S);
      glDisable(GL_TEXTURE_GEN_T);
    }
    glDisable(GL_TEXTURE_2D);
  }

  if (multisample && (currentGBUfferEffectType != NO_GBUFFER_EFFECT))
  {
    glEnable(GL_MULTISAMPLE);
  }
}

bool SuperQuadricsTest::isUsingLOD() const
{
  return (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader() && useLODModels && (currentLightingModel != FIXED_PIPELINE));
}

/*
*  Draws a string in 2D space using GLUT.
*  The projection matrix must be set to orthogonal before call this function.
*/
void SuperQuadricsTest::drawString(const char* str, int x, int y, const GLfloat color[4], void* font) const
{
  glColor4fv(color);   // set text color
  glRasterPos2i(x, y); // place text position

  // loop all characters in the string
  while (*str)
  {
    glutBitmapCharacter(font, *str);
    ++str;
  }
}

/*
*  Draws a string in 3D space using GLUT.
*/
void SuperQuadricsTest::drawString3D(const char *str, float position[3], const GLfloat color[4], void* font) const
{
  glColor4fv(color);        // set text color
  glRasterPos3fv(position); // place text position

  // loop all characters in the string
  while (*str)
  {
    glutBitmapCharacter(font, *str);
    ++str;
  }
}

void SuperQuadricsTest::showInfo()
{
  if (reInitUIInformation)
  {
    // just to clear VRAM memory
    if (glIsList(useUIInformationDisplayList))
    {
      glDeleteLists(useUIInformationDisplayList, 1);
    }
    glNewList(useUIInformationDisplayList, GL_COMPILE);

    glPushAttrib(GL_LIGHTING_BIT | GL_CURRENT_BIT | GL_DEPTH_BUFFER_BIT); // push all relevant attribs
    glDisable(GL_LIGHTING); // need to disable lighting for proper text color
    glDisable(GL_DEPTH_TEST);

    // safety restore-default-texture GL state commands
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, 0);
    glDisable(GL_TEXTURE_2D);

    // backup current model-view matrix
    glPushMatrix();   // save current modelview matrix
    glLoadIdentity(); // reset modelview matrix

    // set to 2D orthogonal projection
    glMatrixMode(GL_PROJECTION); // switch to projection matrix
    glPushMatrix();   // save current projection matrix
    glLoadIdentity(); // reset projection matrix
    gluOrtho2D(0, screenWidth, 0, screenHeight); // set to orthogonal projection

    const GLfloat* color = (blackOrWhiteBackground) ? BLACK_BACKGROUND_COLOR.data() : WHITE_BACKGROUND_COLOR.data();
    ostringstream ss;

    ss << "Number of Rendered Triangles: " << vertices.size() / 3 << ends; // add 0(ends) at the end
    drawString(ss.str().c_str(), (screenWidth - 33 * GLUT_TEXT_WIDTH) / 2, screenHeight - GLUT_TEXT_HEIGHT, color, getFont());
    ss.str(""); // clear buffer

    ss << "Number of Rendered Vertices:  " << vertices.size() << ends;
    drawString(ss.str().c_str(), (screenWidth - 33 * GLUT_TEXT_WIDTH) / 2, screenHeight - (2 * GLUT_TEXT_HEIGHT), color, getFont());
    ss.str(""); // clear buffer

    ss << "Tessellation Level:           " << slices << ends;
    drawString(ss.str().c_str(), (screenWidth - 33 * GLUT_TEXT_WIDTH) / 2, screenHeight - (3 * GLUT_TEXT_HEIGHT), color, getFont());
    ss.str(""); // clear buffer

    ss << "Geometry Repetition:          " << geometryRepetition << ends;
    drawString(ss.str().c_str(), (screenWidth - 33 * GLUT_TEXT_WIDTH) / 2, screenHeight - (4 * GLUT_TEXT_HEIGHT), color, getFont());
    ss.str(""); // clear buffer

    ss << "Current Video Memory Usage:   " << howMuchCurrentlyAvailableVRAMMemoryString() << ends;
    drawString(ss.str().c_str(), (screenWidth - 33 * GLUT_TEXT_WIDTH) / 2, screenHeight - (5 * GLUT_TEXT_HEIGHT), color, getFont());
    ss.str(""); // clear buffer

    ss << "Memory difference from last Test Scenario: " << abs(previousAvailableMemory - howMuchCurrentlyAvailableVRAMMemory()) << ends;
    drawString(ss.str().c_str(), (screenWidth - 33 * GLUT_TEXT_WIDTH) / 2, screenHeight - (6 * GLUT_TEXT_HEIGHT), color, getFont());
    ss.str(""); // clear buffer

    ss << "VSynch: " << (vsynch ? "on" : "off") << ends;
    drawString(ss.str().c_str(), screenWidth - 16 * GLUT_TEXT_WIDTH, 2 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "AutoRotate: " << (autoRotate ? "on" : "off") << ends;
    drawString(ss.str().c_str(), screenWidth - 16 * GLUT_TEXT_WIDTH, GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press F1-F2-F3-F4-F5 for Testing Scenarios" << ends;
    drawString(ss.str().c_str(), 1, screenHeight - GLUT_TEXT_HEIGHT, color, getFont());
    ss.str(""); // clear buffer

    ss << "CurrentVBOTest: " << ((currentCachedRenderingTest == USE_NO_CACHING_METHOD_1) ? "USE NO CACHING METHOD 1" : ((currentCachedRenderingTest == USE_NO_CACHING_METHOD_2) ? "USE NO CACHING METHOD 2" : ((currentCachedRenderingTest == USE_VERTEX_ARRAYS) ? "USE VERTEX ARRAYS" : ((currentCachedRenderingTest == USE_DISPLAY_LISTS)) ? "USE DISPLAY LISTS" : "USE VBOS"))) << ends;
    drawString(ss.str().c_str(), 1, screenHeight - ( 2 * GLUT_TEXT_HEIGHT), color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 1-2-3-4-5-6 for Testing Lighting Models" << ends;
    drawString(ss.str().c_str(), 1, screenHeight - ( 3 * GLUT_TEXT_HEIGHT) - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "CurrentLightingModel: " << ((currentLightingModel == FIXED_PIPELINE) ? "FIXED PIPELINE" : ((currentLightingModel == PHONG) ? "PHONG" : ((currentLightingModel == BLINN_PHONG) ? "BLINN_PHONG" : ((currentLightingModel == GAUSSIAN) ? "GAUSSIAN" : ((currentLightingModel == TOON)) ? "TOON" : "GOOCH")))) << ends;
    drawString(ss.str().c_str(), 1, screenHeight - ( 4 * GLUT_TEXT_HEIGHT) - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'l' key for " << ((usePositionalLights) ? "Positional Lights" : "Directional Lights") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - ( 5 * GLUT_TEXT_HEIGHT) - 10, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'd' key to toggle the Oren Nayar " << (!useColoredObjects ? "& Fresnel Models " : "Model ") << ((useOrenNayarDiffuseModel) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - ( 6 * GLUT_TEXT_HEIGHT) - 15, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press F6 for Scene Motion Blur" << ends;
    drawString(ss.str().c_str(), 1, screenHeight - ( 7 * GLUT_TEXT_HEIGHT) - 15, color, getFont());
    ss.str(""); // clear buffer

    ss << "Motion Blur: " << (useMotionBlurForScene ? "ACCUM BUFFER" : "NO USE") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - ( 8 * GLUT_TEXT_HEIGHT) - 15, color, getFont());
    ss.str(""); // clear buffer

    if (useMotionBlurForScene)
    {
      ss << "Motion Blur Size: " << motionBlurSize << ends;
    }
    else
    {
      ss << "Motion Blur Size: -" << ends;
    }
    drawString(ss.str().c_str(), 1, screenHeight - ( 9 * GLUT_TEXT_HEIGHT) - 15, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press F7-to-F12 for Testing GS & Tess LOD Performance" << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (10 * GLUT_TEXT_HEIGHT) - 20, color, getFont());
    ss.str(""); // clear buffer

    ss << "CurrentGeometryShader: " << ((openGLDriverInfo->supports330Shaders() && openGLDriverInfo->supports_GL_ARB_geometry_shader4()) ? ((currentGeometryShader == NO_USE) ? "NO USE" : ((currentGeometryShader == PASS_THRU) ? "PASS THRU" : "CREATE NORMAL GEOMETRY")) : "Not Supported!") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (11 * GLUT_TEXT_HEIGHT) - 20, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press '9' & '0' for Tess LOD Model" << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (12 * GLUT_TEXT_HEIGHT) - 25, color, getFont());
    ss.str(""); // clear buffer

    ss << "CurrentLODModel: " << ((openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader()) ? ((!useLODModels) ? "NO USE" : (((currentLODModel == PN_TRIANGLES) && useLODModels) ? "PN TRIANGLES" : "PHONG TESSELLATION")) : "Not Supported!") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (13 * GLUT_TEXT_HEIGHT) - 25, color, getFont());
    ss.str(""); // clear buffer

    if (useLODModels)
    {
      ss << "TessellationLevel: " << tessellationLevel << ends;
    }
    else
    {
      ss << "TessellationLevel: -" << ends;
    }
    drawString(ss.str().c_str(), 1, screenHeight - (14 * GLUT_TEXT_HEIGHT) - 25, color, getFont());
    ss.str(""); // clear buffer

    if (useLODModels)
    {
      ss << "TessellationAlpha: " << tessellationAlpha << ends;
    }
    else
    {
      ss << "TessellationAlpha: -" << ends;
    }
    drawString(ss.str().c_str(), 1, screenHeight - (15 * GLUT_TEXT_HEIGHT) - 25, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 't' key to toggle Texturing " << ((useTexturing) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (16 * GLUT_TEXT_HEIGHT) - 30, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 's' key to toggle Spherical Mapping " << ((useSphericalMapping) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (17 * GLUT_TEXT_HEIGHT) - 30, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'j' key to toggle Outline " << ((useOutline) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (18 * GLUT_TEXT_HEIGHT) - 30, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'k' key to toggle High Quality Outline " << ((useHighQualityOutline) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (19 * GLUT_TEXT_HEIGHT) - 30, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press '7' key to lower  Tessellation Alpha " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (20 * GLUT_TEXT_HEIGHT) - 30, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press '8' key to refine Tessellation Alpha " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (21 * GLUT_TEXT_HEIGHT) - 30, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press '[' key to lower  Motion Blur Size " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (22 * GLUT_TEXT_HEIGHT) - 30, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press ']' key to refine Motion Blur Size " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (23 * GLUT_TEXT_HEIGHT) - 30, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'n' key to lower  Tessellation Level " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (24 * GLUT_TEXT_HEIGHT) - 30, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'm' key to refine Tessellation Level " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (25 * GLUT_TEXT_HEIGHT) - 30, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'space' key for AutoLOD " << ((useAdaptiveTessellation) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (26 * GLUT_TEXT_HEIGHT) - 30, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'alt' key to toggle GBuffer Effects: " << ((currentLightingModel != FIXED_PIPELINE) ? (currentGBUfferEffectType == DEPTH_OF_FIELD) ? "DEPTH-OF-FIELD" : (currentGBUfferEffectType == EDGE_ENHANCEMENT) ? "EDGE ENHANCEMENT" : (currentGBUfferEffectType == HSSAO) ? "HSSAO" : (currentGBUfferEffectType == HSSAO_PLUS) ? "HSSAO PLUS" : "NONE" : "NONE") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (27 * GLUT_TEXT_HEIGHT) - 35, color, getFont());
    ss.str(""); // clear buffer

    if ((currentLightingModel != FIXED_PIPELINE) && (currentGBUfferEffectType == DEPTH_OF_FIELD))
    {
      ss << "Press 'Page Up'   key to refine BlurXY Sample Range " << ends;
      drawString(ss.str().c_str(), 1, screenHeight - (30 * GLUT_TEXT_HEIGHT) - 35, color, getFont());
      ss.str(""); // clear buffer

      ss << "Press 'Page Down' key to lower  BlurXY Sample Range " << ends;
      drawString(ss.str().c_str(), 1, screenHeight - (31 * GLUT_TEXT_HEIGHT) - 35, color, getFont());
      ss.str(""); // clear buffer

      ss << "Press 'Up'     key to refine Depth-Of-Field Range " << ends;
      drawString(ss.str().c_str(), 1, screenHeight - (32 * GLUT_TEXT_HEIGHT) - 35, color, getFont());
      ss.str(""); // clear buffer

      ss << "Press 'Down'   key to lower  Depth-Of-Field Range " << ends;
      drawString(ss.str().c_str(), 1, screenHeight - (33 * GLUT_TEXT_HEIGHT) - 35, color, getFont());
      ss.str(""); // clear buffer

      ss << "Press 'Right' key to refine Depth-Of-Field Z Focus " << ends;
      drawString(ss.str().c_str(), 1, screenHeight - (34 * GLUT_TEXT_HEIGHT) - 35, color, getFont());
      ss.str(""); // clear buffer

      ss << "Press 'Left'  key to lower  Depth-Of-Field Z Focus " << ends;
      drawString(ss.str().c_str(), 1, screenHeight - (35 * GLUT_TEXT_HEIGHT) - 35, color, getFont());
      ss.str(""); // clear buffer
    }

    ss << "Press 'f' key to toggle Fog " << ((useFog) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, 16 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'g' key to toggle Gouraud Shading " << ((useGouraudShading) ? "off" : "on") << " (Fixed Pipeline only)" << ends;
    drawString(ss.str().c_str(), 1, 15 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'c' key to toggle Colored Objects " << ((useColoredObjects) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, 14 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'e' key to toggle Geometry Explosion " << ((useGeometryExplosion) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, 13 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'o' key to lower  Geometry Repetition levels" << ends;
    drawString(ss.str().c_str(), 1, 12 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'p' key to refine Geometry Repetition levels" << ends;
    drawString(ss.str().c_str(), 1, 11 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'z' key to lower  Super Quadrics tessellation levels" << ends;
    drawString(ss.str().c_str(), 1, 10 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'x' key to refine Super Quadrics tessellation levels" << ends;
    drawString(ss.str().c_str(), 1,  9 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'b' key to toggle " << ((blackOrWhiteBackground) ? "Black" : "White") << " background" << ends;
    drawString(ss.str().c_str(), 1,  8 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'w' key to toggle Wireframe " << ((wireframe) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1,  7 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'a' key to toggle AutoRotate " << ((autoRotate) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1,  6 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'r' key to reset  the camera" << ends;
    drawString(ss.str().c_str(), 1,  5 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'v' key to toggle VSynch " << ((vsynch) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1,  4 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'u' key to toggle UI Information " << ((useUIInformation) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1,  3 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'Insert' key to toggle FXAA Antialias " << ((useFXAA_Antialias) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1,  2 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'Shift'  key to Take Screenshot " << ends;
    drawString(ss.str().c_str(), 1, GLUT_TEXT_HEIGHT - 5, color, getFont());

    // unset floating format
    ss << resetiosflags(ios_base::fixed | ios_base::floatfield);

    // restore projection matrix
    glPopMatrix(); // restore to previous projection matrix

    // restore modelview matrix
    glMatrixMode(GL_MODELVIEW); // switch to modelview matrix
    glPopMatrix();              // restore to previous modelview matrix

    glPopAttrib(); // pop all relevant attribs

    glEndList();

    reInitUIInformation = false;
  }

  glCallList(useUIInformationDisplayList);
}

int SuperQuadricsTest::howMuchCurrentlyAvailableVRAMMemory() const
{
  openGLDriverInfo->getGLMemoryInfo();
  return (openGLDriverInfo->isNvidia()) ? (openGLDriverInfo->getGPUMemoryInfoCurrentAvailableMemoryNVX() >> 10) : ((openGLDriverInfo->isAMDATI()) ? (openGLDriverInfo->getVBOFreeMemoryATI()[1] >> 10) : 0);
}

string SuperQuadricsTest::howMuchCurrentlyAvailableVRAMMemoryString() const
{
  const int howMuchAvailableVRAMMemoryInt = howMuchCurrentlyAvailableVRAMMemory();
  return (howMuchAvailableVRAMMemoryInt) ? StringAuxiliaryFunctions::toString<int>(howMuchAvailableVRAMMemoryInt) + " Mb VRAM" : "No information available for this GL vendor";
}

void SuperQuadricsTest::showFPS()
{
  // update fpsString every second
  const double elapsedTime = timer.getElapsedTimeInSecs();
  if (elapsedTime < 1.0)
  {
    ++fpsCounter;
  }
  else
  {
    ++fpsCounter;

    ostringstream ss;
    ss << fixed << setprecision(1);
    // ss << ((double)fpsCounter / elapsedTime) << " FPS" << ends; // update fpsString string
    ss << double(fpsCounter) << " FPS" << ends; // update fpsString string
    ss << resetiosflags(ios_base::fixed | ios_base::floatfield);
    fpsString = ss.str();
    fpsCounter = 0; // reset counter
    timer.startTimer();  // restart timer

    DebugConsole_consoleOutLine("Current framerate: ", fpsString.c_str());
  }

  // safety restore-default-texture GL state commands
  glActiveTexture(GL_TEXTURE0);
  glBindTexture(GL_TEXTURE_2D, 0);
  glDisable(GL_TEXTURE_2D);

  glPushAttrib(GL_LIGHTING_BIT | GL_CURRENT_BIT | GL_DEPTH_BUFFER_BIT); // push all relevant attribs
  glDisable(GL_LIGHTING); // need to disable lighting for proper text color
  glDisable(GL_TEXTURE_2D);
  glDisable(GL_DEPTH_TEST);

  // backup current model-view matrix
  glPushMatrix();   // save current modelview matrix
  glLoadIdentity(); // reset modelview matrix

  // set to 2D orthogonal projection
  glMatrixMode(GL_PROJECTION); // switch to projection matrix
  glPushMatrix();   // save current projection matrix
  glLoadIdentity(); // reset projection matrix
  gluOrtho2D(0, screenWidth, 0, screenHeight); // set to orthogonal projection

  const int textWidth  = int(fpsString.size() * GLUT_TEXT_WIDTH);
  const GLfloat* color = (blackOrWhiteBackground) ? BLACK_BACKGROUND_COLOR.data() : WHITE_BACKGROUND_COLOR.data();

  drawString(fpsString.c_str(), screenWidth - textWidth, screenHeight - GLUT_TEXT_HEIGHT, color, getFont());

  // restore projection matrix
  glPopMatrix(); // restore to previous projection matrix

  // restore modelView matrix
  glMatrixMode(GL_MODELVIEW); // switch to modelview matrix
  glPopMatrix();              // restore to previous modelview matrix

  glPopAttrib(); // pop all relevant attribs
}

void SuperQuadricsTest::renderScene()
{
  if (autoRotate)
  {
    cameraAngleX += 1.0;
    cameraAngleY += 1.0;
  }

  if (++printGLMemoryInfoOnSecondUpdate == 2)
  {
    openGLDriverInfo->getGLMemoryInfo();
    openGLDriverInfo->printGLMemoryInfo();
  }

  GLAuxiliaryFunctions::setVSynch(vsynch, openGLDriverInfo->isVSynchSupported());

  if (openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders() &&
      !(useOutline || useHighQualityOutline) && (currentLightingModel != FIXED_PIPELINE) && (currentGBUfferEffectType != NO_GBUFFER_EFFECT))
  {
    processGBufferEffectsFBOs();
  }

  if (openGLDriverInfo->supports_GL_EXT_framebuffer_object() && !useHighQualityOutline && useFXAA_Antialias)
  {
    drawFXAA_AntialiasFBO();
  }
  else
  {
    if (openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders() &&
        !(useOutline || useHighQualityOutline) && (currentLightingModel != FIXED_PIPELINE) && (currentGBUfferEffectType != NO_GBUFFER_EFFECT))
    {
      drawGBufferEffectsFBOs();
    }
    else
    {
      renderSuperQuadricsScene();
    }
  }

  // draw info messages
  if (useUIInformation)
  {
    showInfo();
  }
  showFPS();

  // Take the contents of the current draw buffer and copy it to the accumulation buffer with each pixel modified by a factor
  // The closer the factor is to 1.0f, the longer the trails... Don't exceed 1.0f - you get garbage.
  if (!openGLDriverInfo->isAMDATI()) // recent AMD/ATI cards lack an accumulation buffer
  {
    if (useMotionBlurForScene)
    {
      // Clear the accumulation buffer (we re-grab the screen into the accumulation buffer after drawing our current frame!)
      if (reInitMotionBlurForScene)
      {
        if (blackOrWhiteBackground)
        {
          glClearAccum(WHITE_BACKGROUND_COLOR[0], WHITE_BACKGROUND_COLOR[1], WHITE_BACKGROUND_COLOR[2], WHITE_BACKGROUND_COLOR[3]);
        }
        else
        {
          glClearAccum(BLACK_BACKGROUND_COLOR[0], BLACK_BACKGROUND_COLOR[1], BLACK_BACKGROUND_COLOR[2], BLACK_BACKGROUND_COLOR[3]);
        }
        glClear(GL_ACCUM_BUFFER_BIT);
        glAccum(GL_LOAD, 1.0f);
        reInitMotionBlurForScene = false;
      }

      glAccum(GL_MULT, motionBlurSize);
      glAccum(GL_ACCUM, 1.0f - motionBlurSize);
      glAccum(GL_RETURN, 1.0f);
    }
  }

  if (takeScreenshot)
  {
    writeScreenshotToFile();
    takeScreenshot = false;
  }

  glFlush();
  glFinish();
  glutSwapBuffers();

  const GLenum glError = glGetError();
  if (glError != GL_NO_ERROR)
  {
    const GLubyte* gluError = gluErrorString(glError);
    DebugConsole_consoleOutLine("renderScene() glGetError() returned: ", gluError);
  }
}

void SuperQuadricsTest::changeSize(int w, int h)
{
  if (h == 0) // prevent divide by zero
  {
    h = 1;
  }

  const GLfloat ratio = 1.0f * GLfloat(w) / GLfloat(h);
  screenWidth  = w;
  screenHeight = h;
  DebugConsole_consoleOutLine("New screenWidth: ",  screenWidth);
  DebugConsole_consoleOutLine("New screenHeight: ", screenHeight, "\n");

  glViewport(0, 0, screenWidth, screenHeight);

  GLAuxiliaryFunctions::setVSynch(vsynch, openGLDriverInfo->isVSynchSupported());

  openGLEulerCamera->setRatio(ratio);
  openGLEulerCamera->setMatrices();

  if (openGLDriverInfo->supports_GL_EXT_framebuffer_object())
  {
    if (openGLDriverInfo->supports_GL_EXT_packed_depth_stencil())
    {
      initDepthStencilFBO();
    }
    if (openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders())
    {
      initGBufferEffectsFBOs();
    }
    initFXAA_AntialiasFBO();
  }

  if (!openGLDriverInfo->isAMDATI()) // recent AMD/ATI cards lack an accumulation buffer
  {
    // Clear the accumulation buffer (we re-grab the screen into the accumulation buffer after drawing our current frame!)
    if (blackOrWhiteBackground)
    {
      glClearAccum(WHITE_BACKGROUND_COLOR[0], WHITE_BACKGROUND_COLOR[1], WHITE_BACKGROUND_COLOR[2], WHITE_BACKGROUND_COLOR[3]);
    }
    else
    {
      glClearAccum(BLACK_BACKGROUND_COLOR[0], BLACK_BACKGROUND_COLOR[1], BLACK_BACKGROUND_COLOR[2], BLACK_BACKGROUND_COLOR[3]);
    }
    glClear(GL_ACCUM_BUFFER_BIT);
    glAccum(GL_LOAD, 1.0f);
  }

  glFlush();
  glFinish();

  const GLenum glError = glGetError();
  if (glError != GL_NO_ERROR)
  {
    const GLubyte* gluError = gluErrorString(glError);
    DebugConsole_consoleOutLine("changeSize() glGetError() returned: ", gluError);
  }
}

void SuperQuadricsTest::keyboard(unsigned char key, int x, int y)
{
  if ((key == char('v')) || (key == char('V')))
  {
    vsynch++;
    vsynch = vsynch & 1;
    DebugConsole_consoleOutLine("Toggle VSynch: ", vsynch ? "on" : "off");
  }
  else if ((key == char('a')) || (key == char('A')))
  {
    autoRotate++;
    autoRotate = autoRotate & 1;
    DebugConsole_consoleOutLine("Toggle AutoRotate: ", autoRotate ? "on" : "off");
  }
  else if ((key == char('r')) || (key == char('R')))
  {
    cameraAngleX = cameraAngleY = cameraDistanceX = cameraDistanceY = 0.0;
    cameraDistanceZ = 15.0;
    DebugConsole_consoleOutLine("Camera reset.");
  }
  else if ((key == char('b')) || (key == char('B')))
  {
    blackOrWhiteBackground = !blackOrWhiteBackground;
    DebugConsole_consoleOutLine("Toggle BlackOrWhiteBackground: ", blackOrWhiteBackground ? "on" : "off");
  }
  else if ((key == char('w')) || (key == char('W')))
  {
    wireframe++;
    wireframe = wireframe & 1;
    DebugConsole_consoleOutLine("Toggle Wireframe: ", wireframe ? "on" : "off");
  }
  else if ((key == char('z')) || (key == char('Z')))
  {
    slices--;
    segments = slices;
    if (slices < 3)
    {
      slices = segments = 3;
    }
    reInitSuperQuadricsDataForRendering = true;
    DebugConsole_consoleOutLine("Slices: ", slices, " Segments: ", segments);
  }
  else if ((key == char('x')) || (key == char('X')))
  {
    slices++;
    segments = slices;
    if (slices > UPPER_LIMIT_TESSELLATION)
    {
      slices = segments = UPPER_LIMIT_TESSELLATION;
    }
    reInitSuperQuadricsDataForRendering = true;
    DebugConsole_consoleOutLine("Slices: ", slices, " Segments: ", segments);
  }
  else if ((key == char('o')) || (key == char('O')))
  {
    geometryRepetition--;
    if (geometryRepetition < 1)
    {
      geometryRepetition = 1;
    }
    reInitSuperQuadricsDataForRendering = true;
    DebugConsole_consoleOutLine("Geometry repetition: ", geometryRepetition);
  }
  else if ((key == char('p')) || (key == char('P')))
  {
    geometryRepetition++;
    if (geometryRepetition > UPPER_LIMIT_GEOMETRY_REPETITION)
    {
      geometryRepetition = UPPER_LIMIT_GEOMETRY_REPETITION;
    }
    reInitSuperQuadricsDataForRendering = true;
    DebugConsole_consoleOutLine("Geometry repetition: ", geometryRepetition);
  }
  else if ((key == char('c')) || (key == char('C')))
  {
    useColoredObjects++;
    useColoredObjects = useColoredObjects & 1;
    useFresnelFactorSchlickApproximationSpecularModel = (useOrenNayarDiffuseModel && !useColoredObjects);
    reInitSuperQuadricsDataForRendering = true;
    DebugConsole_consoleOutLine("Toggle Colored Objects: ", useColoredObjects ? "on" : "off");
  }
  else if ((key == char('e')) || (key == char('E')))
  {
    useGeometryExplosion++;
    useGeometryExplosion = useGeometryExplosion & 1;
    reInitSuperQuadricsDataForRendering = true;
    DebugConsole_consoleOutLine("Toggle Geometry Explosion: ", useGeometryExplosion ? "on" : "off");
  }
  else if ((key == char('g')) || (key == char('G')))
  {
    useGouraudShading++;
    useGouraudShading = useGouraudShading & 1;
    DebugConsole_consoleOutLine("Toggle Gouraud Shading: ", useGouraudShading ? "on" : "off");
  }
  else if ((key == char('u')) || (key == char('U')))
  {
    useUIInformation++;
    useUIInformation = useUIInformation & 1;
    DebugConsole_consoleOutLine("Toggle UI Information Display: ", useUIInformation ? "on" : "off");
  }
  else if (key == char('1'))
  {
    reInitSuperQuadricsDataForRendering = true;
    currentLightingModel = FIXED_PIPELINE;
    if ((useOutline || useHighQualityOutline))
    {
      useOutline = false;
      useHighQualityOutline = false;
    }
    DebugConsole_consoleOutLine("currentLightingModel: FIXED_PIPELINE");
  }
  else if (key == char('2'))
  {
    reInitSuperQuadricsDataForRendering = true;
    currentLightingModel = PHONG;
    if ((useOutline || useHighQualityOutline))
    {
      useOutline = false;
      useHighQualityOutline = false;
    }
    DebugConsole_consoleOutLine("currentLightingModel: PHONG");
  }
  else if (key == char('3'))
  {
    reInitSuperQuadricsDataForRendering = true;
    currentLightingModel = BLINN_PHONG;
    if ((useOutline || useHighQualityOutline))
    {
      useOutline = false;
      useHighQualityOutline = false;
    }
    DebugConsole_consoleOutLine("currentLightingModel: BLINN_PHONG");
  }
  else if (key == char('4'))
  {
    reInitSuperQuadricsDataForRendering = true;
    currentLightingModel = GAUSSIAN;
    if ((useOutline || useHighQualityOutline))
    {
      useOutline = false;
      useHighQualityOutline = false;
    }
    DebugConsole_consoleOutLine("currentLightingModel: GAUSSIAN");
  }
  else if (key == char('5'))
  {
    reInitSuperQuadricsDataForRendering = true;
    currentLightingModel = TOON;
    if (!(useOutline || useHighQualityOutline))
    {
      useOutline = true;
      useHighQualityOutline = false;
    }
    DebugConsole_consoleOutLine("currentLightingModel: TOON");
  }
  else if (key == char('6'))
  {
    reInitSuperQuadricsDataForRendering = true;
    currentLightingModel = GOOCH;
    if (!(useOutline || useHighQualityOutline))
    {
      useOutline = true;
      useHighQualityOutline = false;
    }
    DebugConsole_consoleOutLine("currentLightingModel: GOOCH");
  }
  else if (key == char('7'))
  {
    if (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader() && useLODModels)
    {
      if (tessellationAlpha >= 0.05f)
      {
        tessellationAlpha -= 0.05f;
      }
      else
      {
        tessellationAlpha = 0.0f;
      }
      DebugConsole_consoleOutLine("tessellationAlpha: ", tessellationAlpha);
    }
  }
  else if (key == char('8'))
  {
    if (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader() && useLODModels)
    {
      if (tessellationAlpha <= 0.95f)
      {
        tessellationAlpha += 0.05f;
      }
      else
      {
        tessellationAlpha = 1.0f;
      }
      DebugConsole_consoleOutLine("tessellationAlpha: ", tessellationAlpha);
    }
  }
  else if (key == char('9'))
  {
    if (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader())
    {
      currentLODModel = PN_TRIANGLES;
      DebugConsole_consoleOutLine("currentLODModel: PN_TRIANGLES");
    }
  }
  else if (key == char('0'))
  {
    if (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader())
    {
      currentLODModel = PHONG_TESSELLATION;
      DebugConsole_consoleOutLine("currentLODModel: PHONG_TESSELLATION");
    }
  }
  else if ((key == char('l')) || (key == char('L')))
  {
    usePositionalLights++;
    usePositionalLights = usePositionalLights & 1;
    DebugConsole_consoleOutLine("Toggle Lights: ", usePositionalLights ? "Positional Lights" : "Directional Lights");
  }
  else if ((key == char('d')) || (key == char('D')))
  {
    useOrenNayarDiffuseModel++;
    useOrenNayarDiffuseModel = useOrenNayarDiffuseModel & 1;
    useFresnelFactorSchlickApproximationSpecularModel = (useOrenNayarDiffuseModel && !useColoredObjects);
    DebugConsole_consoleOutLine("Toggle Oren Nayar Diffuse Model: ", useOrenNayarDiffuseModel ? "on" : "off");
  }
  else if ((key == char('f')) || (key == char('F')))
  {
    useFog++;
    useFog = useFog & 1;
    DebugConsole_consoleOutLine("Toggle Fog: ", useFog ? "on" : "off");
  }
  else if ((key == char('t')) || (key == char('T')))
  {
    if (useSphericalMapping)
    {
      useTexturing = useSphericalMapping = false;
    }
    useTexturing++;
    useTexturing = useTexturing & 1;
    if (!useTexturing)
    {
      useSphericalMapping = false;
    }
    DebugConsole_consoleOutLine("Toggle Texturing: ", useTexturing ? "on" : "off");
  }
  else if ((key == char('s')) || (key == char('S')))
  {
    useSphericalMapping++;
    useTexturing = useSphericalMapping = useSphericalMapping & 1;
    DebugConsole_consoleOutLine("Toggle Spherical Mapping: ", useSphericalMapping ? "on" : "off");
  }
  else if ((key == char('j')) || (key == char('J')))
  {
    useOutline++;
    useOutline = useOutline & 1;
    if (useOutline)
    {
      useHighQualityOutline = false;
    }
    DebugConsole_consoleOutLine("Toggle Outline: ", useOutline ? "on" : "off");
  }
  else if ((key == char('k')) || (key == char('K')))
  {
    if (openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_packed_depth_stencil())
    {
      useHighQualityOutline++;
      useHighQualityOutline = useHighQualityOutline & 1;
      if (useHighQualityOutline)
      {
        useOutline = false;
      }
      DebugConsole_consoleOutLine("Toggle High Quality Outline: ", useHighQualityOutline ? "on" : "off");
    }
  }
  else if (key == char('['))
  {
    if (!openGLDriverInfo->isAMDATI()) // recent AMD/ATI cards lack an accumulation buffer
    {
      if (motionBlurSize >= 0.10f)
      {
        motionBlurSize -= 0.05f;
      }
      else
      {
        motionBlurSize = 0.05f;
      }
      DebugConsole_consoleOutLine("motionBlurSize: ", motionBlurSize);
    }
  }
  else if (key == char(']'))
  {
    if (!openGLDriverInfo->isAMDATI()) // recent AMD/ATI cards lack an accumulation buffer
    {
      if (motionBlurSize <= 0.90f)
      {
        motionBlurSize += 0.05f;
      }
      else
      {
        motionBlurSize = 0.95f;
      }
      DebugConsole_consoleOutLine("motionBlurSize: ", motionBlurSize);
    }
  }
  else if ((key == char('n')) || (key == char('N')))
  {
    if (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader() && useLODModels)
    {
      if (tessellationLevel > 1.0f)
      {
        tessellationLevel -= 1.0f;
      }
      DebugConsole_consoleOutLine("tessellationLevel: ", tessellationLevel);
    }
  }
  else if ((key == char('m')) || (key == char('M')))
  {
    if (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader() && useLODModels)
    {
      if (tessellationLevel < 64.0f)
      {
        tessellationLevel += 1.0f;
      }
      DebugConsole_consoleOutLine("tessellationLevel: ", tessellationLevel);
    }
  }
  else if (key == char(' '))
  {
    if (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader() && useLODModels)
    {
      useAdaptiveTessellation = !useAdaptiveTessellation;
      DebugConsole_consoleOutLine("useAdaptiveTessellation: ", useAdaptiveTessellation ? "true" : "false");
    }
  }
  else if (key == 27)
  {
    DebugConsole_consoleOutLine("\nEscape pressed, now exiting Test: Super Quadrics Tests.\n");
    glutLeaveMainLoop();
  }

  reInitUIInformation = true;
}

void SuperQuadricsTest::specialKeysKeyboard(int key, int x, int y)
{
  if (glutGetModifiers() == GLUT_ACTIVE_SHIFT)
  {
    takeScreenshot = true;
    DebugConsole_consoleOutLine("takeScreenshot: ", takeScreenshot ? "yes" : "no");
  }
  else if (glutGetModifiers() == GLUT_ACTIVE_ALT)
  {
    if ((currentLightingModel != FIXED_PIPELINE) && openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders())
    {
      currentGBUfferEffectType = GBUfferEffectTypes((int(currentGBUfferEffectType) + 1) % 5);
      DebugConsole_consoleOutLine("Toggle GBuffer Effects: ", (currentGBUfferEffectType == DEPTH_OF_FIELD) ? "DEPTH-OF-FIELD" : (currentGBUfferEffectType == EDGE_ENHANCEMENT) ? "EDGE ENHANCEMENT" : (currentGBUfferEffectType == HSSAO) ? "HSSAO" : (currentGBUfferEffectType == HSSAO_PLUS) ? "HSSAO PLUS" : "NONE");
    }
  }
  else if (key == GLUT_KEY_UP)
  {
    if ((currentLightingModel != FIXED_PIPELINE) && (currentGBUfferEffectType == DEPTH_OF_FIELD) && openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders())
    {
      if (depthOfFieldRange <= 99.0f)
      {
        depthOfFieldRange += 1.0f;
      }
      else
      {
        depthOfFieldRange = 100.0f;
      }
      DebugConsole_consoleOutLine("depthOfFieldRange: ", depthOfFieldRange);
    }
  }
  else if (key == GLUT_KEY_DOWN)
  {
    if ((currentLightingModel != FIXED_PIPELINE) && (currentGBUfferEffectType == DEPTH_OF_FIELD) & openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders())
    {
      if (depthOfFieldRange >= 2.0f)
      {
        depthOfFieldRange -= 1.0f;
      }
      else
      {
        depthOfFieldRange = 1.0f;
      }
      DebugConsole_consoleOutLine("depthOfFieldRange: ", depthOfFieldRange);
    }
  }
  else if (key == GLUT_KEY_RIGHT)
  {
    if ((currentLightingModel != FIXED_PIPELINE) && (currentGBUfferEffectType == DEPTH_OF_FIELD) & openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders())
    {
      if (depthOfFieldZFocus <= 99.9f)
      {
        depthOfFieldZFocus += 0.1f;
      }
      else
      {
        depthOfFieldZFocus = 100.0f;
      }
      DebugConsole_consoleOutLine("depthOfFieldZFocus: ", depthOfFieldZFocus);
    }
  }
  else if (key == GLUT_KEY_LEFT)
  {
    if ((currentLightingModel != FIXED_PIPELINE) && (currentGBUfferEffectType == DEPTH_OF_FIELD) & openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders())
    {
      if (depthOfFieldZFocus >= 0.1f)
      {
        depthOfFieldZFocus -= 0.1f;
      }
      else
      {
        depthOfFieldZFocus = 0.0f;
      }
      DebugConsole_consoleOutLine("depthOfFieldZFocus: ", depthOfFieldZFocus);
    }
  }
  else if (key == GLUT_KEY_PAGE_UP)
  {
    if ((currentLightingModel != FIXED_PIPELINE) && (currentGBUfferEffectType == DEPTH_OF_FIELD) && openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders())
    {
      if (depthOfFieldBlurXYNumberOfSamplesRange <= 63)
      {
        depthOfFieldBlurXYNumberOfSamplesRange++;
      }
      DebugConsole_consoleOutLine("depthOfFieldBlurXYNumberOfSamplesRange: ", depthOfFieldBlurXYNumberOfSamplesRange);
    }
  }
  else if (key == GLUT_KEY_PAGE_DOWN)
  {
    if ((currentLightingModel != FIXED_PIPELINE) && (currentGBUfferEffectType == DEPTH_OF_FIELD) && openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders())
    {
      if (depthOfFieldBlurXYNumberOfSamplesRange >= 1)
      {
        depthOfFieldBlurXYNumberOfSamplesRange--;
      }
      DebugConsole_consoleOutLine("depthOfFieldBlurXYNumberOfSamplesRange: ", depthOfFieldBlurXYNumberOfSamplesRange);
    }
  }
  else if (key == GLUT_KEY_INSERT)
  {
    if (openGLDriverInfo->supports_GL_EXT_framebuffer_object())
    {
      useFXAA_Antialias = !useFXAA_Antialias;
      DebugConsole_consoleOutLine("Toggle FXAA Antialias: ", useFXAA_Antialias ? "on" : "off");
    }
  }
  else if (key == GLUT_KEY_F1)
  {
    reInitSuperQuadricsDataForRendering = true;
    currentCachedRenderingTest = USE_NO_CACHING_METHOD_1;
    DebugConsole_consoleOutLine("currentVBOTest: USE_NO_CACHING_METHOD_1");
  }
  else if (key == GLUT_KEY_F2)
  {
    reInitSuperQuadricsDataForRendering = true;
    currentCachedRenderingTest = USE_NO_CACHING_METHOD_2;
    DebugConsole_consoleOutLine("currentVBOTest: USE_NO_CACHING_METHOD_2");
  }
  else if (key == GLUT_KEY_F3)
  {
    reInitSuperQuadricsDataForRendering = true;
    currentCachedRenderingTest = USE_VERTEX_ARRAYS;
    DebugConsole_consoleOutLine("currentVBOTest: USE_VERTEX_ARRAYS");
  }
  else if (key == GLUT_KEY_F4)
  {
    reInitSuperQuadricsDataForRendering = true;
    currentCachedRenderingTest = USE_DISPLAY_LISTS;
    DebugConsole_consoleOutLine("currentVBOTest: USE_DISPLAY_LISTS");
  }
  else if (key == GLUT_KEY_F5)
  {
    reInitSuperQuadricsDataForRendering = true;
    currentCachedRenderingTest = USE_VBOS;
    DebugConsole_consoleOutLine("currentVBOTest: USE_VBOS");
  }
  else if (key == GLUT_KEY_F6)
  {
    if (!openGLDriverInfo->isAMDATI()) // recent AMD/ATI cards lack an accumulation buffer
    {
      useMotionBlurForScene = !useMotionBlurForScene;
      if (useMotionBlurForScene)
      {
        reInitMotionBlurForScene = true;
      }
      DebugConsole_consoleOutLine("useMotionBlurForScene: ", useMotionBlurForScene ? "ACCUM BUFFER" : "NO_USE");
    }
  }
  else if (key == GLUT_KEY_F7)
  {
    reInitSuperQuadricsDataForRendering = true;
    useLODModels = false;
    currentGeometryShader = NO_USE;
    DebugConsole_consoleOutLine("currentGeometryShader: NO_USE with LOD turned off.");
  }
  else if (key == GLUT_KEY_F8)
  {
    reInitSuperQuadricsDataForRendering = true;
    useLODModels = false;
    currentGeometryShader = PASS_THRU;
    DebugConsole_consoleOutLine("currentGeometryShader: PASS_THRU with LOD turned off.");
  }
  else if (key == GLUT_KEY_F9)
  {
    reInitSuperQuadricsDataForRendering = true;
    useLODModels = false;
    currentGeometryShader = CREATE_NORMAL_GEOMETRY;
    DebugConsole_consoleOutLine("currentGeometryShader: CREATE_NORMAL_GEOMETRY with LOD turned off.");
  }
  else if (key == GLUT_KEY_F10)
  {
    if (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader())
    {
      reInitSuperQuadricsDataForRendering = true;
      useLODModels = true;
      currentGeometryShader = NO_USE;
      DebugConsole_consoleOutLine("currentGeometryShader: NO_USE with LOD turned on.");
    }
  }
  else if (key == GLUT_KEY_F11)
  {
    if (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader())
    {
      reInitSuperQuadricsDataForRendering = true;
      useLODModels = true;
      currentGeometryShader = PASS_THRU;
      DebugConsole_consoleOutLine("currentGeometryShader: PASS_THRU with LOD turned on.");
    }
  }
  else if (key == GLUT_KEY_F12)
  {
    if (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader())
    {
      reInitSuperQuadricsDataForRendering = true;
      useLODModels = true;
      currentGeometryShader = CREATE_NORMAL_GEOMETRY;
      DebugConsole_consoleOutLine("currentGeometryShader: CREATE_NORMAL_GEOMETRY with LOD turned on.");
    }
  }

  reInitUIInformation = true;
}

void SuperQuadricsTest::mouse(int button, int state, int x, int y)
{
  mouseX = float(x);
  mouseY = float(y);

  if (button == GLUT_LEFT_BUTTON)
  {
    if (state == GLUT_DOWN)
    {
      mouseLeftDown = true;
    }
    else if (state == GLUT_UP)
    {
      mouseLeftDown = false;
    }
  }
  else if (button == GLUT_MIDDLE_BUTTON)
  {
    if (state == GLUT_DOWN)
    {
      mouseMiddleDown = true;
    }
    else if (state == GLUT_UP)
    {
      mouseMiddleDown = false;
    }
  }
  else if (button == GLUT_RIGHT_BUTTON)
  {
    if (state == GLUT_DOWN)
    {
      mouseRightDown = true;
    }
    else if (state == GLUT_UP)
    {
      mouseRightDown = false;
    }
  }
  else if ((button == 3) || (button == 4)) // It's a wheel event for FreeGlut
  {
    const double tempCameraDistanceZ = (cameraDistanceZ > 0.0) ? (1.0 + cameraDistanceZ / 10.0) : 1.0;
    if (button == 3)
    {
      cameraDistanceZ -= 0.15 * tempCameraDistanceZ;
    }
    else
    {
      cameraDistanceZ += 0.15 * tempCameraDistanceZ;
    }
  }
}

void SuperQuadricsTest::mouseMotion(int x, int y)
{
  if (mouseLeftDown)
  {
    cameraAngleY += (double(x) - mouseX);
    cameraAngleX -= (double(y) - mouseY);
    mouseX = double(x);
    mouseY = double(y);
  }
  else if (mouseMiddleDown)
  {
    const double tempCameraDistanceZ = (cameraDistanceZ > 0.0) ? (1.0 + cameraDistanceZ / 5.0) : 1.0;
    cameraDistanceX += 0.005 * (double(x) - mouseX) * tempCameraDistanceZ;
    cameraDistanceY -= 0.005 * (double(y) - mouseY) * tempCameraDistanceZ;
    mouseX = double(x);
    mouseY = double(y);
  }
  else if (mouseRightDown)
  {
    const double tempCameraDistanceZ = (cameraDistanceZ > 0.0) ? (1.0 + cameraDistanceZ / 10.0) : 1.0;
    cameraDistanceZ += 0.05 * (double(y) - mouseY) * tempCameraDistanceZ;
    mouseY = double(y);
  }
}

void SuperQuadricsTest::closeFunc()
{
  if (glIsList(allSuperQuadricsDataDisplayList))
  {
    glDeleteLists(allSuperQuadricsDataDisplayList, 1);
  }
  deleteVBOs();

  // for HSSAO rotationTexture data
  if (glIsTexture(rotationTextureID))
  {
    glDeleteTextures(1, &rotationTextureID);
    rotationTextureID = 0;
  }

  delete openGLShaderGBufferEffectsBlurXY;
  openGLShaderGBufferEffectsBlurXY = nullptr;
  delete openGLShaderGBufferEffectsHSSAO;
  openGLShaderGBufferEffectsHSSAO = nullptr;
  delete openGLShaderGBufferEffects;
  openGLShaderGBufferEffects = nullptr;
  delete openGLShaderFXAA_Antialias;
  openGLShaderFXAA_Antialias = nullptr;
  delete openGLModelAmbientLight;
  openGLModelAmbientLight = nullptr;
  for (auto& openGLLight : openGLLights)
  {
    delete openGLLight;
    openGLLight = nullptr;
  }
  delete openGLMaterial;
  openGLMaterial = nullptr;
  delete openGLShaderSurfaceLightingModels;
  openGLShaderSurfaceLightingModels = nullptr;
  delete openGLShaderSurfaceLightingModelsWithGeometry;
  openGLShaderSurfaceLightingModelsWithGeometry = nullptr;
  delete openGLShaderSurfaceLightingModelsWithGeometryAndNormals;
  openGLShaderSurfaceLightingModelsWithGeometryAndNormals = nullptr;
  if (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader())
  {
    delete openGLShaderSurfaceLightingLODModels;
    openGLShaderSurfaceLightingLODModels = nullptr;
    delete openGLShaderSurfaceLightingLODModelsWithGeometry;
    openGLShaderSurfaceLightingLODModelsWithGeometry = nullptr;
    delete openGLShaderSurfaceLightingLODModelsWithGeometryAndNormals;
    openGLShaderSurfaceLightingLODModelsWithGeometryAndNormals = nullptr;
  }
  if (openGLDriverInfo->supports_GL_EXT_framebuffer_object())
  {
    if (openGLDriverInfo->supports_GL_EXT_packed_depth_stencil())
    {
      delete openGLFrameBufferObjectForHighQualityOutline;
      openGLFrameBufferObjectForHighQualityOutline = nullptr;
    }
    if (openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders())
    {
      delete openGLFrameBufferObjectForGBufferPass1;
      openGLFrameBufferObjectForGBufferPass1 = nullptr;
      delete openGLFrameBufferObjectRenderPass2;
      openGLFrameBufferObjectRenderPass2 = nullptr;
      delete openGLFrameBufferObjectRenderPass3;
      openGLFrameBufferObjectRenderPass3 = nullptr;
    }
    delete openGLFrameBufferObjectForFXAA_Antialias;
    openGLFrameBufferObjectForFXAA_Antialias = nullptr;
  }
  delete openGLILTexture;
  openGLILTexture = nullptr;

  TestAbstractBase::releaseAllGLResources();
}