#include "ParticleRenderingModelLoaderTest.h"
#include "OpenGLAssetManager.h"
#include "OpenGLDriverInfo.h"
#include "OpenGLEulerCamera.h"
#include "OpenGLUtilityFunctions.h"
#include "OpenGLModelAmbientLight.h"
#include "OpenGLLight.h"
#include "OpenGLMaterial.h"
#include "OpenGLShaderImpostorModels.h"
#include "OpenGLFrameBufferObject.h"
#include "OpenGLShaderCompileAndLink.h"
#include "OpenGLShaderGLSLPreProcessorCommands.h"
#include "OpenGLUniqueColorsGenerator.h"
#include "OpenGLAssimpModelLoader.h"
#include "OpenGLILTexture.h"
#include "MathConstants.h"
#include "UtilityFunctions.h"
#include <cmath>
#include <array>
#include <memory>
#include <iomanip>
#include <set>
#include <tuple>

using namespace std;
using namespace OpenGLRenderingEngine;
using namespace OpenGLRenderingEngineTests;
using namespace OpenGLRenderingEngine::OpenGLUtilityFunctions;
using namespace Utils::VectorTypes;
using namespace Utils::UtilityFunctions;

namespace // anonymous namespace used instead of deprecated 'static' keyword used for cpp variable locality
{
  const array<GLfloat, 4> BLACK_BACKGROUND_COLOR  = { { 0.0f, 0.0f, 0.0f, 1.0f } };                   // double braces because we initialize an array inside an std::array object
  const array<GLfloat, 4> WHITE_BACKGROUND_COLOR  = { { 1.0f, 1.0f, 1.0f, 1.0f } };                   // double braces because we initialize an array inside an std::array object
  const array<GLfloat, 4> MODEL_COLOR             = { { 0.75f, 0.005f, 0.25f, 1.0f } };               // double braces because we initialize an array inside an std::array object
  const GLfloat MODEL_COLOR_OPACITY               = 0.30f;
  array<GLfloat, 4> MODEL_COLOR_WITH_ALPHA        = { { 0.25f, 0.50f, 0.75f, MODEL_COLOR_OPACITY } }; // double braces because we initialize an array inside an std::array object (remove const for OIT opacity changes from keyboard)
  const GLfloat FOG_DENSITY                       = 0.015f;
  const array<GLfloat, 4> MODEL_AMBIENT_ARRAY     = { { 0.4f, 0.4f, 0.4f, 1.0f } };                   // double braces because we initialize an array inside an std::array object
  const array<GLfloat, 4> LIGHT_AMBIENT_ARRAY     = { { 0.0f, 0.0f, 0.0f, 1.0f } };                   // double braces because we initialize an array inside an std::array object
  const array<GLfloat, 4> LIGHT_DIFFUSE_ARRAY     = { { 1.0f, 1.0f, 1.0f, 1.0f } };                   // double braces because we initialize an array inside an std::array object
  const array<GLfloat, 4> LIGHT_SPECULAR_ARRAY    = { { 1.0f, 1.0f, 1.0f, 1.0f } };                   // double braces because we initialize an array inside an std::array object
  const array<GLfloat, 4> NO_LIGHT_SPECULAR_ARRAY = { { 0.0f, 0.0f, 0.0f, 1.0f } };                   // double braces because we initialize an array inside an std::array object
  const GLfloat LIGHT_SPECULAR_SHININESS          = 25.0f;
  const GLfloat IMPOSTORS_RADIUS_SIZE             = 1.0f / 250.0f;
  const bool    IMPOSTORS_DOUBLE_PRECISION        = false;
  const float PI_4                                = Utils::PI_FLT / 4.0f;
}

ParticleRenderingModelLoaderTest::ParticleRenderingModelLoaderTest(int screenWidth, int screenHeight, const string& textureFileName, const string& modelFileName, const string& modelLoaderDescriptorFileName, bool multisample) noexcept
  : TestAbstractBase(screenWidth, screenHeight, textureFileName, modelFileName, modelLoaderDescriptorFileName, multisample)
  , opacity(MODEL_COLOR_OPACITY)
{
  allModelDataDisplayList = glGenLists(1);
  allModelCylinderDataDisplayList = glGenLists(1);
  if (openGLDriverInfo->supports330Shaders())
  {
    glGenVertexArrays(1, &dummyVao);
  }
  prepareFog();
  prepareLighting();
  preparePointSphereShaders();
  if (openGLDriverInfo->supports430Shaders())
  {
    prepareABuffer3DShaders();
  }
  if (openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders())
  {
    prepareGBufferEffectsShaders();
  }
  prepareFXAA_AntialiasShaders();
  prepareTexture();
  prepareScene();
  if (openGLDriverInfo->supports_GL_EXT_framebuffer_object())
  {
    prepareEnvironmentMappingFBO();
    if (openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders())
    {
      prepareGBufferEffectsFBOs();
      prepareHSSAOData();
    }
    prepareFXAA_AntialiasFBO();
  }
  clearScreen();
}

void ParticleRenderingModelLoaderTest::OpenGLShaderClearABuffer3D::initializeShaderProgram()
{
  if ((particleRenderingModelLoaderTest->currentOITMode == A_BUFFER_3D_USE_TEXTURES) || (particleRenderingModelLoaderTest->currentOITMode == A_BUFFER_3D_USE_BUFFERS))
  {
    openGLShaderGLSLPreProcessorCommands_->addHighestGLVersionDefinition();
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("A_BUFFER_3D_USE_TEXTURES", particleRenderingModelLoaderTest->currentOITMode == A_BUFFER_3D_USE_TEXTURES);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_COUNTER", ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_COUNTER);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D", ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionForStartingLine();

    openGLShaderCompileAndLink_->addShaderLibraryToProgram("OrderIndependentTransparency", "ApplyFullScreenQuad", OpenGLAssetManager::VS);
    openGLShaderCompileAndLink_->addShaderLibraryToProgram("OrderIndependentTransparency", "ApplyClearABuffer3D", OpenGLAssetManager::FS);
    openGLShaderCompileAndLink_->linkShaderProgram();
  }
  else if ((particleRenderingModelLoaderTest->currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_TEXTURES) || (particleRenderingModelLoaderTest->currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_BUFFERS))
  {
    openGLShaderGLSLPreProcessorCommands_->addHighestGLVersionDefinition();
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("A_BUFFER_3D_USE_TEXTURES", particleRenderingModelLoaderTest->currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_TEXTURES);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_ATOMIC_COUNTER", ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_ATOMIC_COUNTER);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET", ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST", ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionForStartingLine();

    openGLShaderCompileAndLink_->addShaderLibraryToProgram("OrderIndependentTransparency", "ApplyFullScreenQuad",           OpenGLAssetManager::VS);
    openGLShaderCompileAndLink_->addShaderLibraryToProgram("OrderIndependentTransparency", "ApplyClearABuffer3DLinkedList", OpenGLAssetManager::FS);
    openGLShaderCompileAndLink_->linkShaderProgram();
  }

  GLAuxiliaryFunctions_checkGLError();
}

void ParticleRenderingModelLoaderTest::OpenGLShaderDisplayABuffer3D::initializeShaderProgram()
{
  if ((particleRenderingModelLoaderTest->currentOITMode == A_BUFFER_3D_USE_TEXTURES) || (particleRenderingModelLoaderTest->currentOITMode == A_BUFFER_3D_USE_BUFFERS))
  {
    openGLShaderGLSLPreProcessorCommands_->addHighestGLVersionDefinition();
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("A_BUFFER_3D_USE_TEXTURES", particleRenderingModelLoaderTest->currentOITMode == A_BUFFER_3D_USE_TEXTURES);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("GPU_SHADER4_COMPATIBILITY_CONDITION", openGLDriverInfo_->supports_GL_EXT_gpu_shader4());
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("GPU_SHADER5_COMPATIBILITY_CONDITION", openGLDriverInfo_->supports_GL_ARB_gpu_shader5());
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("USE_DEFERRED_SHADING", (particleRenderingModelLoaderTest->currentOITMode == A_BUFFER_3D_USE_BUFFERS) && particleRenderingModelLoaderTest->oitUseDeferredShading);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_COUNTER", ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_COUNTER);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D", ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("A_BUFFER_3D_MAX_SIZE", particleRenderingModelLoaderTest->aBuffer3DSize + 1);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionForStartingLine();

    openGLShaderCompileAndLink_->addShaderLibraryToProgram("CommonFunctions", "CommonFunctions", OpenGLAssetManager::FS);
    openGLShaderCompileAndLink_->addShaderLibraryToProgram("SurfaceLightingModels", "SurfaceLightingModels", OpenGLAssetManager::FS);
    openGLShaderCompileAndLink_->addShaderLibraryToProgram("OrderIndependentTransparency", "ApplyFullScreenQuad", OpenGLAssetManager::VS);
    openGLShaderCompileAndLink_->addShaderLibraryToProgram("OrderIndependentTransparency", "ApplyDisplayABuffer3D", OpenGLAssetManager::FS);
    openGLShaderCompileAndLink_->linkShaderProgram();
  }
  else if ((particleRenderingModelLoaderTest->currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_TEXTURES) || (particleRenderingModelLoaderTest->currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_BUFFERS))
  {
    openGLShaderGLSLPreProcessorCommands_->addHighestGLVersionDefinition();
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("A_BUFFER_3D_USE_TEXTURES", particleRenderingModelLoaderTest->currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_TEXTURES);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("GPU_SHADER4_COMPATIBILITY_CONDITION", openGLDriverInfo_->supports_GL_EXT_gpu_shader4());
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("GPU_SHADER5_COMPATIBILITY_CONDITION", openGLDriverInfo_->supports_GL_ARB_gpu_shader5());
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("USE_DEFERRED_SHADING", (particleRenderingModelLoaderTest->currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_BUFFERS) && particleRenderingModelLoaderTest->oitUseDeferredShading);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_ATOMIC_COUNTER", ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_ATOMIC_COUNTER);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET", ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST", ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("A_BUFFER_3D_MAX_SIZE", particleRenderingModelLoaderTest->aBuffer3DinkedListMaxSize + 1);
    openGLShaderGLSLPreProcessorCommands_->addDefinitionForStartingLine();

    openGLShaderCompileAndLink_->addShaderLibraryToProgram("CommonFunctions", "CommonFunctions", OpenGLAssetManager::FS);
    openGLShaderCompileAndLink_->addShaderLibraryToProgram("SurfaceLightingModels", "SurfaceLightingModels", OpenGLAssetManager::FS);
    openGLShaderCompileAndLink_->addShaderLibraryToProgram("OrderIndependentTransparency", "ApplyFullScreenQuad", OpenGLAssetManager::VS);
    openGLShaderCompileAndLink_->addShaderLibraryToProgram("OrderIndependentTransparency", "ApplyDisplayABuffer3DLinkedList", OpenGLAssetManager::FS);
    openGLShaderCompileAndLink_->linkShaderProgram();
  }

  GLAuxiliaryFunctions_checkGLError();
}

void ParticleRenderingModelLoaderTest::OpenGLShaderGBufferEffectsBlurXY::initializeShaderProgram()
{
  openGLShaderGLSLPreProcessorCommands_->addHighestGLVersionDefinition();
  openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("GPU_SHADER4_COMPATIBILITY_CONDITION", openGLDriverInfo_->supports_GL_EXT_gpu_shader4());
  openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("GPU_SHADER5_COMPATIBILITY_CONDITION", openGLDriverInfo_->supports_GL_ARB_gpu_shader5());
  openGLShaderGLSLPreProcessorCommands_->addDefinitionForStartingLine();

  openGLShaderCompileAndLink_->addShaderLibraryToProgram("SurfaceLightingModels", "SurfaceLightingModels", OpenGLAssetManager::FS);
  openGLShaderCompileAndLink_->addShaderLibraryToProgram("GBufferEffects", "ApplyFullScreenQuad", OpenGLAssetManager::VS);
  openGLShaderCompileAndLink_->addShaderLibraryToProgram("GBufferEffects", "ApplyGBufferEffectsBlurXY", OpenGLAssetManager::FS);
  openGLShaderCompileAndLink_->linkShaderProgram();

  GLAuxiliaryFunctions_checkGLError();
}

void ParticleRenderingModelLoaderTest::OpenGLShaderGBufferEffectsHSSAO::initializeShaderProgram()
{
  openGLShaderGLSLPreProcessorCommands_->addHighestGLVersionDefinition();
  openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("GPU_SHADER4_COMPATIBILITY_CONDITION", openGLDriverInfo_->supports_GL_EXT_gpu_shader4());
  openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("GPU_SHADER5_COMPATIBILITY_CONDITION", openGLDriverInfo_->supports_GL_ARB_gpu_shader5());
  openGLShaderGLSLPreProcessorCommands_->addDefinitionForStartingLine();

  openGLShaderCompileAndLink_->addShaderLibraryToProgram("SurfaceLightingModels", "SurfaceLightingModels", OpenGLAssetManager::FS);
  openGLShaderCompileAndLink_->addShaderLibraryToProgram("GBufferEffects", "ApplyFullScreenQuad", OpenGLAssetManager::VS);
  openGLShaderCompileAndLink_->addShaderLibraryToProgram("GBufferEffects", "ApplyGBufferEffectsHSSAO", OpenGLAssetManager::FS);
  openGLShaderCompileAndLink_->linkShaderProgram();

  GLAuxiliaryFunctions_checkGLError();
}

void ParticleRenderingModelLoaderTest::OpenGLShaderGBufferEffects::initializeShaderProgram()
{
  openGLShaderGLSLPreProcessorCommands_->addHighestGLVersionDefinition();
  openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("GPU_SHADER4_COMPATIBILITY_CONDITION", openGLDriverInfo_->supports_GL_EXT_gpu_shader4());
  openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("GPU_SHADER5_COMPATIBILITY_CONDITION", openGLDriverInfo_->supports_GL_ARB_gpu_shader5());
  openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("DEPTH_OF_FIELD_DEBUG_MODE", DEPTH_OF_FIELD_DEBUG_MODE);
  openGLShaderGLSLPreProcessorCommands_->addDefinitionForStartingLine();

  openGLShaderCompileAndLink_->addShaderLibraryToProgram("CommonFunctions", "CommonFunctions", OpenGLAssetManager::FS);
  openGLShaderCompileAndLink_->addShaderLibraryToProgram("SurfaceLightingModels", "SurfaceLightingModels", OpenGLAssetManager::FS);
  openGLShaderCompileAndLink_->addShaderLibraryToProgram("GBufferEffects", "ApplyFullScreenQuad", OpenGLAssetManager::VS);
  openGLShaderCompileAndLink_->addShaderLibraryToProgram("GBufferEffects", "ApplyGBufferEffects", OpenGLAssetManager::FS);
  openGLShaderCompileAndLink_->linkShaderProgram();

  GLAuxiliaryFunctions_checkGLError();
}

void ParticleRenderingModelLoaderTest::OpenGLShaderFXAA_Antialias::initializeShaderProgram()
{
  openGLShaderGLSLPreProcessorCommands_->addHighestGLVersionDefinition();
  openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("OPENGL_330_SUPPORTED", openGLDriverInfo_->supports330Shaders());
  openGLShaderGLSLPreProcessorCommands_->addDefinitionForStartingLine();

  openGLShaderCompileAndLink_->addShaderLibraryToProgram("FXAA_Antialias", "ApplyFXAA_Antialias", OpenGLAssetManager::VS | OpenGLAssetManager::FS);
  openGLShaderCompileAndLink_->linkShaderProgram();

  GLAuxiliaryFunctions_checkGLError();
}

void ParticleRenderingModelLoaderTest::prepareFog() const
{
  glFogi(GL_FOG_MODE, GL_EXP2);        // exponential fog mode
  glFogf(GL_FOG_DENSITY, FOG_DENSITY); // fog density value
}

void ParticleRenderingModelLoaderTest::prepareLighting()
{
  // enable light sources
  glEnable(GL_LIGHTING);
  // disable two-sided lighting vertex program for Point Sphere Model shaders to get a speed-boost
  // glEnable(GL_VERTEX_PROGRAM_TWO_SIDE); // must manually populate the gl_BackColor state variable in the vertex shader for Intel GPUs
  glEnable(GL_NORMALIZE);

  glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_TRUE); // GL_TRUE = infinite distance from scene, optimization trick for OpenGL lighting calculations
  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);     // Enable two-sided lighting
  glShadeModel(GL_SMOOTH);

  openGLModelAmbientLight = new OpenGLModelAmbientLight();
  openGLModelAmbientLight->setModelAmbientColor(MODEL_AMBIENT_ARRAY.data());
  openGLModelAmbientLight->enable();

  // range -2  to -+2 for all lights positions
  for (size_t i = 0; i < NUMBER_OF_LIGHTS; ++i)
  {
    allLightsPositions[i][0] = (i & 1) ?  2.0f : -2.0f; // light position x
    allLightsPositions[i][1] = (i & 2) ? -2.0f :  2.0f; // light position y
    allLightsPositions[i][2] = (i & 3) ?  2.0f : -2.0f; // light position z

    openGLLights[i] = new OpenGLLight();
    openGLLights[i]->setAmbientColor(LIGHT_AMBIENT_ARRAY.data());
    openGLLights[i]->setDiffuseColor(LIGHT_DIFFUSE_ARRAY.data());
    openGLLights[i]->setSpecularColor(LIGHT_SPECULAR_ARRAY.data());
    if (usePositionalLights)
    {
      openGLLights[i]->setPosition(allLightsPositions[i]);
    }
    else
    {
      openGLLights[i]->setDirection(allLightsPositions[i]);
    }
    openGLLights[i]->setConstantAttenuation(1.0f);
    openGLLights[i]->setLinearAttenuation(0.5f);
    openGLLights[i]->setQuadraticAttenuation(0.250f);
    openGLLights[i]->enable(GL_LIGHT0 + int(i), usePositionalLights != 0);
  }

  openGLMaterial = new OpenGLMaterial(USE_COLOR_MATERIAL);
  if (useColoredObjects)
  {
    openGLMaterial->setSpecularColor(LIGHT_SPECULAR_ARRAY.data());
    openGLMaterial->setShininess(LIGHT_SPECULAR_SHININESS);
  }
  else
  {
    openGLMaterial->setSpecularColor(NO_LIGHT_SPECULAR_ARRAY.data());
    openGLMaterial->setShininess(0.0f);
  }
  openGLMaterial->enable();
}

void ParticleRenderingModelLoaderTest::updateLighting()
{
  glShadeModel(GL_SMOOTH);

  if (useSpecularLighting)
  {
    openGLMaterial->setSpecularColor(LIGHT_SPECULAR_ARRAY.data());
    openGLMaterial->setShininess(LIGHT_SPECULAR_SHININESS);
  }
  else
  {
    openGLMaterial->setSpecularColor(NO_LIGHT_SPECULAR_ARRAY.data());
    openGLMaterial->setShininess(0.0f);
  }
  openGLMaterial->enable();

  sinLightAngle += (PI_4 / 75.0f);
  cosLightAngle += (PI_4 / 75.0f);
  float sinLightAngleValue = sin(sinLightAngle);
  float cosLightAngleValue = cos(cosLightAngle);

  if (usePositionalLights)
  {
    for (size_t i = 0; i < NUMBER_OF_LIGHTS; ++i)
    {
      openGLLights[i]->setPosition(allLightsPositions[i]);
      openGLLights[i]->update(GL_LIGHT0 + int(i), true, (i & 1) ? sinLightAngleValue : cosLightAngleValue,
                                                        (i & 2) ? cosLightAngleValue : sinLightAngleValue,
                                                        (i & 3) ? sinLightAngleValue : cosLightAngleValue);
    }
  }
  else
  {
    for (size_t i = 0; i < NUMBER_OF_LIGHTS; ++i)
    {
      openGLLights[i]->setDirection(allLightsPositions[i]);
      openGLLights[i]->update(GL_LIGHT0 + int(i), false, (i & 1) ? sinLightAngleValue : cosLightAngleValue,
                                                         (i & 2) ? cosLightAngleValue : sinLightAngleValue,
                                                         (i & 3) ? sinLightAngleValue : cosLightAngleValue);
    }
  }
}

void ParticleRenderingModelLoaderTest::preparePointSphereShaders()
{
  glEnable(GL_POINT_SPRITE);
  glEnable(GL_VERTEX_PROGRAM_POINT_SIZE);
  glPointSize(radiusFactor);
  // glPointParameterf(GL_POINT_SPRITE_COORD_ORIGIN, GL_LOWER_LEFT);

  if (openGLShaderPointSphereModels)
  {
    delete openGLShaderPointSphereModels;
    openGLShaderPointSphereModels = nullptr;
  }
  openGLShaderPointSphereModels                 = new OpenGLShaderImpostorModels(openGLDriverInfo, false, false, false, false, false, false, useTransparency && currentOITMode != NO_OIT, currentOITMode,
                                                                                 ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_COUNTER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D,
                                                                                 ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_ATOMIC_COUNTER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST,
                                                                                 oitUseDeferredShading);

  if (openGLShaderPointSphereModelsQuadVFFS)
  {
    delete openGLShaderPointSphereModelsQuadVFFS;
    openGLShaderPointSphereModelsQuadVFFS = nullptr;
  }
  openGLShaderPointSphereModelsQuadVFFS         = new OpenGLShaderImpostorModels(openGLDriverInfo, true, false, false, false, false, IMPOSTORS_DOUBLE_PRECISION, useTransparency && currentOITMode != NO_OIT, currentOITMode,
                                                                                 ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_COUNTER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D,
                                                                                 ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_ATOMIC_COUNTER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST,
                                                                                 oitUseDeferredShading);

  if (openGLShaderPointSphereModelsQuadVFGSFS)
  {
    delete openGLShaderPointSphereModelsQuadVFGSFS;
    openGLShaderPointSphereModelsQuadVFGSFS = nullptr;
  }
  openGLShaderPointSphereModelsQuadVFGSFS       = new OpenGLShaderImpostorModels(openGLDriverInfo, true, true, false, false, false, IMPOSTORS_DOUBLE_PRECISION, useTransparency && currentOITMode != NO_OIT, currentOITMode,
                                                                                 ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_COUNTER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D,
                                                                                 ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_ATOMIC_COUNTER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST,
                                                                                 oitUseDeferredShading);

  if (openGLShaderPointSphereModelsQuadVFTESSLODFS)
  {
    delete openGLShaderPointSphereModelsQuadVFTESSLODFS;
    openGLShaderPointSphereModelsQuadVFTESSLODFS = nullptr;
  }
  openGLShaderPointSphereModelsQuadVFTESSLODFS  = new OpenGLShaderImpostorModels(openGLDriverInfo, true, false, true, false, false, false, useTransparency && currentOITMode != NO_OIT, currentOITMode,
                                                                                 ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_COUNTER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D,
                                                                                 ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_ATOMIC_COUNTER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST,
                                                                                 oitUseDeferredShading);

  if (openGLShaderPointSphereModelsQuadVFTESSQuadFS)
  {
    delete openGLShaderPointSphereModelsQuadVFTESSQuadFS;
    openGLShaderPointSphereModelsQuadVFTESSQuadFS = nullptr;
  }
  openGLShaderPointSphereModelsQuadVFTESSQuadFS = new OpenGLShaderImpostorModels(openGLDriverInfo, true, false, true, true, false, IMPOSTORS_DOUBLE_PRECISION, useTransparency && currentOITMode != NO_OIT, currentOITMode,
                                                                                 ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_COUNTER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D,
                                                                                 ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_ATOMIC_COUNTER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST,
                                                                                 oitUseDeferredShading);

  if (openGLShaderCylinderVFGSFS)
  {
    delete openGLShaderCylinderVFGSFS;
    openGLShaderCylinderVFGSFS = nullptr;
  }
  openGLShaderCylinderVFGSFS                    = new OpenGLShaderImpostorModels(openGLDriverInfo, true, true, false, false, true, IMPOSTORS_DOUBLE_PRECISION, useTransparency && currentOITMode != NO_OIT, currentOITMode,
                                                                                 ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_COUNTER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D,
                                                                                 ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_ATOMIC_COUNTER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST,
                                                                                 oitUseDeferredShading);
}

void ParticleRenderingModelLoaderTest::prepareABuffer3DShaders()
{
  if (currentOITMode != NO_OIT)
  {
    // make sure to delete first the A-Buffer-3D shaders for correct re-initializations
    if (openGLShaderClearABuffer3D)
    {
      delete openGLShaderClearABuffer3D;
      openGLShaderClearABuffer3D = nullptr;
    }
    openGLShaderClearABuffer3D = new OpenGLShaderClearABuffer3D(this, openGLDriverInfo);

    if (openGLShaderDisplayABuffer3D)
    {
      delete openGLShaderDisplayABuffer3D;
      openGLShaderDisplayABuffer3D = nullptr;
    }
    openGLShaderDisplayABuffer3D = new OpenGLShaderDisplayABuffer3D(this, openGLDriverInfo);
  }
}

void ParticleRenderingModelLoaderTest::prepareGBufferEffectsShaders()
{
  openGLShaderGBufferEffectsBlurXY = new OpenGLShaderGBufferEffectsBlurXY(openGLDriverInfo);
  openGLShaderGBufferEffectsHSSAO  = new OpenGLShaderGBufferEffectsHSSAO(openGLDriverInfo);
  openGLShaderGBufferEffects       = new OpenGLShaderGBufferEffects(openGLDriverInfo);
}

void ParticleRenderingModelLoaderTest::prepareFXAA_AntialiasShaders()
{
  openGLShaderFXAA_Antialias = new OpenGLShaderFXAA_Antialias(openGLDriverInfo);
}

void ParticleRenderingModelLoaderTest::prepareTexture()
{
  openGLILTexture = new OpenGLILTexture(openGLDriverInfo->supports330Shaders(), openGLDriverInfo->supports_GL_EXT_texture_filter_anisotropic(), openGLDriverInfo->getMaxTextureMaxAnisotropy());
  openGLILTexture->createILTexture(OpenGLAssetManager::getTexturesDefaultDirectory(), textureFileName);
}

void ParticleRenderingModelLoaderTest::prepareScene()
{
  openGLAssimpModelLoader = new OpenGLAssimpModelLoader();
  openGLAssimpModelLoader->load(OpenGLAssetManager::getModelsDefaultDirectory(), modelFileName, modelLoaderDescriptorFileName, true, false, false); // no normals & indexing usage for particles rendering
}

void ParticleRenderingModelLoaderTest::prepareParticles()
{
  createParticlesVerticesArray();
  createParticlesColorsArray();
  createPointCylinderVerticesAndDirectionsArray();
  createParticlesCylinderColorsArray();
  createQuadColorsArray();
  createQuadTextureCoordsArray();
  createQuadVerticesArray();
}

void ParticleRenderingModelLoaderTest::prepareEnvironmentMappingFBO()
{
  openGLFrameBufferObjectForEnvironmentMapping = new OpenGLFrameBufferObject(openGLDriverInfo);
  openGLFrameBufferObjectForEnvironmentMapping->setUseTexture(true);
  openGLFrameBufferObjectForEnvironmentMapping->setDepthRenderBuffer(true);
  openGLFrameBufferObjectForEnvironmentMapping->setUseDepthTexture(true);
  if (!openGLDriverInfo->isIntel() && openGLDriverInfo->supports_GL_EXT_framebuffer_multisample() && openGLDriverInfo->supports_GL_EXT_framebuffer_blit())
  {
    openGLFrameBufferObjectForEnvironmentMapping->setMultisampleFBO(true, 4);
  }
}

void ParticleRenderingModelLoaderTest::initEnviromentMappingFBO() const
{
  DebugConsole_consoleOutLine("Environment Mapping FBO Render-to-texture ratioFactor: ", ENVIRONMENT_MAPPING_RATIO_FACTOR);
  openGLFrameBufferObjectForEnvironmentMapping->disposeFrameBufferObjectResources();
  openGLFrameBufferObjectForEnvironmentMapping->initFrameBufferObjectResources(screenWidth / ENVIRONMENT_MAPPING_RATIO_FACTOR, screenHeight / ENVIRONMENT_MAPPING_RATIO_FACTOR, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
}

void ParticleRenderingModelLoaderTest::createParticlesVerticesArray()
{
  DebugConsole_consoleOutLine("Now creating & pre-caching the Particles Vertices Array\n");

  // 3 is used below because of the enforced triangulation which is used above in the model loader
  // particle need x-y-z-w of which w component is particle size & random colors added in the mix
  vector<GLfloat> modelLoaderVertices = openGLAssimpModelLoader->getAllModelStorageVerticesData()[3];
  const int modelLoaderVerticesLength = int(modelLoaderVertices.size());
  int index = 0;
  if (currentImpostorShader == CYLINDER_VS_GS_FS && currentImpostorWireframe != CYLINDER_EXPLOSION) // use Singleton Vertices to offload rendering a bit for wireframe Sphere Impostors
  {
    float x = 0.0f;
    float y = 0.0f;
    float z = 0.0f;
    float w = 0.0f;
    set<tuple<float, float, float, float>> uniqueVertices;
    for (int i = 0; i < modelLoaderVerticesLength; i += 3)
    {
      x = float(modelLoaderVertices[i    ]);
      y = float(modelLoaderVertices[i + 1]);
      z = float(modelLoaderVertices[i + 2]);
      w = 1.15f * IMPOSTORS_RADIUS_SIZE;
      uniqueVertices.insert(make_tuple(x, y, z, w));
    }

    const int newVerticesArraySize = int(uniqueVertices.size() << 2);
    vertices.clear();
    vertices.resize(newVerticesArraySize);
    for (const auto& uniqueVertex : uniqueVertices)
    {
      vertices[index    ] = get<0>(uniqueVertex);
      vertices[index + 1] = get<1>(uniqueVertex);
      vertices[index + 2] = get<2>(uniqueVertex);
      vertices[index + 3] = get<3>(uniqueVertex);

      index += 4;
    }
  }
  else
  {
    const int originalVerticesArraySize = modelLoaderVerticesLength / 3;
    const int newVerticesArraySize      = originalVerticesArraySize << 2;
    vertices.clear();
    vertices.resize(newVerticesArraySize);
    for (int i = 0; i < modelLoaderVerticesLength; i += 3)
    {
      vertices[index    ] = float(modelLoaderVertices[i    ] + random.getRandomFloat() / 1000.0); // add random offset in X axis
      vertices[index + 1] = float(modelLoaderVertices[i + 1] + random.getRandomFloat() / 1000.0); // add random offset in Y axis
      vertices[index + 2] = float(modelLoaderVertices[i + 2] + random.getRandomFloat() / 1000.0); // add random offset in Z axis
      vertices[index + 3] = float(random.getRandomFloat() / 50.0); // random particle size added in w component of vertex

      index += 4;
    }
  }
}

void ParticleRenderingModelLoaderTest::createParticlesColorsArray()
{
  DebugConsole_consoleOutLine("Now creating & pre-caching the Particles Colors Array\n");

  // 3 is used below because of the enforced triangulation which is used above in the model loader
  // particle need x-y-z-w of which w component is particle size & random colors added in the mix
  const int verticesLength = int(vertices.size());
  colors.clear();
  colors.resize(verticesLength);
  OpenGLUniqueColorsGenerator uniqueColorsGenerator;
  vector<float4> uniqueColors = uniqueColorsGenerator.createUniqueColorsBasedOnPrimeNumbers(verticesLength / 4);
  for (int i = 0; i < verticesLength; i += 4)
  {
    const auto& uniqueColor = uniqueColors[i >> 2]; // means '/ 4' to get the float4 color index
    colors[i    ] = uniqueColor.x;                  // generated particle R color
    colors[i + 1] = uniqueColor.y;                  // generated particle G color
    colors[i + 2] = uniqueColor.z;                  // generated particle B color
    colors[i + 3] = (useTransparency) ? opacity : uniqueColor.w;
  }
}

void ParticleRenderingModelLoaderTest::createPointCylinderVerticesAndDirectionsArray()
{
  DebugConsole_consoleOutLine("Now creating & pre-caching the Particles Cylinder Vertices Array\n");

  vector<GLfloat> modelLoaderVertices = openGLAssimpModelLoader->getAllModelStorageVerticesData()[3];
  const int modelLoaderVerticesLength = int(modelLoaderVertices.size());
  float x  = 0.0f;
  float y  = 0.0f;
  float z  = 0.0f;
  float w  = 0.0f;
  float n1 = 0.0f;
  float n2 = 0.0f;
  float n3 = 0.0f;
  set<tuple<float, float, float, float, float, float, float>> uniqueVerticesAndDirections;
  for (int i = 0; i < modelLoaderVerticesLength; i += 9) // iteration per triangle (3 * 3 points) & use Singleton Vertices & Directions to offload rendering a bit for wireframe Cylinder Impostors
  {
    // vertex & direction 1
    x  = float(modelLoaderVertices[i        ] + modelLoaderVertices[i     + 3]) / 2.0f; // center point for cylinder: (x0 + x1) / 2
    y  = float(modelLoaderVertices[i + 1    ] + modelLoaderVertices[i + 1 + 3]) / 2.0f; // center point for cylinder: (y0 + y1) / 2
    z  = float(modelLoaderVertices[i + 2    ] + modelLoaderVertices[i + 2 + 3]) / 2.0f; // center point for cylinder: (z0 + z1) / 2
    w  = float(sqrt((modelLoaderVertices[i     + 3] - modelLoaderVertices[i        ]) * (modelLoaderVertices[i     + 3] - modelLoaderVertices[i        ]) +
                    (modelLoaderVertices[i + 1 + 3] - modelLoaderVertices[i + 1    ]) * (modelLoaderVertices[i + 1 + 3] - modelLoaderVertices[i + 1    ]) +
                    (modelLoaderVertices[i + 2 + 3] - modelLoaderVertices[i + 2    ]) * (modelLoaderVertices[i + 2 + 3] - modelLoaderVertices[i + 2    ])) / 2.0); // lenght of the two 3D points as the cylinder height: sqrt((x1 - x0)^2 + (y1 - y0)^2 + (z1 - z0)^2), divided 2.0 as it handles half-cylinder height
    n1 = float(modelLoaderVertices[i        ] - modelLoaderVertices[i     + 3]); // direction point for cylinder: (x0 - x1)
    n2 = float(modelLoaderVertices[i + 1    ] - modelLoaderVertices[i + 1 + 3]); // direction point for cylinder: (y0 - y1)
    n3 = float(modelLoaderVertices[i + 2    ] - modelLoaderVertices[i + 2 + 3]); // direction point for cylinder: (z0 - z1)
    uniqueVerticesAndDirections.insert(make_tuple(x, y, z, w, n1, n2, n3));

    // vertex & direction 2
    x  = float(modelLoaderVertices[i     + 3] + modelLoaderVertices[i     + 6]) / 2.0f; // center point for cylinder: (x1 + x2) / 2
    y  = float(modelLoaderVertices[i + 1 + 3] + modelLoaderVertices[i + 1 + 6]) / 2.0f; // center point for cylinder: (y1 + y2) / 2
    z  = float(modelLoaderVertices[i + 2 + 3] + modelLoaderVertices[i + 2 + 6]) / 2.0f; // center point for cylinder: (z1 + z2) / 2
    w  = float(sqrt((modelLoaderVertices[i     + 6] - modelLoaderVertices[i     + 3]) * (modelLoaderVertices[i     + 6] - modelLoaderVertices[i     + 3]) +
                    (modelLoaderVertices[i + 1 + 6] - modelLoaderVertices[i + 1 + 3]) * (modelLoaderVertices[i + 1 + 6] - modelLoaderVertices[i + 1 + 3]) +
                    (modelLoaderVertices[i + 2 + 6] - modelLoaderVertices[i + 2 + 3]) * (modelLoaderVertices[i + 2 + 6] - modelLoaderVertices[i + 2 + 3])) / 2.0); // lenght of the two 3D points as the cylinder height: sqrt((x2 - x1)^2 + (y2 - y1)^2 + (z2 - z1)^2), divided 2.0 as it handles half-cylinder height
    n1 = float(modelLoaderVertices[i     + 3] - modelLoaderVertices[i     + 6]); // direction point for cylinder: (x1 - x2)
    n2 = float(modelLoaderVertices[i + 1 + 3] - modelLoaderVertices[i + 1 + 6]); // direction point for cylinder: (y1 - y2)
    n3 = float(modelLoaderVertices[i + 2 + 3] - modelLoaderVertices[i + 2 + 6]); // direction point for cylinder: (z1 - z2)
    uniqueVerticesAndDirections.insert(make_tuple(x, y, z, w, n1, n2, n3));

    // vertex & direction 3
    x  = float(modelLoaderVertices[i        ] + modelLoaderVertices[i     + 6]) / 2.0f; // center point for cylinder: (x0 + x2) / 2
    y  = float(modelLoaderVertices[i + 1    ] + modelLoaderVertices[i + 1 + 6]) / 2.0f; // center point for cylinder: (y0 + y2) / 2
    z  = float(modelLoaderVertices[i + 2    ] + modelLoaderVertices[i + 2 + 6]) / 2.0f; // center point for cylinder: (z0 + z2) / 2
    w  = float(sqrt((modelLoaderVertices[i     + 6] - modelLoaderVertices[i        ]) * (modelLoaderVertices[i     + 6] - modelLoaderVertices[i        ]) +
                    (modelLoaderVertices[i + 1 + 6] - modelLoaderVertices[i + 1    ]) * (modelLoaderVertices[i + 1 + 6] - modelLoaderVertices[i + 1    ]) +
                    (modelLoaderVertices[i + 2 + 6] - modelLoaderVertices[i + 2    ]) * (modelLoaderVertices[i + 2 + 6] - modelLoaderVertices[i + 2    ])) / 2.0); // lenght of the two 3D points as the cylinder height: sqrt((x2 - x0)^2 + (y2 - y0)^2 + (z2 - z0)^2), divided 2.0 as it handles half-cylinder height
    n1 = float(modelLoaderVertices[i        ] - modelLoaderVertices[i     + 6]); // direction point for cylinder: (x0 - x2)
    n2 = float(modelLoaderVertices[i + 1    ] - modelLoaderVertices[i + 1 + 6]); // direction point for cylinder: (y0 - y2)
    n3 = float(modelLoaderVertices[i + 2    ] - modelLoaderVertices[i + 2 + 6]); // direction point for cylinder: (z0 - z2)
    uniqueVerticesAndDirections.insert(make_tuple(x, y, z, w, n1, n2, n3));
  }

  const int newVerticesArraySize = int(uniqueVerticesAndDirections.size());
  cylinderVertices.clear();
  cylinderVertices.resize(newVerticesArraySize << 2);
  cylinderDirections.clear();
  cylinderDirections.resize(3 * newVerticesArraySize);
  int indexVertex    = 0;
  int indexDirection = 0;
  for (auto& uniqueVertexVertexAndDirection : uniqueVerticesAndDirections)
  {
    tie(cylinderVertices[indexVertex    ],
        cylinderVertices[indexVertex + 1],
        cylinderVertices[indexVertex + 2],
        cylinderVertices[indexVertex + 3],
        cylinderDirections[indexDirection    ],
        cylinderDirections[indexDirection + 1],
        cylinderDirections[indexDirection + 2]
       ) = uniqueVertexVertexAndDirection;

    indexVertex    += 4;
    indexDirection += 3;
  }
}

void ParticleRenderingModelLoaderTest::createParticlesCylinderColorsArray()
{
  DebugConsole_consoleOutLine("Now creating & pre-caching the Particles Cylinder Colors Array\n");

  const int cylinderVerticesLength = int(cylinderVertices.size());
  cylinderColors.clear();
  cylinderColors.resize(cylinderVerticesLength);
  OpenGLUniqueColorsGenerator uniqueColorsGenerator;
  vector<float4> uniqueColors = uniqueColorsGenerator.createUniqueColorsBasedOnPrimeNumbers(cylinderVerticesLength >> 2); // means '/ 4'
  for (int i = 0; i < cylinderVerticesLength; i += 4)
  {
    const auto& uniqueColor = uniqueColors[i >> 2]; // means '/ 4' to get the float4 color index
    cylinderColors[i    ] = uniqueColor.x;          // generated cylinder R color
    cylinderColors[i + 1] = uniqueColor.y;          // generated cylinder G color
    cylinderColors[i + 2] = uniqueColor.z;          // generated cylinder B color
    cylinderColors[i + 3] = (useTransparency) ? opacity : uniqueColor.w;
  }
}

void ParticleRenderingModelLoaderTest::createQuadColorsArray()
{
  DebugConsole_consoleOutLine("Now creating & pre-caching the Quad Colors Array\n");

  const int totalNumberOfVertices = int(vertices.size() >> 2);
  const int numberOfQuadVertices  = 4;
  const int verticesArraySize     = 4 * numberOfQuadVertices * totalNumberOfVertices; // numberOfVertices -> 4 for quads and * 4 for color values
  glCachedColors.clear();
  if (!verticesArraySize) return; // failsafe return for vector of size zero
  glCachedColors.resize(verticesArraySize);
  int arrayIndex1 = 0;
  int arrayIndex2 = 0;
  for (int index = 0; index < totalNumberOfVertices; ++index)
  {
    arrayIndex1 = index << 2;
    for (int vertex = 0; vertex < numberOfQuadVertices; ++vertex)
    {
      glCachedColors[arrayIndex2++] = colors[arrayIndex1    ];
      glCachedColors[arrayIndex2++] = colors[arrayIndex1 + 1];
      glCachedColors[arrayIndex2++] = colors[arrayIndex1 + 2];
      glCachedColors[arrayIndex2++] = colors[arrayIndex1 + 3];
    }
  }
}

void ParticleRenderingModelLoaderTest::createQuadTextureCoordsArray()
{
  DebugConsole_consoleOutLine("Now creating & pre-caching the Quad Texture Coords Array\n");

  const int totalNumberOfVertices = int(vertices.size() >> 2);
  const int texCoordwArraySize    = 2 * totalNumberOfVertices << 2; // numberOfVerticesLeftShift -> 4 (<< 2) for quads and * 2 for texcoord values
  glCachedTexCoords.clear();
  if (!texCoordwArraySize) return; // failsafe return for vector of size zero
  glCachedTexCoords.resize(texCoordwArraySize);
  int arrayIndex = 0;
  for (int index = 0; index < totalNumberOfVertices; ++index)
  {
    // bottom-right
    glCachedTexCoords[arrayIndex++] =  1.0f;
    glCachedTexCoords[arrayIndex++] = -1.0f;

    // top-right
    glCachedTexCoords[arrayIndex++] =  1.0f;
    glCachedTexCoords[arrayIndex++] =  1.0f;

    // top-left
    glCachedTexCoords[arrayIndex++] = -1.0f;
    glCachedTexCoords[arrayIndex++] =  1.0f;

    // bottom-left
    glCachedTexCoords[arrayIndex++] = -1.0f;
    glCachedTexCoords[arrayIndex++] = -1.0f;
  }
}

void ParticleRenderingModelLoaderTest::createQuadVerticesArray()
{
  DebugConsole_consoleOutLine("Now creating & pre-caching the Quad Vertices Array\n");

  const int totalNumberOfVertices = int(vertices.size() >> 2);
  const int numberOfQuadVertices  = 4;
  const int verticesArraySize     = 4 * numberOfQuadVertices * totalNumberOfVertices; // numberOfVertices -> 4 for quads and * 4 for vertex values
  glCachedPointsAndRadii.clear();
  if (!verticesArraySize) return; // failsafe return for vector of size zero
  glCachedPointsAndRadii.resize(verticesArraySize);
  int arrayIndex1 = 0;
  int arrayIndex2 = 0;
  for (int index = 0; index < totalNumberOfVertices; ++index)
  {
    arrayIndex1 = index << 2;
    for (int vertex = 0; vertex < numberOfQuadVertices; ++vertex)
    {
      glCachedPointsAndRadii[arrayIndex2++] = vertices[arrayIndex1    ];
      glCachedPointsAndRadii[arrayIndex2++] = vertices[arrayIndex1 + 1];
      glCachedPointsAndRadii[arrayIndex2++] = vertices[arrayIndex1 + 2];
      glCachedPointsAndRadii[arrayIndex2++] = vertices[arrayIndex1 + 3];
    }
  }
}

void ParticleRenderingModelLoaderTest::drawEnvironmentMappingFBO()
{
  openGLFrameBufferObjectForEnvironmentMapping->startRender();
  renderParticleScene(true);
  openGLFrameBufferObjectForEnvironmentMapping->finishRender();
}

void ParticleRenderingModelLoaderTest::prepareGBufferEffectsFBOs()
{
  // prepare GBufferEffects FBO for pass 1
  openGLFrameBufferObjectForGBufferPass1 = new OpenGLFrameBufferObject(openGLDriverInfo);
  openGLFrameBufferObjectForGBufferPass1->setUseTexture(true);
  openGLFrameBufferObjectForGBufferPass1->setDepthRenderBuffer(true);
  // Warning: has to be an unsignet integer FBO texture for the GBuffer to work ok!
  openGLFrameBufferObjectForGBufferPass1->setTextureFormat1(GL_RGB32UI);
  openGLFrameBufferObjectForGBufferPass1->setTextureFormat2(GL_RGB_INTEGER);
  openGLFrameBufferObjectForGBufferPass1->setTextureFormatType(GL_UNSIGNED_INT);

  // prepare GBufferEffects FBO for pass 2
  openGLFrameBufferObjectRenderPass2 = new OpenGLFrameBufferObject(openGLDriverInfo);
  openGLFrameBufferObjectRenderPass2->setUseTexture(true);

  // prepare GBufferEffects FBO for pass 3
  openGLFrameBufferObjectRenderPass3 = new OpenGLFrameBufferObject(openGLDriverInfo);
  openGLFrameBufferObjectRenderPass3->setUseTexture(true);
}

void ParticleRenderingModelLoaderTest::prepareHSSAOData()
{
  // prepare rotationTextureData
  const int rotationTextureDataSize = 64 * 64;
  rotationTextureData.resize(rotationTextureDataSize << 2);
  float randomAngle = float(random.getRandomFloat()) * 4.0f * PI_4 * 2.0f;

  int index = 0;
  for (int i = 0; i < rotationTextureDataSize; ++i)
  {
    rotationTextureData[index++] =  cos(randomAngle) * 0.5f + 0.5f;
    rotationTextureData[index++] =  sin(randomAngle) * 0.5f + 0.5f;
    rotationTextureData[index++] = -sin(randomAngle) * 0.5f + 0.5f;
    rotationTextureData[index++] =  cos(randomAngle) * 0.5f + 0.5f;

    randomAngle += float(random.getRandomFloat()) * 4.0f * PI_4 * 2.0f;
  }

  if (glIsTexture(rotationTextureID))
  {
    glDeleteTextures(1, &rotationTextureID);
    rotationTextureID = 0;
  }
  glGenTextures(1, &rotationTextureID);
  glActiveTexture(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_PERLIN_NOISE_3D_TEXTURE);
  glBindTexture(GL_TEXTURE_2D, rotationTextureID);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, 64, 64, 0, GL_RGBA, GL_FLOAT, rotationTextureData.data());
  glActiveTexture(GL_TEXTURE0);
  glBindTexture(GL_TEXTURE_2D, 0);

  // prepare depthSamplesData
  const int depthSamplesDataSize = 16;
  depthSamplesData.resize(depthSamplesDataSize << 1);
  randomAngle  = PI_4;
  const float radius = 0.416666f;

  index = 0;
  for (int i = 0; i < depthSamplesDataSize; ++i)
  {
    depthSamplesData[index++] = cos(randomAngle) * float(i + 1) / 16.0f * radius;
    depthSamplesData[index++] = sin(randomAngle) * float(i + 1) / 16.0f * radius;

    randomAngle += (2.0f * PI_4);

    if (((i + 1) & 3) == 0) randomAngle += (2.0f * PI_4);
  }
}

void ParticleRenderingModelLoaderTest::initGBufferEffectsFBOs() const
{
  // init GBufferEffects FBO for pass 1
  DebugConsole_consoleOutLine("FBO GBufferEffects Pass 1 ratioFactor: ", FULLSCREEN_MAPPING_RATIO_FACTOR);
  openGLFrameBufferObjectForGBufferPass1->disposeFrameBufferObjectResources();
  openGLFrameBufferObjectForGBufferPass1->initFrameBufferObjectResources(screenWidth / FULLSCREEN_MAPPING_RATIO_FACTOR, screenHeight / FULLSCREEN_MAPPING_RATIO_FACTOR, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);

  // init GBufferEffects FBO for pass 2
  DebugConsole_consoleOutLine("FBO GBufferEffects Pass 2 ratioFactor: ", FULLSCREEN_MAPPING_RATIO_FACTOR);
  openGLFrameBufferObjectRenderPass2->disposeFrameBufferObjectResources();
  openGLFrameBufferObjectRenderPass2->initFrameBufferObjectResources(screenWidth / FULLSCREEN_MAPPING_RATIO_FACTOR, screenHeight / FULLSCREEN_MAPPING_RATIO_FACTOR, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);

  // init GBufferEffects FBO for pass 3
  DebugConsole_consoleOutLine("FBO GBufferEffects Pass 3 ratioFactor: ", FULLSCREEN_MAPPING_RATIO_FACTOR);
  openGLFrameBufferObjectRenderPass3->disposeFrameBufferObjectResources();
  openGLFrameBufferObjectRenderPass3->initFrameBufferObjectResources(screenWidth / FULLSCREEN_MAPPING_RATIO_FACTOR, screenHeight / FULLSCREEN_MAPPING_RATIO_FACTOR, ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
}

void ParticleRenderingModelLoaderTest::processGBufferEffectsFBOs()
{
  const float gbufferResolution[2] = { float(screenWidth), float(screenHeight) };

  // draw GBufferEffects FBO for pass 1
  openGLFrameBufferObjectForGBufferPass1->startRender();
  renderParticleScene(false, true);
  openGLFrameBufferObjectForGBufferPass1->finishRender();

  if (currentGBUfferEffectType == DEPTH_OF_FIELD)
  {
    // draw GBufferEffects FBO for scene creation pass
    openGLFrameBufferObjectRenderPass3->startRender();
    clearScreen();
    openGLShaderGBufferEffectsBlurXY->enableShaderProgram();
    openGLShaderGBufferEffectsBlurXY->setUniform1i("gbufferSampler", ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLShaderGBufferEffectsBlurXY->setUniform2fv("gbufferResolution", 1, gbufferResolution);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("numberOfSamplesRange", depthOfFieldBlurXYNumberOfSamplesRange);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderPlainScene", true);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderBlurXYScene", false);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("blurX", false);
    openGLFrameBufferObjectForGBufferPass1->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, openGLDriverInfo->isNvidia());
    openGLShaderGBufferEffectsBlurXY->disableShaderProgram();
    openGLFrameBufferObjectRenderPass3->finishRender();

    // draw GBufferEffects FBO for pass 2
    openGLFrameBufferObjectRenderPass2->startRender();
    openGLShaderGBufferEffectsBlurXY->enableShaderProgram();
    openGLShaderGBufferEffectsBlurXY->setUniform1i("sceneSampler", ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
    openGLShaderGBufferEffectsBlurXY->setUniform2fv("gbufferResolution", 1, gbufferResolution);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("numberOfSamplesRange", depthOfFieldBlurXYNumberOfSamplesRange);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderPlainScene", false);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderBlurXYScene", true);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("blurX", true);
    openGLFrameBufferObjectRenderPass3->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE, openGLDriverInfo->isNvidia());
    openGLShaderGBufferEffectsBlurXY->disableShaderProgram();
    openGLFrameBufferObjectRenderPass2->finishRender();

    // draw GBufferEffects FBO for pass 3
    openGLFrameBufferObjectRenderPass3->startRender();
    openGLShaderGBufferEffectsBlurXY->enableShaderProgram();
    openGLShaderGBufferEffectsBlurXY->setUniform1i("sceneSampler", ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
    openGLShaderGBufferEffectsBlurXY->setUniform2fv("gbufferResolution", 1, gbufferResolution);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("numberOfSamplesRange", depthOfFieldBlurXYNumberOfSamplesRange);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderPlainScene", false);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderBlurXYScene", true);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("blurX", false);
    openGLFrameBufferObjectRenderPass2->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE, openGLDriverInfo->isNvidia());
    openGLShaderGBufferEffectsBlurXY->disableShaderProgram();
    openGLFrameBufferObjectRenderPass3->finishRender();
  }
  else if ((currentGBUfferEffectType == HSSAO) || (currentGBUfferEffectType == HSSAO_PLUS))
  {
    // main HSSAO pass
    openGLFrameBufferObjectRenderPass3->startRender();
    clearScreen();
    openGLShaderGBufferEffects->enableShaderProgram();
    openGLShaderGBufferEffects->setUniform1i("gbufferSampler", ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLShaderGBufferEffects->setUniform1i("gbufferEffectType", currentGBUfferEffectType);
    openGLShaderGBufferEffects->setUniform1i("rotationTextureSampler", ACTIVE_TEXTURE_UNIT_FOR_PERLIN_NOISE_3D_TEXTURE);
    openGLShaderGBufferEffects->setUniform2fv("depthSamples", 16, depthSamplesData.data());
    openGLShaderGBufferEffects->setUniform1i("useHSSAO", true);
    const float rotationTextureResolution[2] = { gbufferResolution[0] / 64.0f, gbufferResolution[1] / 64.0f };
    openGLShaderGBufferEffects->setUniform2fv("xyHSSAO", 1, rotationTextureResolution);
    openGLFrameBufferObjectForGBufferPass1->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, openGLDriverInfo->isNvidia());
    openGLShaderGBufferEffects->disableShaderProgram();
    openGLFrameBufferObjectRenderPass3->finishRender();

    const int HSSAONumberOfSamplesRange  = 4;
    const bool renderBlurXYScene         = (currentGBUfferEffectType == HSSAO);
    const bool renderWeightedBlurXYScene = (currentGBUfferEffectType == HSSAO_PLUS);

    // blur X for after HSSAO pass 1
    openGLFrameBufferObjectRenderPass2->startRender();
    openGLFrameBufferObjectForGBufferPass1->enable();
    openGLFrameBufferObjectForGBufferPass1->bindTexture(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLShaderGBufferEffectsBlurXY->enableShaderProgram();
    openGLShaderGBufferEffectsBlurXY->setUniform1i("sceneSampler", ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
    openGLShaderGBufferEffectsBlurXY->setUniform2fv("gbufferResolution", 1, gbufferResolution);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderPlainScene", false);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderBlurXYScene",         renderBlurXYScene);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderWeightedBlurXYScene", renderWeightedBlurXYScene);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("blurX", true);
    if (renderWeightedBlurXYScene)
    {
      const float sxValues[3] = { 1.0f / gbufferResolution[0], 2.0f / gbufferResolution[0], 3.0f / gbufferResolution[0] };
      openGLShaderGBufferEffectsBlurXY->setUniform3fv("sxValues", 1, sxValues);
    }
    else
    {
      openGLShaderGBufferEffectsBlurXY->setUniform1i("numberOfSamplesRange", HSSAONumberOfSamplesRange);
    }
    openGLFrameBufferObjectRenderPass3->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE, openGLDriverInfo->isNvidia());
    openGLShaderGBufferEffectsBlurXY->disableShaderProgram();
    openGLFrameBufferObjectForGBufferPass1->unbind(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLFrameBufferObjectForGBufferPass1->disable();
    openGLFrameBufferObjectRenderPass2->finishRender();

    // blur Y for after HSSAO pass 2
    openGLFrameBufferObjectRenderPass3->startRender();
    openGLFrameBufferObjectForGBufferPass1->enable();
    openGLFrameBufferObjectForGBufferPass1->bindTexture(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLShaderGBufferEffectsBlurXY->enableShaderProgram();
    openGLShaderGBufferEffectsBlurXY->setUniform1i("sceneSampler", ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
    openGLShaderGBufferEffectsBlurXY->setUniform2fv("gbufferResolution", 1, gbufferResolution);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderPlainScene", false);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderBlurXYScene",         renderBlurXYScene);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("renderWeightedBlurXYScene", renderWeightedBlurXYScene);
    openGLShaderGBufferEffectsBlurXY->setUniform1i("blurX", false);
    if (renderWeightedBlurXYScene)
    {
      const float syValues[3] = { 1.0f / gbufferResolution[1], 2.0f / gbufferResolution[1], 3.0f / gbufferResolution[1] };
      openGLShaderGBufferEffectsBlurXY->setUniform3fv("syValues", 1, syValues);
    }
    else
    {
      openGLShaderGBufferEffectsBlurXY->setUniform1i("numberOfSamplesRange", HSSAONumberOfSamplesRange);
    }
    openGLFrameBufferObjectRenderPass2->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE, openGLDriverInfo->isNvidia());
    openGLShaderGBufferEffectsBlurXY->disableShaderProgram();
    openGLFrameBufferObjectForGBufferPass1->unbind(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLFrameBufferObjectForGBufferPass1->disable();
    openGLFrameBufferObjectRenderPass3->finishRender();
  }
}

void ParticleRenderingModelLoaderTest::drawGBufferEffectsFBOs()
{
  // main GBuffer render code
  if ((currentGBUfferEffectType == HSSAO) || (currentGBUfferEffectType == HSSAO_PLUS))
  {
    clearScreen();
    disableLights(false);
    openGLFrameBufferObjectForGBufferPass1->enable();
    openGLFrameBufferObjectForGBufferPass1->bindTexture(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLFrameBufferObjectRenderPass3->enable();
    openGLFrameBufferObjectRenderPass3->bindTexture(ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
    openGLShaderGBufferEffectsHSSAO->enableShaderProgram();
    openGLShaderGBufferEffectsHSSAO->setUniform1i("gbufferSampler", ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLShaderGBufferEffectsHSSAO->setUniform1i("HSSAOBlurredXYSampler", ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
    openGLFrameBufferObjectForGBufferPass1->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, openGLDriverInfo->isNvidia());
    openGLShaderGBufferEffectsHSSAO->disableShaderProgram();
    openGLFrameBufferObjectRenderPass3->unbind(ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
    openGLFrameBufferObjectRenderPass3->disable();
    openGLFrameBufferObjectForGBufferPass1->unbind(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLFrameBufferObjectForGBufferPass1->disable();
    enableLights(false);
  }
  else // for Depth-Of-Field & Edge Enhancement effects
  {
    clearScreen();
    disableLights(false);
    openGLFrameBufferObjectForGBufferPass1->enable();
    openGLFrameBufferObjectForGBufferPass1->bindTexture(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    if (currentGBUfferEffectType == DEPTH_OF_FIELD)
    {
      openGLFrameBufferObjectRenderPass3->enable();
      openGLFrameBufferObjectRenderPass3->bindTexture(ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
    }
    openGLShaderGBufferEffects->enableShaderProgram();
    openGLShaderGBufferEffects->setUniform1i("gbufferSampler", ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLShaderGBufferEffects->setUniform1i("gbufferEffectType", currentGBUfferEffectType);
    if (currentGBUfferEffectType == DEPTH_OF_FIELD)
    {
      openGLShaderGBufferEffects->setUniform1i("sceneBlurredXYSampler", ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
      openGLShaderGBufferEffects->setUniform1f("zNear", openGLEulerCamera->getViewNear());
      openGLShaderGBufferEffects->setUniform1f("zFar",  openGLEulerCamera->getViewFar());
      openGLShaderGBufferEffects->setUniform1f("range", depthOfFieldRange);
      openGLShaderGBufferEffects->setUniform1f("zFocus", depthOfFieldZFocus);
    }
    else if (currentGBUfferEffectType == EDGE_ENHANCEMENT)
    {
      const float gbufferResolution[2] = { float(screenWidth), float(screenHeight) };
      openGLShaderGBufferEffects->setUniform2fv("gbufferResolution", 1, gbufferResolution);
      double ratio = (cameraDistanceZ > 1.0) ? cameraDistanceZ  : 1.0;
      if (ratio > 6.0) ratio = 6.0;
      const float edgeEnhancementSize[2] = { float((9.0 / 3.0) / ratio), float((9.0 / 3.0) / ratio) };
      openGLShaderGBufferEffects->setUniform2fv("edgeEnhancementSize", 1, edgeEnhancementSize);
    }
    openGLFrameBufferObjectForGBufferPass1->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, openGLDriverInfo->isNvidia());
    openGLShaderGBufferEffects->disableShaderProgram();
    if (currentGBUfferEffectType == DEPTH_OF_FIELD)
    {
      openGLFrameBufferObjectRenderPass3->unbind(ACTIVE_TEXTURE_UNIT_FOR_BLUR_XY_TEXTURE);
      openGLFrameBufferObjectRenderPass3->disable();
    }
    openGLFrameBufferObjectForGBufferPass1->unbind(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLFrameBufferObjectForGBufferPass1->disable();
    enableLights(false);
  }
}

void ParticleRenderingModelLoaderTest::prepareFXAA_AntialiasFBO()
{
  openGLFrameBufferObjectForFXAA_Antialias = new OpenGLFrameBufferObject(openGLDriverInfo);
  openGLFrameBufferObjectForFXAA_Antialias->setUseTexture(true);
  openGLFrameBufferObjectForFXAA_Antialias->setDepthRenderBuffer(true);
  if (!openGLDriverInfo->isIntel() && openGLDriverInfo->supports_GL_EXT_framebuffer_multisample() && openGLDriverInfo->supports_GL_EXT_framebuffer_blit())
  {
    openGLFrameBufferObjectForFXAA_Antialias->setMultisampleFBO(true, 4);
  }
}

void ParticleRenderingModelLoaderTest::initFXAA_AntialiasFBO() const
{
  DebugConsole_consoleOutLine("FBO FXAA Antialias ratioFactor: ", FULLSCREEN_MAPPING_RATIO_FACTOR);
  openGLFrameBufferObjectForFXAA_Antialias->disposeFrameBufferObjectResources();
  openGLFrameBufferObjectForFXAA_Antialias->initFrameBufferObjectResources(screenWidth / FULLSCREEN_MAPPING_RATIO_FACTOR, screenHeight / FULLSCREEN_MAPPING_RATIO_FACTOR, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
}

void ParticleRenderingModelLoaderTest::drawFXAA_AntialiasFBO()
{
  // draw FXAA Antialias FBO for pass 1
  openGLFrameBufferObjectForFXAA_Antialias->startRender();
  if (openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders() &&
       (currentGBUfferEffectType != NO_GBUFFER_EFFECT))
  {
    drawGBufferEffectsFBOs();
  }
  else
  {
    renderParticleScene(false);
  }
  openGLFrameBufferObjectForFXAA_Antialias->finishRender();

  // draw FXAA Antialias FBO for pass 2
  clearScreen();
  disableLights(false);
  openGLShaderFXAA_Antialias->enableShaderProgram();
  openGLShaderFXAA_Antialias->setUniform1i("frameBufferSampler", ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
  const float frameBufferResolution[2] = { float(screenWidth), float(screenHeight) };
  openGLShaderFXAA_Antialias->setUniform2fv("frameBufferResolution", 1, frameBufferResolution);
  openGLFrameBufferObjectForFXAA_Antialias->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, openGLDriverInfo->isNvidia());
  openGLShaderFXAA_Antialias->disableShaderProgram();
  enableLights(false);
}

void ParticleRenderingModelLoaderTest::initABuffer3D()
{
  if (currentOITMode != NO_OIT)
  {
    // A-Buffer3D storage
    if (currentOITMode == A_BUFFER_3D_USE_TEXTURES) // texture storage path
    {
      // A-Buffer3D per-pixel counter image
      if (glIsTexture(aBuffer3DCounterTextureID))
      {
        glDeleteTextures(1, &aBuffer3DCounterTextureID);
        aBuffer3DCounterTextureID = 0;
      }
      glGenTextures(1, &aBuffer3DCounterTextureID);
      glActiveTexture(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_COUNTER);
      glBindTexture(GL_TEXTURE_2D, aBuffer3DCounterTextureID);
      // set filter
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
      glTexImage2D(GL_TEXTURE_2D, 0, GL_R32UI, screenWidth, screenHeight, 0, GL_RED_INTEGER, GL_UNSIGNED_INT, nullptr);
      glBindImageTexture(ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_COUNTER, aBuffer3DCounterTextureID, 0, GL_FALSE, 0, GL_READ_WRITE, GL_R32UI);
      glActiveTexture(GL_TEXTURE0);
      glBindTexture(GL_TEXTURE_2D, 0);


      // A-Buffer3D image
      if (glIsTexture(aBuffer3DTextureID))
      {
        glDeleteTextures(1, &aBuffer3DTextureID);
        aBuffer3DTextureID = 0;
      }
      glGenTextures(1, &aBuffer3DTextureID);
      glActiveTexture(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D);
      glBindTexture(GL_TEXTURE_2D_ARRAY, aBuffer3DTextureID);
      // set filter
      glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
      glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
      glTexImage3D(GL_TEXTURE_2D_ARRAY, 0, GL_RGBA32UI, screenWidth, screenHeight, aBuffer3DSize, 0, GL_RGBA_INTEGER, GL_UNSIGNED_INT, nullptr);
      glBindImageTexture(ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D, aBuffer3DTextureID, 0, GL_TRUE, 0, GL_READ_WRITE, GL_RGBA32UI);
      glActiveTexture(GL_TEXTURE0);
      glBindTexture(GL_TEXTURE_2D_ARRAY, 0);
    }
    else if (currentOITMode == A_BUFFER_3D_USE_BUFFERS) // global memory (buffer) storage path
    {
      // A-Buffer3D per-pixel counter buffer
      if (glIsBuffer(aBuffer3DCounterBufferID))
      {
        glDeleteBuffers(1, &aBuffer3DCounterBufferID);
        aBuffer3DCounterBufferID = 0;
      }
      glGenBuffers(1, &aBuffer3DCounterBufferID);
      glBindBufferBase(GL_SHADER_STORAGE_BUFFER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_COUNTER, aBuffer3DCounterBufferID);
      glBufferData(GL_SHADER_STORAGE_BUFFER, screenWidth * screenHeight * sizeof(GLuint), nullptr, GL_DYNAMIC_DRAW);
      glBindBuffer(GL_SHADER_STORAGE_BUFFER, 0);


      // A-Buffer3D buffer
      if (glIsBuffer(aBuffer3DBufferID))
      {
        glDeleteBuffers(1, &aBuffer3DBufferID);
        aBuffer3DBufferID = 0;
      }
      glGenBuffers(1, &aBuffer3DBufferID);
      glBindBufferBase(GL_SHADER_STORAGE_BUFFER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D, aBuffer3DBufferID);
      const int numberOfElements = (oitUseDeferredShading) ? DEFERRED_SHADING_BUFFER_ELEMENTS : NORMAL_SHADING_BUFFER_ELEMENTS;
      glBufferData(GL_SHADER_STORAGE_BUFFER, GLsizeiptr(screenWidth * screenHeight * aBuffer3DSize * sizeof(GLfloat) * numberOfElements), nullptr, GL_DYNAMIC_DRAW);
      glBindBuffer(GL_SHADER_STORAGE_BUFFER, 0);
    }
    else
    {
      // A-Buffer3D atomic counter buffer
      if (glIsBuffer(aBuffer3DLinkedListAtomicCounterBufferID))
      {
        glDeleteBuffers(1, &aBuffer3DLinkedListAtomicCounterBufferID);
        aBuffer3DLinkedListAtomicCounterBufferID = 0;
      }
      glGenBuffers(1, &aBuffer3DLinkedListAtomicCounterBufferID);
      glBindBufferBase(GL_ATOMIC_COUNTER_BUFFER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_ATOMIC_COUNTER, aBuffer3DLinkedListAtomicCounterBufferID);
      glBufferData(GL_ATOMIC_COUNTER_BUFFER, sizeof(GLuint), nullptr, GL_DYNAMIC_DRAW);
      glBindBuffer(GL_ATOMIC_COUNTER_BUFFER, 0);

      if (currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_TEXTURES) // texture storage path with linked lists
      {
        // A-Buffer3D linked list offset image
        if (glIsTexture(aBuffer3DLinkedListOffsetTextureID))
        {
          glDeleteTextures(1, &aBuffer3DLinkedListOffsetTextureID);
          aBuffer3DLinkedListOffsetTextureID = 0;
        }
        glGenTextures(1, &aBuffer3DLinkedListOffsetTextureID);
        glActiveTexture(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET);
        glBindTexture(GL_TEXTURE_2D, aBuffer3DLinkedListOffsetTextureID);
        // set filter
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
        glTexImage2D(GL_TEXTURE_2D, 0, GL_R32UI, screenWidth, screenHeight, 0, GL_RED_INTEGER, GL_UNSIGNED_INT, nullptr);
        glBindImageTexture(ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET, aBuffer3DLinkedListOffsetTextureID, 0, GL_FALSE, 0, GL_READ_WRITE, GL_R32UI);
        glActiveTexture(GL_TEXTURE0);
        glBindTexture(GL_TEXTURE_2D, 0);


        // A-Buffer3D linked list image
        if (glIsBuffer(aBuffer3DLinkedListBufferID))
        {
          glDeleteBuffers(1, &aBuffer3DLinkedListBufferID);
          aBuffer3DLinkedListBufferID = 0;
        }
        glGenBuffers(1, &aBuffer3DLinkedListBufferID);
        glBindBuffer(GL_TEXTURE_BUFFER, aBuffer3DLinkedListBufferID);
        glBufferData(GL_TEXTURE_BUFFER, GLsizeiptr(aBuffer3DLinkedListSize * sizeof(GLuint) << 2), nullptr, GL_DYNAMIC_DRAW);
        if (glIsTexture(aBuffer3DLinkedListTextureID))
        {
          glDeleteTextures(1, &aBuffer3DLinkedListTextureID);
          aBuffer3DLinkedListTextureID = 0;
        }
        glGenTextures(1, &aBuffer3DLinkedListTextureID);
        glActiveTexture(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST);
        glBindTexture(GL_TEXTURE_BUFFER, aBuffer3DLinkedListTextureID);
        // Associate Buffer Object storage with the texture
        glTexBuffer(GL_TEXTURE_BUFFER, GL_RGBA32F, aBuffer3DLinkedListBufferID);
        glBindImageTexture(ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST, aBuffer3DLinkedListTextureID, 0, GL_FALSE, 0, GL_READ_WRITE, GL_RGBA32UI);
        glActiveTexture(GL_TEXTURE0);
        glBindTexture(GL_TEXTURE_BUFFER, 0);
      }
      else if (currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_BUFFERS) // global memory (buffer) storage path with linked lists
      {
        // A-Buffer3D linked list offset buffer
        if (glIsBuffer(aBuffer3DLinkedListOffsetBufferID))
        {
          glDeleteBuffers(1, &aBuffer3DLinkedListOffsetBufferID);
          aBuffer3DLinkedListOffsetBufferID = 0;
        }
        glGenBuffers(1, &aBuffer3DLinkedListOffsetBufferID);
        glBindBufferBase(GL_SHADER_STORAGE_BUFFER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET, aBuffer3DLinkedListOffsetBufferID);
        glBufferData(GL_SHADER_STORAGE_BUFFER, screenWidth * screenHeight * sizeof(GLuint), nullptr, GL_DYNAMIC_DRAW);
        glBindBuffer(GL_SHADER_STORAGE_BUFFER, 0);


        // A-Buffer3D linked list buffer
        if (glIsBuffer(aBuffer3DLinkedListBufferID))
        {
          glDeleteBuffers(1, &aBuffer3DLinkedListBufferID);
          aBuffer3DLinkedListBufferID = 0;
        }
        glGenBuffers(1, &aBuffer3DLinkedListBufferID);
        glBindBufferBase(GL_SHADER_STORAGE_BUFFER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST, aBuffer3DLinkedListBufferID);
        const int numberOfElements = (oitUseDeferredShading) ? DEFERRED_SHADING_LINKED_LIST_BUFFER_ELEMENTS : NORMAL_SHADING_LINKED_LIST_BUFFER_ELEMENTS;
        glBufferData(GL_SHADER_STORAGE_BUFFER, GLsizeiptr(aBuffer3DLinkedListSize * sizeof(GLuint) * numberOfElements), nullptr, GL_DYNAMIC_DRAW);
        glBindBuffer(GL_SHADER_STORAGE_BUFFER, 0);
      }
    }
  }

  GLAuxiliaryFunctions_checkGLError();
}

bool ParticleRenderingModelLoaderTest::autoManageABuffer3D()
{
  bool aBuffer3DNeedsResizing     = false;
  GLuint maximumNumberOfFragments = 0;
  if (currentOITMode == A_BUFFER_3D_USE_TEXTURES) // texture storage path
  {
    const int size = screenWidth * screenHeight;
    const auto counters = unique_ptr<GLuint[]>(new GLuint[size]); // avoid enforcing the default constructor through the make_unique() call for the primitive GLchar (make_unique() is using the C++03 array initialization syntax)
    glActiveTexture(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_COUNTER);
    glBindTexture(GL_TEXTURE_2D, aBuffer3DCounterTextureID);
    glGetTexImage(GL_TEXTURE_2D, 0, GL_RED_INTEGER, GL_UNSIGNED_INT, counters.get());
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D_ARRAY, 0);
    maximumNumberOfFragments = counters[0];
    for (int i = 1; i < size; ++i)
    {
      if (maximumNumberOfFragments < counters[i])
      {
        maximumNumberOfFragments = counters[i];
      }
    }
  }
  else if (currentOITMode == A_BUFFER_3D_USE_BUFFERS) // global memory (buffer) storage path
  {
    glBindBufferBase(GL_SHADER_STORAGE_BUFFER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_COUNTER, aBuffer3DCounterBufferID);
    const GLuint* counters = (const GLuint*)glMapBuffer(GL_SHADER_STORAGE_BUFFER, GL_READ_ONLY);
    glUnmapBuffer(GL_SHADER_STORAGE_BUFFER);
    glBindBuffer(GL_SHADER_STORAGE_BUFFER, 0);
    const int size = screenWidth * screenHeight;
    maximumNumberOfFragments = counters[0];
    for (int i = 1; i < size; ++i)
    {
      if (maximumNumberOfFragments < counters[i])
      {
        maximumNumberOfFragments = counters[i];
      }
    }
  }
  else if ((currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_TEXTURES) || (currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_BUFFERS)) // texture or buffer storage path with linked lists
  {
    glBindBufferBase(GL_SHADER_STORAGE_BUFFER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_ATOMIC_COUNTER, aBuffer3DLinkedListAtomicCounterBufferID);
    const GLuint* counters = (const GLuint*)glMapBuffer(GL_SHADER_STORAGE_BUFFER, GL_READ_ONLY);
    glUnmapBuffer(GL_SHADER_STORAGE_BUFFER);
    glBindBuffer(GL_SHADER_STORAGE_BUFFER, 0);
    maximumNumberOfFragments = counters[0];
  }

  if ((currentOITMode == A_BUFFER_3D_USE_TEXTURES) || (currentOITMode == A_BUFFER_3D_USE_BUFFERS))
  {
    if (GLsizei(maximumNumberOfFragments) > aBuffer3DSize)
    {
      aBuffer3DSize = BitManipulationFunctions::getNextPowerOfTwo(aBuffer3DSize + 2) - 1;
      if (aBuffer3DSize > GLsizei(A_BUFFER_3D_MAX_SIZE - 1)) aBuffer3DSize = GLsizei(A_BUFFER_3D_MAX_SIZE - 1);

      DebugConsole_consoleOutLine("New maximumNumberOfFragments: ", maximumNumberOfFragments);
      DebugConsole_consoleOutLine("New aBuffer3DSize: ",            aBuffer3DSize);

      prepareABuffer3DShaders();
      deleteABuffer3D();
      initABuffer3D();

      reInitUIInformation = true;
      aBuffer3DNeedsResizing = aBuffer3DSize < GLsizei(A_BUFFER_3D_MAX_SIZE - 1);
    }
  }
  else if ((currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_TEXTURES) || (currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_BUFFERS))
  {
    if (GLuint64EXT(maximumNumberOfFragments) > aBuffer3DLinkedListSize)
    {
      aBuffer3DinkedListMaxSize = BitManipulationFunctions::getNextPowerOfTwo(aBuffer3DinkedListMaxSize + 2) - 1;
      if (aBuffer3DinkedListMaxSize > GLsizei(A_BUFFER_3D_MAX_SIZE - 1)) aBuffer3DinkedListMaxSize = GLsizei(A_BUFFER_3D_MAX_SIZE - 1);
      aBuffer3DLinkedListSize += (maximumNumberOfFragments - aBuffer3DLinkedListSize) << (oitUseDeferredShading ? 2 : 0); // << 2 for deferred shading to avoid some nasty rendering issues

      DebugConsole_consoleOutLine("New maximumNumberOfFragments: ",  maximumNumberOfFragments);
      DebugConsole_consoleOutLine("New aBuffer3DinkedListMaxSize: ", aBuffer3DinkedListMaxSize);
      DebugConsole_consoleOutLine("New aBuffer3DLinkedListSize: ",   aBuffer3DLinkedListSize);

      prepareABuffer3DShaders();
      deleteABuffer3D();
      initABuffer3D();

      reInitUIInformation = true;
      aBuffer3DNeedsResizing = aBuffer3DLinkedListSize < GLuint64EXT(A_BUFFER_3D_MAX_SIZE - 1);
    }
  }

  return aBuffer3DNeedsResizing;
}

void ParticleRenderingModelLoaderTest::deleteABuffer3D()
{
  // for A-Buffer3D texture version
  if (glIsTexture(aBuffer3DCounterTextureID))
  {
    glDeleteTextures(1, &aBuffer3DCounterTextureID);
    aBuffer3DCounterTextureID = 0;
  }
  if (glIsTexture(aBuffer3DTextureID))
  {
    glDeleteTextures(1, &aBuffer3DTextureID);
    aBuffer3DTextureID = 0;
  }
  // for A-Buffer3D buffer version
  if (glIsBuffer(aBuffer3DCounterBufferID))
  {
    glDeleteBuffers(1, &aBuffer3DCounterBufferID);
    aBuffer3DCounterBufferID = 0;
  }
  if (glIsBuffer(aBuffer3DBufferID))
  {
    glDeleteBuffers(1, &aBuffer3DBufferID);
    aBuffer3DBufferID = 0;
  }
  // for A-Buffer3D linked list texture & buffer version
  if (glIsBuffer(aBuffer3DLinkedListAtomicCounterBufferID))
  {
    glDeleteBuffers(1, &aBuffer3DLinkedListAtomicCounterBufferID);
    aBuffer3DLinkedListAtomicCounterBufferID = 0;
  }
  // for A-Buffer3D linked list texture version
  if (glIsTexture(aBuffer3DLinkedListOffsetTextureID))
  {
    glDeleteTextures(1, &aBuffer3DLinkedListOffsetTextureID);
    aBuffer3DLinkedListOffsetTextureID = 0;
  }
  if (glIsTexture(aBuffer3DLinkedListTextureID))
  {
    glDeleteTextures(1, &aBuffer3DLinkedListTextureID);
    aBuffer3DLinkedListTextureID = 0;
  }
  // for A-Buffer3D linked list buffer version
  if (glIsBuffer(aBuffer3DLinkedListOffsetBufferID))
  {
    glDeleteBuffers(1, &aBuffer3DLinkedListOffsetBufferID);
    aBuffer3DLinkedListOffsetBufferID = 0;
  }
  if (glIsBuffer(aBuffer3DLinkedListBufferID))
  {
    glDeleteBuffers(1, &aBuffer3DLinkedListBufferID);
    aBuffer3DLinkedListBufferID = 0;
  }
}

void ParticleRenderingModelLoaderTest::writeABuffer3DBufferToFile(GLbitfield barriers) const
{
  if (currentOITMode == A_BUFFER_3D_USE_BUFFERS)
  {
    // ensure that all global memory write are done before resolving
    glMemoryBarrier(barriers);

    openGLShaderDisplayABuffer3D->enableShaderProgram();
    openGLShaderDisplayABuffer3D->setUniform1i("screenWidth", screenWidth);
    openGLShaderDisplayABuffer3D->setUniform1i("screenHeight", screenHeight);
    openGLShaderDisplayABuffer3D->setUniform1i("sortGlobalFragmentArrayAndNotRender", true);
    GLAuxiliaryFunctions::createFullScreenQuadWithDummyVAO(openGLDriverInfo->isNvidia(), dummyVao);
    openGLShaderDisplayABuffer3D->disableShaderProgram();

    // ensure that all global memory write are done before resolving
    glMemoryBarrier(barriers);

    glBindBufferBase(GL_SHADER_STORAGE_BUFFER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D, aBuffer3DBufferID);
    const GLuint* pixelColors = (const GLuint*)glMapBuffer(GL_SHADER_STORAGE_BUFFER, GL_READ_ONLY);
    const int arraySize3D     = 3 * screenWidth * screenHeight * aBuffer3DSize; // 3 channels per packed-fragment
    vector<GLubyte> imagePixels;
    imagePixels.resize(arraySize3D << 1); // 4 bytes per pixel
    int imagePixelsIndex = 0;
    GLubyte pixelR = 0, pixelG = 0, pixelB = 0, pixelA = 0;
    for (int i = 0; i < arraySize3D; i += 3) // 3 channels per packed-fragment
    {
      pixelR = GLubyte((pixelColors[i] >> 24) & 255);
      pixelG = GLubyte((pixelColors[i] >> 16) & 255);
      pixelB = GLubyte((pixelColors[i] >>  8) & 255);
      pixelA = GLubyte((pixelColors[i]      ) & 255); // GLubyte(255.0f * *(reinterpret_cast<GLfloat*>(&pixelColors[i + 1]))); // convert GLuint to GLfloat representation, depth found on 3rd channel

      if ((pixelR == 0) && (pixelG == 0) && (pixelB == 0) && (pixelA == 0))
      {
        const GLfloat* color = (blackOrWhiteBackground) ? WHITE_BACKGROUND_COLOR.data() : BLACK_BACKGROUND_COLOR.data();
        imagePixels[imagePixelsIndex    ] = GLubyte(255.0f * color[0]);
        imagePixels[imagePixelsIndex + 1] = GLubyte(255.0f * color[1]);
        imagePixels[imagePixelsIndex + 2] = GLubyte(255.0f * color[2]);
        imagePixels[imagePixelsIndex + 3] = GLubyte(255.0f * color[3]);
      }
      else
      {
        imagePixels[imagePixelsIndex    ] = pixelR;
        imagePixels[imagePixelsIndex + 1] = pixelG;
        imagePixels[imagePixelsIndex + 2] = pixelB;
        imagePixels[imagePixelsIndex + 3] = pixelA;
      }
      imagePixelsIndex += 4;
    }
    glUnmapBuffer(GL_SHADER_STORAGE_BUFFER);
    glBindBuffer(GL_SHADER_STORAGE_BUFFER, 0);

    const int arraySize2D = screenWidth * screenHeight << 2; // 4 bytes per pixel per layer
    for (int sliceNumber = 0; sliceNumber < aBuffer3DSize; ++sliceNumber)
    {
      const string numberOfImagesString = StringAuxiliaryFunctions::formatNumberString(sliceNumber + 1, aBuffer3DSize);
      OpenGLILTexture::saveILImage(OpenGLAssetManager::getImagesDefaultDirectory(), string("Slice" + numberOfImagesString), OpenGLAssetManager::getDefaultScreenshotFormat(), 4, screenWidth, screenHeight, &imagePixels[arraySize2D * sliceNumber]);
    }
  }
}

void ParticleRenderingModelLoaderTest::prepareVBOs()
{
  if (isQuadVSFS()) // draw Quads
  {
    // Bind our buffers much like we would for texturing
    glGenBuffers(1, &VBOColorsID);
    glBindBuffer(GL_ARRAY_BUFFER, VBOColorsID);
    glBufferData(GL_ARRAY_BUFFER, glCachedColors.size() * sizeof(GLfloat), glCachedColors.data(), GL_STATIC_DRAW);

    glGenBuffers(1, &VBOTexCoordsID);
    glBindBuffer(GL_ARRAY_BUFFER, VBOTexCoordsID);
    glBufferData(GL_ARRAY_BUFFER, glCachedTexCoords.size() * sizeof(GLfloat), glCachedTexCoords.data(), GL_STATIC_DRAW);

    glGenBuffers(1, &VBOVerticesAndRadiiID);
    glBindBuffer(GL_ARRAY_BUFFER, VBOVerticesAndRadiiID);
    glBufferData(GL_ARRAY_BUFFER, glCachedPointsAndRadii.size() * sizeof(GLfloat), glCachedPointsAndRadii.data(), GL_STATIC_DRAW);
  }
  else
  {
    // Bind our buffers much like we would for texturing
    glGenBuffers(1, &VBOColorsID);
    glBindBuffer(GL_ARRAY_BUFFER, VBOColorsID);
    glBufferData(GL_ARRAY_BUFFER, colors.size() * sizeof(GLfloat), colors.data(), GL_STATIC_DRAW);

    glGenBuffers(1, &VBOVerticesAndRadiiID);
    glBindBuffer(GL_ARRAY_BUFFER, VBOVerticesAndRadiiID);
    glBufferData(GL_ARRAY_BUFFER, vertices.size() * sizeof(GLfloat), vertices.data(), GL_STATIC_DRAW);
  }

  glBindBuffer(GL_ARRAY_BUFFER, 0);
}

void ParticleRenderingModelLoaderTest::deleteVBOs()
{
  if (glIsBuffer(VBOColorsID))
  {
    glDeleteBuffers(1, &VBOColorsID);
  }
  if (glIsBuffer(VBOTexCoordsID))
  {
    glDeleteBuffers(1, &VBOTexCoordsID);
  }
  if (glIsBuffer(VBOVerticesAndRadiiID))
  {
    glDeleteBuffers(1, &VBOVerticesAndRadiiID);
  }

  VBOColorsID           = 0;
  VBOTexCoordsID        = 0;
  VBOVerticesAndRadiiID = 0;
}

void ParticleRenderingModelLoaderTest::renderNoCaching(bool skipReInitModelDataForRendering)
{
  if (reInitModelDataForRendering && !skipReInitModelDataForRendering)
  {
    previousAvailableMemory = howMuchCurrentlyAvailableVRAMMemory();
    // just to clear VRAM memory
    if (glIsList(allModelDataDisplayList))
    {
      glDeleteLists(allModelDataDisplayList, 1);
    }
    deleteVBOs();
    if (reInitColorArrays)
    {
      createParticlesColorsArray();
      createQuadColorsArray();
    }
    else
    {
      prepareParticles();
    }
    reInitColorArrays = false;
    reInitModelDataForRendering = false;
    printGLMemoryInfoOnSecondUpdate = 0;
  }

  if (isQuadVSFS()) // draw Quads
  {
    glBegin(GL_QUADS);
    for (size_t i = 0; i < vertices.size(); i += 4)
    {
      if (currentCachedRenderingTest == USE_NO_CACHING_METHOD_1)
      {
        // bottom-right
        glMultiTexCoord2d(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE,  1.0f, -1.0f);
        if (!useColoredObjects)
        {
          glColor4f(colors[i], colors[i + 1], colors[i + 2], colors[i + 3]);
        }
        glVertex4f(vertices[i], vertices[i + 1], vertices[i + 2], vertices[i + 3]);

        // top-right
        glMultiTexCoord2d(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE,  1.0f,  1.0f);
        if (!useColoredObjects)
        {
          glColor4f(colors[i], colors[i + 1], colors[i + 2], colors[i + 3]);
        }
        glVertex4f(vertices[i], vertices[i + 1], vertices[i + 2], vertices[i + 3]);

        // top-left
        glMultiTexCoord2d(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, -1.0f,  1.0f);
        if (!useColoredObjects)
        {
          glColor4f(colors[i], colors[i + 1], colors[i + 2], colors[i + 3]);
        }
        glVertex4f(vertices[i], vertices[i + 1], vertices[i + 2], vertices[i + 3]);

        // bottom-left
        glMultiTexCoord2d(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, -1.0f, -1.0f);
        if (!useColoredObjects)
        {
          glColor4f(colors[i], colors[i + 1], colors[i + 2], colors[i + 3]);
        }
        glVertex4f(vertices[i], vertices[i + 1], vertices[i + 2], vertices[i + 3]);
      }
      else // if (currentCachedRenderingTest == USE_NO_CACHING_METHOD_2)
      {
        // bottom-right
        glMultiTexCoord2d(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE,  1.0f, -1.0f);
        if (!useColoredObjects)
        {
          glColor4fv(&colors[i]);
        }
        glVertex4fv(&vertices[i]);

        // top-right
        glMultiTexCoord2d(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE,  1.0f,  1.0f);
        if (!useColoredObjects)
        {
          glColor4fv(&colors[i]);
        }
        glVertex4fv(&vertices[i]);

        // top-left
        glMultiTexCoord2d(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, -1.0f,  1.0f);
        if (!useColoredObjects)
        {
          glColor4fv(&colors[i]);
        }
        glVertex4fv(&vertices[i]);

        // bottom-left
        glMultiTexCoord2d(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, -1.0f, -1.0f);
        if (!useColoredObjects)
        {
          glColor4fv(&colors[i]);
        }
        glVertex4fv(&vertices[i]);
      }
    }
    glEnd();
  }
  else // draw Points
  {
    GLenum faceMode = 0;
    if (isQuadVSTESSFS())
    {
      glPatchParameteri(GL_PATCH_VERTICES, 1);
      faceMode = GL_PATCHES;
    }
    else
    {
      faceMode = GL_POINTS;
    }

    glBegin(faceMode);
    for (size_t i = 0; i < vertices.size(); i += 4)
    {
      if (currentCachedRenderingTest == USE_NO_CACHING_METHOD_1)
      {
        if (!useColoredObjects)
        {
          glColor4f(colors[i], colors[i + 1], colors[i + 2], colors[i + 3]);
        }
        glVertex4f(vertices[i], vertices[i + 1], vertices[i + 2], vertices[i + 3]);
      }
      else // if (currentCachedRenderingTest == USE_NO_CACHING_METHOD_2)
      {
        if (!useColoredObjects)
        {
          glColor4fv(&colors[i]);
        }
        glVertex4fv(&vertices[i]);
      }
    }
    glEnd();
  }
}

void ParticleRenderingModelLoaderTest::renderVertexArrays()
{
  if (reInitModelDataForRendering)
  {
    previousAvailableMemory = howMuchCurrentlyAvailableVRAMMemory();
    // just to clear VRAM memory
    if (glIsList(allModelDataDisplayList))
    {
      glDeleteLists(allModelDataDisplayList, 1);
    }
    deleteVBOs();
    if (reInitColorArrays)
    {
      createParticlesColorsArray();
      createQuadColorsArray();
    }
    else
    {
      prepareParticles();
    }
    reInitColorArrays = false;
    reInitModelDataForRendering = false;
    printGLMemoryInfoOnSecondUpdate = 0;
  }

  if (isQuadVSFS()) // draw Quads
  {
    if (!useColoredObjects)
    {
      glEnableClientState(GL_COLOR_ARRAY);
    }
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);
    glEnableClientState(GL_VERTEX_ARRAY);

    if (!useColoredObjects)
    {
      glColorPointer(4, GL_FLOAT, 0, glCachedColors.data());
    }
    glClientActiveTexture(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    glTexCoordPointer(2, GL_FLOAT, 0, glCachedTexCoords.data());
    glVertexPointer(4, GL_FLOAT, 0, glCachedPointsAndRadii.data());

    glDrawArrays(GL_QUADS, 0, GLsizei(glCachedPointsAndRadii.size() / 4));

    // Disable our client state back to normal drawing (only for ensuring further state-based changes)
    if (!useColoredObjects)
    {
      glDisableClientState(GL_COLOR_ARRAY);
    }
    glDisableClientState(GL_TEXTURE_COORD_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);
  }
  else // draw Points
  {
    GLenum faceMode = 0;
    if (isQuadVSTESSFS())
    {
      glPatchParameteri(GL_PATCH_VERTICES, 1);
      faceMode = GL_PATCHES;
    }
    else
    {
      faceMode = GL_POINTS;
    }

    if (!useColoredObjects)
    {
      glEnableClientState(GL_COLOR_ARRAY);
    }
    glEnableClientState(GL_VERTEX_ARRAY);

    if (!useColoredObjects)
    {
      glColorPointer(4, GL_FLOAT, 0, colors.data());
    }
    glVertexPointer(4, GL_FLOAT, 0, vertices.data());

    glDrawArrays(faceMode, 0, GLsizei(vertices.size() / 4));

    // Disable our client state back to normal drawing (only for ensuring further state-based changes)
    if (!useColoredObjects)
    {
      glDisableClientState(GL_COLOR_ARRAY);
    }
    glDisableClientState(GL_VERTEX_ARRAY);
  }
}

void ParticleRenderingModelLoaderTest::renderDisplayLists()
{
  if (reInitModelDataForRendering)
  {
    previousAvailableMemory = howMuchCurrentlyAvailableVRAMMemory();
    // just to clear VRAM memory
    if (glIsList(allModelDataDisplayList))
    {
      glDeleteLists(allModelDataDisplayList, 1);
    }
    deleteVBOs();
    if (reInitColorArrays)
    {
      createParticlesColorsArray();
      createQuadColorsArray();
    }
    else
    {
      prepareParticles();
    }
    glNewList(allModelDataDisplayList, GL_COMPILE);
    renderNoCaching(true);
    glEndList();
    reInitColorArrays = false;
    reInitModelDataForRendering = false;
    printGLMemoryInfoOnSecondUpdate = 0;
  }

  glCallList(allModelDataDisplayList);
}

void ParticleRenderingModelLoaderTest::renderVBOs()
{
  if (reInitModelDataForRendering)
  {
    previousAvailableMemory = howMuchCurrentlyAvailableVRAMMemory();
    // just to clear VRAM memory
    if (glIsList(allModelDataDisplayList))
    {
      glDeleteLists(allModelDataDisplayList, 1);
    }
    deleteVBOs();
    if (reInitColorArrays)
    {
      createParticlesColorsArray();
      createQuadColorsArray();
    }
    else
    {
      prepareParticles();
    }
    prepareVBOs();
    reInitColorArrays = false;
    reInitModelDataForRendering = false;
    printGLMemoryInfoOnSecondUpdate = 0;
  }

  if (isQuadVSFS()) // draw Quads
  {
    if (!useColoredObjects)
    {
      glEnableClientState(GL_COLOR_ARRAY);
    }
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);
    glEnableClientState(GL_VERTEX_ARRAY);

    // Bind our buffers much like we would for texturing
    if (!useColoredObjects)
    {
      glBindBuffer(GL_ARRAY_BUFFER, VBOColorsID);
      glColorPointer(4, GL_FLOAT, 0, nullptr);
    }
    glBindBuffer(GL_ARRAY_BUFFER, VBOTexCoordsID);
    glClientActiveTexture(GL_TEXTURE0 + ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    glTexCoordPointer(2, GL_FLOAT, 0, nullptr);
    glBindBuffer(GL_ARRAY_BUFFER, VBOVerticesAndRadiiID);
    glVertexPointer(4, GL_FLOAT, 0, nullptr);

    glDrawArrays(GL_QUADS, 0, GLsizei(glCachedPointsAndRadii.size() / 4));

    glBindBuffer(GL_ARRAY_BUFFER, 0);

    // Disable our client state back to normal drawing (only for ensuring further state-based changes)
    if (!useColoredObjects)
    {
      glDisableClientState(GL_COLOR_ARRAY);
    }
    glDisableClientState(GL_TEXTURE_COORD_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);
  }
  else // draw Points
  {
    GLenum faceMode = 0;
    if (isQuadVSTESSFS())
    {
      glPatchParameteri(GL_PATCH_VERTICES, 1);
      faceMode = GL_PATCHES;
    }
    else
    {
      faceMode = GL_POINTS;
    }

    if (!useColoredObjects)
    {
      glEnableClientState(GL_COLOR_ARRAY);
    }
    glEnableClientState(GL_VERTEX_ARRAY);

    // Bind our buffers much like we would for texturing
    if (!useColoredObjects)
    {
      glBindBuffer(GL_ARRAY_BUFFER, VBOColorsID);
      glColorPointer(4, GL_FLOAT, 0, nullptr);
    }
    glBindBuffer(GL_ARRAY_BUFFER, VBOVerticesAndRadiiID);
    glVertexPointer(4, GL_FLOAT, 0, nullptr);

    glDrawArrays(faceMode, 0, GLsizei(vertices.size() / 4));

    glBindBuffer(GL_ARRAY_BUFFER, 0);

    // Disable our client state back to normal drawing (only for ensuring further state-based changes)
    if (!useColoredObjects)
    {
      glDisableClientState(GL_COLOR_ARRAY);
    }
    glDisableClientState(GL_VERTEX_ARRAY);
  }
}

void ParticleRenderingModelLoaderTest::prepareCylinderVBOs()
{
  // Bind our buffers much like we would for texturing
  glGenBuffers(1, &cylinderVBOColorsID);
  glBindBuffer(GL_ARRAY_BUFFER, cylinderVBOColorsID);
  glBufferData(GL_ARRAY_BUFFER, cylinderColors.size() * sizeof(GLfloat), cylinderColors.data(), GL_STATIC_DRAW);

  glGenBuffers(1, &cylinderVBODirectionsID);
  glBindBuffer(GL_ARRAY_BUFFER, cylinderVBODirectionsID);
  glBufferData(GL_ARRAY_BUFFER, cylinderDirections.size() * sizeof(GLfloat), cylinderDirections.data(), GL_STATIC_DRAW);

  glGenBuffers(1, &cylinderVBOVerticesAndHeightsID);
  glBindBuffer(GL_ARRAY_BUFFER, cylinderVBOVerticesAndHeightsID);
  glBufferData(GL_ARRAY_BUFFER, cylinderVertices.size() * sizeof(GLfloat), cylinderVertices.data(), GL_STATIC_DRAW);

  glBindBuffer(GL_ARRAY_BUFFER, 0);
}

void ParticleRenderingModelLoaderTest::deleteCylinderVBOs()
{
  if (glIsBuffer(cylinderVBOColorsID))
  {
    glDeleteBuffers(1, &cylinderVBOColorsID);
  }
  if (glIsBuffer(cylinderVBODirectionsID))
  {
    glDeleteBuffers(1, &cylinderVBODirectionsID);
  }
  if (glIsBuffer(cylinderVBOVerticesAndHeightsID))
  {
    glDeleteBuffers(1, &cylinderVBOVerticesAndHeightsID);
  }

  cylinderVBOColorsID             = 0;
  cylinderVBODirectionsID         = 0;
  cylinderVBOVerticesAndHeightsID = 0;
}

void ParticleRenderingModelLoaderTest::renderCylinderNoCaching(bool skipReInitModelDataForRendering)
{
  if (reInitModelCylinderDataForRendering && !skipReInitModelDataForRendering)
  {
    previousAvailableMemory = howMuchCurrentlyAvailableVRAMMemory();
    // just to clear VRAM memory
    if (glIsList(allModelCylinderDataDisplayList))
    {
      glDeleteLists(allModelCylinderDataDisplayList, 1);
    }
    deleteCylinderVBOs();
    reInitModelCylinderDataForRendering = false;
  }

  int normalIndex = 0;
  glBegin(GL_POINTS);
  for (size_t vertexIndex = 0; vertexIndex < cylinderVertices.size(); vertexIndex += 4)
  {
    if (currentCachedRenderingTest == USE_NO_CACHING_METHOD_1)
    {
      if (!useColoredObjects)
      {
        glColor4f(cylinderColors[vertexIndex], cylinderColors[vertexIndex + 1], cylinderColors[vertexIndex + 2], cylinderColors[vertexIndex + 3]);
      }
      glNormal3f(cylinderDirections[normalIndex], cylinderDirections[normalIndex + 1], cylinderDirections[normalIndex + 2]);
      glVertex4f(cylinderVertices[vertexIndex], cylinderVertices[vertexIndex + 1], cylinderVertices[vertexIndex + 2], cylinderVertices[vertexIndex + 3]);
    }
    else // if (currentCachedRenderingTest == USE_NO_CACHING_METHOD_2)
    {
      if (!useColoredObjects)
      {
        glColor4fv(&cylinderColors[vertexIndex]);
      }
      glNormal3fv(&cylinderDirections[normalIndex]);
      glVertex4fv(&cylinderVertices[vertexIndex]);
    }
    normalIndex += 3;
  }
  glEnd();
}

void ParticleRenderingModelLoaderTest::renderCylinderVertexArrays()
{
  if (reInitModelCylinderDataForRendering)
  {
    previousAvailableMemory = howMuchCurrentlyAvailableVRAMMemory();
    // just to clear VRAM memory
    if (glIsList(allModelCylinderDataDisplayList))
    {
      glDeleteLists(allModelCylinderDataDisplayList, 1);
    }
    deleteCylinderVBOs();
    reInitModelCylinderDataForRendering = false;
  }

  if (!useColoredObjects)
  {
    glEnableClientState(GL_COLOR_ARRAY);
  }
  glEnableClientState(GL_NORMAL_ARRAY);
  glEnableClientState(GL_VERTEX_ARRAY);

  if (!useColoredObjects)
  {
    glColorPointer(4, GL_FLOAT, 0, cylinderColors.data());
  }
  glNormalPointer(GL_FLOAT, 0, cylinderDirections.data());
  glVertexPointer(4, GL_FLOAT, 0, cylinderVertices.data());

  glDrawArrays(GL_POINTS, 0, GLsizei(cylinderVertices.size() / 4));

  // Disable our client state back to normal drawing (only for ensuring further state-based changes)
  if (!useColoredObjects)
  {
    glDisableClientState(GL_COLOR_ARRAY);
  }
  glDisableClientState(GL_NORMAL_ARRAY);
  glDisableClientState(GL_VERTEX_ARRAY);
}

void ParticleRenderingModelLoaderTest::renderCylinderDisplayLists()
{
  if (reInitModelCylinderDataForRendering)
  {
    previousAvailableMemory = howMuchCurrentlyAvailableVRAMMemory();
    // just to clear VRAM memory
    if (glIsList(allModelCylinderDataDisplayList))
    {
      glDeleteLists(allModelCylinderDataDisplayList, 1);
    }
    deleteCylinderVBOs();
    glNewList(allModelCylinderDataDisplayList, GL_COMPILE);
    renderCylinderNoCaching(true);
    glEndList();
    reInitModelCylinderDataForRendering = false;
  }

  glCallList(allModelCylinderDataDisplayList);
}

void ParticleRenderingModelLoaderTest::renderCylinderVBOs()
{
  if (reInitModelCylinderDataForRendering)
  {
    previousAvailableMemory = howMuchCurrentlyAvailableVRAMMemory();
    // just to clear VRAM memory
    if (glIsList(allModelCylinderDataDisplayList))
    {
      glDeleteLists(allModelCylinderDataDisplayList, 1);
    }
    deleteCylinderVBOs();
    prepareCylinderVBOs();
    reInitModelCylinderDataForRendering = false;
  }

  if (!useColoredObjects)
  {
    glEnableClientState(GL_COLOR_ARRAY);
  }
  glEnableClientState(GL_NORMAL_ARRAY);
  glEnableClientState(GL_VERTEX_ARRAY);

  // Bind our buffers much like we would for texturing
  if (!useColoredObjects)
  {
    glBindBuffer(GL_ARRAY_BUFFER, cylinderVBOColorsID);
    glColorPointer(4, GL_FLOAT, 0, nullptr);
  }
  glBindBuffer(GL_ARRAY_BUFFER, cylinderVBODirectionsID);
  glNormalPointer(GL_FLOAT, 0, nullptr);
  glBindBuffer(GL_ARRAY_BUFFER, cylinderVBOVerticesAndHeightsID);
  glVertexPointer(4, GL_FLOAT, 0, nullptr);

  glDrawArrays(GL_POINTS, 0, GLsizei(cylinderVertices.size() / 4));

  glBindBuffer(GL_ARRAY_BUFFER, 0);

  // Disable our client state back to normal drawing (only for ensuring further state-based changes)
  if (!useColoredObjects)
  {
    glDisableClientState(GL_COLOR_ARRAY);
  }
  glDisableClientState(GL_NORMAL_ARRAY);
  glDisableClientState(GL_VERTEX_ARRAY);
}

void ParticleRenderingModelLoaderTest::clearScreen(bool useGBuffer) const
{
  if (useGBuffer || blackOrWhiteBackground)
  {
    glClearColor(WHITE_BACKGROUND_COLOR[0], WHITE_BACKGROUND_COLOR[1], WHITE_BACKGROUND_COLOR[2], WHITE_BACKGROUND_COLOR[3]);
  }
  else
  {
    glClearColor(BLACK_BACKGROUND_COLOR[0], BLACK_BACKGROUND_COLOR[1], BLACK_BACKGROUND_COLOR[2], BLACK_BACKGROUND_COLOR[3]);
  }
  glClearStencil(0); // clear stencil buffer
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
}

void ParticleRenderingModelLoaderTest::disableLights(bool disableAllLightSources) const
{
  glDisable(GL_LIGHTING);
  // disable two-sided lighting vertex program for Point Sphere Model shaders to get a speed-boost
  // glDisable(GL_VERTEX_PROGRAM_TWO_SIDE); // Intel GPUs may not honor this command, must manually populate the gl_BackColor state variable in the vertex shader
  glDisable(GL_NORMALIZE);

  if (disableAllLightSources)
  {
    openGLModelAmbientLight->disable();
    for (const auto& openGLLight : openGLLights)
    {
      openGLLight->disable();
    }
    openGLMaterial->disable();
  }
}

void ParticleRenderingModelLoaderTest::enableLights(bool enableAllLightSources)
{
  glEnable(GL_LIGHTING);
  // disable two-sided lighting vertex program for Point Sphere Model shaders to get a speed-boost
  // glEnable(GL_VERTEX_PROGRAM_TWO_SIDE); // Intel GPUs may not honor this command, must manually populate the gl_BackColor state variable in the vertex shader
  glEnable(GL_NORMALIZE);

  if (enableAllLightSources)
  {
    openGLModelAmbientLight->enable();
    for (auto& openGLLight : openGLLights)
    {
      openGLLight->enable(GL_LIGHT0, usePositionalLights != 0);
    }
    openGLMaterial->enable();
  }
}

void ParticleRenderingModelLoaderTest::renderParticleScene(bool environmentMappingRenderPass, bool useGBuffer)
{
  clearScreen(useGBuffer);

  if (wireframe || !useBackFaceCulling) // don't cull faces in wireframe mode
  {
    glDisable(GL_CULL_FACE);
  }
  else
  {
    glFrontFace(GL_CCW); // implied
    glEnable(GL_CULL_FACE);
    glCullFace(GL_BACK);
  }

  if (!environmentMappingRenderPass)
  {
    updateLighting();
  }

  glPushMatrix();

  openGLEulerCamera->setEulerCamera(cameraDistanceX, cameraDistanceY, cameraDistanceZ, cameraAngleX, cameraAngleY,
                                    openGLAssimpModelLoader->getSceneScaleFactor(),
                                    openGLAssimpModelLoader->getSceneCenterX(),
                                    openGLAssimpModelLoader->getSceneCenterY(),
                                    openGLAssimpModelLoader->getSceneCenterZ());

  if (useFog)
  {
    glFogfv(GL_FOG_COLOR, blackOrWhiteBackground ? WHITE_BACKGROUND_COLOR.data() : BLACK_BACKGROUND_COLOR.data());
    glEnable(GL_FOG);
  }
  else
  {
    glDisable(GL_FOG);
  }

  if (openGLDriverInfo->supports430Shaders() && useTransparency)
  {
    bool frameOK = true;
    do
    {
      renderParticleGeometry(environmentMappingRenderPass, useGBuffer);
      frameOK = (currentOITMode == NO_OIT) || !oitAutoManageABuffer3D || !isNavigating || !autoManageABuffer3D();
    }
    while (!frameOK);
  }
  else
  {
    renderParticleGeometry(environmentMappingRenderPass, useGBuffer);
  }

  glPopMatrix();
}

void ParticleRenderingModelLoaderTest::renderParticleGeometry(bool environmentMappingRenderPass, bool useGBuffer)
{
  if (multisample && (currentGBUfferEffectType != NO_GBUFFER_EFFECT))
  {
    glDisable(GL_MULTISAMPLE);
  }

  if (useTransparency)
  {
    // disable depth mask & alpha test for semi-transparent object
    glDepthMask(GL_FALSE);
    glDisable(GL_ALPHA_TEST);
  }

  if (useTexturing && !environmentMappingRenderPass)
  {
    if (openGLDriverInfo->supports_GL_EXT_framebuffer_object() && useEnvironmentMapping)
    {
      // Use the GL_MODULATE texture function to effectively multiply
      // each pixel in the texture by the current alpha value
      glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

      openGLFrameBufferObjectForEnvironmentMapping->bindTexture(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    }
    else
    {
      // Use the GL_DECAL texture function to effectively blend between the vertex color and the texture color
      // based on the textures alpha channel, but leave the alpha channel set to the vertex color
      glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);

      openGLILTexture->bindTexture(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    }

    glEnable(GL_TEXTURE_2D);
  }
  else
  {
    glDisable(GL_TEXTURE_2D);
  }

  GLbitfield barriers = 0;
  if (useTransparency && (currentOITMode != NO_OIT))
  {
    if (multisample)
    {
      glDisable(GL_MULTISAMPLE);
    }

    // OIT Shader Step 1
    if (currentOITMode == A_BUFFER_3D_USE_TEXTURES) // texture storage path
    {
      barriers = GL_SHADER_IMAGE_ACCESS_BARRIER_BIT;

      openGLShaderClearABuffer3D->enableShaderProgram();
      openGLShaderClearABuffer3D->setUniform1i("screenWidth", screenWidth);
      openGLShaderClearABuffer3D->setUniform1i("screenHeight", screenHeight);
      GLAuxiliaryFunctions::createFullScreenQuadWithDummyVAO(openGLDriverInfo->isNvidia(), dummyVao);
      openGLShaderClearABuffer3D->disableShaderProgram();
    }
    else if (currentOITMode == A_BUFFER_3D_USE_BUFFERS) // buffer storage path uses GL 4.3 faster delete (non-shader usage)
    {
      barriers = GL_SHADER_STORAGE_BARRIER_BIT;

      GLuint zero = 0;
      glBindBufferBase(GL_SHADER_STORAGE_BUFFER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_COUNTER, aBuffer3DCounterBufferID);
      glClearBufferData(GL_SHADER_STORAGE_BUFFER, GL_R32UI, GL_RED_INTEGER, GL_UNSIGNED_INT, &zero);
      if (oitCaptureABuffer3DBufferToFile)
      {
        // the A-Buffer-3D doesn't really need cleaning for rendering, do it for write-to-file data dumping only
        glBindBufferBase(GL_SHADER_STORAGE_BUFFER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D, aBuffer3DBufferID);
        glClearBufferData(GL_SHADER_STORAGE_BUFFER, GL_RGB32UI, GL_RGB_INTEGER, GL_UNSIGNED_INT, &zero);
      }
      glBindBuffer(GL_SHADER_STORAGE_BUFFER, 0);
    }
    else if (currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_TEXTURES) // texture storage path with linked lists
    {
      barriers = GL_SHADER_IMAGE_ACCESS_BARRIER_BIT | GL_SHADER_STORAGE_BARRIER_BIT;

      GLuint zero = 0;
      glBindBufferBase(GL_ATOMIC_COUNTER_BUFFER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_ATOMIC_COUNTER, aBuffer3DLinkedListAtomicCounterBufferID);
      glClearBufferData(GL_ATOMIC_COUNTER_BUFFER, GL_R32UI, GL_RED_INTEGER, GL_UNSIGNED_INT, &zero);
      glBindBuffer(GL_ATOMIC_COUNTER_BUFFER, 0);

      openGLShaderClearABuffer3D->enableShaderProgram();
      openGLShaderClearABuffer3D->setUniform1i("screenWidth", screenWidth);
      openGLShaderClearABuffer3D->setUniform1i("screenHeight", screenHeight);
      GLAuxiliaryFunctions::createFullScreenQuadWithDummyVAO(openGLDriverInfo->isNvidia(), dummyVao);
      openGLShaderClearABuffer3D->disableShaderProgram();
    }
    else if (currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_BUFFERS) // buffer storage path uses GL 4.3 faster delete (non-shader usage) with linked lists
    {
      barriers = GL_SHADER_STORAGE_BARRIER_BIT;

      GLuint zero     = GLuint(0);
      GLuint minusOne = GLuint (-1);
      glBindBufferBase(GL_ATOMIC_COUNTER_BUFFER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_ATOMIC_COUNTER, aBuffer3DLinkedListAtomicCounterBufferID);
      glClearBufferData(GL_ATOMIC_COUNTER_BUFFER, GL_R32UI, GL_RED_INTEGER, GL_UNSIGNED_INT, &zero);
      glBindBuffer(GL_ATOMIC_COUNTER_BUFFER, 0);
      glBindBufferBase(GL_SHADER_STORAGE_BUFFER, ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET, aBuffer3DLinkedListOffsetBufferID);
      glClearBufferData(GL_SHADER_STORAGE_BUFFER, GL_R32UI, GL_RED_INTEGER, GL_UNSIGNED_INT, &minusOne); // reset offset to -1
      glBindBuffer(GL_SHADER_STORAGE_BUFFER, 0);
    }

    // ensure that all global memory write are done before starting to render
    glMemoryBarrier(barriers);
  }

  if (openGLDriverInfo->isAMDATI())
  {
    enableLights(); // make sure lights are enabled for AMD-ATI hardware
  }
  if (currentImpostorShader == CYLINDER_VS_GS_FS && currentImpostorWireframe != CYLINDER_EXPLOSION)
  {
    if (currentImpostorWireframe == FULL_IMPOSTOR_WIREFRAME || currentImpostorWireframe == IMPOSTOR_POINTS_ONLY)
    {
      openGLShaderPointSphereModelsQuadVFGSFS->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, USE_COLOR_MATERIAL, openGLAssimpModelLoader->getSceneScaleFactor(),
                                                                     screenWidth, screenHeight, radiusFactor, usePointAntialias, usePerspectiveCorrection,
                                                                     useTexturing && !environmentMappingRenderPass, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, useEnvironmentMapping,
                                                                     useFog, false, 1.0f,
                                                                     useGBuffer,
                                                                     screenWidth, screenHeight, aBuffer3DSize, oitCaptureABuffer3DBufferToFile,
                                                                     false, IMPOSTORS_RADIUS_SIZE);

      if (useColoredObjects)
      {
        if (useTransparency)
        {
          MODEL_COLOR_WITH_ALPHA[3] = opacity;
          glColor4fv(MODEL_COLOR_WITH_ALPHA.data());
        }
        else
        {
          glColor4fv(MODEL_COLOR.data());
        }
      }

      if (wireframe)
      {
        glPolygonMode(GL_FRONT_AND_BACK, GL_LINE); // set wireframe
      }

      if ((currentCachedRenderingTest == USE_NO_CACHING_METHOD_1) || (currentCachedRenderingTest == USE_NO_CACHING_METHOD_2))
      {
        renderNoCaching(false);
      }
      else if (currentCachedRenderingTest == USE_VERTEX_ARRAYS)
      {
        renderVertexArrays();
      }
      else if (currentCachedRenderingTest == USE_DISPLAY_LISTS)
      {
        renderDisplayLists();
      }
      else // if (currentCachedRenderingTest == USE_VBOS)
      {
        renderVBOs();
      }

      if (wireframe)
      {
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL); // reset wireframe state
      }

      openGLShaderPointSphereModelsQuadVFGSFS->disableShaderProgram();
    }

    if (currentImpostorWireframe == FULL_IMPOSTOR_WIREFRAME || currentImpostorWireframe == IMPOSTOR_LINES_ONLY)
    {
      openGLShaderCylinderVFGSFS->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, USE_COLOR_MATERIAL, openGLAssimpModelLoader->getSceneScaleFactor(),
                                                        screenWidth, screenHeight, radiusFactor, usePointAntialias, usePerspectiveCorrection,
                                                        useTexturing && !environmentMappingRenderPass, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, useEnvironmentMapping,
                                                        useFog, false, 1.0f,
                                                        useGBuffer,
                                                        screenWidth, screenHeight, aBuffer3DSize, oitCaptureABuffer3DBufferToFile,
                                                        false, IMPOSTORS_RADIUS_SIZE);

      if (useColoredObjects)
      {
        if (useTransparency)
        {
          MODEL_COLOR_WITH_ALPHA[3] = opacity;
          glColor4fv(MODEL_COLOR_WITH_ALPHA.data());
        }
        else
        {
          glColor4fv(MODEL_COLOR.data());
        }
      }

      if (wireframe)
      {
        glPolygonMode(GL_FRONT_AND_BACK, GL_LINE); // set wireframe
      }

      if ((currentCachedRenderingTest == USE_NO_CACHING_METHOD_1) || (currentCachedRenderingTest == USE_NO_CACHING_METHOD_2))
      {
        renderCylinderNoCaching(false);
      }
      else if (currentCachedRenderingTest == USE_VERTEX_ARRAYS)
      {
        renderCylinderVertexArrays();
      }
      else if (currentCachedRenderingTest == USE_DISPLAY_LISTS)
      {
        renderCylinderDisplayLists();
      }
      else // if (currentCachedRenderingTest == USE_VBOS)
      {
        renderCylinderVBOs();
      }

      if (wireframe)
      {
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL); // reset wireframe state
      }

      openGLShaderCylinderVFGSFS->disableShaderProgram();
    }
  }
  else
  {
    if (currentImpostorShader == POINT_SPHERE)
    {
      openGLShaderPointSphereModels->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, USE_COLOR_MATERIAL, openGLAssimpModelLoader->getSceneScaleFactor(),
                                                           screenWidth, screenHeight, radiusFactor, usePointAntialias, usePerspectiveCorrection,
                                                           useTexturing && !environmentMappingRenderPass, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, useEnvironmentMapping,
                                                           useFog, false, 1.0f,
                                                           useGBuffer,
                                                           screenWidth, screenHeight, aBuffer3DSize, oitCaptureABuffer3DBufferToFile);
    }
    else if (currentImpostorShader == QUAD_SPHERE_VS_FS)
    {
      openGLShaderPointSphereModelsQuadVFFS->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, USE_COLOR_MATERIAL, openGLAssimpModelLoader->getSceneScaleFactor(),
                                                                   screenWidth, screenHeight, radiusFactor, usePointAntialias, usePerspectiveCorrection,
                                                                   useTexturing && !environmentMappingRenderPass, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, useEnvironmentMapping,
                                                                   useFog, false, 1.0f,
                                                                   useGBuffer,
                                                                   screenWidth, screenHeight, aBuffer3DSize, oitCaptureABuffer3DBufferToFile);
    }
    else if (currentImpostorShader == QUAD_SPHERE_VS_GS_FS)
    {
      openGLShaderPointSphereModelsQuadVFGSFS->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, USE_COLOR_MATERIAL, openGLAssimpModelLoader->getSceneScaleFactor(),
                                                                     screenWidth, screenHeight, radiusFactor, usePointAntialias, usePerspectiveCorrection,
                                                                     useTexturing && !environmentMappingRenderPass, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, useEnvironmentMapping,
                                                                     useFog, false, 1.0f,
                                                                     useGBuffer,
                                                                     screenWidth, screenHeight, aBuffer3DSize, oitCaptureABuffer3DBufferToFile);
    }
    else if (currentImpostorShader == QUAD_SPHERE_VS_TESS_LOD_FS)
    {
      openGLShaderPointSphereModelsQuadVFTESSLODFS->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, USE_COLOR_MATERIAL, openGLAssimpModelLoader->getSceneScaleFactor(),
                                                                          screenWidth, screenHeight, radiusFactor, usePointAntialias, usePerspectiveCorrection,
                                                                          useTexturing && !environmentMappingRenderPass, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, useEnvironmentMapping,
                                                                          useFog, false, 1.0f,
                                                                          useGBuffer,
                                                                          screenWidth, screenHeight, aBuffer3DSize, oitCaptureABuffer3DBufferToFile);
    }
    else if (currentImpostorShader == QUAD_SPHERE_VS_TESS_QUAD_FS)
    {
      openGLShaderPointSphereModelsQuadVFTESSQuadFS->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, USE_COLOR_MATERIAL, openGLAssimpModelLoader->getSceneScaleFactor(),
                                                                           screenWidth, screenHeight, radiusFactor, usePointAntialias, usePerspectiveCorrection,
                                                                           useTexturing && !environmentMappingRenderPass, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, useEnvironmentMapping,
                                                                           useFog, false, 1.0f,
                                                                           useGBuffer,
                                                                           screenWidth, screenHeight, aBuffer3DSize, oitCaptureABuffer3DBufferToFile);
    }
    else if (currentImpostorShader == CYLINDER_VS_GS_FS)
    {
      openGLShaderCylinderVFGSFS->useProgramAndUniforms(NUMBER_OF_LIGHTS, currentLightingModel, useOrenNayarDiffuseModel, USE_COLOR_MATERIAL, openGLAssimpModelLoader->getSceneScaleFactor(),
                                                        screenWidth, screenHeight, radiusFactor, usePointAntialias, usePerspectiveCorrection,
                                                        useTexturing && !environmentMappingRenderPass, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, useSphericalMapping, useEnvironmentMapping,
                                                        useFog, false, 1.0f,
                                                        useGBuffer,
                                                        screenWidth, screenHeight, aBuffer3DSize, oitCaptureABuffer3DBufferToFile);
    }

    if (useColoredObjects)
    {
      if (useTransparency)
      {
        MODEL_COLOR_WITH_ALPHA[3] = opacity;
        glColor4fv(MODEL_COLOR_WITH_ALPHA.data());
      }
      else
      {
        glColor4fv(MODEL_COLOR.data());
      }
    }

    if (wireframe)
    {
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE); // set wireframe
    }

    if ((currentCachedRenderingTest == USE_NO_CACHING_METHOD_1) || (currentCachedRenderingTest == USE_NO_CACHING_METHOD_2))
    {
      renderNoCaching(false);
    }
    else if (currentCachedRenderingTest == USE_VERTEX_ARRAYS)
    {
      renderVertexArrays();
    }
    else if (currentCachedRenderingTest == USE_DISPLAY_LISTS)
    {
      renderDisplayLists();
    }
    else // if (currentCachedRenderingTest == USE_VBOS)
    {
      renderVBOs();
    }

    if (wireframe)
    {
      glPolygonMode(GL_FRONT_AND_BACK, GL_FILL); // reset wireframe state
    }

    if (currentImpostorShader == POINT_SPHERE)
    {
      openGLShaderPointSphereModels->disableShaderProgram();
    }
    else if (currentImpostorShader == QUAD_SPHERE_VS_FS)
    {
      openGLShaderPointSphereModelsQuadVFFS->disableShaderProgram();
    }
    else if (currentImpostorShader == QUAD_SPHERE_VS_GS_FS)
    {
      openGLShaderPointSphereModelsQuadVFGSFS->disableShaderProgram();
    }
    else if (currentImpostorShader == QUAD_SPHERE_VS_TESS_LOD_FS)
    {
      openGLShaderPointSphereModelsQuadVFTESSLODFS->disableShaderProgram();
    }
    else if (currentImpostorShader == QUAD_SPHERE_VS_TESS_QUAD_FS)
    {
      openGLShaderPointSphereModelsQuadVFTESSQuadFS->disableShaderProgram();
    }
    else if (currentImpostorShader == CYLINDER_VS_GS_FS)
    {
      openGLShaderCylinderVFGSFS->disableShaderProgram();
    }
  }
  if (openGLDriverInfo->isAMDATI())
  {
    disableLights(); // make sure lights are disabled for AMD-ATI hardware
  }

  if (useTransparency && (currentOITMode != NO_OIT))
  {
    // ensure that all global memory write are done before resolving
    glMemoryBarrier(barriers);

    if (oitUseDeferredShading && openGLDriverInfo->isAMDATI())
    {
      enableLights(); // make sure lights are enabled for AMD-ATI hardware
    }

    openGLShaderDisplayABuffer3D->enableShaderProgram();
    openGLShaderDisplayABuffer3D->setUniform1i("screenWidth", screenWidth);
    openGLShaderDisplayABuffer3D->setUniform1i("screenHeight", screenHeight);
    openGLShaderDisplayABuffer3D->setUniform1i("sortFragments", oitSortFragments);
    openGLShaderDisplayABuffer3D->setUniform1i("resolveTranslucency", oitResolveTranslucency && !wireframe); // translucency doesn't work correctly with wireframe mode
    openGLShaderDisplayABuffer3D->setUniform1f("zNear", openGLEulerCamera->getViewNear());
    openGLShaderDisplayABuffer3D->setUniform1f("zFar",  openGLEulerCamera->getViewFar());
    openGLShaderDisplayABuffer3D->setUniform1f("translucencyAbsorptionCoefficient", oitTranslucencyAbsorptionCoefficient);
    openGLShaderDisplayABuffer3D->setUniform4fv("backgroundColor", 1, (blackOrWhiteBackground) ? WHITE_BACKGROUND_COLOR.data() : BLACK_BACKGROUND_COLOR.data());
    openGLShaderDisplayABuffer3D->setUniform1i("useGBuffer", useGBuffer);
    if ((currentOITMode == A_BUFFER_3D_USE_TEXTURES) || (currentOITMode == A_BUFFER_3D_USE_BUFFERS))
    {
      openGLShaderDisplayABuffer3D->setUniform1i("aBuffer3DSize", aBuffer3DSize);
      openGLShaderDisplayABuffer3D->setUniform1i("useOwnPackingMethodsForScreenshots", oitCaptureABuffer3DBufferToFile);
      openGLShaderDisplayABuffer3D->setUniform1i("sortGlobalFragmentArrayAndNotRender", false);
    }
    if (oitUseDeferredShading)
    {
      const float tanFieldOfView = tanf(openGLEulerCamera->getFieldOfView() * 0.5f * 4.0f * PI_4 / 180.0f);
      openGLShaderDisplayABuffer3D->setUniform1f("tanFieldOfView", tanFieldOfView);
      openGLShaderDisplayABuffer3D->setUniform1i("numberOfLights", NUMBER_OF_LIGHTS);
      openGLShaderDisplayABuffer3D->setUniform1i("lightingModel", currentLightingModel);
      openGLShaderDisplayABuffer3D->setUniform1i("useOrenNayarDiffuseModel", useOrenNayarDiffuseModel);
      openGLShaderDisplayABuffer3D->setUniform1i("useFresnelFactorSchlickApproximationSpecularModel", false);
      openGLShaderDisplayABuffer3D->setUniform1i("useColorMaterial", USE_COLOR_MATERIAL);
      openGLShaderDisplayABuffer3D->setUniform1f("extraGaussianSpecularFactor", 1.0f);
      openGLShaderDisplayABuffer3D->setUniform1i("texturing", useTexturing);
      if (useTexturing)
      {
        openGLShaderDisplayABuffer3D->setUniform1i("texture", ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
        openGLShaderDisplayABuffer3D->setUniform1i("sphericalMapping", useSphericalMapping);
        openGLShaderDisplayABuffer3D->setUniform1i("texturingMode", GLAuxiliaryFunctions::convertTexEnvModeGLConstantToShaderEnum((useSphericalMapping && false) ? GL_MODULATE : GL_DECAL));
        openGLShaderDisplayABuffer3D->setUniform1i("environmentMapping", false);
      }
      openGLShaderDisplayABuffer3D->setUniform1i("fog", useFog);
      openGLShaderDisplayABuffer3D->setUniform1i("gammaCorrection", false);
    }
    GLAuxiliaryFunctions::createFullScreenQuadWithDummyVAO(openGLDriverInfo->isNvidia(), dummyVao);
    openGLShaderDisplayABuffer3D->disableShaderProgram();

    if (oitUseDeferredShading && openGLDriverInfo->isAMDATI())
    {
      disableLights(); // make sure lights are disabled for AMD-ATI hardware
    }

    if (multisample)
    {
      glEnable(GL_MULTISAMPLE);
    }

    if (oitCaptureABuffer3DBufferToFile)
    {
      writeABuffer3DBufferToFile(barriers);
      oitCaptureABuffer3DBufferToFile = false;
    }
  }

  if (useTexturing && !environmentMappingRenderPass)
  {
    glDisable(GL_TEXTURE_2D);
  }

  if (useTransparency)
  {
    // re-enable depth mask & alpha test for rest of scene
    glDepthMask(GL_TRUE);
    glEnable(GL_ALPHA_TEST);
  }

  if (multisample && (currentGBUfferEffectType != NO_GBUFFER_EFFECT))
  {
    glEnable(GL_MULTISAMPLE);
  }
}

bool ParticleRenderingModelLoaderTest::isQuadVSFS() const
{
  return (currentImpostorShader == QUAD_SPHERE_VS_FS);
}

bool ParticleRenderingModelLoaderTest::isQuadVSTESSFS() const
{
  return (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader() && ((currentImpostorShader == QUAD_SPHERE_VS_TESS_LOD_FS) || (currentImpostorShader == QUAD_SPHERE_VS_TESS_QUAD_FS)));
}

/*
*  Draws a string in 2D space using GLUT.
*  The projection matrix must be set to orthogonal before call this function.
*/
void ParticleRenderingModelLoaderTest::drawString(const char* str, int x, int y, const float color[4], void* font) const
{
  glColor4fv(color);   // set text color
  glRasterPos2i(x, y); // place text position

  // loop all characters in the string
  while (*str)
  {
    glutBitmapCharacter(font, *str);
    ++str;
  }
}

/*
*  Draws a string in 3D space using GLUT.
*/
void ParticleRenderingModelLoaderTest::drawString3D(const char *str, float position[3], const float color[4], void* font) const
{
  glColor4fv(color);        // set text color
  glRasterPos3fv(position); // place text position

  // loop all characters in the string
  while (*str)
  {
    glutBitmapCharacter(font, *str);
    ++str;
  }
}

void ParticleRenderingModelLoaderTest::showInfo()
{
  if (reInitUIInformation)
  {
    // just to clear VRAM memory
    if (glIsList(useUIInformationDisplayList))
    {
      glDeleteLists(useUIInformationDisplayList, 1);
    }
    glNewList(useUIInformationDisplayList, GL_COMPILE);

    glPushAttrib(GL_LIGHTING_BIT | GL_CURRENT_BIT | GL_DEPTH_BUFFER_BIT); // push all relevant attribs
    glDisable(GL_LIGHTING); // need to disable lighting for proper text color
    glDisable(GL_DEPTH_TEST);

    // safety restore-default-texture GL state commands
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, 0);
    glDisable(GL_TEXTURE_2D);

    // backup current model-view matrix
    glPushMatrix();   // save current modelview matrix
    glLoadIdentity(); // reset modelview matrix

    // set to 2D orthogonal projection
    glMatrixMode(GL_PROJECTION);    // switch to projection matrix
    glPushMatrix();  // save current projection matrix
    glLoadIdentity();// reset projection matrix
    gluOrtho2D(0, screenWidth, 0, screenHeight); // set to orthogonal projection

    const GLfloat* color = (blackOrWhiteBackground) ? BLACK_BACKGROUND_COLOR.data() : WHITE_BACKGROUND_COLOR.data();
    ostringstream ss;

    ss << "Total Number of Rendered Faces: " << openGLAssimpModelLoader->getTotalNumberOfFaces() << ends; // add 0(ends) at the end
    drawString(ss.str().c_str(), (screenWidth - 33 * GLUT_TEXT_WIDTH) / 2, screenHeight - GLUT_TEXT_HEIGHT, color, getFont());
    ss.str(""); // clear buffer

    ss << "Total Number of Rendered Particles:  " << (vertices.size() >> 2) << ends;
    drawString(ss.str().c_str(), (screenWidth - 33 * GLUT_TEXT_WIDTH) / 2, screenHeight - (2 * GLUT_TEXT_HEIGHT), color, getFont());
    ss.str(""); // clear buffer

    ss << "Current Video Memory Usage:   " << howMuchCurrentlyAvailableVRAMMemoryString() << ends;
    drawString(ss.str().c_str(), (screenWidth - 33 * GLUT_TEXT_WIDTH) / 2, screenHeight - (3 * GLUT_TEXT_HEIGHT), color, getFont());
    ss.str(""); // clear buffer

    ss << "Memory difference from last Test Scenario: " << abs(previousAvailableMemory - howMuchCurrentlyAvailableVRAMMemory()) << ends;
    drawString(ss.str().c_str(), (screenWidth - 33 * GLUT_TEXT_WIDTH) / 2, screenHeight - (4 * GLUT_TEXT_HEIGHT), color, getFont());
    ss.str(""); // clear buffer

    if (currentOITMode != NO_OIT)
    {
      const float aBuffer3DVRAMUsage = ((currentOITMode == A_BUFFER_3D_USE_TEXTURES) || (currentOITMode == A_BUFFER_3D_USE_BUFFERS)) ? (screenWidth * screenHeight * aBuffer3DSize * sizeof(GLfloat) * ((oitUseDeferredShading && (currentOITMode == A_BUFFER_3D_USE_BUFFERS)) ? DEFERRED_SHADING_BUFFER_ELEMENTS : NORMAL_SHADING_BUFFER_ELEMENTS)) / float(1 << 20) : (aBuffer3DLinkedListSize * sizeof(GLuint) * ((oitUseDeferredShading && (currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_BUFFERS)) ? DEFERRED_SHADING_LINKED_LIST_BUFFER_ELEMENTS : NORMAL_SHADING_LINKED_LIST_BUFFER_ELEMENTS)) / float(1 << 20);
      ss << fixed << setprecision(1) << "A-Buffer3D Video Memory Usage: " << aBuffer3DVRAMUsage << " MB" << ends;
      drawString(ss.str().c_str(), (screenWidth - 33 * GLUT_TEXT_WIDTH) / 2, screenHeight - (5 * GLUT_TEXT_HEIGHT), color, getFont());
      ss.str(""); // clear buffer
    }

    ss << "VSynch: " << (vsynch ? "on" : "off") << ends;
    drawString(ss.str().c_str(), screenWidth - 16 * GLUT_TEXT_WIDTH, 2 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "AutoRotate: " << (autoRotate ? "on" : "off") << ends;
    drawString(ss.str().c_str(), screenWidth - 16 * GLUT_TEXT_WIDTH, GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press F1-F2-F3-F4-F5 for Testing Scenarios" << ends;
    drawString(ss.str().c_str(), 1, screenHeight - GLUT_TEXT_HEIGHT, color, getFont());
    ss.str(""); // clear buffer

    ss << "CurrentVBOTest: " << ((currentCachedRenderingTest == USE_NO_CACHING_METHOD_1) ? "USE NO CACHING METHOD 1" : ((currentCachedRenderingTest == USE_NO_CACHING_METHOD_2) ? "USE NO CACHING METHOD 2" : ((currentCachedRenderingTest == USE_VERTEX_ARRAYS) ? "USE VERTEX ARRAYS" : ((currentCachedRenderingTest == USE_DISPLAY_LISTS)) ? "USE DISPLAY LISTS" : "USE VBOS"))) << ends;
    drawString(ss.str().c_str(), 1, screenHeight - ( 2 * GLUT_TEXT_HEIGHT), color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 1-2-3-4-5-6 for Testing Lighting Models" << ends;
    drawString(ss.str().c_str(), 1, screenHeight - ( 3 * GLUT_TEXT_HEIGHT) - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "CurrentLightingModel: " << ((currentLightingModel == SIMPLE_LIGHTING) ? "SIMPLE LIGHTING" : ((currentLightingModel == PHONG) ? "PHONG" : ((currentLightingModel == BLINN_PHONG) ? "BLINN_PHONG" : ((currentLightingModel == GAUSSIAN) ? "GAUSSIAN" : ((currentLightingModel == TOON)) ? "TOON" : "GOOCH")))) << ends;
    drawString(ss.str().c_str(), 1, screenHeight - ( 4 * GLUT_TEXT_HEIGHT) - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'l' key for " << ((usePositionalLights) ? "Positional Lights" : "Directional Lights") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - ( 5 * GLUT_TEXT_HEIGHT) - 10, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'd' key to toggle the Oren Nayar Model " << ((useOrenNayarDiffuseModel) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - ( 6 * GLUT_TEXT_HEIGHT) - 10, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press F6 for Scene Motion Blur" << ends;
    drawString(ss.str().c_str(), 1, screenHeight - ( 7 * GLUT_TEXT_HEIGHT) - 15, color, getFont());
    ss.str(""); // clear buffer

    ss << "Motion Blur: " << (useMotionBlurForScene ? "ACCUM BUFFER" : "NO USE") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - ( 8 * GLUT_TEXT_HEIGHT) - 15, color, getFont());
    ss.str(""); // clear buffer

    if (useMotionBlurForScene)
    {
      ss << "Motion Blur Size: " << motionBlurSize << ends;
    }
    else
    {
      ss << "Motion Blur Size: " << "-" << ends;
    }
    drawString(ss.str().c_str(), 1, screenHeight - ( 9 * GLUT_TEXT_HEIGHT) - 15, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press F7-to-F12 for Testing Particle Rendering Performance" << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (10 * GLUT_TEXT_HEIGHT) - 20, color, getFont());
    ss.str(""); // clear buffer

    ss << "CurrentImpostorShader: " << ((currentImpostorShader == POINT_SPHERE) ? "POINT SPHERE" : ((currentImpostorShader == QUAD_SPHERE_VS_FS) ? "QUAD SPHERE VS FS" : ((currentImpostorShader == QUAD_SPHERE_VS_GS_FS) ? "QUAD SPHERE VS GS FS" : ((currentImpostorShader == QUAD_SPHERE_VS_TESS_LOD_FS) ? "QUAD SPHERE VS TESS LOD FS" : ((currentImpostorShader == QUAD_SPHERE_VS_TESS_QUAD_FS) ? "QUAD SPHERE VS TESS QUAD FS" : ((currentImpostorShader == QUAD_SPHERE_VS_TESS_QUAD_FS) ? "CYLINDER VS GS FS" : "NONE")))))) << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (11 * GLUT_TEXT_HEIGHT) - 20, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 't' key to toggle Texturing " << ((useTexturing) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (12 * GLUT_TEXT_HEIGHT) - 25, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'h' key to toggle Spherical Mapping " << ((useSphericalMapping) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (13 * GLUT_TEXT_HEIGHT) - 25, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'j' key to toggle Jitter Environment Mapping " << ((useEnvironmentMapping) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (14 * GLUT_TEXT_HEIGHT) - 25, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press '[' key to lower  Motion Blur Size " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (15 * GLUT_TEXT_HEIGHT) - 25, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press ']' key to refine Motion Blur Size " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (16 * GLUT_TEXT_HEIGHT) - 25, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'n' key to lower Sphere Radius Factor " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (17 * GLUT_TEXT_HEIGHT) - 25, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'm' key to higher Sphere Radius Factor " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (18 * GLUT_TEXT_HEIGHT) - 25, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'ctrl'  key to toggle Point Antialias " << ((usePointAntialias) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (19 * GLUT_TEXT_HEIGHT) - 25, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'space' key to toggle Perspective Correction " << ((usePerspectiveCorrection) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (20 * GLUT_TEXT_HEIGHT) - 25, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'alt' key to toggle GBuffer Effects: " << ((currentGBUfferEffectType == DEPTH_OF_FIELD) ? "DEPTH-OF-FIELD" : (currentGBUfferEffectType == EDGE_ENHANCEMENT) ? "EDGE ENHANCEMENT" : (currentGBUfferEffectType == HSSAO) ? "HSSAO" : (currentGBUfferEffectType == HSSAO_PLUS) ? "HSSAO PLUS" : "NONE") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (21 * GLUT_TEXT_HEIGHT) - 35, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'Page Up'   key to refine BlurXY Sample Range " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (22 * GLUT_TEXT_HEIGHT) - 35, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'Page Down' key to lower  BlurXY Sample Range " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (23 * GLUT_TEXT_HEIGHT) - 35, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'Up'     key to refine Depth-Of-Field Range " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (24 * GLUT_TEXT_HEIGHT) - 35, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'Down'   key to lower  Depth-Of-Field Range " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (25 * GLUT_TEXT_HEIGHT) - 35, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'Right' key to refine Depth-Of-Field Z Focus " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (26 * GLUT_TEXT_HEIGHT) - 35, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'Left'  key to lower  Depth-Of-Field Z Focus " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (27 * GLUT_TEXT_HEIGHT) - 35, color, getFont());
    ss.str(""); // clear buffer

    if (currentImpostorShader == CYLINDER_VS_GS_FS)
    {
      ss << "Press 'i' key to toggle Impostor Wireframe: " << ((currentImpostorWireframe == FULL_IMPOSTOR_WIREFRAME) ? "FULL_IMPOSTOR_WIREFRAME" : (currentImpostorWireframe == IMPOSTOR_POINTS_ONLY) ? "IMPOSTOR_POINTS_ONLY" : (currentImpostorWireframe == IMPOSTOR_LINES_ONLY) ? "IMPOSTOR_LINES_ONLY" : "CYLINDER_EXPLOSION") << ends;
      drawString(ss.str().c_str(), 1, screenHeight - (28 * GLUT_TEXT_HEIGHT) - 40, color, getFont());
      ss.str(""); // clear buffer
    }

    ss << "Press '+'  key to toggle Transparency " << ((useTransparency) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (29 * GLUT_TEXT_HEIGHT) - 45, color, getFont());
    ss.str(""); // clear buffer

    if (useTransparency)
    {
      ss << "Press 'Home'    key to refine Opacity Level" << ends;
      drawString(ss.str().c_str(), 1, screenHeight - (30 * GLUT_TEXT_HEIGHT) - 45, color, getFont());
      ss.str(""); // clear buffer

      ss << "Press 'End'     key to lower  Opacity Level " << ends;
      drawString(ss.str().c_str(), 1, screenHeight - (31 * GLUT_TEXT_HEIGHT) - 45, color, getFont());
      ss.str(""); // clear buffer

      ss << "Press '7-0'  key to toggle OIT Rendering Modes " << ends;
      drawString(ss.str().c_str(), 1, screenHeight - (32 * GLUT_TEXT_HEIGHT) - 45, color, getFont());
      ss.str(""); // clear buffer

      ss << "CurrentOITMode: " << ((currentOITMode == NO_OIT) ? "NO OIT" : ((currentOITMode == A_BUFFER_3D_USE_TEXTURES) ? "A-BUFFER-3D TEXTURES: " : ((currentOITMode == A_BUFFER_3D_USE_BUFFERS) ? "A-BUFFER-3D BUFFERS: " : ((currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_TEXTURES) ? "A-BUFFER-3D LINKED LIST TEXTURES: " : "A-BUFFER-3D LINKED LIST BUFFERS: ")))) << ((currentOITMode == NO_OIT) ? "" : (!oitSortFragments ? "Opaque Mode" : (!oitResolveTranslucency ? "Transparent Mode" : "Translucent Mode")))<< ends;
      drawString(ss.str().c_str(), 1, screenHeight - (33 * GLUT_TEXT_HEIGHT) - 45, color, getFont());
      ss.str(""); // clear buffer

      if (((currentOITMode == A_BUFFER_3D_USE_BUFFERS) || (currentOITMode == A_BUFFER_3D_USE_LINKED_LIST_BUFFERS))&& !oitUseDeferredShading)
      {
        ss << "Press '-' key to toggle OIT Rendering Method " << ends;
        drawString(ss.str().c_str(), 1, screenHeight - (34 * GLUT_TEXT_HEIGHT) - 45, color, getFont());
        ss.str(""); // clear buffer

        if (currentOITMode == A_BUFFER_3D_USE_BUFFERS)
        {
          ss << "Press 'k' key to Capture the current A-Buffer-3D to file " << ends;
          drawString(ss.str().c_str(), 1, screenHeight - (35 * GLUT_TEXT_HEIGHT) - 45, color, getFont());
          ss.str(""); // clear buffer
        }
      }
    }

    ss << "Press 'f' key to toggle Fog " << ((useFog) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, 12 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'q' key to toggle BackFace Culling " << ((useBackFaceCulling) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, 11 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'c' key to toggle Colored Objects " << ((useColoredObjects) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, 10 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 's' key to toggle Specular Lighting " << ((useSpecularLighting) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1,  9 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'b' key to toggle " << ((blackOrWhiteBackground) ? "Black" : "White") << " background" << ends;
    drawString(ss.str().c_str(), 1,  8 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'w' key to toggle Wireframe " << ((wireframe) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1,  7 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'a' key to toggle AutoRotate " << ((autoRotate) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1,  6 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'r' key to reset  the camera" << ends;
    drawString(ss.str().c_str(), 1,  5 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'v' key to toggle VSynch " << ((vsynch) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1,  4 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'u' key to toggle UI Information " << ((useUIInformation) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1,  3 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'Insert' key to toggle FXAA Antialias " << ((useFXAA_Antialias) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1,  2 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'Shift'  key to Take Screenshot " << ends;
    drawString(ss.str().c_str(), 1, GLUT_TEXT_HEIGHT - 5, color, getFont());

    // unset floating format
    ss << resetiosflags(ios_base::fixed | ios_base::floatfield);

    // restore projection matrix
    glPopMatrix(); // restore to previous projection matrix

    // restore modelview matrix
    glMatrixMode(GL_MODELVIEW); // switch to modelview matrix
    glPopMatrix();              // restore to previous modelview matrix

    glPopAttrib(); // pop all relevant attribs

    glEndList();

    reInitUIInformation = false;
  }

  glCallList(useUIInformationDisplayList);
}

int ParticleRenderingModelLoaderTest::howMuchCurrentlyAvailableVRAMMemory() const
{
  openGLDriverInfo->getGLMemoryInfo();
  return (openGLDriverInfo->isNvidia()) ? (openGLDriverInfo->getGPUMemoryInfoCurrentAvailableMemoryNVX() >> 10) : ((openGLDriverInfo->isAMDATI()) ? (openGLDriverInfo->getVBOFreeMemoryATI()[1] >> 10) : 0);
}

string ParticleRenderingModelLoaderTest::howMuchCurrentlyAvailableVRAMMemoryString() const
{
  const int howMuchAvailableVRAMMemoryInt = howMuchCurrentlyAvailableVRAMMemory();
  return (howMuchAvailableVRAMMemoryInt) ? StringAuxiliaryFunctions::toString<int>(howMuchAvailableVRAMMemoryInt) + " Mb VRAM" : "No information available for this GL vendor";
}

void ParticleRenderingModelLoaderTest::showFPS()
{
  // update fpsString every second
  const double elapsedTime = timer.getElapsedTimeInSecs();
  if (elapsedTime < 1.0)
  {
    ++fpsCounter;
  }
  else
  {
    ++fpsCounter;

    ostringstream ss;
    ss << fixed << setprecision(1);
    // ss << ((double)fpsCounter / elapsedTime) << " FPS" << ends; // update fpsString string
    ss << double(fpsCounter) << " FPS" << ends; // update fpsString string
    ss << resetiosflags(ios_base::fixed | ios_base::floatfield);
    fpsString = ss.str();
    fpsCounter = 0; // reset counter
    timer.startTimer();  // restart timer

    DebugConsole_consoleOutLine("Current framerate: ", fpsString.c_str());
  }

  // safety restore-default-texture GL state commands
  glActiveTexture(GL_TEXTURE0);
  glBindTexture(GL_TEXTURE_2D, 0);
  glDisable(GL_TEXTURE_2D);

  glPushAttrib(GL_LIGHTING_BIT | GL_CURRENT_BIT | GL_DEPTH_BUFFER_BIT); // push all relevant attribs
  glDisable(GL_LIGHTING); // need to disable lighting for proper text color
  glDisable(GL_TEXTURE_2D);
  glDisable(GL_DEPTH_TEST);

  // backup current model-view matrix
  glPushMatrix();   // save current modelview matrix
  glLoadIdentity(); // reset modelview matrix

  // set to 2D orthogonal projection
  glMatrixMode(GL_PROJECTION); // switch to projection matrix
  glPushMatrix();   // save current projection matrix
  glLoadIdentity(); // reset projection matrix
  gluOrtho2D(0, screenWidth, 0, screenHeight); // set to orthogonal projection

  const int textWidth  = int(fpsString.size() * GLUT_TEXT_WIDTH);
  const GLfloat* color = (blackOrWhiteBackground) ? BLACK_BACKGROUND_COLOR.data() : WHITE_BACKGROUND_COLOR.data();

  drawString(fpsString.c_str(), screenWidth - textWidth, screenHeight - GLUT_TEXT_HEIGHT, color, getFont());

  // restore projection matrix
  glPopMatrix();      // restore to previous projection matrix

  // restore modelView matrix
  glMatrixMode(GL_MODELVIEW); // switch to modelview matrix
  glPopMatrix();              // restore to previous modelview matrix

  glPopAttrib(); // pop all relevant attribs
}

void ParticleRenderingModelLoaderTest::renderScene()
{
  if (autoRotate)
  {
    cameraAngleX += 1.0;
    cameraAngleY += 1.0;
  }

  if (++printGLMemoryInfoOnSecondUpdate == 2)
  {
    openGLDriverInfo->getGLMemoryInfo();
    openGLDriverInfo->printGLMemoryInfo();
    if (openGLDriverInfo->supports430Shaders())
    {
      autoManageABuffer3D(); // do first-time update on A-Buffer3D
    }
  }

  GLAuxiliaryFunctions::setVSynch(vsynch, openGLDriverInfo->isVSynchSupported());

  if (openGLDriverInfo->supports430Shaders() && reInitOITShaders)
  {
    preparePointSphereShaders();
    prepareABuffer3DShaders();
    deleteABuffer3D();
    initABuffer3D();
    reInitOITShaders = false;
  }

  if (openGLDriverInfo->supports_GL_EXT_framebuffer_object() && useEnvironmentMapping)
  {
    drawEnvironmentMappingFBO();
  }
  if (openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders() &&
      (currentGBUfferEffectType != NO_GBUFFER_EFFECT))
  {
    processGBufferEffectsFBOs();
  }

  if (openGLDriverInfo->supports_GL_EXT_framebuffer_object() && useFXAA_Antialias)
  {
    drawFXAA_AntialiasFBO();
  }
  else
  {
    if (openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders() &&
        (currentGBUfferEffectType != NO_GBUFFER_EFFECT))
    {
      drawGBufferEffectsFBOs();
    }
    else
    {
      renderParticleScene(false);
    }
  }

  // draw info messages
  if (useUIInformation)
  {
    showInfo();
  }
  showFPS();

  // Take the contents of the current draw buffer and copy it to the accumulation buffer with each pixel modified by a factor
  // The closer the factor is to 1.0f, the longer the trails... Don't exceed 1.0f - you get garbage.
  if (!openGLDriverInfo->isAMDATI()) // recent AMD/ATI cards lack an accumulation buffer
  {
    if (useMotionBlurForScene)
    {
      // Clear the accumulation buffer (we re-grab the screen into the accumulation buffer after drawing our current frame!)
      if (reInitMotionBlurForScene)
      {
        if (blackOrWhiteBackground)
        {
          glClearAccum(WHITE_BACKGROUND_COLOR[0], WHITE_BACKGROUND_COLOR[1], WHITE_BACKGROUND_COLOR[2], WHITE_BACKGROUND_COLOR[3]);
        }
        else
        {
          glClearAccum(BLACK_BACKGROUND_COLOR[0], BLACK_BACKGROUND_COLOR[1], BLACK_BACKGROUND_COLOR[2], BLACK_BACKGROUND_COLOR[3]);
        }
        glClear(GL_ACCUM_BUFFER_BIT);
        glAccum(GL_LOAD, 1.0f);
        reInitMotionBlurForScene = false;
      }

      glAccum(GL_MULT, motionBlurSize);
      glAccum(GL_ACCUM, 1.0f - motionBlurSize);
      glAccum(GL_RETURN, 1.0f);
    }
  }

  if (takeScreenshot)
  {
    writeScreenshotToFile();
    takeScreenshot = false;
  }

  isNavigating = false;

  glFlush();
  glFinish();
  glutSwapBuffers();

  const GLenum glError = glGetError();
  if (glError != GL_NO_ERROR)
  {
    const GLubyte* gluError = gluErrorString(glError);
    DebugConsole_consoleOutLine("renderScene() glGetError() returned: ", gluError);
  }
}

void ParticleRenderingModelLoaderTest::changeSize(int w, int h)
{
  if (h == 0) // Prevent divide by zero
  {
    h = 1;
  }

  const GLfloat ratio = 1.0f * GLfloat(w) / GLfloat(h);
  screenWidth  = w;
  screenHeight = h;
  DebugConsole_consoleOutLine("New screenWidth: ",  screenWidth);
  DebugConsole_consoleOutLine("New screenHeight: ", screenHeight, "\n");

  glViewport(0, 0, screenWidth, screenHeight);

  GLAuxiliaryFunctions::setVSynch(vsynch, openGLDriverInfo->isVSynchSupported());

  openGLEulerCamera->setRatio(ratio);
  openGLEulerCamera->setMatrices();

  if (openGLDriverInfo->supports430Shaders())
  {
    initABuffer3D();
  }
  if (openGLDriverInfo->supports_GL_EXT_framebuffer_object())
  {
    initEnviromentMappingFBO();
    if (openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders())
    {
      initGBufferEffectsFBOs();
    }
    initFXAA_AntialiasFBO();
  }

  if (!openGLDriverInfo->isAMDATI()) // recent AMD/ATI cards lack an accumulation buffer
  {
    // Clear the accumulation buffer (we re-grab the screen into the accumulation buffer after drawing our current frame!)
    if (blackOrWhiteBackground)
    {
      glClearAccum(WHITE_BACKGROUND_COLOR[0], WHITE_BACKGROUND_COLOR[1], WHITE_BACKGROUND_COLOR[2], WHITE_BACKGROUND_COLOR[3]);
    }
    else
    {
      glClearAccum(BLACK_BACKGROUND_COLOR[0], BLACK_BACKGROUND_COLOR[1], BLACK_BACKGROUND_COLOR[2], BLACK_BACKGROUND_COLOR[3]);
    }
    glClear(GL_ACCUM_BUFFER_BIT);
    glAccum(GL_LOAD, 1.0f);
  }

  glFlush();
  glFinish();

  const GLenum glError = glGetError();
  if (glError != GL_NO_ERROR)
  {
    const GLubyte* gluError = gluErrorString(glError);
    DebugConsole_consoleOutLine("changeSize() glGetError() returned: ", gluError);
  }
}

void ParticleRenderingModelLoaderTest::keyboard(unsigned char key, int x, int y)
{
  if ((key == char('v')) || (key == char('V')))
  {
    vsynch++;
    vsynch = vsynch & 1;
    DebugConsole_consoleOutLine("Toggle VSynch: ", vsynch ? "on" : "off");
  }
  else if ((key == char('a')) || (key == char('A')))
  {
    autoRotate++;
    autoRotate = autoRotate & 1;
    DebugConsole_consoleOutLine("Toggle AutoRotate: ", autoRotate ? "on" : "off");
  }
  else if ((key == char('r')) || (key == char('R')))
  {
    cameraAngleX = cameraAngleY = cameraDistanceX = cameraDistanceY = 0.0;
    cameraDistanceZ = 15.0;
    DebugConsole_consoleOutLine("Camera reset.");
  }
  else if ((key == char('b')) || (key == char('B')))
  {
    blackOrWhiteBackground = !blackOrWhiteBackground;
    DebugConsole_consoleOutLine("Toggle BlackOrWhiteBackground: ", blackOrWhiteBackground ? "on" : "off");
  }
  else if ((key == char('w')) || (key == char('W')))
  {
    wireframe++;
    wireframe = wireframe & 1;
    DebugConsole_consoleOutLine("Toggle Wireframe: ", wireframe ? "on" : "off");
  }
  else if ((key == char('c')) || (key == char('C')))
  {
    reInitModelDataForRendering = true;
    reInitModelCylinderDataForRendering = true;
    useColoredObjects++;
    useColoredObjects = useColoredObjects & 1;
    DebugConsole_consoleOutLine("Toggle Colored Objects: ", useColoredObjects ? "on" : "off");
  }
  else if ((key == char('s')) || (key == char('S')))
  {
    useSpecularLighting++;
    useSpecularLighting = useSpecularLighting & 1;
    DebugConsole_consoleOutLine("Toggle Specular Lighting: ", useSpecularLighting ? "on" : "off");
  }
  else if ((key == char('q')) || (key == char('Q')))
  {
    useBackFaceCulling++;
    useBackFaceCulling = useBackFaceCulling & 1;
    DebugConsole_consoleOutLine("Toggle Back Face Culling: ", useBackFaceCulling ? "on" : "off");
  }
  else if ((key == char('u')) || (key == char('U')))
  {
    useUIInformation++;
    useUIInformation = useUIInformation & 1;
    DebugConsole_consoleOutLine("Toggle UI Information: ", useUIInformation ? "on" : "off");
  }
  else if (key == char('1'))
  {
    currentLightingModel = SIMPLE_LIGHTING;
    DebugConsole_consoleOutLine("currentLightingModel: SIMPLE LIGHTING");
  }
  else if (key == char('2'))
  {
    currentLightingModel = PHONG;
    DebugConsole_consoleOutLine("currentLightingModel: PHONG");
  }
  else if (key == char('3'))
  {
    currentLightingModel = BLINN_PHONG;
    DebugConsole_consoleOutLine("currentLightingModel: BLINN_PHONG");
  }
  else if (key == char('4'))
  {
    currentLightingModel = GAUSSIAN;
    DebugConsole_consoleOutLine("currentLightingModel: GAUSSIAN");
  }
  else if (key == char('5'))
  {
    currentLightingModel = TOON;
    DebugConsole_consoleOutLine("currentLightingModel: TOON");
  }
  else if (key == char('6'))
  {
    currentLightingModel = GOOCH;
    DebugConsole_consoleOutLine("currentLightingModel: GOOCH");
  }
  else if (key == char('+'))
  {
    useTransparency = !useTransparency;
    reInitColorArrays = true;
    reInitModelDataForRendering = true;
    reInitModelCylinderDataForRendering = true;
    reInitOITShaders = true;
    DebugConsole_consoleOutLine("useTransparency: ", useTransparency);
  }
  else if (key == char('7'))
  {
    if (useTransparency)
    {
      currentOITMode = NO_OIT;
      oitSortFragments = false;
      oitResolveTranslucency = false;
      reInitOITShaders = true;
      DebugConsole_consoleOutLine("current transparency model: NO_OIT");
    }
  }
  else if (key == char('8'))
  {
    if (useTransparency && openGLDriverInfo->supports430Shaders())
    {
      currentOITMode = A_BUFFER_3D_USE_LINKED_LIST_BUFFERS;
      oitSortFragments = false;
      oitResolveTranslucency = false;
      reInitOITShaders = true;
      DebugConsole_consoleOutLine("current transparency model: A_BUFFER_3D_USE_LINKED_LIST_BUFFERS with no fragments sorting");
    }
  }
  else if (key == char('9'))
  {
    if (useTransparency && openGLDriverInfo->supports430Shaders())
    {
      currentOITMode = A_BUFFER_3D_USE_LINKED_LIST_BUFFERS;
      oitSortFragments = true;
      oitResolveTranslucency = false;
      reInitOITShaders = true;
      DebugConsole_consoleOutLine("current transparency model: A_BUFFER_3D_USE_LINKED_LIST_BUFFERS with fragments sorting");
    }
  }
  else if (key == char('0'))
  {
    if (useTransparency && openGLDriverInfo->supports430Shaders())
    {
      currentOITMode = A_BUFFER_3D_USE_LINKED_LIST_BUFFERS;
      oitSortFragments = true;
      oitResolveTranslucency = true;
      reInitOITShaders = true;
      DebugConsole_consoleOutLine("current transparency model: A_BUFFER_3D_USE_LINKED_LIST_BUFFERS with translucency fragments sorting");
    }
  }
  else if (key == char('-'))
  {
    if (useTransparency && openGLDriverInfo->supports430Shaders() && (currentOITMode != NO_OIT))
    {
      if (currentOITMode == A_BUFFER_3D_USE_BUFFERS)
      {
        currentOITMode = A_BUFFER_3D_USE_LINKED_LIST_BUFFERS;
      }
      else
      {
        currentOITMode = A_BUFFER_3D_USE_BUFFERS;
      }
      reInitOITShaders = true;
    }
  }
  else if ((key == char('k')) || (key == char('K')))
  {
    if (useTransparency && openGLDriverInfo->supports430Shaders())
    {
      if ((currentOITMode == A_BUFFER_3D_USE_BUFFERS) && !oitUseDeferredShading)
      {
        oitCaptureABuffer3DBufferToFile = true;
        DebugConsole_consoleOutLine("oitCaptureABuffer3DBufferToFile: ", oitCaptureABuffer3DBufferToFile ? "yes" : "no");
      }
    }
  }
  else if ((key == char('l')) || (key == char('L')))
  {
    usePositionalLights++;
    usePositionalLights = usePositionalLights & 1;
    DebugConsole_consoleOutLine("Toggle Lights: ", usePositionalLights ? "Positional Lights" : "Directional Lights");
  }
  else if ((key == char('d')) || (key == char('D')))
  {
    useOrenNayarDiffuseModel++;
    useOrenNayarDiffuseModel = useOrenNayarDiffuseModel & 1;
    DebugConsole_consoleOutLine("Toggle Oren Nayar Diffure Model: ", useOrenNayarDiffuseModel ? "on" : "off");
  }
  else if ((key == char('f')) || (key == char('F')))
  {
    useFog++;
    useFog = useFog & 1;
    DebugConsole_consoleOutLine("Toggle Fog: ", useFog ? "on" : "off");
  }
  else if ((key == char('t')) || (key == char('T')))
  {
    if (useSphericalMapping)
    {
      useTexturing = useSphericalMapping = useEnvironmentMapping = false;
    }
    useTexturing++;
    useTexturing = useTexturing & 1;
    if (!useTexturing)
    {
      useSphericalMapping = useEnvironmentMapping = false;
    }
    DebugConsole_consoleOutLine("Toggle Texturing: ", useTexturing ? "on" : "off");
  }
  else if ((key == char('h')) || (key == char('H')))
  {
    if (useEnvironmentMapping)
    {
      useTexturing = useSphericalMapping = useEnvironmentMapping = false;
    }
    useSphericalMapping++;
    useTexturing = useSphericalMapping = useSphericalMapping & 1;
    if (useSphericalMapping)
    {
      useEnvironmentMapping = false;
    }
    DebugConsole_consoleOutLine("Toggle Spherical Mapping: ", useSphericalMapping ? "on" : "off");
  }
  else if ((key == char('j')) || (key == char('J')))
  {
    if (openGLDriverInfo->supports_GL_EXT_framebuffer_object())
    {
      useEnvironmentMapping++;
      useTexturing = useSphericalMapping = useEnvironmentMapping = useEnvironmentMapping & 1;
      DebugConsole_consoleOutLine("Toggle Jitter Environment Mapping: ", useEnvironmentMapping ? "on" : "off");
    }
  }
  else if (key == char('['))
  {
    if (!openGLDriverInfo->isAMDATI()) // recent AMD/ATI cards lack an accumulation buffer
    {
      if (motionBlurSize >= 0.10f)
      {
        motionBlurSize -= 0.05f;
      }
      else
      {
        motionBlurSize = 0.05f;
      }
      DebugConsole_consoleOutLine("motionBlurSize: ", motionBlurSize);
    }
  }
  else if (key == char(']'))
  {
    if (!openGLDriverInfo->isAMDATI()) // recent AMD/ATI cards lack an accumulation buffer
    {
      if (motionBlurSize <= 0.90f)
      {
        motionBlurSize += 0.05f;
      }
      else
      {
        motionBlurSize = 0.95f;
      }
      DebugConsole_consoleOutLine("motionBlurSize: ", motionBlurSize);
    }
  }
  else if ((key == char('n')) || (key == char('N')))
  {
    if (radiusFactor > 1.0f)
    {
      radiusFactor -= 1.0f;
    }
    DebugConsole_consoleOutLine("radiusFactor: ", radiusFactor);
  }
  else if ((key == char('m')) || (key == char('M')))
  {
    if (radiusFactor < 32.0f)
    {
      radiusFactor += 1.0f;
    }
    DebugConsole_consoleOutLine("radiusFactor: ", radiusFactor);
  }
  else if ((key == char('i')) || (key == char('I')))
  {
    currentImpostorWireframe = ImpostorWireframeUsage((currentImpostorWireframe + 1) & 3);
    reInitModelDataForRendering = true;
    reInitModelCylinderDataForRendering = true;
    DebugConsole_consoleOutLine("Toggle Impostor Wireframe: ", (currentImpostorWireframe == FULL_IMPOSTOR_WIREFRAME) ? "FULL_IMPOSTOR_WIREFRAME" : (currentImpostorWireframe == IMPOSTOR_POINTS_ONLY) ? "IMPOSTOR_POINTS_ONLY" : (currentImpostorWireframe == IMPOSTOR_LINES_ONLY) ? "IMPOSTOR_LINES_ONLY" : "CYLINDER_EXPLOSION");
  }
  else if (key == char(' '))
  {
    usePerspectiveCorrection++;
    usePerspectiveCorrection = usePerspectiveCorrection & 1;
    DebugConsole_consoleOutLine("Toggle Perspective Correction: ", usePerspectiveCorrection ? "on" : "off");
  }
  else if (key == 27)
  {
    DebugConsole_consoleOutLine("\nEscape pressed, now exiting Test: Particle Rendering Model Loader Tests.\n");
    glutLeaveMainLoop();
  }

  reInitUIInformation = true;
  isNavigating = true;
}

void ParticleRenderingModelLoaderTest::specialKeysKeyboard(int key, int x, int y)
{
  if (glutGetModifiers() == GLUT_ACTIVE_SHIFT)
  {
    takeScreenshot = true;
    DebugConsole_consoleOutLine("takeScreenshot: ", takeScreenshot ? "yes" : "no");
  }
  else if (glutGetModifiers() == GLUT_ACTIVE_CTRL)
  {
    usePointAntialias++;
    usePointAntialias = usePointAntialias & 1;
    DebugConsole_consoleOutLine("Toggle Point Antialias: ", usePointAntialias ? "on" : "off");
  }
  else if (glutGetModifiers() == GLUT_ACTIVE_ALT)
  {
    if (openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders())
    {
      currentGBUfferEffectType = GBUfferEffectTypes((currentGBUfferEffectType + 1) % 5);
      DebugConsole_consoleOutLine("Toggle GBuffer Effects: ", (currentGBUfferEffectType == DEPTH_OF_FIELD) ? "DEPTH-OF-FIELD" : (currentGBUfferEffectType == EDGE_ENHANCEMENT) ? "EDGE ENHANCEMENT" : (currentGBUfferEffectType == HSSAO) ? "HSSAO" : (currentGBUfferEffectType == HSSAO_PLUS) ? "HSSAO PLUS" : "NONE");
    }
  }
  else if (key == GLUT_KEY_UP)
  {
    if ((currentGBUfferEffectType == DEPTH_OF_FIELD) && openGLDriverInfo->supports_GL_EXT_framebuffer_object())
    {
      if (depthOfFieldRange <= 99.0f)
      {
        depthOfFieldRange += 1.0f;
      }
      else
      {
        depthOfFieldRange = 100.0f;
      }
      DebugConsole_consoleOutLine("depthOfFieldRange: ", depthOfFieldRange);
    }
  }
  else if (key == GLUT_KEY_DOWN)
  {
    if ((currentGBUfferEffectType == DEPTH_OF_FIELD) && openGLDriverInfo->supports_GL_EXT_framebuffer_object())
    {
      if (depthOfFieldRange >= 2.0f)
      {
        depthOfFieldRange -= 1.0f;
      }
      else
      {
        depthOfFieldRange = 1.0f;
      }
      DebugConsole_consoleOutLine("depthOfFieldRange: ", depthOfFieldRange);
    }
  }
  else if (key == GLUT_KEY_RIGHT)
  {
    if ((currentGBUfferEffectType == DEPTH_OF_FIELD) && openGLDriverInfo->supports_GL_EXT_framebuffer_object())
    {
      if (depthOfFieldZFocus <= 99.9f)
      {
        depthOfFieldZFocus += 0.1f;
      }
      else
      {
        depthOfFieldZFocus = 100.0f;
      }
      DebugConsole_consoleOutLine("depthOfFieldZFocus: ", depthOfFieldZFocus);
    }
  }
  else if (key == GLUT_KEY_LEFT)
  {
    if ((currentGBUfferEffectType == DEPTH_OF_FIELD) && openGLDriverInfo->supports_GL_EXT_framebuffer_object())
    {
      if (depthOfFieldZFocus >= 0.1f)
      {
        depthOfFieldZFocus -= 0.1f;
      }
      else
      {
        depthOfFieldZFocus = 0.0f;
      }
      DebugConsole_consoleOutLine("depthOfFieldZFocus: ", depthOfFieldZFocus);
    }
  }
  else if (key == GLUT_KEY_PAGE_UP)
  {
    if ((currentGBUfferEffectType == DEPTH_OF_FIELD) && openGLDriverInfo->supports_GL_EXT_framebuffer_object())
    {
      if (depthOfFieldBlurXYNumberOfSamplesRange <= 63)
      {
        depthOfFieldBlurXYNumberOfSamplesRange++;
      }
      DebugConsole_consoleOutLine("depthOfFieldBlurXYNumberOfSamplesRange: ", depthOfFieldBlurXYNumberOfSamplesRange);
    }
  }
  else if (key == GLUT_KEY_PAGE_DOWN)
  {
    if ((currentGBUfferEffectType == DEPTH_OF_FIELD) && openGLDriverInfo->supports_GL_EXT_framebuffer_object())
    {
      if (depthOfFieldBlurXYNumberOfSamplesRange >= 1)
      {
        depthOfFieldBlurXYNumberOfSamplesRange--;
      }
      DebugConsole_consoleOutLine("depthOfFieldBlurXYNumberOfSamplesRange: ", depthOfFieldBlurXYNumberOfSamplesRange);
    }
  }
  else if (key == GLUT_KEY_HOME)
  {
    if (useTransparency)
    {
      if (opacity <= 0.95f)
      {
        opacity += 0.05f;
      }
      else
      {
        opacity = 0.999f;
      }

      reInitColorArrays = true;
      reInitModelDataForRendering = true;
      reInitModelCylinderDataForRendering = true;
      DebugConsole_consoleOutLine("opacity: ", opacity);
    }
  }
  else if (key == GLUT_KEY_END)
  {
    if (useTransparency)
    {
      if (opacity >= 0.10f)
      {
        opacity -= 0.05f;
      }
      else
      {
        opacity = 0.001f;
      }

      reInitColorArrays = true;
      reInitModelDataForRendering = true;
      reInitModelCylinderDataForRendering = true;
      DebugConsole_consoleOutLine("opacity: ", opacity);
    }
  }
  else if (key == GLUT_KEY_INSERT)
  {
    if (openGLDriverInfo->supports_GL_EXT_framebuffer_object())
    {
      useFXAA_Antialias = !useFXAA_Antialias;
      DebugConsole_consoleOutLine("Toggle FXAA Antialias: ", useFXAA_Antialias ? "on" : "off");
    }
  }
  else if (key == GLUT_KEY_F1)
  {
    reInitModelDataForRendering = true;
    reInitModelCylinderDataForRendering = true;
    currentCachedRenderingTest = USE_NO_CACHING_METHOD_1;
    DebugConsole_consoleOutLine("currentVBOTest: USE_NO_CACHING_METHOD_1");
  }
  else if (key == GLUT_KEY_F2)
  {
    reInitModelDataForRendering = true;
    reInitModelCylinderDataForRendering = true;
    currentCachedRenderingTest = USE_NO_CACHING_METHOD_2;
    DebugConsole_consoleOutLine("currentVBOTest: USE_NO_CACHING_METHOD_2");
  }
  else if (key == GLUT_KEY_F3)
  {
    reInitModelDataForRendering = true;
    reInitModelCylinderDataForRendering = true;
    currentCachedRenderingTest = USE_VERTEX_ARRAYS;
    DebugConsole_consoleOutLine("currentVBOTest: USE_VERTEX_ARRAYS");
  }
  else if (key == GLUT_KEY_F4)
  {
    reInitModelDataForRendering = true;
    reInitModelCylinderDataForRendering = true;
    currentCachedRenderingTest = USE_DISPLAY_LISTS;
    DebugConsole_consoleOutLine("currentVBOTest: USE_DISPLAY_LISTS");
  }
  else if (key == GLUT_KEY_F5)
  {
    reInitModelDataForRendering = true;
    reInitModelCylinderDataForRendering = true;
    currentCachedRenderingTest = USE_VBOS;
    DebugConsole_consoleOutLine("currentVBOTest: USE_VBOS");
  }
  else if (key == GLUT_KEY_F6)
  {
    if (!openGLDriverInfo->isAMDATI()) // recent AMD/ATI cards lack an accumulation buffer
    {
      useMotionBlurForScene = !useMotionBlurForScene;
      if (useMotionBlurForScene)
      {
        reInitMotionBlurForScene = true;
      }
      DebugConsole_consoleOutLine("useMotionBlurForScene: ", useMotionBlurForScene ? "ACCUM BUFFER" : "NO_USE");
    }
  }
  else if (key == GLUT_KEY_F7)
  {
    reInitModelDataForRendering = true;
    reInitModelCylinderDataForRendering = true;
    usePerspectiveCorrection = previousUsePerspectiveCorrection; // restore perspective correction state
    currentImpostorShader = POINT_SPHERE;
    DebugConsole_consoleOutLine("currentImpostorShader: POINT SPHERE");
  }
  else if (key == GLUT_KEY_F8)
  {
    reInitModelDataForRendering = true;
    reInitModelCylinderDataForRendering = true;
    usePerspectiveCorrection = previousUsePerspectiveCorrection; // restore perspective correction state
    currentImpostorShader = QUAD_SPHERE_VS_FS;
    DebugConsole_consoleOutLine("currentImpostorShader: QUAD SPHERE VS FS");
  }
  else if (key == GLUT_KEY_F9)
  {
    reInitModelDataForRendering = true;
    reInitModelCylinderDataForRendering = true;
    usePerspectiveCorrection = previousUsePerspectiveCorrection; // restore perspective correction state
    currentImpostorShader = QUAD_SPHERE_VS_GS_FS;
    DebugConsole_consoleOutLine("currentImpostorShader: QUAD SPHERE VS GS FS");
  }
  else if (key == GLUT_KEY_F10)
  {
    if (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader())
    {
      reInitModelDataForRendering = true;
      reInitModelCylinderDataForRendering = true;
      usePerspectiveCorrection = previousUsePerspectiveCorrection; // restore perspective correction state
      currentImpostorShader = QUAD_SPHERE_VS_TESS_LOD_FS;
      DebugConsole_consoleOutLine("currentImpostorShader: QUAD SPHERE VS TESS LOD FS");
    }
  }
  else if (key == GLUT_KEY_F11)
  {
    if (openGLDriverInfo->supports420Shaders() && openGLDriverInfo->supports_GL_ARB_tessellation_shader())
    {
      reInitModelDataForRendering = true;
      reInitModelCylinderDataForRendering = true;
      usePerspectiveCorrection = previousUsePerspectiveCorrection; // restore perspective correction state
      currentImpostorShader = QUAD_SPHERE_VS_TESS_QUAD_FS;
      DebugConsole_consoleOutLine("currentImpostorShader: QUAD SPHERE VS TESS QUAD FS");
    }
  }
  else if (key == GLUT_KEY_F12)
  {
    reInitModelDataForRendering = true;
    reInitModelCylinderDataForRendering = true;
    previousUsePerspectiveCorrection = usePerspectiveCorrection; // save perspective correction state
    usePerspectiveCorrection = true; // enforce perspective correction for Sphere Impostors to look 100% together with Cylinder Impostors
    currentImpostorShader = CYLINDER_VS_GS_FS;
    DebugConsole_consoleOutLine("currentImpostorShader: CYLINDER VS GS FS");
  }

  reInitUIInformation = true;
  isNavigating = true;
}

void ParticleRenderingModelLoaderTest::mouse(int button, int state, int x, int y)
{
  mouseX = float(x);
  mouseY = float(y);

  if (button == GLUT_LEFT_BUTTON)
  {
    if (state == GLUT_DOWN)
    {
      mouseLeftDown = true;
    }
    else if (state == GLUT_UP)
    {
      mouseLeftDown = false;
    }
  }
  else if (button == GLUT_MIDDLE_BUTTON)
  {
    if (state == GLUT_DOWN)
    {
      mouseMiddleDown = true;
    }
    else if (state == GLUT_UP)
    {
      mouseMiddleDown = false;
    }
  }
  else if (button == GLUT_RIGHT_BUTTON)
  {
    if (state == GLUT_DOWN)
    {
      mouseRightDown = true;
    }
    else if (state == GLUT_UP)
    {
      mouseRightDown = false;
    }
  }
  else if ((button == 3) || (button == 4)) // It's a wheel event for FreeGlut
  {
    const double tempCameraDistanceZ = (cameraDistanceZ > 0.0) ? (1.0 + cameraDistanceZ / 10.0) : 1.0;
    if (button == 3)
    {
      cameraDistanceZ -= 0.15 * tempCameraDistanceZ;
    }
    else
    {
      cameraDistanceZ += 0.15 * tempCameraDistanceZ;
    }
  }

  isNavigating = true;
}

void ParticleRenderingModelLoaderTest::mouseMotion(int x, int y)
{
  if (mouseLeftDown)
  {
    cameraAngleY += (double(x) - mouseX);
    cameraAngleX -= (double(y) - mouseY);
    mouseX = double(x);
    mouseY = double(y);
  }
  else if (mouseMiddleDown)
  {
    const double tempCameraDistanceZ = (cameraDistanceZ > 0.0) ? (1.0 + cameraDistanceZ / 5.0) : 1.0;
    cameraDistanceX += 0.005 * (double(x) - mouseX) * tempCameraDistanceZ;
    cameraDistanceY -= 0.005 * (double(y) - mouseY) * tempCameraDistanceZ;
    mouseX = double(x);
    mouseY = double(y);
  }
  else if (mouseRightDown)
  {
    const double tempCameraDistanceZ = (cameraDistanceZ > 0.0) ? (1.0 + cameraDistanceZ / 10.0) : 1.0;
    cameraDistanceZ += 0.05 * (double(y) - mouseY) * tempCameraDistanceZ;
    mouseY = double(y);
  }

  isNavigating = true;
}

void ParticleRenderingModelLoaderTest::closeFunc()
{
  if (glIsList(allModelDataDisplayList))
  {
    glDeleteLists(allModelDataDisplayList, 1);
  }
  deleteVBOs();
  if (glIsList(allModelCylinderDataDisplayList))
  {
    glDeleteLists(allModelCylinderDataDisplayList, 1);
  }
  if (openGLDriverInfo->supports330Shaders())
  {
    if (glIsVertexArray(dummyVao))
    {
      glDeleteVertexArrays(1, &dummyVao);
    }
  }
  deleteCylinderVBOs();
  if (openGLDriverInfo->supports430Shaders())
  {
    deleteABuffer3D();
  }

  // for HSSAO rotationTexture data
  if (glIsTexture(rotationTextureID))
  {
    glDeleteTextures(1, &rotationTextureID);
    rotationTextureID = 0;
  }

  delete openGLShaderClearABuffer3D;
  openGLShaderClearABuffer3D = nullptr;
  delete openGLShaderDisplayABuffer3D;
  openGLShaderDisplayABuffer3D = nullptr;
  delete openGLShaderGBufferEffectsBlurXY;
  openGLShaderGBufferEffectsBlurXY = nullptr;
  delete openGLShaderGBufferEffectsHSSAO;
  openGLShaderGBufferEffectsHSSAO = nullptr;
  delete openGLShaderGBufferEffects;
  openGLShaderGBufferEffects = nullptr;
  delete openGLShaderFXAA_Antialias;
  openGLShaderFXAA_Antialias = nullptr;
  delete openGLModelAmbientLight;
  openGLModelAmbientLight = nullptr;
  for (auto& openGLLight : openGLLights)
  {
    delete openGLLight;
    openGLLight = nullptr;
  }
  delete openGLMaterial;
  openGLMaterial = nullptr;
  delete openGLShaderPointSphereModels;
  openGLShaderPointSphereModels = nullptr;
  delete openGLShaderPointSphereModelsQuadVFFS;
  openGLShaderPointSphereModelsQuadVFFS = nullptr;
  delete openGLShaderPointSphereModelsQuadVFGSFS;
  openGLShaderPointSphereModelsQuadVFGSFS = nullptr;
  delete openGLShaderPointSphereModelsQuadVFTESSLODFS;
  openGLShaderPointSphereModelsQuadVFTESSLODFS = nullptr;
  delete openGLShaderPointSphereModelsQuadVFTESSQuadFS;
  openGLShaderPointSphereModelsQuadVFTESSQuadFS = nullptr;
  delete openGLShaderCylinderVFGSFS;
  openGLShaderCylinderVFGSFS = nullptr;
  if (openGLDriverInfo->supports_GL_EXT_framebuffer_object())
  {
    delete openGLFrameBufferObjectForEnvironmentMapping;
    openGLFrameBufferObjectForEnvironmentMapping = nullptr;
    if (openGLDriverInfo->supports_GL_EXT_gpu_shader4() && openGLDriverInfo->supports330Shaders())
    {
      delete openGLFrameBufferObjectForGBufferPass1;
      openGLFrameBufferObjectForGBufferPass1 = nullptr;
      delete openGLFrameBufferObjectRenderPass2;
      openGLFrameBufferObjectRenderPass2 = nullptr;
      delete openGLFrameBufferObjectRenderPass3;
      openGLFrameBufferObjectRenderPass3 = nullptr;
    }
    delete openGLFrameBufferObjectForFXAA_Antialias;
    openGLFrameBufferObjectForFXAA_Antialias = nullptr;
  }
  delete openGLILTexture;
  openGLILTexture = nullptr;
  delete openGLAssimpModelLoader;
  openGLAssimpModelLoader = nullptr;

  TestAbstractBase::releaseAllGLResources();
}