#include "CubeCappingTest.h"
#include "OpenGLAssetManager.h"
#include "OpenGLDriverInfo.h"
#include "OpenGLEulerCamera.h"
#include "OpenGLFrameBufferObject.h"
#include "OpenGLUtilityFunctions.h"
#include "OpenGLShaderCompileAndLink.h"
#include "OpenGLShaderGLSLPreProcessorCommands.h"
#include "UtilityFunctions.h"
#include <algorithm>
#include <array>
#include <iomanip>

using namespace std;
using namespace OpenGLRenderingEngine;
using namespace OpenGLRenderingEngineTests;
using namespace OpenGLRenderingEngine::OpenGLUtilityFunctions;

namespace // anonymous namespace used instead of deprecated 'static' keyword used for cpp variable locality
{
  const array<GLfloat, 4> BLACK_BACKGROUND_COLOR = { { 0.0f, 0.0f, 0.0f, 1.0f } }; // double braces because we initialize an array inside an std::array object
  const array<GLfloat, 4> WHITE_BACKGROUND_COLOR = { { 1.0f, 1.0f, 1.0f, 1.0f } }; // double braces because we initialize an array inside an std::array object
  const GLfloat CUBE_SIZE                        = 0.5f; // must be 0.5 for cube capping to work properly!
  const unsigned int VERTICES_DATA_SIZE          = 3 * 24;
  const array<GLfloat, VERTICES_DATA_SIZE> VERTICES_DATA = { {
                                                              -CUBE_SIZE, -CUBE_SIZE,  CUBE_SIZE,
                                                               CUBE_SIZE, -CUBE_SIZE,  CUBE_SIZE,
                                                               CUBE_SIZE,  CUBE_SIZE,  CUBE_SIZE,
                                                              -CUBE_SIZE,  CUBE_SIZE,  CUBE_SIZE,

                                                               CUBE_SIZE, -CUBE_SIZE,  CUBE_SIZE,
                                                               CUBE_SIZE, -CUBE_SIZE, -CUBE_SIZE,
                                                               CUBE_SIZE,  CUBE_SIZE, -CUBE_SIZE,
                                                               CUBE_SIZE,  CUBE_SIZE,  CUBE_SIZE,

                                                               CUBE_SIZE, -CUBE_SIZE, -CUBE_SIZE,
                                                              -CUBE_SIZE, -CUBE_SIZE, -CUBE_SIZE,
                                                              -CUBE_SIZE,  CUBE_SIZE, -CUBE_SIZE,
                                                               CUBE_SIZE,  CUBE_SIZE, -CUBE_SIZE,

                                                              -CUBE_SIZE, -CUBE_SIZE,  CUBE_SIZE,
                                                              -CUBE_SIZE, -CUBE_SIZE, -CUBE_SIZE,
                                                              -CUBE_SIZE,  CUBE_SIZE, -CUBE_SIZE,
                                                              -CUBE_SIZE,  CUBE_SIZE,  CUBE_SIZE,

                                                              -CUBE_SIZE,  CUBE_SIZE,  CUBE_SIZE,
                                                               CUBE_SIZE,  CUBE_SIZE,  CUBE_SIZE,
                                                               CUBE_SIZE,  CUBE_SIZE, -CUBE_SIZE,
                                                              -CUBE_SIZE,  CUBE_SIZE, -CUBE_SIZE,

                                                              -CUBE_SIZE, -CUBE_SIZE, -CUBE_SIZE,
                                                               CUBE_SIZE, -CUBE_SIZE, -CUBE_SIZE,
                                                               CUBE_SIZE, -CUBE_SIZE,  CUBE_SIZE,
                                                              -CUBE_SIZE, -CUBE_SIZE,  CUBE_SIZE
                                                          } };
  const unsigned int NORMALS_DATA_SIZE = 3 * 24;
  const array<GLfloat, NORMALS_DATA_SIZE> NORMALS_DATA = { {
                                                             0.0f,  1.0f, 0.0f,
                                                             0.0f,  1.0f, 0.0f,
                                                             0.0f,  1.0f, 0.0f,
                                                             0.0f,  1.0f, 0.0f,

                                                             0.0f, -1.0f, 0.0f,
                                                             0.0f, -1.0f, 0.0f,
                                                             0.0f, -1.0f, 0.0f,
                                                             0.0f, -1.0f, 0.0f,

                                                             0.0f,  0.0f,  1.0f,
                                                             0.0f,  0.0f,  1.0f,
                                                             0.0f,  0.0f,  1.0f,
                                                             0.0f,  0.0f,  1.0f,

                                                             0.0f,  0.0f, -1.0f,
                                                             0.0f,  0.0f, -1.0f,
                                                             0.0f,  0.0f, -1.0f,
                                                             0.0f,  0.0f, -1.0f,

                                                            -1.0f,  0.0f,  0.0f,
                                                            -1.0f,  0.0f,  0.0f,
                                                            -1.0f,  0.0f,  0.0f,
                                                            -1.0f,  0.0f,  0.0f,

                                                             1.0f,  0.0f,  0.0f,
                                                             1.0f,  0.0f,  0.0f,
                                                             1.0f,  0.0f,  0.0f,
                                                             1.0f,  0.0f,  0.0f
                                                         } };
  const unsigned int TEXCOORDS_DATA_SIZE = 2 * 24;
  const array<GLfloat, TEXCOORDS_DATA_SIZE> TEXCOORDS_DATA = { {
                                                                 0.0f,  0.0f,
                                                                 0.0f,  1.0f,
                                                                 1.0f,  1.0f,
                                                                 1.0f,  0.0f,

                                                                 0.0f,  0.0f,
                                                                 0.0f,  1.0f,
                                                                 1.0f,  1.0f,
                                                                 1.0f,  0.0f,

                                                                 0.0f,  0.0f,
                                                                 0.0f,  1.0f,
                                                                 1.0f,  1.0f,
                                                                 1.0f,  0.0f,

                                                                 0.0f,  0.0f,
                                                                 0.0f,  1.0f,
                                                                 1.0f,  1.0f,
                                                                 1.0f,  0.0f,

                                                                 0.0f,  0.0f,
                                                                 0.0f,  1.0f,
                                                                 1.0f,  1.0f,
                                                                 1.0f,  0.0f,

                                                                 0.0f,  0.0f,
                                                                 0.0f,  1.0f,
                                                                 1.0f,  1.0f,
                                                                 1.0f,  0.0f
                                                             } };
  const unsigned int COLORS_DATA_SIZE = 3 * 24;
  const array<GLfloat, COLORS_DATA_SIZE> COLORS_DATA = { {
                                                           0.0f, 0.0f, 0.0f,
                                                           1.0f, 0.0f, 0.0f,
                                                           1.0f, 1.0f, 0.0f,
                                                           0.0f, 1.0f, 0.0f,

                                                           1.0f, 0.0f, 0.0f,
                                                           1.0f, 0.0f, 1.0f,
                                                           1.0f, 1.0f, 1.0f,
                                                           1.0f, 1.0f, 0.0f,

                                                           1.0f, 0.0f, 1.0f,
                                                           0.0f, 0.0f, 1.0f,
                                                           0.0f, 1.0f, 1.0f,
                                                           1.0f, 1.0f, 1.0f,

                                                           0.0f, 0.0f, 0.0f,
                                                           0.0f, 0.0f, 1.0f,
                                                           0.0f, 1.0f, 1.0f,
                                                           0.0f, 1.0f, 0.0f,

                                                           0.0f, 1.0f, 0.0f,
                                                           1.0f, 1.0f, 0.0f,
                                                           1.0f, 1.0f, 1.0f,
                                                           0.0f, 1.0f, 1.0f,

                                                           0.0f, 0.0f, 1.0f,
                                                           1.0f, 0.0f, 1.0f,
                                                           1.0f, 0.0f, 0.0f,
                                                           0.0f, 0.0f, 0.0f
                                                       } };
  const unsigned int INDICES_DATA_SIZE = 3 * 12;
  const array<GLubyte, INDICES_DATA_SIZE> INDICES_DATA = { {
                                                             0,  1,  2,
                                                             2,  3,  0,

                                                             4,  5,  6,
                                                             6,  7,  4,

                                                             8,  9, 10,
                                                            10, 11,  8,

                                                            12, 13, 14,
                                                            14, 15, 12,

                                                            16, 17, 18,
                                                            18, 19, 16,

                                                            20, 21, 22,
                                                            22, 23, 20
                                                        } };
  const GLfloat VIEW_NEAR                             = 1.0f; // use a different viewNear value than default 0.01f in OpenGLCameraAbstractBase to make the Cube Clipping more pronounced to the viewer
  const GLfloat VIEW_NEAR_OFFSET                      = 0.001f;
  const bool USE_CUBE_CAPPING_FBO_FOR_RENDERING       = true;
  const bool USE_TEXTURE_MATRIX_PLANE_TRANSFORMATIONS = false;
  array<array<GLfloat, 3>, 4> CLIP_PLANE_VERTICES     = { { { { 1.0f,  1.0f, 0.0f } },
                                                            { {-1.0f,  1.0f, 0.0f } },
                                                            { {-1.0f, -1.0f, 0.0f } },
                                                            { { 1.0f, -1.0f, 0.0f } }
                                                        } };
}

CubeCappingTest::CubeCappingTest(int screenWidth, int screenHeight, bool multisample) noexcept
  : TestAbstractBase(screenWidth, screenHeight, "", "", "", multisample)
{
  prepareCubeCappingShaders();
  if (USE_CUBE_CAPPING_FBO_FOR_RENDERING && openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_packed_depth_stencil())
  {
    prepareCubeCappingFBO();
  }
  prepareVBOs();
  
  cameraAngleY = 180.0f;
  cameraDistanceZ = 2.5; // 2.5 to emulate original glLookAt() behavior for this test
  openGLEulerCamera->setViewNear(VIEW_NEAR + clipPlaneZ);

  clearScreen();
}

void CubeCappingTest::OpenGLShaderCubeCapping::initializeShaderProgram()
{
  openGLShaderGLSLPreProcessorCommands_->addHighestGLVersionDefinition();
  openGLShaderGLSLPreProcessorCommands_->addPreprocessorLine(string("#define VS_VARYING ") + string((openGLDriverInfo_->supports330Shaders() || openGLDriverInfo_->supports420Shaders()) ? "out" : "varying"));
  openGLShaderGLSLPreProcessorCommands_->addPreprocessorLine(string("#define FS_VARYING ") + string((openGLDriverInfo_->supports330Shaders() || openGLDriverInfo_->supports420Shaders()) ? "in" : "varying"));
  openGLShaderGLSLPreProcessorCommands_->addDefinitionAndCondition("USE_TEXTURE_MATRIX_PLANE_TRANSFORMATIONS", USE_TEXTURE_MATRIX_PLANE_TRANSFORMATIONS);
  openGLShaderGLSLPreProcessorCommands_->addDefinitionForStartingLine();

  openGLShaderCompileAndLink_->addShaderLibraryToProgram("CubeCapping", "CubeCapping", OpenGLAssetManager::VS | OpenGLAssetManager::FS);
  openGLShaderCompileAndLink_->linkShaderProgram();

  GLAuxiliaryFunctions_checkGLError();
}

void CubeCappingTest::prepareCubeCappingShaders()
{
  openGLShaderCubeCapping = new OpenGLShaderCubeCapping(openGLDriverInfo);
}

void CubeCappingTest::prepareCubeCappingFBO()
{
  openGLFrameBufferObjectForCubeCapping = new OpenGLFrameBufferObject(openGLDriverInfo);
  openGLFrameBufferObjectForCubeCapping->setUseTexture(true);
  openGLFrameBufferObjectForCubeCapping->setDepthStencilRenderBuffer(true);
  if (!openGLDriverInfo->isIntel() && openGLDriverInfo->supports_GL_EXT_framebuffer_multisample() && openGLDriverInfo->supports_GL_EXT_framebuffer_blit())
  {
    openGLFrameBufferObjectForCubeCapping->setMultisampleFBO(true, 4);
  }
}

void CubeCappingTest::initCubeCappingFBO() const
{
  DebugConsole_consoleOutLine("Cube Capping FBO Render-to-texture ratioFactor:\t  ", FULLSCREEN_MAPPING_RATIO_FACTOR);
  DebugConsole_consoleOutLine("Cube Capping FBO Render-to-texture multiplicationFactor: ", FULLSCREEN_MAPPING_RATIO_FACTOR);
  openGLFrameBufferObjectForCubeCapping->disposeFrameBufferObjectResources();
  openGLFrameBufferObjectForCubeCapping->initFrameBufferObjectResources(screenWidth / FULLSCREEN_MAPPING_RATIO_FACTOR, screenHeight / FULLSCREEN_MAPPING_RATIO_FACTOR, ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
}

void CubeCappingTest::prepareVBOs()
{
  glGenBuffers(1, &VBOVerticesID);
  glBindBuffer(GL_ARRAY_BUFFER, VBOVerticesID);
  glBufferData(GL_ARRAY_BUFFER, VERTICES_DATA_SIZE * sizeof(GLfloat), VERTICES_DATA.data(), GL_STATIC_DRAW);

  glGenBuffers(1, &VBONormalsID);
  glBindBuffer(GL_ARRAY_BUFFER, VBONormalsID);
  glBufferData(GL_ARRAY_BUFFER, NORMALS_DATA_SIZE * sizeof(GLfloat), NORMALS_DATA.data(), GL_STATIC_DRAW);

  glGenBuffers(1, &VBOTexCoordsID);
  glBindBuffer(GL_ARRAY_BUFFER, VBOTexCoordsID);
  glBufferData(GL_ARRAY_BUFFER, TEXCOORDS_DATA_SIZE * sizeof(GLfloat), TEXCOORDS_DATA.data(), GL_STATIC_DRAW);

  glGenBuffers(1, &VBOColorsID);
  glBindBuffer(GL_ARRAY_BUFFER, VBOColorsID);
  glBufferData(GL_ARRAY_BUFFER, COLORS_DATA_SIZE * sizeof(GLfloat), COLORS_DATA.data(), GL_STATIC_DRAW);

  glGenBuffers(1, &VBOIndicesID);
  glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, VBOIndicesID);
  glBufferData(GL_ELEMENT_ARRAY_BUFFER, INDICES_DATA_SIZE * sizeof(GLubyte), INDICES_DATA.data(), GL_STATIC_DRAW);

  glBindBuffer(GL_ARRAY_BUFFER, 0);
  glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
}

void CubeCappingTest::deleteVBOs()
{
  glDeleteBuffers(1, &VBOIndicesID);
  glDeleteBuffers(1, &VBOColorsID);
  glDeleteBuffers(1, &VBOTexCoordsID);
  glDeleteBuffers(1, &VBONormalsID);
  glDeleteBuffers(1, &VBOVerticesID);
}

void CubeCappingTest::clearScreen() const
{
  if (blackOrWhiteBackground)
  {
    glClearColor(WHITE_BACKGROUND_COLOR[0], WHITE_BACKGROUND_COLOR[1], WHITE_BACKGROUND_COLOR[2], WHITE_BACKGROUND_COLOR[3]);
  }
  else
  {
    glClearColor(BLACK_BACKGROUND_COLOR[0], BLACK_BACKGROUND_COLOR[1], BLACK_BACKGROUND_COLOR[2], BLACK_BACKGROUND_COLOR[3]);
  }
  glClearStencil(0); // clear stencil buffer
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
}

void CubeCappingTest::renderCubeClippingPlane() const
{
  if (USE_TEXTURE_MATRIX_PLANE_TRANSFORMATIONS)
  {
    // clip plane translate tranformation on texture matrix
    glMatrixMode(GL_TEXTURE);
    glPushMatrix();
    glLoadIdentity();
    glTranslatef(0.0f, 0.0f, -(VIEW_NEAR + clipPlaneZ) - VIEW_NEAR_OFFSET);
  }
  else
  {
    // manual translate tranformation on clip plane
    CLIP_PLANE_VERTICES[0][2] = CLIP_PLANE_VERTICES[1][2] = CLIP_PLANE_VERTICES[2][2] = CLIP_PLANE_VERTICES[3][2] = -(VIEW_NEAR + clipPlaneZ) - VIEW_NEAR_OFFSET;
  }

  glBegin(GL_QUADS);
    glVertex3fv(CLIP_PLANE_VERTICES[0].data());
    glVertex3fv(CLIP_PLANE_VERTICES[1].data());
    glVertex3fv(CLIP_PLANE_VERTICES[2].data());
    glVertex3fv(CLIP_PLANE_VERTICES[3].data());
  glEnd();

  if (USE_TEXTURE_MATRIX_PLANE_TRANSFORMATIONS)
  {
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
  }
}

void CubeCappingTest::renderCube() const
{
  // Enable our client state for VBOs
  glEnableClientState(GL_TEXTURE_COORD_ARRAY);
  glEnableClientState(GL_COLOR_ARRAY);
  glEnableClientState(GL_NORMAL_ARRAY);
  glEnableClientState(GL_VERTEX_ARRAY);

  // Bind our buffers much like we would for texturing
  glBindBuffer(GL_ARRAY_BUFFER, VBOTexCoordsID);
  glTexCoordPointer(2, GL_FLOAT, 0, nullptr);

  glBindBuffer(GL_ARRAY_BUFFER, VBONormalsID);
  glNormalPointer(GL_FLOAT, 0, nullptr);

  glBindBuffer(GL_ARRAY_BUFFER, VBOVerticesID);
  glVertexPointer(3, GL_FLOAT, 0, nullptr);

  glBindBuffer(GL_ARRAY_BUFFER, VBOColorsID);
  glColorPointer(3, GL_FLOAT, 0, nullptr);

  if (currentCachedRenderingTest != DRAW_ARRAYS)
  {
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, VBOIndicesID);
  }

  if (currentCachedRenderingTest == DRAW_ARRAYS)
  {
    glDrawArrays(GL_QUADS, 0, 24);
  }
  else if (currentCachedRenderingTest == DRAW_ELEMENTS)
  {
    glDrawElements(GL_TRIANGLES, 36, GL_UNSIGNED_BYTE, nullptr);
  }
  else // if (currentCachedRenderingTest == DRAW_RANGE_ELEMENTS)
  {
    glDrawRangeElements(GL_TRIANGLES, 0, 12, 12, GL_UNSIGNED_BYTE, nullptr);
  }

  if (currentCachedRenderingTest != DRAW_ARRAYS)
  {
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
  }
  glBindBuffer(GL_ARRAY_BUFFER, 0);

  // Disable our client state back to normal drawing
  glDisableClientState(GL_TEXTURE_COORD_ARRAY);
  glDisableClientState(GL_COLOR_ARRAY);
  glDisableClientState(GL_NORMAL_ARRAY);
  glDisableClientState(GL_VERTEX_ARRAY);
}

void CubeCappingTest::renderCubeScene() const
{
  openGLEulerCamera->setViewNear(VIEW_NEAR + clipPlaneZ);
  openGLEulerCamera->setMatrices();

  if (USE_CUBE_CAPPING_FBO_FOR_RENDERING && openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_packed_depth_stencil())
  {
    openGLFrameBufferObjectForCubeCapping->bindTexture(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE);
    openGLFrameBufferObjectForCubeCapping->startRender();
  }

  clearScreen();

  // if (currentCachedRenderingTest != DRAW_RANGE_ELEMENTS)
  // {
  //   glEnable(GL_CULL_FACE);
  //   glCullFace(GL_BACK);
  // }
  // else
  // {
  glDisable(GL_CULL_FACE); // always disable GL_CULL_FACE for Cube Capping to work ok
  // }

  glPushMatrix();

  if (useCubeCapping && (currentCachedRenderingTest != DRAW_RANGE_ELEMENTS))
  {
    // Use a stencil func/op which leaves a bit set in the stencil buffer
    // wherever the inside of the object is exposed by the clipping plane.
    glEnable(GL_STENCIL_TEST);
    glStencilFunc(GL_ALWAYS, GL_ONE, GL_ONE);
    glStencilOp(GL_KEEP, GL_INVERT, GL_INVERT);
  }

  // object modeling transformations via the Euler Camera
  openGLEulerCamera->setEulerCamera(cameraDistanceX, cameraDistanceY, cameraDistanceZ, cameraAngleX, cameraAngleY);

  if (wireframe)
  {
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE); // set wireframe
  }

  renderCube();

  if (wireframe)
  {
    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL); // reset wireframe state
  }

  if (useCubeCapping && (currentCachedRenderingTest != DRAW_RANGE_ELEMENTS))
  {
    // Use a stencil func/op which passes only where there is a bit set
    // in the stencil buffer. We also zero the bits as we go so that we
    // don't have to explicitly clear the stencil buffer each frame.
    glStencilFunc(GL_EQUAL, GL_ONE, GL_ONE);
    glStencilOp(GL_KEEP, GL_ZERO, GL_ZERO);

    openGLShaderCubeCapping->enableShaderProgram();
    renderCubeClippingPlane();
    openGLShaderCubeCapping->disableShaderProgram();

    glDisable(GL_STENCIL_TEST);
  }

  glPopMatrix();

  if (USE_CUBE_CAPPING_FBO_FOR_RENDERING && openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_packed_depth_stencil())
  {
    openGLFrameBufferObjectForCubeCapping->finishRender();

    clearScreen();
    openGLFrameBufferObjectForCubeCapping->renderTextureToFullScreenQuad(ACTIVE_TEXTURE_UNIT_FOR_2D_TEXTURE, openGLDriverInfo->isNvidia(), false); // don't use a dummyVAO for quad rendering as no shader is being used here
  }
}

/*
*  Draws a string in 2D space using GLUT.
*  The projection matrix must be set to orthogonal before call this function.
*/
void CubeCappingTest::drawString(const char* str, int x, int y, const GLfloat color[4], void* font) const
{
  glColor4fv(color);      // set text color
  glRasterPos2i(x, y);    // place text position

  // loop all characters in the string
  while (*str)
  {
    glutBitmapCharacter(font, *str);
    ++str;
  }
}

/*
*  Draws a string in 3D space using GLUT.
*/
void CubeCappingTest::drawString3D(const char *str, float position[3], const GLfloat color[4], void* font) const
{
  glColor4fv(color);        // set text color
  glRasterPos3fv(position); // place text position

  // loop all characters in the string
  while (*str)
  {
    glutBitmapCharacter(font, *str);
    ++str;
  }
}

void CubeCappingTest::showInfo()
{
  if (reInitUIInformation)
  {
    // just to clear VRAM memory
    if (glIsList(useUIInformationDisplayList))
    {
      glDeleteLists(useUIInformationDisplayList, 1);
    }
    glNewList(useUIInformationDisplayList, GL_COMPILE);

    glPushAttrib(GL_LIGHTING_BIT | GL_CURRENT_BIT | GL_DEPTH_BUFFER_BIT); // push all relevant attribs
    glDisable(GL_LIGHTING); // need to disable lighting for proper text color
    glDisable(GL_DEPTH_TEST);

    // safety restore-default-texture GL state commands
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, 0);
    glDisable(GL_TEXTURE_2D);

    // backup current model-view matrix
    glPushMatrix();                 // save current modelview matrix
    glLoadIdentity();               // reset modelview matrix

    // set to 2D orthogonal projection
    glMatrixMode(GL_PROJECTION);    // switch to projection matrix
    glPushMatrix();                 // save current projection matrix
    glLoadIdentity();               // reset projection matrix
    gluOrtho2D(0, screenWidth, 0, screenHeight); // set to orthogonal projection

    const GLfloat* color = (blackOrWhiteBackground) ? BLACK_BACKGROUND_COLOR.data() : WHITE_BACKGROUND_COLOR.data();
    ostringstream ss;

    ss << "VBO: " << (true ? "on" : "off") << ends;  // add 0(ends) at the end
    drawString(ss.str().c_str(), 1, screenHeight - GLUT_TEXT_HEIGHT, color, getFont());
    ss.str(""); // clear buffer

    ss << "VSynch: " << (vsynch ? "on" : "off") << ends;  // add 0(ends) at the end
    drawString(ss.str().c_str(), 1, screenHeight - (2 * GLUT_TEXT_HEIGHT), color, getFont());
    ss.str(""); // clear buffer

    ss << "AutoRotate: " << (autoRotate ? "on" : "off") << ends;  // add 0(ends) at the end
    drawString(ss.str().c_str(), 1, screenHeight - (3 * GLUT_TEXT_HEIGHT), color, getFont());
    ss.str(""); // clear buffer

    ss << "currentCachedRenderingTest: " << ((currentCachedRenderingTest == DRAW_ARRAYS) ? "DRAW_ARRAYS" : ((currentCachedRenderingTest == DRAW_ELEMENTS) ? "DRAW_ELEMENTS" : "DRAW_RANGE_ELEMENTS")) << ends;  // add 0(ends) at the end
    drawString(ss.str().c_str(), 1, screenHeight - (4 * GLUT_TEXT_HEIGHT), color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'c'    key to toggle Cube Capping " << ((useCubeCapping) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (5 * GLUT_TEXT_HEIGHT) - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'Up'   key to move away from the viewer the Clip Plane Z " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (6 * GLUT_TEXT_HEIGHT) - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'Down' key to move nearer to the viewer the Clip Plane Z " << ends;
    drawString(ss.str().c_str(), 1, screenHeight - (7 * GLUT_TEXT_HEIGHT) - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'b' key to toggle " << ((blackOrWhiteBackground) ? "Black" : "White") << " background" << ends;
    drawString(ss.str().c_str(), 1, 7 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'w' key to toggle Wireframe " << ((wireframe) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, 6* GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'a' key to toggle AutoRotate on/off" << ends;
    drawString(ss.str().c_str(), 1, 5 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'r' key to reset the camera" << ends;
    drawString(ss.str().c_str(), 1, 4 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'v' key to toggle VSynch on/off" << ends;
    drawString(ss.str().c_str(), 1, 3 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'u' key to toggle UI Information Display " << ((useUIInformation) ? "off" : "on") << ends;
    drawString(ss.str().c_str(), 1, 2 * GLUT_TEXT_HEIGHT - 5, color, getFont());
    ss.str(""); // clear buffer

    ss << "Press 'Shift'  key to Take Screenshot " << ends;
    drawString(ss.str().c_str(), 1, GLUT_TEXT_HEIGHT - 5, color, getFont());

    // unset floating format
    ss << resetiosflags(ios_base::fixed | ios_base::floatfield);

    // restore projection matrix
    glPopMatrix();                   // restore to previous projection matrix

    // restore modelview matrix
    glMatrixMode(GL_MODELVIEW);      // switch to modelview matrix
    glPopMatrix();                   // restore to previous modelview matrix

    glPopAttrib(); // pop all relevant attribs

    glEndList();

    reInitUIInformation = false;
  }

  glCallList(useUIInformationDisplayList);
}

void CubeCappingTest::showFPS()
{
  // update fpsString every second
  const double elapsedTime = timer.getElapsedTimeInSecs();
  if(elapsedTime < 1.0)
  {
    ++fpsCounter;
  }
  else
  {
    ++fpsCounter;

    ostringstream ss;
    ss << fixed << setprecision(1);
    // ss << ((double)fpsCounter / elapsedTime) << " FPS" << ends; // update fpsString string
    ss << double(fpsCounter) << " FPS" << ends; // update fpsString string
    ss << resetiosflags(ios_base::fixed | ios_base::floatfield);
    fpsString = ss.str();
    fpsCounter = 0; // reset counter
    timer.startTimer();  // restart timer

    DebugConsole_consoleOutLine("Current framerate: ", fpsString.c_str());
  }

  // safety restore-default-texture GL state commands
  glActiveTexture(GL_TEXTURE0);
  glBindTexture(GL_TEXTURE_2D, 0);
  glDisable(GL_TEXTURE_2D);

  glPushAttrib(GL_LIGHTING_BIT | GL_CURRENT_BIT | GL_DEPTH_BUFFER_BIT); // push all relevant attribs
  glDisable(GL_LIGHTING); // need to disable lighting for proper text color
  glDisable(GL_TEXTURE_2D);
  glDisable(GL_DEPTH_TEST);

  // backup current model-view matrix
  glPushMatrix();                     // save current modelview matrix
  glLoadIdentity();                   // reset modelview matrix

  // set to 2D orthogonal projection
  glMatrixMode(GL_PROJECTION);        // switch to projection matrix
  glPushMatrix();                     // save current projection matrix
  glLoadIdentity();                   // reset projection matrix
  gluOrtho2D(0, screenWidth, 0, screenHeight); // set to orthogonal projection

  const int textWidth  = int(fpsString.size() * GLUT_TEXT_WIDTH);
  const GLfloat* color = (blackOrWhiteBackground) ? BLACK_BACKGROUND_COLOR.data() : WHITE_BACKGROUND_COLOR.data();

  drawString(fpsString.c_str(), screenWidth - textWidth, screenHeight - GLUT_TEXT_HEIGHT, color, getFont());

  // restore projection matrix
  glPopMatrix();                      // restore to previous projection matrix

  // restore modelView matrix
  glMatrixMode(GL_MODELVIEW);         // switch to modelview matrix
  glPopMatrix();                      // restore to previous modelview matrix

  glPopAttrib(); // pop all relevant attribs
}

void CubeCappingTest::renderScene()
{
  if (autoRotate)
  {
    cameraAngleX += 1.0;
    cameraAngleY += 1.0;
  }

  GLAuxiliaryFunctions::setVSynch(vsynch, openGLDriverInfo->isVSynchSupported());

  renderCubeScene();

  // draw info messages
  if (useUIInformation)
  {
    showInfo();
  }
  showFPS();

  if (takeScreenshot)
  {
    writeScreenshotToFile();
    takeScreenshot = false;
  }

  glFlush();
  glFinish();
  glutSwapBuffers();

  const GLenum glError = glGetError();
  if (glError != GL_NO_ERROR)
  {
    const GLubyte* gluError = gluErrorString(glError);
    DebugConsole_consoleOutLine("renderScene() glGetError() returned: ", gluError);
  }
}

void CubeCappingTest::changeSize(int w, int h)
{
  if (h == 0) // Prevent divide by zero
  {
    h = 1;
  }

  const GLfloat ratio = 1.0f * GLfloat(w) / GLfloat(h);
  screenWidth  = w;
  screenHeight = h;
  DebugConsole_consoleOutLine("New screenWidth: ",  screenWidth);
  DebugConsole_consoleOutLine("New screenHeight: ", screenHeight, "\n");

  glViewport(0, 0, screenWidth, screenHeight);

  GLAuxiliaryFunctions::setVSynch(vsynch, openGLDriverInfo->isVSynchSupported());

  openGLEulerCamera->setRatio(ratio);
  openGLEulerCamera->setMatrices();

  if (USE_CUBE_CAPPING_FBO_FOR_RENDERING && openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_packed_depth_stencil())
  {
    initCubeCappingFBO();
  }

  glFlush();
  glFinish();

  const GLenum glError = glGetError();
  if (glError != GL_NO_ERROR)
  {
    const GLubyte* gluError = gluErrorString(glError);
    DebugConsole_consoleOutLine("changeSize() glGetError() returned: ", gluError);
  }
}

void CubeCappingTest::keyboard(unsigned char key, int x, int y)
{
  if ((key == char('v')) || (key == char('V')))
  {
    vsynch++;
    vsynch = vsynch & 1;
    DebugConsole_consoleOutLine("Toggle VSynch: ", vsynch ? "on" : "off");
  }
  else if ((key == char('a')) || (key == char('A')))
  {
    autoRotate++;
    autoRotate = autoRotate & 1;
    DebugConsole_consoleOutLine("Toggle AutoRotate: ", autoRotate ? "true" : "false");
  }
  else if ((key == char('r')) || (key == char('R')))
  {
    cameraAngleX = cameraDistanceX = cameraDistanceY = 0.0;
    cameraAngleY = 180.0f;
    cameraDistanceZ = 2.5; // 2.5 to emulate original glLookAt() behavior for this test
    clipPlaneZ = 0.0f;     // also reset the clipPlaneZ value
    DebugConsole_consoleOutLine("Camera reset.");
  }
  else if ((key == char('c')) || (key == char('C')))
  {
    useCubeCapping = !useCubeCapping;
    DebugConsole_consoleOutLine("Toggle Cube Capping: ", useCubeCapping ? "on" : "off");
  }
  else if ((key == char('b')) || (key == char('B')))
  {
    blackOrWhiteBackground = !blackOrWhiteBackground;
    DebugConsole_consoleOutLine("Toggle BlackOrWhiteBackground: ", blackOrWhiteBackground ? "on" : "off");
  }
  else if ((key == char('w')) || (key == char('W')))
  {
    wireframe++;
    wireframe = wireframe & 1;
    DebugConsole_consoleOutLine("Toggle Wireframe: ", wireframe ? "on" : "off");
  }
  else if ((key == char('u')) || (key == char('U')))
  {
    useUIInformation++;
    useUIInformation = useUIInformation & 1;
    DebugConsole_consoleOutLine("Toggle UI Information: ", useUIInformation ? "on" : "off");
  }
  else if (key == 27)
  {
    DebugConsole_consoleOutLine("\nEscape pressed, now exiting Test: VBO Tests.\n");
    glutLeaveMainLoop();
  }

  reInitUIInformation = true;
}

void CubeCappingTest::specialKeysKeyboard(int key, int x, int y)
{
  if (glutGetModifiers() == GLUT_ACTIVE_SHIFT)
  {
    takeScreenshot = true;
    DebugConsole_consoleOutLine("takeScreenshot: ", takeScreenshot ? "yes" : "no");
  }
  else if (key == GLUT_KEY_F1)
  {
    currentCachedRenderingTest = DRAW_ARRAYS;
    DebugConsole_consoleOutLine("currentCachedRenderingTest: DRAW_ARRAYS");
  }
  else if (key == GLUT_KEY_F2)
  {
    currentCachedRenderingTest = DRAW_ELEMENTS;
    DebugConsole_consoleOutLine("currentCachedRenderingTest: DRAW_ELEMENTS");
  }
  else if (key == GLUT_KEY_F3)
  {
    currentCachedRenderingTest = DRAW_RANGE_ELEMENTS;
    DebugConsole_consoleOutLine("currentCachedRenderingTest: DRAW_RANGE_ELEMENTS");
  }
  else if (key == GLUT_KEY_UP)
  {
    if (clipPlaneZ <= 0.99f)
    {
      clipPlaneZ += 0.01f;
    }
    else
    {
      clipPlaneZ = 0.999f;
    }
    DebugConsole_consoleOutLine("clipPlaneZ: ", clipPlaneZ);
  }
  else if (key == GLUT_KEY_DOWN)
  {
    if (clipPlaneZ >= -0.99f)
    {
      clipPlaneZ -= 0.01f;
    }
    else
    {
      clipPlaneZ = -0.999f;
    }
    DebugConsole_consoleOutLine("clipPlaneZ: ", clipPlaneZ);
  }

  reInitUIInformation = true;
}

void CubeCappingTest::mouse(int button, int state, int x, int y)
{
  mouseX = float(x);
  mouseY = float(y);

  if (button == GLUT_LEFT_BUTTON)
  {
    if (state == GLUT_DOWN)
    {
      mouseLeftDown = true;
    }
    else if (state == GLUT_UP)
    {
      mouseLeftDown = false;
    }
  }
  else if (button == GLUT_MIDDLE_BUTTON)
  {
    if (state == GLUT_DOWN)
    {
      mouseMiddleDown = true;
    }
    else if (state == GLUT_UP)
    {
      mouseMiddleDown = false;
    }
  }
  else if (button == GLUT_RIGHT_BUTTON)
  {
    if (state == GLUT_DOWN)
    {
      mouseRightDown = true;
    }
    else if (state == GLUT_UP)
    {
      mouseRightDown = false;
    }
  }
  else if ((button == 3) || (button == 4)) // It's a wheel event for FreeGlut
  {
    const double tempCameraDistanceZ = (cameraDistanceZ > 0.0) ? (1.0 + cameraDistanceZ / 10.0) : 1.0;
    if (button == 3)
    {
      cameraDistanceZ -= 0.15 * tempCameraDistanceZ;
    }
    else
    {
      cameraDistanceZ += 0.15 * tempCameraDistanceZ;
    }
  }
}

void CubeCappingTest::mouseMotion(int x, int y)
{
  if (mouseLeftDown)
  {
    cameraAngleY += (double(x) - mouseX);
    cameraAngleX -= (double(y) - mouseY);
    mouseX = double(x);
    mouseY = double(y);
  }
  else if (mouseMiddleDown)
  {
    const double tempCameraDistanceZ = (cameraDistanceZ > 0.0) ? (1.0 + cameraDistanceZ / 5.0) : 1.0;
    cameraDistanceX += 0.005 * (double(x) - mouseX) * tempCameraDistanceZ;
    cameraDistanceY -= 0.005 * (double(y) - mouseY) * tempCameraDistanceZ;
    mouseX = double(x);
    mouseY = double(y);
  }
  else if (mouseRightDown)
  {
    const double tempCameraDistanceZ = (cameraDistanceZ > 0.0) ? (1.0 + cameraDistanceZ / 10.0) : 1.0;
    cameraDistanceZ += 0.05 * (double(y) - mouseY) * tempCameraDistanceZ;
    mouseY = double(y);
  }
}

void CubeCappingTest::closeFunc()
{
  deleteVBOs();

  delete openGLShaderCubeCapping;
  openGLShaderCubeCapping = nullptr;

  if (USE_CUBE_CAPPING_FBO_FOR_RENDERING && openGLDriverInfo->supports_GL_EXT_framebuffer_object() && openGLDriverInfo->supports_GL_EXT_packed_depth_stencil())
  {
    delete openGLFrameBufferObjectForCubeCapping;
    openGLFrameBufferObjectForCubeCapping = nullptr;
  }

  TestAbstractBase::releaseAllGLResources();
}