#include "ConfigFile.h"
#include "UtilityFunctions.h"
#include <sstream>

using namespace std;
using namespace OpenGLRenderingEngineTests;
using namespace Utils::UtilityFunctions;

namespace // anonymous namespace used instead of deprecated 'static' keyword used for cpp variable locality
{
  const string CONFIG_FILE_NAME = "OpenGLRenderingEngine.cfg";
}

string ConfigFile::getConfigFileName()
{
  return CONFIG_FILE_NAME;
}

ConfigFile::ConfigFile()
{
  list<string> configFileLines = StdReadWriteFileFunctions::readTextFile(CONFIG_FILE_NAME);
  if (configFileLines.empty())
  {
    DebugConsole_consoleOutLine("Config file does not exist or is empty.\nNow creating a new config file.\n");
    StdReadWriteFileFunctions::writeTextFile(CONFIG_FILE_NAME, createDefaultConfigFileFromParameters());
  }
  else
  {
    DebugConsole_consoleOutLine("Now parsing config file parameters:\n");
    parseParametersFromConfigFile(configFileLines);
  }
}

string ConfigFile::createDefaultConfigFileFromParameters() const
{
  ostringstream ss;
  ss << "# OpenGLRenderingEngine Version 14.0.0.0 config file, (#) used for commenting"                           << endl;
  ss                                                                                                              << endl;
  ss << "fullScreen = "                    << StringAuxiliaryFunctions::toString<bool>(fullScreen_)               << endl;
  ss << "multiSample = "                   << StringAuxiliaryFunctions::toString<bool>(multiSample_)              << endl;
#ifdef GPU_FRAMEWORK_DEBUG
  ss << "glDebug = "                       << StringAuxiliaryFunctions::toString<bool>(glDebug_)                  << endl;
#endif // GPU_FRAMEWORK_DEBUG
  ss << "test = "                          << test_                                                               << endl;
  ss << "textureFileName = "               << textureFileName_                                                    << endl;
  ss << "modelFileName = "                 << modelFileName_                                                      << endl;
  ss << "useModelLoaderDescriptor = "      << StringAuxiliaryFunctions::toString<bool>(useModelLoaderDescriptor_) << endl;
  ss << "modelLoaderDescriptorFileName = " << modelLoaderDescriptorFileName_                                      << endl;

  return ss.str();
}

void ConfigFile::parseParametersFromConfigFile(const list<string>& configFileLines)
{
  for (auto& line : configFileLines)
  {
    if (!line.empty() && (line[0] != '#')) // skip empty and comment lines
    {
      vector<string> results = StringAuxiliaryFunctions::tokenize<vector<string>>(line, "=");
      results[0] = StringAuxiliaryFunctions::trim(results[0]);
      results[1] = StringAuxiliaryFunctions::trim(results[1]);
      if (results[0] == "fullScreen")
      {
        fullScreen_ = StringAuxiliaryFunctions::fromString<bool>(results[1]);
        DebugConsole_consoleOutLine("fullScreen: ", results[1]);
      }
      else if (results[0] == "multiSample")
      {
        multiSample_ = StringAuxiliaryFunctions::fromString<bool>(results[1]);
        DebugConsole_consoleOutLine("multiSample: ", results[1]);
      }
#ifdef GPU_FRAMEWORK_DEBUG
      else if (results[0] == "glDebug")
      {
        glDebug_ = StringAuxiliaryFunctions::fromString<bool>(results[1]);
        DebugConsole_consoleOutLine("glDebug: ", results[1]);
      }
#endif // GPU_FRAMEWORK_DEBUG
      else if (results[0] == "test")
      {
        test_ = StringAuxiliaryFunctions::fromString<size_t>(results[1]);
        DebugConsole_consoleOutLine("test: ", results[1]);
      }
      else if (results[0] == "textureFileName")
      {
        textureFileName_ = results[1];
        DebugConsole_consoleOutLine("textureFileName: ", results[1]);
      }
      else if (results[0] == "modelFileName")
      {
        modelFileName_ = results[1];
        DebugConsole_consoleOutLine("modelFileName: ", results[1]);
      }
      else if (results[0] == "useModelLoaderDescriptor")
      {
        useModelLoaderDescriptor_ = StringAuxiliaryFunctions::fromString<bool>(results[1]);
        DebugConsole_consoleOutLine("useModelLoaderDescriptor: ", results[1]);
      }
      else if (results[0] == "modelLoaderDescriptorFileName")
      {
        modelLoaderDescriptorFileName_ = results[1];
        DebugConsole_consoleOutLine("modelLoaderDescriptorFileName: ", results[1]);
      }
    }
  }
  DebugConsole_consoleOutLine();
}