// Note that the OpenGLUtilityFunctions.h include already includes glew.h which is crucial for the wglew.h & glxew.h includes below
#include "OpenGLUtilityFunctions.h"
#include <sstream>
#ifdef _WINDOWS
  #include "wglew.h"
#elif !defined(__APPLE__) || defined(GLEW_APPLE_GLX)
  #include "glxew.h"
#endif
#include "UtilityFunctions.h"
#include <cstdlib>

using namespace std;
using namespace OpenGLRenderingEngine::OpenGLUtilityFunctions;
using namespace Utils::UtilityFunctions;

namespace // anonymous namespace used instead of deprecated 'static' keyword used for cpp variable locality
{
  union i10i10i10i2
  {
    struct
    {
      int32_t x : 10;
      int32_t y : 10;
      int32_t z : 10;
      int32_t w : 2;
    } data;
    uint32_t pack;
  };

  inline string glGetGLErrorString(GLenum errnum)
  {
    switch (errnum)
    {
      case GL_INVALID_ENUM:
        return "GL_INVALID_ENUM";

      case GL_INVALID_VALUE:
        return "GL_INVALID_VALUE";

      case GL_INVALID_OPERATION:
        return "GL_INVALID_OPERATION";

      case GL_STACK_OVERFLOW:
        return "GL_STACK_OVERFLOW";

      case GL_STACK_UNDERFLOW:
        return "GL_STACK_UNDERFLOW";

      case GL_INVALID_FRAMEBUFFER_OPERATION:
        return "GL_INVALID_FRAMEBUFFER_OPERATION";

      case GL_OUT_OF_MEMORY:
        return "GL_OUT_OF_MEMORY";

      case GL_NO_ERROR:
        return "GL_NO_ERROR";

      default:
        return "GL_NO_ERROR";
    }
  }
}

void GLAuxiliaryFunctions::findCurrentActiveTextureUnit(int textureValues[3])
{
  // get current active texture
  GLint currentActiveTexture = 0;
  glGetIntegerv(GL_ACTIVE_TEXTURE, &currentActiveTexture);

  for (int textureUnit = 0; textureUnit < 32; ++textureUnit)
  {
    glActiveTexture(GL_TEXTURE0 + textureUnit);
    if (glIsEnabled(GL_TEXTURE_2D))
    {
      textureValues[0] = GL_TRUE;
      textureValues[1] = textureUnit;
      textureValues[2] = currentTexEnvModeGLConstantToShaderEnum();

      break;
    }
  }

  // restore current active texture
  glActiveTexture(currentActiveTexture);
}

int GLAuxiliaryFunctions::currentTexEnvModeGLConstantToShaderEnum()
{
  int texEnvMode = 0;
  glGetTexEnviv(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, &texEnvMode);

  return convertTexEnvModeGLConstantToShaderEnum(texEnvMode);
}

int GLAuxiliaryFunctions::convertTexEnvModeGLConstantToShaderEnum(int texEnvMode)
{
  switch (texEnvMode)
  {
    case GL_MODULATE:
      return 0;

    case GL_DECAL:
      return 1;

    case GL_BLEND:
      return 2;

    case GL_REPLACE:
      return 3;

    case GL_ADD:
      return 4;

    case GL_COMBINE:
      return 5;

    default:
      return 0;
  }
}

int GLAuxiliaryFunctions::getCurrentGLState(GLenum mode)
{
  GLint currentGLMode = 0;
  glGetIntegerv(mode, &currentGLMode);

  return currentGLMode;
}

void GLAuxiliaryFunctions::setVSynch(int enableVSynch, bool isVSyncSupported)
{
  if (isVSyncSupported)
  {
  #ifdef _WINDOWS
    wglSwapIntervalEXT(enableVSynch);
  #else
    glXSwapIntervalSGI(enableVSynch);
  #endif
  }
}

void GLAuxiliaryFunctions::prepareHighQualityRendering(bool isNvidia)
{
  // don't enable/disable the options below to avoid depth related problems
  // glEnable(GL_POINT_SMOOTH);
  // glEnable(GL_LINE_SMOOTH);
  // glEnable(GL_POLYGON_SMOOTH);

  // glHint(GL_POINT_SMOOTH_HINT, GL_NICEST);
  // glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
  // glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);

  glHint(GL_FOG_HINT, GL_NICEST);
  glHint(GL_GENERATE_MIPMAP_HINT, GL_NICEST);
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
  if (isNvidia)
  {
    glHint(GL_FRAGMENT_SHADER_DERIVATIVE_HINT, GL_NICEST); // warning, the AMD/ATI driver does not like this setting, and it's only for OpenGL 2.1 and above
  }
}

void GLAuxiliaryFunctions::prepareLowQualityRendering(bool isNvidia)
{
  // don't enable/disable the options below to avoid depth related problems
  // glDisable(GL_POINT_SMOOTH);
  // glDisable(GL_LINE_SMOOTH);
  // glDisable(GL_POLYGON_SMOOTH);

  // glHint(GL_POINT_SMOOTH_HINT, GL_FASTEST);
  // glHint(GL_LINE_SMOOTH_HINT, GL_FASTEST);
  // glHint(GL_POLYGON_SMOOTH_HINT, GL_FASTEST);

  glHint(GL_FOG_HINT, GL_FASTEST);
  glHint(GL_GENERATE_MIPMAP_HINT, GL_FASTEST);
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST);
  if (isNvidia)
  {
    glHint(GL_FRAGMENT_SHADER_DERIVATIVE_HINT, GL_FASTEST); // warning, the AMD/ATI driver does not like this setting, and it's only for OpenGL 2.1 and above
  }
}

void GLAuxiliaryFunctions::createFullScreenQuad()
{
  const int blendState = getCurrentGLState(GL_BLEND);
  if (blendState) glDisable(GL_BLEND);
  const int depthState = getCurrentGLState(GL_DEPTH_TEST);
  if (depthState) glDisable(GL_DEPTH_TEST);

  // render the quad with normalized texcoords
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
      glBegin(GL_QUADS);
        glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
        glTexCoord2f(0.0f, 0.0f); glVertex2f(-1.0f, -1.0f);
        glTexCoord2f(1.0f, 0.0f); glVertex2f( 1.0f, -1.0f);
        glTexCoord2f(1.0f, 1.0f); glVertex2f( 1.0f,  1.0f);
        glTexCoord2f(0.0f, 1.0f); glVertex2f(-1.0f,  1.0f);
      glEnd();
    glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();

  if (depthState) glEnable(GL_DEPTH_TEST);
  if (blendState) glEnable(GL_BLEND);
}

void GLAuxiliaryFunctions::createFullScreenQuadWithDummyVAO(bool isNvidia, GLuint dummyVao)
{
  const int blendState = getCurrentGLState(GL_BLEND);
  if (blendState) glDisable(GL_BLEND);
  const int depthState = getCurrentGLState(GL_DEPTH_TEST);
  if (depthState) glDisable(GL_DEPTH_TEST);

  glBindVertexArray(dummyVao);
  if (!isNvidia) glEnableVertexAttribArray(0);  // dummy VAOs fix for non-NVidia hardware
  glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
  if (!isNvidia) glDisableVertexAttribArray(0); // dummy VAOs fix for non-NVidia hardware
  glBindVertexArray(0);

  if (depthState) glEnable(GL_DEPTH_TEST);
  if (blendState) glEnable(GL_BLEND);
}

uint32_t GLAuxiliaryFunctions::packNormalToUInt(float x, float y, float z)
{
  const i10i10i10i2 value{ { int32_t(round(MathFunctions::clamp(x, -1.0f, 1.0f) * 511.0f)),
                             int32_t(round(MathFunctions::clamp(y, -1.0f, 1.0f) * 511.0f)),
                             int32_t(round(MathFunctions::clamp(z, -1.0f, 1.0f) * 511.0f)),
                             0
                         } };
  return value.pack;
}

void GLAuxiliaryFunctions::checkGLErrorImpl(const char* file, const char* function, int line, GLenum errnum)
{
  if (errnum)
  {
    ostringstream ss;
    ss << "\nFile:" << file << endl;
    ss << "Function:" << function << endl;
    ss << "Line:" << line << endl;
    ss << "GL error reported: " << glGetGLErrorString(errnum) << endl << endl;
    DebugConsole_consoleOutLine(ss.str());
  }
}