#include "OpenGLUniqueColorsGenerator.h"
#include "UtilityFunctions.h"
#include <array>
#include <set>

using namespace std;
using namespace OpenGLRenderingEngine;
using namespace Utils::VectorTypes;
using namespace Utils::UtilityFunctions;

namespace // anonymous namespace used instead of deprecated 'static' keyword used for cpp variable locality
{
  /**
  *  Calculates all the prime numbers up to given threshold.
  */
  inline vector<size_t> findPrimeNumbersUpToThreshold(size_t threshold)
  {
    vector<size_t> allPrimeNumbers;
    size_t x = 0;
    size_t y = 0;
    for (x = 2; x < threshold; ++x)
    {
      if (((x & 1) != 0) || (x == 2))
      {
        for (y = 2; y <= x / 2; ++y)
        {
          if ((x % y) == 0)
          {
            break;
          }
        }

        if (y > x / 2)
        {
          allPrimeNumbers.emplace_back(x);
        }
      }
    }

    return allPrimeNumbers;
  }

  const size_t COLOR_UPPER_RANGE               = (1 << 8) - 1; // implies '255'
  const vector<size_t> COLOR_PRIME_NUMBERS     = findPrimeNumbersUpToThreshold(COLOR_UPPER_RANGE);
  const float COLOR_LOWER_INTENSITY_PERCENTAGE = 0.05f;
  const float COLOR_UPPER_INTENSITY_PERCENTAGE = 0.90f;
  const size_t LOWER_THRESHOLD                 = size_t(255.0f * COLOR_LOWER_INTENSITY_PERCENTAGE);
  const size_t UPPER_THRESHOLD                 = size_t(255.0f * COLOR_UPPER_INTENSITY_PERCENTAGE);
  const size_t PRIME_NUMBERS_INDEX_LENGTH      = size_t(COLOR_UPPER_INTENSITY_PERCENTAGE * COLOR_PRIME_NUMBERS.size());
  const size_t COLOR_LOOKUP_THRESHOLD          = 1000;

  // struct comparing float4 colors in the set
  struct ColorSetComparator
  {
    inline bool operator()(const float4& color1, const float4& color2) const noexcept
    {
      if (!MathFunctions::equal(color1.x, color2.x)) return (color1.x < color2.x);
      if (!MathFunctions::equal(color1.y, color2.y)) return (color1.y < color2.y);
      if (!MathFunctions::equal(color1.z, color2.z)) return (color1.z < color2.z);
      if (!MathFunctions::equal(color1.w, color2.w)) return (color1.w < color2.w);

      return false;
    }
  };

  size_t getRandomRange(double randomValue, size_t minimum, size_t maximum)
  {
    return size_t(randomValue * (maximum - minimum + 1) + minimum);
  }

  float4 createRandomColor(double randomValue1, double randomValue2, double randomValue3)
  {
    return float4{ getRandomRange(randomValue1, LOWER_THRESHOLD, UPPER_THRESHOLD) / 255.0f,
                   getRandomRange(randomValue2, LOWER_THRESHOLD, UPPER_THRESHOLD) / 255.0f,
                   getRandomRange(randomValue3, LOWER_THRESHOLD, UPPER_THRESHOLD) / 255.0f,
                   1.0f
                 };
  }

  /**
  *  Creates one unique color based on a prime number generator, same color hues reproducible for every run.
  *  Parameters primeNumberIndex1, primeNumberIndex2 & primeNumberIndex3 need to be used by reference, as they are being chanhed in this function.
  */
  float4 createColorBasedOnPrimeNumbers(size_t* __restrict colorIndicesRGB, size_t& primeNumberIndex1, size_t& primeNumberIndex2, size_t& primeNumberIndex3, bool incrementPrimeNumberIndices)
  {
    if (incrementPrimeNumberIndices)
    {
      if (++primeNumberIndex1 >= PRIME_NUMBERS_INDEX_LENGTH)
      {
        primeNumberIndex1 = 0;
      }
      if (++primeNumberIndex2 >= PRIME_NUMBERS_INDEX_LENGTH)
      {
        primeNumberIndex2 = 0;
      }
      if (++primeNumberIndex3 >= PRIME_NUMBERS_INDEX_LENGTH)
      {
        primeNumberIndex3 = 0;
      }
    }

    colorIndicesRGB[0] += COLOR_PRIME_NUMBERS[primeNumberIndex1];
    colorIndicesRGB[1] += COLOR_PRIME_NUMBERS[primeNumberIndex2];
    colorIndicesRGB[2] += COLOR_PRIME_NUMBERS[primeNumberIndex3];

    if (colorIndicesRGB[0] >= UPPER_THRESHOLD)
    {
      colorIndicesRGB[0] = (LOWER_THRESHOLD + (colorIndicesRGB[0] - UPPER_THRESHOLD)) & 255; // means module 256 to avoid out-of-bounds errors
    }
    if (colorIndicesRGB[1] >= UPPER_THRESHOLD)
    {
      colorIndicesRGB[1] = (LOWER_THRESHOLD + (colorIndicesRGB[1] - UPPER_THRESHOLD)) & 255; // means module 256 to avoid out-of-bounds errors
    }
    if (colorIndicesRGB[2] >= UPPER_THRESHOLD)
    {
      colorIndicesRGB[2] = (LOWER_THRESHOLD + (colorIndicesRGB[2] - UPPER_THRESHOLD)) & 255; // means module 256 to avoid out-of-bounds errors
    }

    return float4{ colorIndicesRGB[0] / 255.0f, colorIndicesRGB[1] / 255.0f, colorIndicesRGB[2] / 255.0f, 1.0f };
  }
}

vector<float4> OpenGLUniqueColorsGenerator::createUniqueColorsBasedOnPrimeNumbers(size_t numberOfColors)
{
  size_t sameColours                = 0;
  const size_t colorRange           = UPPER_THRESHOLD - LOWER_THRESHOLD;
  const size_t RGBColorCombinations = colorRange * colorRange * colorRange;
  array<size_t, 3> colorIndicesRGB  = { { (UPPER_THRESHOLD - LOWER_THRESHOLD) / 2 + COLOR_PRIME_NUMBERS[primeNumberIndex1_],
                                          (UPPER_THRESHOLD - LOWER_THRESHOLD) / 4 + COLOR_PRIME_NUMBERS[primeNumberIndex2_],
                                          (UPPER_THRESHOLD - LOWER_THRESHOLD) / 2 + COLOR_PRIME_NUMBERS[primeNumberIndex3_]
                                      } };
  set<float4, ColorSetComparator> sensorColorsSet;
  vector<float4> sensorColorsResult;
  for (size_t i = 0; i < numberOfColors; ++i)
  {
    sameColours = 0;
    float4 sensorColor = createColorBasedOnPrimeNumbers(colorIndicesRGB.data(), primeNumberIndex1_, primeNumberIndex2_, primeNumberIndex3_, false);
    auto result = sensorColorsSet.emplace(sensorColor);
    while (!result.second && sensorColorsSet.size() <= RGBColorCombinations) // if sensorColor not emplaced, element was already in
    {
      if (++sameColours > COLOR_LOOKUP_THRESHOLD)
      {
        sameColours = 0;
        sensorColor = createRandomColor(random_(), random_(), random_());
      }
      else
      {
        sensorColor = createColorBasedOnPrimeNumbers(colorIndicesRGB.data(), primeNumberIndex1_, primeNumberIndex2_, primeNumberIndex3_, true);
      }
      result = sensorColorsSet.emplace(sensorColor); // re-emplace the newly generated color
    }
    sensorColorsResult.emplace_back(sensorColor);
  }

  return sensorColorsResult;
}