#include "OpenGLShaderProgram.h"
#include "OpenGLDriverInfo.h"
#include "OpenGLShaderGLSLPreProcessorCommands.h"
#include "OpenGLShaderCompileAndLink.h"

using namespace std;
using namespace OpenGLRenderingEngine;

OpenGLShaderProgram::OpenGLShaderProgram(OpenGLDriverInfo* openGLDriverInfo, bool enableVertexProgramTwoSidedLighting) noexcept
  : openGLDriverInfo_(openGLDriverInfo)
{
  if (enableVertexProgramTwoSidedLighting)
  {
    glEnable(GL_VERTEX_PROGRAM_TWO_SIDE);
  }
  else
  {
    glDisable(GL_VERTEX_PROGRAM_TWO_SIDE);
  }

  createGLShaderProgram();

  openGLShaderGLSLPreProcessorCommands_ = new OpenGLShaderGLSLPreProcessorCommands(openGLDriverInfo_);
  openGLShaderCompileAndLink_           = new OpenGLShaderCompileAndLink(openGLDriverInfo_, openGLShaderGLSLPreProcessorCommands_, shaderProgram_);
}

OpenGLShaderProgram::~OpenGLShaderProgram() noexcept
{
  disableShaderProgram();

  if (openGLShaderGLSLPreProcessorCommands_)
  {
    delete openGLShaderGLSLPreProcessorCommands_;
    openGLShaderGLSLPreProcessorCommands_ = nullptr;
  }

  if (openGLShaderCompileAndLink_)
  {
    delete openGLShaderCompileAndLink_; // will also deallocate all GL shader objects
    openGLShaderCompileAndLink_ = nullptr;
  }

  releaseGLShaderProgram();
}

void OpenGLShaderProgram::createGLShaderProgram()
{
  if (!glIsProgram(shaderProgram_))
  {
    shaderProgram_ = glCreateProgram();
  }
}

GLint OpenGLShaderProgram::getUniformLocation(const string& name)
{
  const auto& position = allUniformLocationsMap_.find(name);
  if (position == allUniformLocationsMap_.end())
  {
    allUniformLocationsMap_.emplace(name, glGetUniformLocation(shaderProgram_, name.c_str()));
  }

  return allUniformLocationsMap_[name];
}

void OpenGLShaderProgram::setUniform1i(const string& name, GLint value)
{
  GLint location;
  if ((location = getUniformLocation(name)) >= 0)
  {
    glUniform1i(location, value);
  }
}

void OpenGLShaderProgram::setUniform1iv(const string& name, GLsizei count, const GLint* values)
{
  GLint location;
  if ((location = getUniformLocation(name)) >= 0)
  {
    glUniform1iv(location, count, values);
  }
}

void OpenGLShaderProgram::setUniform2iv(const string& name, GLsizei count, const GLint* values)
{
  GLint location;
  if ((location = getUniformLocation(name)) >= 0)
  {
    glUniform2iv(location, count, values);
  }
}

void OpenGLShaderProgram::setUniform3iv(const string& name, GLsizei count, const GLint* values)
{
  GLint location;
  if ((location = getUniformLocation(name)) >= 0)
  {
    glUniform3iv(location, count, values);
  }
}

void OpenGLShaderProgram::setUniform4iv(const string& name, GLsizei count, const GLint* values)
{
  GLint location;
  if ((location = getUniformLocation(name)) >= 0)
  {
    glUniform4iv(location, count, values);
  }
}

void OpenGLShaderProgram::setUniform1ui(const string& name, GLuint value)
{
  GLint location;
  if ((location = getUniformLocation(name)) >= 0)
  {
    glUniform1ui(location, value);
  }
}

void OpenGLShaderProgram::setUniform1uiv(const string& name, GLsizei count, const GLuint* values)
{
  GLint location;
  if ((location = getUniformLocation(name)) >= 0)
  {
    glUniform1uiv(location, count, values);
  }
}

void OpenGLShaderProgram::setUniform2uiv(const string& name, GLsizei count, const GLuint* values)
{
  GLint location;
  if ((location = getUniformLocation(name)) >= 0)
  {
    glUniform2uiv(location, count, values);
  }
}

void OpenGLShaderProgram::setUniform3uiv(const string& name, GLsizei count, const GLuint* values)
{
  GLint location;
  if ((location = getUniformLocation(name)) >= 0)
  {
    glUniform3uiv(location, count, values);
  }
}

void OpenGLShaderProgram::setUniform4uiv(const string& name, GLsizei count, const GLuint* values)
{
  GLint location;
  if ((location = getUniformLocation(name)) >= 0)
  {
    glUniform4uiv(location, count, values);
  }
}

void OpenGLShaderProgram::setUniform1i64NV(const string& name, GLint64EXT value)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getUniformLocation(name)) >= 0)
    {
      glUniform1i64NV(location, value);
    }
  }
}

void OpenGLShaderProgram::setUniform1i64vNV(const string& name, GLsizei count, const GLint64EXT* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getUniformLocation(name)) >= 0)
    {
      glUniform1i64vNV(location, count, values);
    }
  }
}

void OpenGLShaderProgram::setUniform2i64vNV(const string& name, GLsizei count, const GLint64EXT* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getUniformLocation(name)) >= 0)
    {
      glUniform2i64vNV(location, count, values);
    }
  }
}

void OpenGLShaderProgram::setUniform3i64vNV(const string& name, GLsizei count, const GLint64EXT* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getUniformLocation(name)) >= 0)
    {
      glUniform3i64vNV(location, count, values);
    }
  }
}

void OpenGLShaderProgram::setUniform4i64vNV(const string& name, GLsizei count, const GLint64EXT* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getUniformLocation(name)) >= 0)
    {
      glUniform4i64vNV(location, count, values);
    }
  }
}

void OpenGLShaderProgram::setUniform1ui64NV(const string& name, GLuint64EXT value)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getUniformLocation(name)) >= 0)
    {
      glUniform1ui64NV(location, value);
    }
  }
}

void OpenGLShaderProgram::setUniform1ui64vNV(const string& name, GLsizei count, const GLuint64EXT* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getUniformLocation(name)) >= 0)
    {
      glUniform1ui64vNV(location, count, values);
    }
  }
}

void OpenGLShaderProgram::setUniform2ui64vNV(const string& name, GLsizei count, const GLuint64EXT* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getUniformLocation(name)) >= 0)
    {
      glUniform2ui64vNV(location, count, values);
    }
  }
}

void OpenGLShaderProgram::setUniform3ui64vNV(const string& name, GLsizei count, const GLuint64EXT* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getUniformLocation(name)) >= 0)
    {
      glUniform3ui64vNV(location, count, values);
    }
  }
}

void OpenGLShaderProgram::setUniform4ui64vNV(const string& name, GLsizei count, const GLuint64EXT* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getUniformLocation(name)) >= 0)
    {
      glUniform4ui64vNV(location, count, values);
    }
  }
}

void OpenGLShaderProgram::setUniform1f(const string& name, GLfloat value)
{
  GLint location;
  if ((location = getUniformLocation(name)) >= 0)
  {
    glUniform1f(location, value);
  }
}

void OpenGLShaderProgram::setUniform1fv(const string& name, GLsizei count, const GLfloat* values)
{
  GLint location;
  if ((location = getUniformLocation(name)) >= 0)
  {
    glUniform1fv(location, count, values);
  }
}

void OpenGLShaderProgram::setUniform2fv(const string& name, GLsizei count, const GLfloat* values)
{
  GLint location;
  if ((location = getUniformLocation(name)) >= 0)
  {
    glUniform2fv(location, count, values);
  }
}

void OpenGLShaderProgram::setUniform3fv(const string& name, GLsizei count, const GLfloat* values)
{
  GLint location;
  if ((location = getUniformLocation(name)) >= 0)
  {
    glUniform3fv(location, count, values);
  }
}

void OpenGLShaderProgram::setUniform4fv(const string& name, GLsizei count, const GLfloat* values)
{
  GLint location;
  if ((location = getUniformLocation(name)) >= 0)
  {
    glUniform4fv(location, count, values);
  }
}

void OpenGLShaderProgram::setUniform1d(const string& name, GLdouble value)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getUniformLocation(name)) >= 0)
    {
      glUniform1d(location, value);
    }
  }
}

void OpenGLShaderProgram::setUniform1dv(const string& name, GLsizei count, const GLdouble* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getUniformLocation(name)) >= 0)
    {
      glUniform1dv(location, count, values);
    }
  }
}

void OpenGLShaderProgram::setUniform2dv(const string& name, GLsizei count, const GLdouble* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getUniformLocation(name)) >= 0)
    {
      glUniform2dv(location, count, values);
    }
  }
}

void OpenGLShaderProgram::setUniform3dv(const string& name, GLsizei count, const GLdouble* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getUniformLocation(name)) >= 0)
    {
      glUniform3dv(location, count, values);
    }
  }
}

void OpenGLShaderProgram::setUniform4dv(const string& name, GLsizei count, const GLdouble* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getUniformLocation(name)) >= 0)
    {
      glUniform4dv(location, count, values);
    }
  }
}

GLint OpenGLShaderProgram::getAttributeLocation(const string& name)
{
  const auto& position = allAttribLocationsMap_.find(name);
  if (position == allAttribLocationsMap_.end())
  {
    allAttribLocationsMap_.emplace(name, glGetAttribLocation(shaderProgram_, name.c_str()));
  }

  return allAttribLocationsMap_[name];
}

void OpenGLShaderProgram::setAttribute1i(const string& name, GLint value)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttribI1i(location, value);
  }
}

void OpenGLShaderProgram::setAttribute1iv(const string& name, const GLint* values)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttribI1iv(location, values);
  }
}

void OpenGLShaderProgram::setAttribute2iv(const string& name, const GLint* values)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttribI2iv(location, values);
  }
}

void OpenGLShaderProgram::setAttribute3iv(const string& name, const GLint* values)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttribI3iv(location, values);
  }
}

void OpenGLShaderProgram::setAttribute4iv(const string& name, const GLint* values)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttribI4iv(location, values);
  }
}

void OpenGLShaderProgram::setAttribute1ui(const string& name, GLuint value)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttribI1ui(location, value);
  }
}

void OpenGLShaderProgram::setAttribute1uiv(const string& name, const GLuint* values)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttribI1uiv(location, values);
  }
}

void OpenGLShaderProgram::setAttribute2uiv(const string& name, const GLuint* values)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttribI2uiv(location, values);
  }
}

void OpenGLShaderProgram::setAttribute3uiv(const string& name, const GLuint* values)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttribI3uiv(location, values);
  }
}

void OpenGLShaderProgram::setAttribute4uiv(const string& name, const GLuint* values)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttribI4uiv(location, values);
  }
}

void OpenGLShaderProgram::setAttributeL1i64NV(const string& name, GLint64EXT value)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getAttributeLocation(name)) >= 0)
    {
      glVertexAttribL1i64NV(location, value);
    }
  }
}

void OpenGLShaderProgram::setAttributeL1i64vNV(const string& name, const GLint64EXT* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getAttributeLocation(name)) >= 0)
    {
      glVertexAttribL1i64vNV(location, values);
    }
  }
}

void OpenGLShaderProgram::setAttributeL2i64vNV(const string& name, const GLint64EXT* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getAttributeLocation(name)) >= 0)
    {
      glVertexAttribL2i64vNV(location, values);
    }
  }
}

void OpenGLShaderProgram::setAttributeL3i64vNV(const string& name, const GLint64EXT* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getAttributeLocation(name)) >= 0)
    {
      glVertexAttribL3i64vNV(location, values);
    }
  }
}

void OpenGLShaderProgram::setAttributeL4i64vNV(const string& name, const GLint64EXT* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getAttributeLocation(name)) >= 0)
    {
      glVertexAttribL4i64vNV(location, values);
    }
  }
}

void OpenGLShaderProgram::setAttributeL1ui64NV(const string& name, GLuint64EXT value)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getAttributeLocation(name)) >= 0)
    {
      glVertexAttribL1ui64NV(location, value);
    }
  }
}

void OpenGLShaderProgram::setAttributeL1ui64vNV(const string& name, const GLuint64EXT* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getAttributeLocation(name)) >= 0)
    {
      glVertexAttribL1ui64vNV(location, values);
    }
  }
}

void OpenGLShaderProgram::setAttributeL2ui64vNV(const string& name, const GLuint64EXT* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getAttributeLocation(name)) >= 0)
    {
      glVertexAttribL2ui64vNV(location, values);
    }
  }
}

void OpenGLShaderProgram::setAttributeL3ui64vNV(const string& name, const GLuint64EXT* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getAttributeLocation(name)) >= 0)
    {
      glVertexAttribL3ui64vNV(location, values);
    }
  }
}

void OpenGLShaderProgram::setAttributeL4ui64vNV(const string& name, const GLuint64EXT* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getAttributeLocation(name)) >= 0)
    {
      glVertexAttribL4ui64vNV(location, values);
    }
  }
}

void OpenGLShaderProgram::setAttributeP1ui(const string& name, GLenum type, GLboolean normalized, GLuint value)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttribP1ui(location, type, normalized, value);
  }
}

void OpenGLShaderProgram::setAttributeP1uiv(const string& name, GLenum type, GLboolean normalized, const GLuint* values)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttribP1uiv(location, type, normalized, values);
  }
}

void OpenGLShaderProgram::setAttributeP2uiv(const string& name, GLenum type, GLboolean normalized, const GLuint* values)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttribP2uiv(location, type, normalized, values);
  }
}

void OpenGLShaderProgram::setAttributeP3uiv(const string& name, GLenum type, GLboolean normalized, const GLuint* values)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttribP3uiv(location, type, normalized, values);
  }
}

void OpenGLShaderProgram::setAttributeP4uiv(const string& name, GLenum type, GLboolean normalized, const GLuint* values)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttribP4uiv(location, type, normalized, values);
  }
}

void OpenGLShaderProgram::setAttribute1f(const string& name, GLfloat value)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttrib1f(location, value);
  }
}

void OpenGLShaderProgram::setAttribute1fv(const string& name, const GLfloat* values)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttrib1fv(location, values);
  }
}

void OpenGLShaderProgram::setAttribute2fv(const string& name, const GLfloat* values)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttrib2fv(location, values);
  }
}

void OpenGLShaderProgram::setAttribute3fv(const string& name, const GLfloat* values)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttrib3fv(location, values);
  }
}

void OpenGLShaderProgram::setAttribute4fv(const string& name, const GLfloat* values)
{
  GLint location;
  if ((location = getAttributeLocation(name)) >= 0)
  {
    glVertexAttrib4fv(location, values);
  }
}

void OpenGLShaderProgram::setAttribute1d(const string& name, GLdouble value)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getAttributeLocation(name)) >= 0)
    {
      glVertexAttrib1d(location, value);
    }
  }
}

void OpenGLShaderProgram::setAttribute1dv(const string& name, const GLdouble* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getAttributeLocation(name)) >= 0)
    {
      glVertexAttrib1dv(location, values);
    }
  }
}

void OpenGLShaderProgram::setAttribute2dv(const string& name, const GLdouble* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getAttributeLocation(name)) >= 0)
    {
      glVertexAttrib2dv(location, values);
    }
  }
}

void OpenGLShaderProgram::setAttribute3dv(const string& name, const GLdouble* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getAttributeLocation(name)) >= 0)
    {
      glVertexAttrib3dv(location, values);
    }
  }
}

void OpenGLShaderProgram::setAttribute4dv(const string& name, const GLdouble* values)
{
  if (openGLDriverInfo_->supports400Shaders())
  {
    GLint location;
    if ((location = getAttributeLocation(name)) >= 0)
    {
      glVertexAttrib4dv(location, values);
    }
  }
}

void OpenGLShaderProgram::enableShaderProgram() const
{
  if (glIsProgram(shaderProgram_))
  {
    glUseProgram(shaderProgram_);
  }
}

void OpenGLShaderProgram::disableShaderProgram() const
{
  glUseProgram(0);
}

void OpenGLShaderProgram::releaseGLShaderProgram() const
{
  if (glIsProgram(shaderProgram_))
  {
    glDeleteProgram(shaderProgram_);
  }
}