#include "OpenGLShaderCompileAndLink.h"
#include "OpenGLAssetManager.h"
#include "EnvironmentConfig.h"
#include "UtilityFunctions.h"
#include "AllGLSLShaderFiles.h"
#include "OpenGLDriverInfo.h"
#include "OpenGLShaderGLSLPreProcessorCommands.h"
#include "OpenGLShaderObjects.h"
#include <tuple>
#include <memory>

using namespace std;
using namespace Utils;
using namespace Utils::UtilityFunctions;
using namespace OpenGLRenderingEngine;
using namespace OpenGLRenderingEngine::GLSLShaderFiles;

namespace // anonymous namespace used instead of deprecated 'static' keyword used for cpp variable locality
{
  // const string for empty string
  const string EMPTY_STRING = "";

  inline void createShader(int shaderTypeEnum, GLenum glShaderType, OpenGLShaderObjects* openGLShaderObjects)
  {
    GLuint shaderObject = 0;
    if (openGLShaderObjects->isUsingShaderType(shaderTypeEnum))
    {
      shaderObject = glCreateShader(glShaderType);
    }

    if (shaderObject > 0)
    {
      openGLShaderObjects->setShaderObject(shaderObject, shaderTypeEnum);
    }
  }

  inline void releaseShaderObject(int shaderTypeEnum, const OpenGLShaderObjects* openGLShaderObjects)
  {
    if (openGLShaderObjects->isUsingShaderType(shaderTypeEnum))
    {
      const GLuint shaderObject = openGLShaderObjects->getShaderObject(shaderTypeEnum);
      if (glIsShader(shaderObject))
      {
        glDeleteShader(shaderObject);
      }
    }
  }

#ifdef GPU_FRAMEWORK_GLSL_EXTERNAL_FILES
  /**
  *  Reads the shader file.
  */
  inline string readShaderFile(const string& GLSLShadersFilePath, const string& shaderLibraryPathName, const string& shaderLibraryName)
  {
    string fullShaderLibraryNameWithPath(GLSLShadersFilePath + shaderLibraryPathName + "/" + shaderLibraryName);
    ifstream in;
    in.open(fullShaderLibraryNameWithPath.c_str(), ios::in);
    if (!StdReadWriteFileFunctions::assure(in, fullShaderLibraryNameWithPath))
    {
      return EMPTY_STRING;
    }

    string line;
    ostringstream ss;
    while (getline(in, line)) // getline() removes '\n'
    {
      ss << line << endl; // must add end-of-line
    }
    ss << '\0'; // terminate shader file
    in.close();

    return ss.str();
  }
#else
  inline string decodeLine(const string& line)
  {
    switch (OpenGLAssetManager::CHARACTER_ENCODING_METHOD)
    {
      case OpenGLAssetManager::NONE:
        return line;

      case OpenGLAssetManager::BASE64:
        return Base64CompressorScrambler::decodeBase64String(line);

      case OpenGLAssetManager::FLIP_BITS:
        return Base64CompressorScrambler::flipString(line);

      case OpenGLAssetManager::FLIP_XOR_SWAP_BITS:
        return Base64CompressorScrambler::xorSwapString(Base64CompressorScrambler::flipString(line));

      case OpenGLAssetManager::BASE64_FLIP_XOR_SWAP_BITS:
        return Base64CompressorScrambler::xorSwapString(Base64CompressorScrambler::flipString(Base64CompressorScrambler::decodeBase64String(line)));

      case OpenGLAssetManager::BASE64_COMPRESSION:
        return Base64CompressorScrambler::decompressString(Base64CompressorScrambler::decodeBase64String(line));

      case OpenGLAssetManager::BASE64_FLIP_XOR_SWAP_BITS_COMPRESSION:
        return Base64CompressorScrambler::decompressString(Base64CompressorScrambler::xorSwapString(Base64CompressorScrambler::flipString(Base64CompressorScrambler::decodeBase64String(line))));

      default:
        return line;
    }
  }

  inline string unscrambleShaderFileLines(const string& name)
  {
    const auto& shaderFileLinesTuple = AllGLSLShaderFiles::getSingleton().getShader(name);
    const auto& shaderFileLines      = get<0>(shaderFileLinesTuple);
    const size_t shaderFileLinesSize = get<1>(shaderFileLinesTuple);
    ostringstream ss;
    for (size_t i = 0; i < shaderFileLinesSize; ++i)
    {
      ss << decodeLine(string(shaderFileLines[i])) << '\n';
    }
    ss << '\0'; // terminate shader file

    return ss.str();
  }
#endif // GPU_FRAMEWORK_GLSL_EXTERNAL_FILES
}

OpenGLShaderCompileAndLink::OpenGLShaderCompileAndLink(OpenGLDriverInfo* openGLDriverInfo, OpenGLShaderGLSLPreProcessorCommands* openGLShaderGLSLPreProcessorCommands, GLuint shaderProgram) noexcept
  : openGLDriverInfo_(openGLDriverInfo)
  , openGLShaderGLSLPreProcessorCommands_(openGLShaderGLSLPreProcessorCommands)
  , shaderProgram_(shaderProgram)
{
}

OpenGLShaderCompileAndLink::~OpenGLShaderCompileAndLink() noexcept
{
  releaseAllShaderObjects();
}

void OpenGLShaderCompileAndLink::addShaderLibraryToProgram(const string& shaderLibraryPathName, const string& shaderLibraryName, int shaderType)
{
  string shaderLibraryStringKey = shaderLibraryPathName + "_" + shaderLibraryName;
  const auto& position = allOpenGLShaderObjectsMap_.find(shaderLibraryStringKey);
  if (position != allOpenGLShaderObjectsMap_.end())
  {
    DebugConsole_consoleOutLine("This Shader Library has already been added:\n", shaderLibraryStringKey);
    return;
  }

  // shader objects storing creation
  OpenGLShaderObjects* openGLShaderObjects = new OpenGLShaderObjects(shaderType);

  // VS creation part
  createShader(OpenGLAssetManager::VS,  GL_VERTEX_SHADER,          openGLShaderObjects);

  // TCS creation part
  createShader(OpenGLAssetManager::TCS, GL_TESS_CONTROL_SHADER,    openGLShaderObjects);

  // TES creation part
  createShader(OpenGLAssetManager::TES, GL_TESS_EVALUATION_SHADER, openGLShaderObjects);

  // GS creation part
  createShader(OpenGLAssetManager::GS,  GL_GEOMETRY_SHADER,        openGLShaderObjects);

  // FS creation part
  createShader(OpenGLAssetManager::FS,  GL_FRAGMENT_SHADER,        openGLShaderObjects);

  // CS creation part
  createShader(OpenGLAssetManager::CS,  GL_COMPUTE_SHADER,         openGLShaderObjects);

  // store in shader objects map if shader objects were created && if the number of the shader type programmable stages equals the created shader objects
  if (openGLShaderObjects->hasCreatedShaderObjects() && openGLShaderObjects->isEqualNumberOfShaderTypeProgrammableStagesAndCreatedShaderObjects())
  {
    allOpenGLShaderObjectsMap_.emplace(move(shaderLibraryStringKey), openGLShaderObjects);
  }
  else
  {
    DebugConsole_consoleOutLine("Error: No shader object was created or the number of the shader type programmable stages do not match the created shader objects.\nShader loading process now aborted.");
    delete openGLShaderObjects;
    openGLShaderObjects = nullptr;

    return;
  }

  // VS compilation part
  compileShader(shaderLibraryPathName, shaderLibraryName, OpenGLAssetManager::getVertexShadersFileNameExtension(),                 OpenGLAssetManager::getVertexShadersFileName(),                 OpenGLAssetManager::VS,  openGLShaderObjects, "Vertex File '");

  // TCS compilation part
  compileShader(shaderLibraryPathName, shaderLibraryName, OpenGLAssetManager::getTessellationControlShadersFileNameExtension(),    OpenGLAssetManager::getTessellationControlShadersFileName(),    OpenGLAssetManager::TCS, openGLShaderObjects, "Tessellation Control File '");

  // TES compilation part
  compileShader(shaderLibraryPathName, shaderLibraryName, OpenGLAssetManager::getTessellationEvaluationShadersFileNameExtension(), OpenGLAssetManager::getTessellationEvaluationShadersFileName(), OpenGLAssetManager::TES, openGLShaderObjects, "Tessellation Evaluation File '");

  // GS compilation part
  compileShader(shaderLibraryPathName, shaderLibraryName, OpenGLAssetManager::getGeometryShadersFileNameExtension(),               OpenGLAssetManager::getGeometryShadersFileName(),               OpenGLAssetManager::GS,  openGLShaderObjects, "Geometry File '");

  // FS compilation part
  compileShader(shaderLibraryPathName, shaderLibraryName, OpenGLAssetManager::getFragmentShadersFileNameExtension(),               OpenGLAssetManager::getFragmentShadersFileName(),               OpenGLAssetManager::FS,  openGLShaderObjects, "Fragment File '");

  // CS compilation part
  compileShader(shaderLibraryPathName, shaderLibraryName, OpenGLAssetManager::getComputeShadersFileNameExtension(),                OpenGLAssetManager::getComputeShadersFileName(),                OpenGLAssetManager::CS,  openGLShaderObjects, "Compute File '");

  // attach all shader objects to program
  if (shaderProgram_ > 0)
  {
    openGLShaderObjects->attachShaderObjectsToProgram(shaderProgram_);
  }

#ifdef GPU_FRAMEWORK_GLSL_EXTERNAL_FILES
  if (mergedShaderLibraryName_.empty())
  {
    mergedShaderLibraryName_ = shaderLibraryName;
  }
  else
  {
    mergedShaderLibraryName_ += " + " + shaderLibraryName;
  }
#endif // GPU_FRAMEWORK_GLSL_EXTERNAL_FILES
}

void OpenGLShaderCompileAndLink::compileShader(const string& shaderLibraryPathName, const string& shaderLibraryName, const string& shaderFileNameExtension, const string& shaderFileName, int shaderTypeEnum, const OpenGLShaderObjects* openGLShaderObjects, const string& shaderTypeString) const
{
  const GLuint shaderObject = openGLShaderObjects->getShaderObject(shaderTypeEnum);
  if (shaderObject > 0)
  {
#ifdef GPU_FRAMEWORK_GLSL_EXTERNAL_FILES
    string shaderSource = openGLShaderGLSLPreProcessorCommands_->getFinalizedGLSLPreProcessorCommands() + readShaderFile(OpenGLAssetManager::getGLSLDefaultDirectory(), shaderLibraryPathName, shaderLibraryName + shaderFileNameExtension);
#else
    string shaderSource = _openGLShaderGLSLPreProcessorCommands->getFinalizedGLSLPreProcessorCommands() + unscrambleShaderFileLines(shaderLibraryPathName + "_" + shaderLibraryName + "::" + shaderFileName);
#endif // GPU_FRAMEWORK_GLSL_EXTERNAL_FILES
    if (!shaderSource.empty())
    {
      const GLchar* shaderSourceString = shaderSource.c_str();
      GLint shaderSourceLength = GLint(shaderSource.size());
      glShaderSource(shaderObject, 1, &shaderSourceString, &shaderSourceLength);
      glCompileShader(shaderObject);
#ifdef GPU_FRAMEWORK_GLSL_EXTERNAL_FILES
      checkInfoLog(shaderTypeString + shaderLibraryName + shaderFileNameExtension + "'", shaderObject);
#endif // GPU_FRAMEWORK_GLSL_EXTERNAL_FILES
    }
  }
}

bool OpenGLShaderCompileAndLink::checkUsageOfGeometryShaderObject()
{
  for (const auto& mapEntry : allOpenGLShaderObjectsMap_)
  {
    OpenGLShaderObjects* openGLShaderObjects = mapEntry.second;
    if (openGLShaderObjects->isUsingShaderType(OpenGLAssetManager::GS))
    {
      return true;
    }
  }

  return false;
}

void OpenGLShaderCompileAndLink::linkShaderProgram(GLint inputTopology, GLint outputTopology, GLint maxVerticesOut)
{
  if (shaderProgram_ > 0)
  {
    if (checkUsageOfGeometryShaderObject())
    {
      if ((inputTopology != GL_POINTS) && (inputTopology != GL_LINES) && (inputTopology != GL_LINES_ADJACENCY) && (inputTopology != GL_TRIANGLES) && (inputTopology != GL_TRIANGLES_ADJACENCY))
      {
        DebugConsole_consoleOutLine("Error: You have not specified a supported Geometry Shader Input Topology.\nShader loading process now aborted.");
        return;
      }
      if ((outputTopology != GL_POINTS) && (outputTopology != GL_LINE_STRIP) && (outputTopology != GL_TRIANGLE_STRIP))
      {
        DebugConsole_consoleOutLine("Error: You have not specified a supported Geometry Shader Output Topology.\nShader loading process now aborted.");
        return;
      }
      if (maxVerticesOut > openGLDriverInfo_->getMaxGeometryOutputVertices())
      {
        DebugConsole_consoleOutLine("Error: You have specified a larger number of Geometry Shader output vertices than this GPU can handle.\nGL_MAX_GEOMETRY_OUTPUT_VERTICES: ", openGLDriverInfo_->getMaxGeometryOutputVertices(), "\nShader loading process now aborted.");
        return;
      }

      // warning, Intel GPU driver crashes with the GL calls below, have to use the shader layout GL 3.3+ syntax
      if (!openGLDriverInfo_->isIntel())
      {
        // don't use the non _ARB/_EXT variety to avoid GL errors
        glProgramParameteri(shaderProgram_, GL_GEOMETRY_INPUT_TYPE_ARB, inputTopology);
        glProgramParameteri(shaderProgram_, GL_GEOMETRY_OUTPUT_TYPE_ARB, outputTopology);
        glProgramParameteri(shaderProgram_, GL_GEOMETRY_VERTICES_OUT_ARB, maxVerticesOut);
      }
    }

    glLinkProgram(shaderProgram_);
#ifdef GPU_FRAMEWORK_GLSL_EXTERNAL_FILES
    if (!openGLDriverInfo_->isNvidia()) // Nvidia GPUs may produce more compile error information if the glValidateProgram() func is not run
    {
      glValidateProgram(shaderProgram_);
    }
    checkInfoLog("Modular Shader Program '" + mergedShaderLibraryName_ + "'", shaderProgram_);
#endif // GPU_FRAMEWORK_GLSL_EXTERNAL_FILES
  }
  else
  {
    DebugConsole_consoleOutLine("Link Shader Error: Shader objects were attempted to be linked to a non-valid shader program!");
  }
}

/**
*  Checks the OpenGL info log of the shader loading process.
*/
void OpenGLShaderCompileAndLink::checkInfoLog(const string& shaderName, GLuint obj) const
{
  GLsizei infoLogLength = 0;
  if (glIsShader(obj))
  {
    glGetShaderiv(obj, GL_INFO_LOG_LENGTH, &infoLogLength);
  }
  else
  {
    glGetProgramiv(obj, GL_INFO_LOG_LENGTH, &infoLogLength);
  }

  if (infoLogLength <= 1)
  {
    return;
  }

  GLsizei charactersWritten = 0;
  auto infoLog = unique_ptr<GLchar[]>(new GLchar[infoLogLength]); // avoid enforcing the default constructor through the make_unique() call for the primitive GLchar (make_unique() is using the C++03 array initialization syntax)
  if (glIsShader(obj))
  {
    glGetShaderInfoLog(obj, infoLogLength, &charactersWritten, infoLog.get());
  }
  else
  {
    glGetProgramInfoLog(obj, infoLogLength, &charactersWritten, infoLog.get());
  }

  if (charactersWritten)
  {
    DebugConsole_consoleOutLine("\nGLSL Validation Report for the Shader ", shaderName, ":\n", infoLog.get());
  }
}

void OpenGLShaderCompileAndLink::releaseAllShaderObjects()
{
  for (const auto& mapEntry : allOpenGLShaderObjectsMap_)
  {
    OpenGLShaderObjects* openGLShaderObjects = mapEntry.second;

    // detach all shader objects from program
    if (shaderProgram_ > 0)
    {
      openGLShaderObjects->detachShaderObjectsFromProgram(shaderProgram_);
    }

    // VS deletion part
    releaseShaderObject(OpenGLAssetManager::VS, openGLShaderObjects);

    // TCS deletion part
    releaseShaderObject(OpenGLAssetManager::TCS, openGLShaderObjects);

    // TES deletion part
    releaseShaderObject(OpenGLAssetManager::TES, openGLShaderObjects);

    // GS deletion part
    releaseShaderObject(OpenGLAssetManager::GS, openGLShaderObjects);

    // FS deletion part
    releaseShaderObject(OpenGLAssetManager::FS, openGLShaderObjects);

    // CS deletion part
    releaseShaderObject(OpenGLAssetManager::CS, openGLShaderObjects);

    delete openGLShaderObjects;
    openGLShaderObjects = nullptr;
  }
}