#include "OpenGLQueryTimer.h"
#include "UtilityFunctions.h"

using namespace OpenGLRenderingEngine;
using namespace Utils::UtilityFunctions;

OpenGLQueryTimer::OpenGLQueryTimer() noexcept
{
  if (!glIsQuery(query_))
  {
    glGenQueries(1, &query_);
  }
}

OpenGLQueryTimer::~OpenGLQueryTimer() noexcept
{
  if (glIsQuery(query_))
  {
    glDeleteQueries(1, &query_);
  }

  query_         = 0;
  renderingTime_ = 0;
}

void OpenGLQueryTimer::startTimer()
{
  stopped_ = false; // reset stop flag
  glBeginQuery(GL_TIME_ELAPSED, query_);
}

void OpenGLQueryTimer::stopTimer()
{
  if (stopped_)
  {
    return;
  }

  stopped_ = true; // set timer stopped flag
  glEndQuery(GL_TIME_ELAPSED);
}

double OpenGLQueryTimer::getElapsedTime()
{
  if (!glIsQuery(query_))
  {
    return 0.0;
  }

  if (!stopped_)
  {
    stopTimer();
  }

  GLint available = 0;
  while (!available) // wait for all results to become available
  {
    glGetQueryObjectiv(query_, GL_QUERY_RESULT_AVAILABLE, &available);
  }
  glGetQueryObjectui64v(query_, GL_QUERY_RESULT, &renderingTime_);

  return double(renderingTime_);
}

double OpenGLQueryTimer::getElapsedTimeInNanoSecs()
{
  return getElapsedTime();
}

double OpenGLQueryTimer::getElapsedTimeInMicroSecs()
{
  return getElapsedTime() / 1000.0;
}

double OpenGLQueryTimer::getElapsedTimeInMilliSecs()
{
  return getElapsedTime() / 1000000.0;
}

double OpenGLQueryTimer::getElapsedTimeInSecs()
{
  return getElapsedTime() / 1000000000.0;
}

double OpenGLQueryTimer::getMeanTimeInNanoSecs()
{
  const auto index = StdAuxiliaryFunctions::toUnsignedType(TimerTypes::NANOSECS);
  return calculateMeanTime(getElapsedTimeInNanoSecs(), timersBookKeeping_[index], timersBookKeepingIndex_[index], firstTimersBookKeepingIterationCompleted_[index]);
}

double OpenGLQueryTimer::getMeanTimeInMicroSecs()
{
  const auto index = StdAuxiliaryFunctions::toUnsignedType(TimerTypes::MICROSECS);
  return calculateMeanTime(getElapsedTimeInMicroSecs(), timersBookKeeping_[index], timersBookKeepingIndex_[index], firstTimersBookKeepingIterationCompleted_[index]);
}

double OpenGLQueryTimer::getMeanTimeInMilliSecs()
{
  const auto index = StdAuxiliaryFunctions::toUnsignedType(TimerTypes::MILLISECS);
  return calculateMeanTime(getElapsedTimeInMilliSecs(), timersBookKeeping_[index], timersBookKeepingIndex_[index], firstTimersBookKeepingIterationCompleted_[index]);
}

double OpenGLQueryTimer::getMeanTimeInSecs()
{
  const auto index = StdAuxiliaryFunctions::toUnsignedType(TimerTypes::SECS);
  return calculateMeanTime(getElapsedTimeInSecs(), timersBookKeeping_[index], timersBookKeepingIndex_[index], firstTimersBookKeepingIterationCompleted_[index]);
}

double OpenGLQueryTimer::getDecimalElapsedTimeInMicroSecs()
{
  return getElapsedTimeInNanoSecs() / NANO_TO_MICROSECS_CONVERSION;
}

double OpenGLQueryTimer::getDecimalElapsedTimeInMilliSecs()
{
  return getElapsedTimeInNanoSecs() / NANO_TO_MILLISECS_CONVERSION;
}

double OpenGLQueryTimer::getDecimalElapsedTimeInSecs()
{
  return getElapsedTimeInNanoSecs() / NANO_TO_SECS_CONVERSION;
}

double OpenGLQueryTimer::getDecimalMeanTimeInMicroSecs()
{
  return getMeanTimeInNanoSecs() / NANO_TO_MICROSECS_CONVERSION;
}

double OpenGLQueryTimer::getDecimalMeanTimeInMilliSecs()
{
  return getMeanTimeInNanoSecs() / NANO_TO_MILLISECS_CONVERSION;
}

double OpenGLQueryTimer::getDecimalMeanTimeInSecs()
{
  return getMeanTimeInNanoSecs() / NANO_TO_SECS_CONVERSION;
}