#include "OpenGLPerlinNoise.h"
#include "OpenGLUtilityFunctions.h"
#include "OpenGLSimplexNoise.h"
#include "CPUParallelism/CPUParallelismNCP.h"
#include "UtilityFunctions.h"
#include <mutex>
#include <thread>
#include <cmath>

using namespace std;
using namespace Utils::CPUParallelism;
using namespace OpenGLRenderingEngine;

namespace // anonymous namespace used instead of deprecated 'static' keyword used for cpp variable locality
{
  // Perlin Noise related variables
  const int NOISE_3D_NUMBER_OF_OCTAVES =   4; // noise 3D number of octaves
  const int NOISE_3D_TEXTURE_SIZE      = 128; // noise 3D texture size
  const int MAXB = 0x100;
  const int N    = 0x1000;
  const int ARRAY_SIZE = MAXB + MAXB + 2;
  // N-CP related variables
  const unsigned int NUMBER_OF_NCP_THREAD_PROCESSES = (unsigned int)((numberOfHardwareThreads() > NOISE_3D_NUMBER_OF_OCTAVES) ? NOISE_3D_NUMBER_OF_OCTAVES : numberOfHardwareThreads()); // variables needed for N-CP (up to 4 cores in this case, splitting the 4 frequencies to up to 4 cores)
  // dummy vector to avoid non-const reference default parameters in initPerlinNoise()
  auto DUMMY_VECTOR = vector<vector<double>>();

  /**
  *  Hermite blending function (or a fifth degree polynomial for the improved version of Perlin Noise so as to have a continuous 2nd derivative, C2 polynomial).
  */
  inline double sCurve(double t)
  {
    // return (t * t * (3.0 - 2.0 * t));
    return (t * t * t * (t * (t * 6.0 - 15.0) + 10.0));
  }

  /**
  *  Linear interpolation (mix (1 - t) * a + t * b) function.
  */
  inline double lerp(double t, double a, double b)
  {
    return a + t * (b - a);
  }

  /**
  *  2D dot product function.
  */
  inline double at2(double rx, double ry, double q1, double q2)
  {
    return (rx * q1 + ry * q2);
  }

  /**
  *  3D dot product function.
  */
  inline double at3(double rx, double ry, double rz, double q1, double q2, double q3)
  {
    return (rx * q1 + ry * q2 + rz * q3);
  }

  /**
  *  2D normalize function.
  */
  inline void normalize2(vector<double>& v)
  {
    const double s = sqrt(v[0] * v[0] + v[1] * v[1]);
    v[0] /= s;
    v[1] /= s;
  }

  /**
  *  3D normalize function.
  */
  inline void normalize3(vector<double>& v)
  {
    const double s = sqrt(v[0] * v[0] + v[1] * v[1] + v[2] * v[2]);
    v[0] /= s;
    v[1] /= s;
    v[2] /= s;
  }

  /**
  *  1D Perlin Noise function.
  *
  *  Overloaded version where the p/g1 arrays are being passed to the function as parameters.
  */
  inline double perlinNoise1Internal(double xIn, int frequency, vector<int>& p, vector<double>& g1)
  {
    int bx0 = 0, bx1 = 0;
    double rx0 = 0.0, rx1 = 0.0, sx = 0.0, t = 0.0, u = 0.0, v = 0.0;

    // setup(0, bx0, bx1, rx0, rx1)
    t   = xIn + N;
    bx0 = int(t) & (frequency - 1);
    bx1 = (bx0 + 1) & (frequency - 1);
    rx0 = t - int(t);
    rx1 = rx0 - 1.0;

    sx = sCurve(rx0);
    u = rx0 * g1[p[bx0]];
    v = rx1 * g1[p[bx1]];

    return lerp(sx, u, v);
  }

  /**
  *  1D Perlin Noise harmonic summing function.
  *  In what follows "alpha" is the weight when the sum is formed.
  *  Typically it is 2, as this approaches 1 the function is noisier.
  *  "beta" is the harmonic scaling/spacing, typically 2.
  *
  *  Version where the p/g1 arrays are being passed to the function as parameters.
  */
  inline double perlinNoise1DInternal(double xIn, double alpha, double beta, int n, int frequency, vector<int>& p, vector<double>& g1)
  {
    int i = 0;
    double val = 0.0, sum = 0.0;
    double pp1 = 0.0, scale = 1.0;

    pp1 = xIn;
    for (i = 0; i < n; ++i)
    {
      val = perlinNoise1Internal(pp1, frequency, p, g1);
      sum += val / scale;
      scale *= alpha;
      pp1 *= beta;
    }

    return sum;
  }

  /**
  *  2D Perlin Noise function.
  *
  *  Version where the p/g1/g2 arrays are being passed to the function as parameters.
  */
  inline double perlinNoise2Internal(double xIn, double yIn, int frequency, vector<int>& p, vector<double>& g1, vector<vector<double>>& g2)
  {
    int bx0 = 0, bx1 = 0, by0 = 0, by1 = 0, b00 = 0, b10 = 0, b01 = 0, b11 = 0;
    double rx0 = 0.0, rx1 = 0.0, ry0 = 0.0, ry1 = 0.0, sx = 0.0, sy = 0.0, a = 0.0, b = 0.0, t = 0.0, u = 0.0, v = 0.0;
    double q1 = 0.0, q2 = 0.0;
    int i = 0, j = 0;

    // setup(0, bx0, bx1, rx0, rx1)
    t   = xIn + N;
    bx0 = int(t) & (frequency - 1);
    bx1 = (bx0 + 1) & (frequency - 1);
    rx0 = t - int(t);
    rx1 = rx0 - 1.0;

    // setup(1, by0, by1, ry0, ry1)
    t = yIn + N;
    by0 = int(t) & (frequency - 1);
    by1 = (by0 + 1) & (frequency - 1);
    ry0 = t - int(t);
    ry1 = ry0 - 1.0;

    i = p[bx0];
    j = p[bx1];

    b00 = p[i + by0];
    b10 = p[j + by0];
    b01 = p[i + by1];
    b11 = p[j + by1];

    sx = sCurve(rx0);
    sy = sCurve(ry0);

    q1 = g2[b00][0];
    q2 = g2[b00][1];
    u = at2(rx0, ry0, q1, q2);
    q1 = g2[b10][0];
    q2 = g2[b10][1];
    v = at2(rx1, ry0, q1, q2);
    a = lerp(sx, u, v);

    q1 = g2[b01][0];
    q2 = g2[b01][1];
    u = at2(rx0, ry1, q1, q2);
    q1 = g2[b11][0];
    q2 = g2[b11][1];
    v = at2(rx1, ry1, q1, q2);
    b = lerp(sx, u, v);

    return lerp(sy, a, b);
  }

  /**
  *  2D Perlin Noise harmonic summing function.
  *  In what follows "alpha" is the weight when the sum is formed.
  *  Typically it is 2, as this approaches 1 the function is noisier.
  *  "beta" is the harmonic scaling/spacing, typically 2.
  *
  *  Version where the p/g1/g2 arrays are being passed to the function as parameters.
  */
  inline double perlinNoise2DInternal(double xIn, double yIn, double alpha, double beta, int n, int frequency, vector<int>& p, vector<double>& g1, vector<vector<double>>& g2)
  {
    int i = 0;
    double val = 0.0, sum = 0.0;
    double scale = 1.0;
    double pp1 = 0.0, pp2 = 0.0;

    pp1 = xIn;
    pp2 = yIn;
    for (i = 0; i < n; ++i)
    {
      val = perlinNoise2Internal(pp1, pp2, frequency, p, g1, g2);
      sum += val / scale;
      scale *= alpha;
      pp1 *= beta;
      pp2 *= beta;
    }

    return sum;
  }

  /**
  *  3D Perlin Noise function.
  *
  *  Version where the p/g1/g2/g3 arrays are being passed to the function as parameters.
  */
  inline double perlinNoise3Internal(double xIn, double yIn, double zIn, int frequency, vector<int>& p, vector<double>& g1, vector<vector<double>>& g2, vector<vector<double>>& g3)
  {
    int bx0 = 0, bx1 = 0, by0 = 0, by1 = 0, bz0 = 0, bz1 = 0, b00 = 0, b10 = 0, b01 = 0, b11 = 0;
    double rx0 = 0.0, rx1 = 0.0, ry0 = 0.0, ry1 = 0.0, rz0 = 0.0, rz1 = 0.0, sy = 0.0, sz = 0.0, a = 0.0, b = 0.0, c = 0.0, d = 0.0, t = 0.0, u = 0.0, v = 0.0;
    double q1 = 0.0, q2 = 0.0, q3 = 0.0;
    int i = 0, j = 0;

    // setup(0, bx0, bx1, rx0, rx1)
    t   = xIn + N;
    bx0 = int(t) & (frequency - 1);
    bx1 = (bx0 + 1) & (frequency - 1);
    rx0 = t - int(t);
    rx1 = rx0 - 1.0;

    // setup(1, by0, by1, ry0, ry1)
    t   = yIn + N;
    by0 = int(t) & (frequency - 1);
    by1 = (by0 + 1) & (frequency - 1);
    ry0 = t - int(t);
    ry1 = ry0 - 1.0;

    // setup(2, bz0, bz1, rz0, rz1)
    t   = zIn + N;
    bz0 = int(t) & (frequency - 1);
    bz1 = (bz0 + 1) & (frequency - 1);
    rz0 = t - int(t);
    rz1 = rz0 - 1.0;

    i = p[bx0];
    j = p[bx1];

    b00 = p[i + by0];
    b10 = p[j + by0];
    b01 = p[i + by1];
    b11 = p[j + by1];

    t = sCurve(rx0);
    sy = sCurve(ry0);
    sz = sCurve(rz0);

    q1 = g3[b00 + bz0][0];
    q2 = g3[b00 + bz0][1];
    q3 = g3[b00 + bz0][2];
    u = at3(rx0, ry0, rz0, q1, q2, q3);
    q1 = g3[b10 + bz0][0];
    q2 = g3[b10 + bz0][1];
    q3 = g3[b10 + bz0][2];
    v = at3(rx1, ry0, rz0, q1, q2, q3);
    a = lerp(t, u, v);

    q1 = g3[b01 + bz0][0];
    q2 = g3[b01 + bz0][1];
    q3 = g3[b01 + bz0][2];
    u = at3(rx0, ry1, rz0, q1, q2, q3);
    q1 = g3[b11 + bz0][0];
    q2 = g3[b11 + bz0][1];
    q3 = g3[b11 + bz0][2];
    v = at3(rx1, ry1, rz0, q1, q2, q3);
    b = lerp(t, u, v);

    c = lerp(sy, a, b);

    q1 = g3[b00 + bz1][0];
    q2 = g3[b00 + bz1][1];
    q3 = g3[b00 + bz1][2];
    u = at3(rx0, ry0, rz1, q1, q2, q3);
    q1 = g3[b10 + bz1][0];
    q2 = g3[b10 + bz1][1];
    q3 = g3[b10 + bz1][2];
    v = at3(rx1, ry0, rz1, q1, q2, q3);
    a = lerp(t, u, v);

    q1 = g3[b01 + bz1][0];
    q2 = g3[b01 + bz1][1];
    q3 = g3[b01 + bz1][2];
    u = at3(rx0, ry1, rz1, q1, q2, q3);
    q1 = g3[b11 + bz1][0];
    q2 = g3[b11 + bz1][1];
    q3 = g3[b11 + bz1][2];
    v = at3(rx1, ry1, rz1, q1, q2, q3);
    b = lerp(t, u, v);

    d = lerp(sy, a, b);

    return lerp(sz, c, d);
  }

  /**
  *  3D Perlin Noise harmonic summing function.
  *  In what follows "alpha" is the weight when the sum is formed.
  *  Typically it is 2, as this approaches 1 the function is noisier.
  *  "beta" is the harmonic scaling/spacing, typically 2.
  *
  *  Version where the p/g1/g2/g3 arrays are being passed to the function as parameters.
  */
  inline double perlinNoise3DInternal(double xIn, double yIn, double zIn, double alpha, double beta, int n, int frequency, vector<int>& p, vector<double>& g1, vector<vector<double>>& g2, vector<vector<double>>& g3)
  {
    int i = 0;
    double val = 0.0, sum = 0.0;
    double scale = 1.0;
    double pp1 = 0.0, pp2 = 0.0, pp3 = 0.0;

    pp1 = xIn;
    pp2 = yIn;
    pp3 = zIn;
    for (i = 0; i < n; ++i)
    {
      val = perlinNoise3Internal(pp1, pp2, pp3, frequency, p, g1, g2, g3);
      sum += val / scale;
      scale *= alpha;
      pp1 *= beta;
      pp2 *= beta;
      pp3 *= beta;
    }

    return sum;
  }
}

OpenGLPerlinNoise::OpenGLPerlinNoise(bool useSimplexNoise) noexcept
  : useSimplexNoise_(useSimplexNoise)
{
}

OpenGLPerlinNoise::~OpenGLPerlinNoise() noexcept
{
  // free the Perlin Noise 3D texture
  glDeleteTextures(1, &perlinNoise3DTextureID_);
  perlinNoise3DTextureID_ = 0;

  perlinNoise3DTextureData_.clear();
}

/**
*  Initializes all the Perlin Noise arrays.
*/
void OpenGLPerlinNoise::initPerlinNoise(int frequency, vector<int>& p, vector<double>& g1, vector<vector<double>>& g2, vector<vector<double>>& g3)
{
  int i = 0, j = 0, k = 0;
  for (i = 0; i < frequency; ++i)
  {
    p[i] = i;
    g1[i] = Utils::Randomizers::RandomRNGWELL512::getRandomMax() * double((random_.getRandomInteger() % (frequency + frequency)) - frequency) / frequency;

    if (!g2.empty())
    {
      for (j = 0; j < 2; ++j)
      {
        g2[i][j] = Utils::Randomizers::RandomRNGWELL512::getRandomMax() * double((random_.getRandomInteger() % (frequency + frequency)) - frequency) / frequency;
      }
      normalize2(g2[i]);
    }

    if (!g3.empty())
    {
      for (j = 0; j < 3; ++j)
      {
        g3[i][j] = Utils::Randomizers::RandomRNGWELL512::getRandomMax() * double((random_.getRandomInteger() % (frequency + frequency)) - frequency) / frequency;
      }
      normalize3(g3[i]);
    }
  }

  while (--i != 0)
  {
    k = p[i];
    p[i] = p[j = (Utils::Randomizers::RandomRNGWELL512::getRandomMax() * random_.getRandomInteger()) % frequency];
    p[j] = k;
  }

  for (i = 0; i < frequency + 2; ++i)
  {
    p[frequency + i] = p[i];
    g1[frequency + i] = g1[i];

    if (!g2.empty())
    {
      for (j = 0; j < 2; ++j)
      {
        g2[frequency + i][j] = g2[i][j];
      }
    }

    if (!g3.empty())
    {
      for (j = 0; j < 3; ++j)
      {
        g3[frequency + i][j] = g3[i][j];
      }
    }
  }
}

/**
*  1D Perlin Noise function.
*/
double OpenGLPerlinNoise::perlinNoise1(double xIn, int frequency)
{
  vector<int> p;
  p.resize(ARRAY_SIZE);
  vector<double> g1;
  g1.resize(ARRAY_SIZE);
  initPerlinNoise(frequency, p, g1, DUMMY_VECTOR, DUMMY_VECTOR);

  return perlinNoise1Internal(xIn, frequency, p, g1);
}

/**
*  2D Perlin Noise function.
*/
double OpenGLPerlinNoise::perlinNoise2(double xIn, double yIn, int frequency)
{
  vector<int> p;
  p.resize(ARRAY_SIZE);
  vector<double> g1;
  g1.resize(ARRAY_SIZE);
  vector<vector<double>> g2;
  g2.resize(ARRAY_SIZE);
  for (int i = 0; i < ARRAY_SIZE; ++i)
  {
    g2[i].resize(2);
  }
  initPerlinNoise(frequency, p, g1, g2, DUMMY_VECTOR);

  return perlinNoise2Internal(xIn, yIn, frequency, p, g1, g2);
}

/**
*  3D Perlin Noise function.
*/
double OpenGLPerlinNoise::perlinNoise3(double xIn, double yIn, double zIn, int frequency)
{
  vector<int> p;
  p.resize(ARRAY_SIZE);
  vector<double> g1;
  g1.resize(ARRAY_SIZE);
  vector<vector<double>> g2;
  g2.resize(ARRAY_SIZE);
  for (int i = 0; i < ARRAY_SIZE; ++i)
  {
    g2[i].resize(2);
  }
  vector<vector<double>> g3;
  g3.resize(ARRAY_SIZE);
  for (int i = 0; i < ARRAY_SIZE; ++i)
  {
    g3[i].resize(3);
  }
  initPerlinNoise(frequency, p, g1, g2, g3);

  return perlinNoise3Internal(xIn, yIn, zIn, frequency, p, g1, g2, g3);
}

/**
*  1D Perlin Noise harmonic summing function.
*  In what follows "alpha" is the weight when the sum is formed.
*  Typically it is 2, as this approaches 1 the function is noisier.
*  "beta" is the harmonic scaling/spacing, typically 2.
*/
double OpenGLPerlinNoise::perlinNoise1D(double xIn, double alpha, double beta, int n, int frequency)
{
  vector<int> p;
  p.resize(ARRAY_SIZE);
  vector<double> g1;
  g1.resize(ARRAY_SIZE);
  initPerlinNoise(frequency, p, g1, DUMMY_VECTOR, DUMMY_VECTOR);

  return perlinNoise1DInternal(xIn, alpha, beta, n, frequency, p, g1);
}

/**
*  2D Perlin Noise harmonic summing function.
*  In what follows "alpha" is the weight when the sum is formed.
*  Typically it is 2, as this approaches 1 the function is noisier.
*  "beta" is the harmonic scaling/spacing, typically 2.
*/
double OpenGLPerlinNoise::perlinNoise2D(double xIn, double yIn, double alpha, double beta, int n, int frequency)
{
  vector<int> p;
  p.resize(ARRAY_SIZE);
  vector<double> g1;
  g1.resize(ARRAY_SIZE);
  vector<vector<double>> g2;
  g2.resize(ARRAY_SIZE);
  for (int i = 0; i < ARRAY_SIZE; ++i)
  {
    g2[i].resize(2);
  }
  initPerlinNoise(frequency, p, g1, g2, DUMMY_VECTOR);

  return perlinNoise2DInternal(xIn, yIn, alpha, beta, n, frequency, p, g1, g2);
}

/**
*  3D Perlin Noise harmonic summing function.
*  In what follows "alpha" is the weight when the sum is formed.
*  Typically it is 2, as this approaches 1 the function is noisier.
*  "beta" is the harmonic scaling/spacing, typically 2.
*/
double OpenGLPerlinNoise::perlinNoise3D(double xIn, double yIn, double zIn, double alpha, double beta, int n, int frequency)
{
  vector<int> p;
  p.resize(ARRAY_SIZE);
  vector<double> g1;
  g1.resize(ARRAY_SIZE);
  vector<vector<double>> g2;
  g2.resize(ARRAY_SIZE);
  for (int i = 0; i < ARRAY_SIZE; ++i)
  {
    g2[i].resize(2);
  }
  vector<vector<double>> g3;
  g3.resize(ARRAY_SIZE);
  for (int i = 0; i < ARRAY_SIZE; ++i)
  {
    g3[i].resize(3);
  }
  initPerlinNoise(frequency, p, g1, g2, g3);

  return perlinNoise3DInternal(xIn, yIn, zIn, alpha, beta, n, frequency, p, g1, g2, g3);
}

/**
*  Makes the Perlin Noise 3D texture.
*/
void OpenGLPerlinNoise::makePerlinNoise3DTexture()
{
  if (perlinNoise3DTextureData_.empty())
  {
    perlinNoise3DTextureData_.resize(4 * NOISE_3D_TEXTURE_SIZE * NOISE_3D_TEXTURE_SIZE * NOISE_3D_TEXTURE_SIZE);
  }
  else
  {
    perlinNoise3DTextureData_.clear();
  }

  if (numberOfHardwareThreads() < NOISE_3D_NUMBER_OF_OCTAVES)
  {
    int f         = 0;
    int inc       = 0;
    int frequency = 4;
    double amp    = 0.5;
    for (f = 0, inc = 0; f < NOISE_3D_NUMBER_OF_OCTAVES; ++f, frequency *= 2, ++inc, amp *= 0.5)
    {
      makePerlinNoise3DTextureCalculation(f, inc, frequency, amp);
    }
  }
  else
  {
    parallelFor(NOISE_3D_NUMBER_OF_OCTAVES, [&](size_t i)
    {
      const int index     = int(i);
      const int f         = index;
      const int inc       = index;
      const int frequency = 2 << (index + 1);
      const double amp    = 1.0 / double(1 << (index + 1));
      makePerlinNoise3DTextureCalculation(f, inc, frequency, amp);
    }, NUMBER_OF_NCP_THREAD_PROCESSES);
  }
}

void OpenGLPerlinNoise::makePerlinNoise3DTextureCalculation(int f, int inc, int frequency, double amp)
{
  DebugConsole_consoleOutLine("Generating Perlin Noise 3D Texture: Octave ", (f + 1), "/", NOISE_3D_NUMBER_OF_OCTAVES, "...");

  int i = 0, j = 0, k = 0;
  int offset = 0;
  double ni1 = 0.0, ni2 = 0.0, ni3 = 0.0;
  double inci = 0.0, incj = 0.0, inck = 0.0;
  vector<int> p;
  vector<double> g1;
  vector<vector<double>> g2;
  vector<vector<double>> g3;
  if (!useSimplexNoise_)
  {
    p.resize(ARRAY_SIZE);
    g1.resize(ARRAY_SIZE);
    g2.resize(ARRAY_SIZE);
    for (i = 0; i < ARRAY_SIZE; ++i)
    {
      g2[i].resize(2);
    }
    g3.resize(ARRAY_SIZE);
    for (i = 0; i < ARRAY_SIZE; ++i)
    {
      g3[i].resize(3);
    }
    initPerlinNoise(frequency, p, g1, g2, g3);
  }

  inci = 1.0 / (double(NOISE_3D_TEXTURE_SIZE) / frequency);
  for (i = 0; i < NOISE_3D_TEXTURE_SIZE; ++i, ni1 += inci)
  {
    incj = 1.0 / (double(NOISE_3D_TEXTURE_SIZE) / frequency);
    for (j = 0; j < NOISE_3D_TEXTURE_SIZE; ++j, ni2 += incj)
    {
      inck = 1.0 / (double(NOISE_3D_TEXTURE_SIZE) / frequency);
      for (k = 0; k < NOISE_3D_TEXTURE_SIZE; ++k, ni3 += inck, offset += 4)
      {
        perlinNoise3DTextureData_[offset + inc] = (useSimplexNoise_) ? GLubyte((((OpenGLSimplexNoise::perlinNoise3(ni1, ni2, ni3) + 1.0)                                   * amp) * 128.0))
                                                                     : GLubyte((((                    perlinNoise3Internal(ni1, ni2, ni3, frequency, p, g1, g2, g3) + 1.0) * amp) * 128.0));
      }
    }
  }
}

void OpenGLPerlinNoise::bindTexture(GLenum textureUnit) const
{
  glActiveTexture(GL_TEXTURE0 + textureUnit);
  glBindTexture(GL_TEXTURE_2D, perlinNoise3DTextureID_);
}

void OpenGLPerlinNoise::unbind(GLenum textureUnit) const
{
  glActiveTexture(GL_TEXTURE0 + textureUnit);
  glBindTexture(GL_TEXTURE_2D, 0);
}

/**
*  Initializes the Perlin Noise 3D texture with a given active texture unit.
*  Overloaded version of the method above that selects an active texture unit for the Perlin Noise 3D texture.
*/
void OpenGLPerlinNoise::initPerlinNoise3DTexture(GLenum textureUnit)
{
  if (!perlinNoise3DTextureData_.empty())
  {
    // allocate the Perlin Noise 3D texture
    glGenTextures(1, &perlinNoise3DTextureID_);

    bindTexture(textureUnit);

    glTexParameterf(GL_TEXTURE_3D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameterf(GL_TEXTURE_3D, GL_TEXTURE_WRAP_T, GL_REPEAT);
    glTexParameterf(GL_TEXTURE_3D, GL_TEXTURE_WRAP_R, GL_REPEAT);
    glTexParameterf(GL_TEXTURE_3D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_3D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

    glTexImage3D(GL_TEXTURE_3D, 0, GL_RGBA, NOISE_3D_TEXTURE_SIZE, NOISE_3D_TEXTURE_SIZE, NOISE_3D_TEXTURE_SIZE, 0, GL_RGBA, GL_UNSIGNED_BYTE, &perlinNoise3DTextureData_[0]);

    unbind(textureUnit);

    perlinNoise3DTextureData_.clear();
  }
}