#include "OpenGLILTexture.h"
#include "UtilityFunctions.h"
#include "devil_internal_exports.h"
#include <unordered_map>
#include <algorithm>

using namespace std;
using namespace OpenGLRenderingEngine;
using namespace Utils::UtilityFunctions;

namespace // anonymous namespace used instead of deprecated 'static' keyword used for cpp variable locality
{
  unordered_map<string, ILenum> allFormatsMap;
  ILint MaxTexW = 256;
  ILint MaxTexH = 256;

  inline ILenum ilGetTypeSave(const string& fileNameExtension)
  {
    const auto& position = allFormatsMap.find(fileNameExtension);
    if (position == allFormatsMap.end())
    {
      return IL_TYPE_UNKNOWN;
    }

    return allFormatsMap[fileNameExtension];
  }

  inline ILimage* makeGLCompliant(ILimage* src, bool enforceImageFlip)
  {
    ILimage* dest = src, *Temp;
    ILboolean Created = IL_FALSE;

    if ((src->Pal.Palette != nullptr) && (src->Pal.PalSize != 0) && (src->Pal.PalType != IL_PAL_NONE))
    {
      dest = iConvertImage(src, ilGetPalBaseType(src->Pal.PalType), IL_UNSIGNED_BYTE);
      if (dest == nullptr)
      {
        return nullptr;
      }
      Created = IL_TRUE;

      // Set dest's palette stuff here
      dest->Pal.PalType = IL_PAL_NONE;
    }

    if ((src->Width != ilNextPower2(src->Width)) || (src->Height != ilNextPower2(src->Height)) || (ILint(src->Width) > MaxTexW) || (ILint(src->Height) > MaxTexH))
    {
      if (!Created)
      {
        dest = ilCopyImage_(src);
        if (dest == nullptr)
        {
          return nullptr;
        }
        Created = IL_TRUE;
      }

      const ILenum Filter = iluGetInteger(ILU_FILTER);
      if (src->Format == IL_COLOUR_INDEX)
      {
        iluImageParameter(ILU_FILTER, ILU_NEAREST);
        Temp = iluScale_(dest, min<ILint>(MaxTexW, ILint(ilNextPower2(dest->Width))), min<ILint>(MaxTexH, ILint(ilNextPower2(dest->Height))), 1);
        iluImageParameter(ILU_FILTER, Filter);
      }
      else
      {
        iluImageParameter(ILU_FILTER, ILU_BILINEAR);
        Temp = iluScale_(dest, min<ILint>(MaxTexW, ILint(ilNextPower2(dest->Width))), min<ILint>(MaxTexH, ILint(ilNextPower2(dest->Height))), 1);
        iluImageParameter(ILU_FILTER, Filter);
      }

      ilCloseImage(dest);
      if (!Temp)
      {
        return nullptr;
      }
      dest = Temp;
    }

    // changed 2003-08-25: images passed to openGL have to be upper-left
    if ((dest->Origin != IL_ORIGIN_UPPER_LEFT) || enforceImageFlip)
    {
      ILubyte* Flipped = iGetFlipped(dest);
      ifree(dest->Data);
      dest->Data = Flipped;
      dest->Origin = IL_ORIGIN_UPPER_LEFT;
    }

    return dest;
  }

  inline bool initializeDevIL()
  {
    // initialization of allFormatsMap
    allFormatsMap.emplace("bmp", IL_BMP);
    allFormatsMap.emplace("jpg", IL_JPG);
    allFormatsMap.emplace("png", IL_PNG);
    allFormatsMap.emplace("tga", IL_TGA);

    // initialization of DevIL
    if (ilGetInteger(IL_VERSION_NUM) < IL_VERSION)
    {
      DebugConsole_consoleOutLine("Wrong DevIL version, now aborting ilInit().");
      return false;
    }

    ilInit();
    glGetIntegerv(GL_MAX_TEXTURE_SIZE, (GLint*)&MaxTexW);
    glGetIntegerv(GL_MAX_TEXTURE_SIZE, (GLint*)&MaxTexH);
    if ((MaxTexW == 0) || (MaxTexH == 0))
    {
      MaxTexW = MaxTexH = 256; // Trying this because of the VooDoo series of cards...
    }

    return true;
  }

  const bool hasDevILInitialized = initializeDevIL();
}

OpenGLILTexture::OpenGLILTexture(bool generateMipmap, bool supportsMaxTextureMaxAnisotropy, GLint maxTextureMaxAnisotropy) noexcept
  : generateMipmap_(generateMipmap)
  , supportsMaxTextureMaxAnisotropy_(supportsMaxTextureMaxAnisotropy)
  , maxTextureMaxAnisotropy_(maxTextureMaxAnisotropy)
{
}

OpenGLILTexture::~OpenGLILTexture() noexcept
{
  deleteILTexture();
}

void OpenGLILTexture::createILTexture(const string& textureDirectory, const string& textureFileName, bool enforceImageFlip, GLenum textureUnit)
{
  if (hasDevILInitialized)
  {
    ILuint imageId = 0;
    ilGenImages(1, &imageId); // generation of one image name
    ilBindImage(imageId);     // binding of image name
    string textureFullPathFileName = (textureDirectory + textureFileName);
    ILboolean success = ilLoadImage((const ILstring)textureFullPathFileName.c_str()); // loading of image with filename
    if (success)
    {
      // convert every color component into unsigned byte,.if the image contains alpha channel IL_RGB can be replaced with IL_RGBA
      success = ilConvertImage(IL_RGBA, IL_UNSIGNED_BYTE);
      if (!success)
      {
        DebugConsole_consoleOutLine("Error occurred with ilConvertImage('", textureFullPathFileName, "') in prepareTexture().\nNow aborting loading texture '", textureFullPathFileName, "'.");
      }
      ilSetCurImage(makeGLCompliant(ilGetCurImage(), enforceImageFlip));

      if (!glIsTexture(ILtextureID_))
      {
        glGenTextures(1, &ILtextureID_);
      }
      glActiveTexture(GL_TEXTURE0 + textureUnit);
      glBindTexture(GL_TEXTURE_2D, ILtextureID_);
      glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
      glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
      if (supportsMaxTextureMaxAnisotropy_)
      {
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, maxTextureMaxAnisotropy_);
      }
      if (generateMipmap_)
      {
        glGenerateMipmap(GL_TEXTURE_2D);
        // glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);
      }
      glTexImage2D(GL_TEXTURE_2D, 0, ilGetInteger(IL_IMAGE_BPP), ilGetInteger(IL_IMAGE_WIDTH), ilGetInteger(IL_IMAGE_HEIGHT), 0, ilGetInteger(IL_IMAGE_FORMAT), GL_UNSIGNED_BYTE, ilGetData());

      // fixed pipeline spherical mapping
      glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
      glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);

      DebugConsole_consoleOutLine("\nprepareTexture('", textureFullPathFileName, "') initialized ok.\n");
    }
    else
    {
      DebugConsole_consoleOutLine("Error occurred with ilLoadImage('", textureFullPathFileName, "') in prepareTexture().\nNow aborting loading texture '", textureFullPathFileName, "'.");
    }

    ilDeleteImages(1, &imageId); // because the image data has already been copied into texture data, it's memory can be released
  }
}

void OpenGLILTexture::deleteILTexture()
{
  if (hasDevILInitialized)
  {
    if (glIsTexture(ILtextureID_))
    {
      glDeleteTextures(1, &ILtextureID_);
      ILtextureID_ = 0;
    }
  }
}

void OpenGLILTexture::enable() const
{
  if (hasDevILInitialized)
  {
    glEnable(GL_TEXTURE_2D);
  }
}

void OpenGLILTexture::disable() const
{
  if (hasDevILInitialized)
  {
    glDisable(GL_TEXTURE_2D);
  }
}

void OpenGLILTexture::bindTexture(GLenum textureUnit) const
{
  if (hasDevILInitialized)
  {
    glActiveTexture(GL_TEXTURE0 + textureUnit);
    glBindTexture(GL_TEXTURE_2D, ILtextureID_);
  }
}

void OpenGLILTexture::unbind(GLenum textureUnit) const
{
  if (hasDevILInitialized)
  {
    glActiveTexture(GL_TEXTURE0 + textureUnit);
    glBindTexture(GL_TEXTURE_2D, 0);
  }
}

void OpenGLILTexture::saveILImage(const string& imageDirectory, const string& imageFileNamePrefix, const string& imageFileNameExtension, ILubyte imageFormat, ILuint imageWidth, ILuint imageHeight, ILubyte* imagePixels)
{
  if (hasDevILInitialized)
  {
    const string imageFileName         = imageFileNamePrefix + "." + imageFileNameExtension;
    const string imageFullPathFileName = imageDirectory + imageFileName;
    if (imageFileNamePrefix.empty() || imageFileNameExtension.empty() || !imagePixels)
    {
      DebugConsole_consoleOutLine("Error occurred in saveILTexture().\nNow aborting saving image with name '", imageDirectory, "'.");
      return;
    }

    if (StdReadWriteFileFunctions::createDirectory(imageDirectory))
    {
      DebugConsole_consoleOutLine("Directory '", imageDirectory, "' was created for saving image with name '", imageFileName, "'.");
    }

    ILenum textureType = 0;
    if ((textureType = ilGetTypeSave(imageFileNameExtension)) == IL_TYPE_UNKNOWN)
    {
      DebugConsole_consoleOutLine("Error occurred in saveILTexture().\nNow aborting saving image with type '", imageFileNameExtension, "'.");
      return;
    }

    int ilImageFormat = 0;
    switch (imageFormat)
    {
      case 1:
      {
        ilImageFormat = IL_LUMINANCE;
        break;
      }
      case 3:
      {
        ilImageFormat = IL_RGB;
        break;
      }
      case 4:
      {
        ilImageFormat = IL_RGBA;
        break;
      }
      default:
      {
        DebugConsole_consoleOutLine("Error occurred in saveILTexture().\nNow aborting saving image with imageFormat '", imageFormat, "'.");
        return;
      }
    }

    ILuint imageName = 0;
    ilEnable(IL_FILE_OVERWRITE);
    ilGenImages(1, &imageName);
    ilBindImage(imageName);
    ilTexImage(imageWidth, imageHeight, 1, imageFormat, ilImageFormat, IL_UNSIGNED_BYTE, imagePixels);
    ilSave(textureType, (const ILstring)imageFullPathFileName.c_str());

    const ILenum possibleError = ilGetError();
    ilDeleteImages(1, &imageName);

    if (possibleError != IL_NO_ERROR)
    {
      DebugConsole_consoleOutLine("Error occurred in saveILTexture() while saving image with name '", imageFullPathFileName, "'.");
    }
    else
    {
      DebugConsole_consoleOutLine("saveILTexture() saved image with name '", imageFullPathFileName, "'.");
    }
  }
}