#include "OpenGLFrameBufferObject.h"
#include "OpenGLDriverInfo.h"
#include "OpenGLUtilityFunctions.h"
#include "UtilityFunctions.h"

using namespace std;
using namespace OpenGLRenderingEngine;
using namespace OpenGLRenderingEngine::OpenGLUtilityFunctions;

namespace // anonymous namespace used instead of deprecated 'static' keyword used for cpp variable locality
{
  /**
  *  Checks the status of the Frame Buffer Object (FBO) initialization.
  */
  inline bool checkFrameBufferStatus()
  {
    const GLenum fboStatus = glCheckFramebufferStatus(GL_FRAMEBUFFER);
    switch (fboStatus)
    {
      case GL_FRAMEBUFFER_COMPLETE:
        DebugConsole_consoleOutLine("\nFramebuffer initialization is complete.");
        return true;

      case GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT:
        DebugConsole_consoleOutLine("\nFramebuffer incomplete, incomplete attachment.");
        return false;

      case GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT:
        DebugConsole_consoleOutLine("\nFramebuffer incomplete, missing attachment.");
        return false;

      case GL_FRAMEBUFFER_INCOMPLETE_DIMENSIONS_EXT:
        DebugConsole_consoleOutLine("\nFramebuffer incomplete, attached images must have same dimensions.");
        return false;

      case GL_FRAMEBUFFER_INCOMPLETE_FORMATS_EXT:
        DebugConsole_consoleOutLine("\nFramebuffer incomplete, attached images must have same format.");
        return false;

      case GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER:
        DebugConsole_consoleOutLine("\nFramebuffer incomplete, missing draw buffer.");
        return false;

      case GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER:
        DebugConsole_consoleOutLine("\nFramebuffer incomplete, missing read buffer.");
        return false;

      case GL_FRAMEBUFFER_UNSUPPORTED:
        DebugConsole_consoleOutLine("\nUnsupported framebuffer format.");
        return false;

      default:
        DebugConsole_consoleOutLine("\nFramebuffer unknown error: ", fboStatus);
        return false;
    }
  }

  /**
  *  Converts the OpenGL internal format enum to a string.
  */
  inline string convertInternalFormatToString(GLsizei format)
  {
    switch (format)
    {
      case GL_STENCIL_INDEX:
        return "GL_STENCIL_INDEX";

      case GL_STENCIL_INDEX1:
        return "GL_STENCIL_INDEX1";

      case GL_STENCIL_INDEX4:
        return "GL_STENCIL_INDEX4";

      case GL_STENCIL_INDEX8:
        return "GL_STENCIL_INDEX8";

      case GL_STENCIL_INDEX16:
        return "GL_STENCIL_INDEX16";

      case GL_DEPTH_COMPONENT:
        return "GL_DEPTH_COMPONENT";

      case GL_DEPTH_COMPONENT16:
        return "GL_DEPTH_COMPONENT16";

      case GL_DEPTH_COMPONENT24:
        return "GL_DEPTH_COMPONENT24";

      case GL_DEPTH_COMPONENT32:
        return "GL_DEPTH_COMPONENT32";

      case GL_DEPTH_COMPONENT32F:
        return "GL_DEPTH_COMPONENT32F";

      case GL_DEPTH_STENCIL:
        return "GL_DEPTH_STENCIL";

      case GL_DEPTH24_STENCIL8:
        return "GL_DEPTH24_STENCIL8";

      case GL_DEPTH32F_STENCIL8:
        return "GL_DEPTH32F_STENCIL8";

      case GL_ALPHA:
        return "GL_ALPHA";

      case GL_RGB:
        return "GL_RGB";

      case GL_RGBA:
        return "GL_RGBA";

      case GL_LUMINANCE:
        return "GL_LUMINANCE";

      case GL_LUMINANCE_ALPHA:
        return "GL_LUMINANCE_ALPHA";

      case GL_ALPHA4:
        return "GL_ALPHA4";

      case GL_ALPHA8:
        return "GL_ALPHA8";

      case GL_ALPHA12:
        return "GL_ALPHA12";

      case GL_ALPHA16:
        return "GL_ALPHA16";

      case GL_LUMINANCE4:
        return "GL_LUMINANCE4";

      case GL_LUMINANCE8:
        return "GL_LUMINANCE8";

      case GL_LUMINANCE12:
        return "GL_LUMINANCE12";

      case GL_LUMINANCE16:
        return "GL_LUMINANCE16";

      case GL_LUMINANCE4_ALPHA4:
        return "GL_LUMINANCE4_ALPHA4";

      case GL_LUMINANCE6_ALPHA2:
        return "GL_LUMINANCE6_ALPHA2";

      case GL_LUMINANCE8_ALPHA8:
        return "GL_LUMINANCE8_ALPHA8";

      case GL_LUMINANCE12_ALPHA4:
        return "GL_LUMINANCE12_ALPHA4";

      case GL_LUMINANCE12_ALPHA12:
        return "GL_LUMINANCE12_ALPHA12";

      case GL_LUMINANCE16_ALPHA16:
        return "GL_LUMINANCE16_ALPHA16";

      case GL_INTENSITY:
        return "GL_INTENSITY";

      case GL_INTENSITY4:
        return "GL_INTENSITY4";

      case GL_INTENSITY8:
        return "GL_INTENSITY8";

      case GL_INTENSITY12:
        return "GL_INTENSITY12";

      case GL_INTENSITY16:
        return "GL_INTENSITY16";

      case GL_R3_G3_B2:
        return "GL_R3_G3_B2";

      case GL_R32F:
        return "GL_R32F";

      case GL_R32I:
        return "GL_R32I";

      case GL_R32UI:
        return "GL_R32UI";

      case GL_RG32F:
        return "GL_RG32F";

      case GL_RG32I:
        return "GL_RG32I";

      case GL_RG32UI:
        return "GL_RG32UI";

      case GL_RGB4:
        return "GL_RGB4";

      case GL_RGB5:
        return "GL_RGB4";

      case GL_RGB8:
        return "GL_RGB8";

      case GL_RGB10:
        return "GL_RGB10";

      case GL_RGB12:
        return "GL_RGB12";

      case GL_RGB16:
        return "GL_RGB16";

      case GL_RGB32F:
        return "GL_RGB32F";

      case GL_RGB32I:
        return "GL_RGB32I";

      case GL_RGB32UI:
        return "GL_RGB32UI";

      case GL_RGBA2:
        return "GL_RGBA2";

      case GL_RGBA4:
        return "GL_RGBA4";

      case GL_RGB5_A1:
        return "GL_RGB5_A1";

      case GL_RGBA8:
        return "GL_RGBA8";

      case GL_RGB10_A2:
        return "GL_RGB10_A2";

      case GL_RGBA12:
        return "GL_RGBA12";

      case GL_RGBA16:
        return "GL_RGBA16";

      case GL_RGBA32F:
        return "GL_RGBA32F";

      case GL_RGBA32I:
        return "GL_RGBA32I";

      case GL_RGBA32UI:
        return "GL_RGBA32UI";

      default:
        return "Unknown Format";
    }
  }
}

OpenGLFrameBufferObject::OpenGLFrameBufferObject(OpenGLDriverInfo* openGLDriverInfo,
                                                 bool useTexture, bool depthRenderBuffer, bool useDepthTexture, bool depthStencilRenderBuffer,
                                                 GLint textureFormat1, GLenum textureFormat2, GLenum textureFormatType,
                                                 bool generateMipmap, bool multisampleFBO, GLint numberOfSamples) noexcept
  : openGLDriverInfo_(openGLDriverInfo)
  , depthRenderBuffer_(depthRenderBuffer)
  , useTexture_(useTexture)
  , useDepthTexture_(useDepthTexture)
  , depthStencilRenderBuffer_(depthStencilRenderBuffer)
  , textureFormat1_(textureFormat1)
  , textureFormat2_(textureFormat2)
  , textureFormatType_(textureFormatType)
  , generateMipmap_(generateMipmap)
  , multisampleFBO_(multisampleFBO)
  , numberOfSamples_(numberOfSamples)
{
  // choose which boolean to turn on at the same time
  if (depthRenderBuffer_)
  {
    depthStencilRenderBuffer_ = false;
  }
  if (depthStencilRenderBuffer_)
  {
    useDepthTexture_ = depthRenderBuffer_ = false;
  }
}

OpenGLFrameBufferObject::~OpenGLFrameBufferObject() noexcept
{
  disposeFrameBufferObjectResources();
}

/**
*  Sets the depthRenderBuffer value.
*/
void OpenGLFrameBufferObject::setDepthRenderBuffer(bool depthRenderBuffer)
{
  depthRenderBuffer_ = depthRenderBuffer;

  if (depthRenderBuffer_)
  {
    depthStencilRenderBuffer_ = false;
  }
}

/**
*  Sets the shadowMap value.
*/
void OpenGLFrameBufferObject::setUseDepthTexture(bool useDepthTexture)
{
  useDepthTexture_ = useDepthTexture;

  if (useDepthTexture_)
  {
    depthRenderBuffer_ = true;
  }

  if (depthRenderBuffer_)
  {
    depthStencilRenderBuffer_ = false;
  }
}

/**
*  Sets the depthStencilRenderBuffer value.
*/
void OpenGLFrameBufferObject::setDepthStencilRenderBuffer(bool depthStencilRenderBuffer)
{
  depthStencilRenderBuffer_ = depthStencilRenderBuffer;

  if (depthStencilRenderBuffer_)
  {
    useDepthTexture_ = depthRenderBuffer_ = false;
  }
}

/**
*  Sets the multisampleFBO value.
*/
void OpenGLFrameBufferObject::setMultisampleFBO(bool multisampleFBO, GLint numberOfSamples)
{
  multisampleFBO_  = multisampleFBO;
  numberOfSamples_ = numberOfSamples;
}

/**
*  Initializes all Frame Buffer Object resources.
*/
void OpenGLFrameBufferObject::initFrameBufferObjectResources(GLsizei width, GLsizei height, GLenum textureUnit, GLenum depthTextureUnit)
{
  width_  = width;
  height_ = height;

  if (openGLDriverInfo_->supports330Shaders())
  {
    // generate dummy vao
    if (!glIsVertexArray(dummyVao_))
    {
      glGenVertexArrays(1, &dummyVao_);
    }
  }

  //  allocate a framebuffer object
  glGenFramebuffers(1, &fboID_);
  glBindFramebuffer(GL_FRAMEBUFFER, fboID_);

  if (multisampleFBO_)
  {
    glGenFramebuffers(1, &fboMultiSampleID_);
    glBindFramebuffer(GL_FRAMEBUFFER, fboMultiSampleID_);

    //  allocate a depth renderbuffer for our depth buffer the same size as our texture
    glGenRenderbuffers(1, &renderBufferMultiSampleID_);
    glBindRenderbuffer(GL_RENDERBUFFER, renderBufferMultiSampleID_);
    glRenderbufferStorageMultisample(GL_RENDERBUFFER, numberOfSamples_, GL_RGBA, width_, height_);
    glBindRenderbuffer(GL_RENDERBUFFER, 0);

    //  attach a renderbuffer to depth attachment point
    glFramebufferRenderbuffer(GL_DRAW_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_RENDERBUFFER, renderBufferMultiSampleID_);
  }

  glGenRenderbuffers(1, &renderBufferID_);
  glBindRenderbuffer(GL_RENDERBUFFER, renderBufferID_);
  //  create a renderbuffer object to store depth & stencil info with GL_DEPTH_STENCIL
  //  check "GL_EXT_packed_depth_stencil" extension spec for more details.
  //  Note: A depth renderable image should be attached the FBO for depth test.
  //  if we don't attach a depth renderable image to the FBO, then
  //  the rendering output will be corrupted because of missing depth test.
  //  if you also need stencil test for your rendering, then you must
  //  attach additional image to the stencil attachment point, too.
  //  Note that we need to allocate a renderbuffer for our buffer the same size as our texture
  const GLenum format = (depthRenderBuffer_) ? GL_DEPTH_COMPONENT : (depthStencilRenderBuffer_) ? GL_DEPTH_STENCIL : GL_RGBA;
  if (multisampleFBO_)
  {
    glRenderbufferStorageMultisample(GL_RENDERBUFFER, numberOfSamples_, format, width_, height_);
  }
  else
  {
    glRenderbufferStorage(GL_RENDERBUFFER, format, width_, height_);
  }
  glBindRenderbuffer(GL_RENDERBUFFER, 0);
  if (depthRenderBuffer_)
  {
    //  attach a renderbuffer to depth attachment point
    glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, renderBufferID_);
  }
  else if (depthStencilRenderBuffer_)
  {
    // attach a renderbuffer to depth stencil attachment point
    glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_STENCIL_ATTACHMENT, GL_RENDERBUFFER, renderBufferID_);
  }

  if (multisampleFBO_)
  {
    glBindFramebuffer(GL_FRAMEBUFFER, fboID_);
  }
  if (useTexture_)
  {
    initTextureResouces(textureUnit);
  }
  if (useDepthTexture_)
  {
    initDepthTextureResouces(depthTextureUnit);
  }

  if (checkFrameBufferStatus())
  {
    DebugConsole_consoleOutLine("checkFrameBufferStatus() reported all ok with FBO status:");
  }
  else
  {
    DebugConsole_consoleOutLine("checkFrameBufferStatus() reported an FBO error:");
  }

  printFramebufferInfo();
  glBindFramebuffer(GL_FRAMEBUFFER, 0);
}

/**
*  Initializes the Frame Buffer Object texture resources.
*/
void OpenGLFrameBufferObject::initTextureResouces(GLenum textureUnit)
{
  //  allocate the texture that we will render into
  if (!glIsTexture(textureID_))
  {
    glGenTextures(1, &textureID_);
  }
  bindTexture(textureUnit);
  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
  glTexImage2D(GL_TEXTURE_2D, 0, textureFormat1_, width_, height_, 0, textureFormat2_, textureFormatType_, nullptr);
  initTextureParameters();

  // attach the texture to FBO color attachment point
  glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, textureID_, 0);
}

/**
*  Initializes the Frame Buffer Object depth texture resources.
*/
void OpenGLFrameBufferObject::initDepthTextureResouces(GLenum depthTextureUnit)
{
  //  allocate the texture that we will render into
  if (!glIsTexture(depthTextureID_))
  {
    glGenTextures(1, &depthTextureID_);
  }
  bindDepthTexture(depthTextureUnit);
  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
  glTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT, width_, height_, 0, GL_DEPTH_COMPONENT, GL_FLOAT, nullptr);
  initTextureParameters();

  // attach the depth texture to FBO depth attachment point
  glFramebufferTexture2D(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_TEXTURE_2D, depthTextureID_, 0);
  //  disable color buffer if we don't attach any color buffer image,
  //  for example, rendering depth buffer only to a texture
  //  otherwise, glCheckFramebufferStatus will not be complete.
  if (!useTexture_)
  {
    glDrawBuffer(GL_NONE);
    glReadBuffer(GL_NONE);
  }
}

/**
*  Initializes the Frame Buffer Object texture parameters.
*/
void OpenGLFrameBufferObject::initTextureParameters() const
{
  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, (generateMipmap_) ? GL_LINEAR_MIPMAP_LINEAR : GL_LINEAR);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  if (generateMipmap_)
  {
    glGenerateMipmap(GL_TEXTURE_2D);
    // glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);
  }
  glBindTexture(GL_TEXTURE_2D, 0);
}

/**
*  Prints information about the Frame Buffer Object (FBO).
*/
void OpenGLFrameBufferObject::printFramebufferInfo() const
{
  DebugConsole_consoleOutLine("\n===== FBO STATUS =====");

  // print max # of color buffers supported by FBO
  GLint colorBufferCount = 0;
  glGetIntegerv(GL_MAX_COLOR_ATTACHMENTS, &colorBufferCount);
  DebugConsole_consoleOutLine("Max Number of Color Buffer Attachment Points: ", colorBufferCount);

  GLint objectType = 0;
  GLint objectId   = 0;

  // print info of the color buffer attachable image
  for (int i = 0; i < colorBufferCount; ++i)
  {
    glGetFramebufferAttachmentParameteriv(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0 + i, GL_FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE, &objectType);
    if (objectType != GL_NONE)
    {
      glGetFramebufferAttachmentParameteriv(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0 + i, GL_FRAMEBUFFER_ATTACHMENT_OBJECT_NAME, &objectId);

      if (objectType == GL_TEXTURE)
      {
        DebugConsole_consoleOutLine("Color Attachment ", i, ": GL_TEXTURE, ", getTextureParameters(objectId));
      }
      else if (objectType == int(GL_RENDERBUFFER))
      {
        DebugConsole_consoleOutLine("Color Attachment ", i, ": GL_RENDERBUFFER, ", getRenderbufferParameters(objectId));
      }
    }
  }

  // print info of the depth buffer attachable image
  glGetFramebufferAttachmentParameteriv(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE, &objectType);
  if (objectType != GL_NONE)
  {
    glGetFramebufferAttachmentParameteriv(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_FRAMEBUFFER_ATTACHMENT_OBJECT_NAME, &objectId);

    switch(objectType)
    {
      case GL_TEXTURE:
        DebugConsole_consoleOutLine("Depth Attachment:   GL_TEXTURE, ", getTextureParameters(objectId));
        break;

      case int(GL_RENDERBUFFER):
        DebugConsole_consoleOutLine("Depth Attachment:   GL_RENDERBUFFER, ", getRenderbufferParameters(objectId));
        break;

      default:
        DebugConsole_consoleOutLine("Depth Attachment:   GL_TEXTURE, ", getTextureParameters(objectId));
        break;
    }
  }

  // print info of the stencil buffer attachable image
  glGetFramebufferAttachmentParameteriv(GL_FRAMEBUFFER, GL_STENCIL_ATTACHMENT, GL_FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE, &objectType);
  if (objectType != GL_NONE)
  {
    glGetFramebufferAttachmentParameteriv(GL_FRAMEBUFFER, GL_STENCIL_ATTACHMENT, GL_FRAMEBUFFER_ATTACHMENT_OBJECT_NAME, &objectId);

    switch(objectType)
    {
      case GL_TEXTURE:
        DebugConsole_consoleOutLine("Stencil Attachment: GL_TEXTURE, ", getTextureParameters(objectId));
        break;

      case int(GL_RENDERBUFFER):
        DebugConsole_consoleOutLine("Stencil Attachment: GL_RENDERBUFFER, ", getRenderbufferParameters(objectId));
        break;

      default:
        DebugConsole_consoleOutLine("Stencil Attachment: GL_TEXTURE, ", getTextureParameters(objectId));
        break;
    }
  }

  DebugConsole_consoleOutLine();
}

/**
*  Returns the texture parameters as string using glGetTexLevelParameteriv().
*/
string OpenGLFrameBufferObject::getTextureParameters(GLuint id) const
{
  if (glIsTexture(id) == GL_FALSE)
  {
    return "Not a Texture object";
  }

  GLsizei width = 0, height = 0, format = 0;
  glBindTexture(GL_TEXTURE_2D, id);
  glGetTexLevelParameteriv(GL_TEXTURE_2D, 0, GL_TEXTURE_WIDTH, &width);            // get texture _width
  glGetTexLevelParameteriv(GL_TEXTURE_2D, 0, GL_TEXTURE_HEIGHT, &height);          // get texture _height
  glGetTexLevelParameteriv(GL_TEXTURE_2D, 0, GL_TEXTURE_INTERNAL_FORMAT, &format); // get texture internal format
  glBindTexture(GL_TEXTURE_2D, 0);

  ostringstream ss;
  ss << width << " x " << height << ", " << convertInternalFormatToString(format);

  return ss.str();
}

/**
*  Returns the renderbuffer parameters as string using glGetRenderbufferParameteriv().
*/
string OpenGLFrameBufferObject::getRenderbufferParameters(GLuint id) const
{
  if (glIsRenderbuffer(id) == GL_FALSE)
  {
    return "Not a Renderbuffer object";
  }

  GLsizei width = 0, height = 0, format = 0;
  glBindRenderbuffer(GL_RENDERBUFFER, id);
  glGetRenderbufferParameteriv(GL_RENDERBUFFER, GL_RENDERBUFFER_WIDTH, &width);            // get render buffer _width
  glGetRenderbufferParameteriv(GL_RENDERBUFFER, GL_RENDERBUFFER_HEIGHT, &height);          // get render buffer _height
  glGetRenderbufferParameteriv(GL_RENDERBUFFER, GL_RENDERBUFFER_INTERNAL_FORMAT, &format); // get render buffer internal format
  glBindRenderbuffer(GL_RENDERBUFFER, 0);

  ostringstream ss;
  ss << width << " x " << height << ", " << convertInternalFormatToString(format);

  return ss.str();
}

/**
*  Binds the framebuffer & sets the viewport to the given texture dimensions (uses glPushAttrib).
*/
void OpenGLFrameBufferObject::startRender() const
{
  if (multisampleFBO_)
  {
    glBindFramebuffer(GL_FRAMEBUFFER, fboMultiSampleID_);
  }
  else
  {
    glBindFramebuffer(GL_FRAMEBUFFER, fboID_);
  }
  glPushAttrib(GL_VIEWPORT_BIT);
  glViewport(0, 0, width_, height_);
}

/**
*  Unbinds the framebuffer & returns to default state.
*  Always restore the viewport when ready to render to the screen (uses glPopAttrib).
*/
void OpenGLFrameBufferObject::finishRender() const
{
  glPopAttrib();
  if (multisampleFBO_)
  {
    glReadBuffer(GL_COLOR_ATTACHMENT0);
    glBindFramebuffer(GL_DRAW_FRAMEBUFFER, fboID_);
    GLenum drawBuffer[1] = { GL_COLOR_ATTACHMENT0 };
    glDrawBuffers(1, &drawBuffer[0]);
    GLbitfield mask = GL_COLOR_BUFFER_BIT;
    if (depthRenderBuffer_ || depthStencilRenderBuffer_)
    {
      mask |= GL_DEPTH_BUFFER_BIT;
    }
    if (depthStencilRenderBuffer_)
    {
      mask |= GL_STENCIL_BUFFER_BIT;
    }
    glBlitFramebuffer(0, 0, width_, height_, 0, 0, width_, height_, mask, (useDepthTexture_ || depthRenderBuffer_ || depthStencilRenderBuffer_) ? GL_NEAREST : GL_LINEAR);
  }
  glBindFramebuffer(GL_FRAMEBUFFER, 0);
}

/**
*  Enable the fbo texture.
*/
void OpenGLFrameBufferObject::enable() const
{
  glEnable(GL_TEXTURE_2D);
}

/**
*  Disable the fbo texture.
*/
void OpenGLFrameBufferObject::disable() const
{
  glDisable(GL_TEXTURE_2D);
}

/**
*  Binds the fbo texture with a given active texture unit.
*/
void OpenGLFrameBufferObject::bindTexture(GLenum textureUnit) const
{
  glActiveTexture(GL_TEXTURE0 + textureUnit);
  glBindTexture(GL_TEXTURE_2D, textureID_);
}

/**
*  Binds the fbo depth texture with a given active depth texture unit.
*/
void OpenGLFrameBufferObject::bindDepthTexture(GLenum depthTextureUnit) const
{
  glActiveTexture(GL_TEXTURE0 + depthTextureUnit);
  glBindTexture(GL_TEXTURE_2D, depthTextureID_);
}

/**
*  Unbinds the fbo texture with a given active texture unit.
*/
void OpenGLFrameBufferObject::unbind(GLenum textureUnit) const
{
  glActiveTexture(GL_TEXTURE0 + textureUnit);
  glBindTexture(GL_TEXTURE_2D, 0);
}

void OpenGLFrameBufferObject::renderTextureToFullScreenQuad(GLenum textureUnit, bool isNvidia, bool useDummyVAO) const
{
  enable();
  bindTexture(textureUnit);
  if (openGLDriverInfo_->supports330Shaders() & useDummyVAO)
  {
    GLAuxiliaryFunctions::createFullScreenQuadWithDummyVAO(isNvidia, dummyVao_);
  }
  else
  {
    GLAuxiliaryFunctions::createFullScreenQuad();
  }
  unbind(textureUnit);
  disable();
}

void OpenGLFrameBufferObject::renderDepthTextureToFullScreenQuad(GLenum depthTextureUnit, bool isNvidia, bool useDummyVAO) const
{
  enable();
  bindDepthTexture(depthTextureUnit);
  if (openGLDriverInfo_->supports330Shaders() & useDummyVAO)
  {
    GLAuxiliaryFunctions::createFullScreenQuadWithDummyVAO(isNvidia, dummyVao_);
  }
  else
  {
    GLAuxiliaryFunctions::createFullScreenQuad();
  }
  unbind(depthTextureUnit);
  disable();
}

void OpenGLFrameBufferObject::disposeFrameBufferObjectResources()
{
  if (openGLDriverInfo_->supports330Shaders())
  {
    // delete dummy vao
    if (glIsVertexArray(dummyVao_))
    {
      glDeleteVertexArrays(1, &dummyVao_);
    }
  }

  // delete the framebuffer
  if (glIsFramebuffer(fboID_))
  {
    glDeleteFramebuffers(1, &fboID_);
    fboID_ = 0;
  }

  if (multisampleFBO_)
  {
    if (glIsFramebuffer(fboMultiSampleID_))
    {
      glDeleteFramebuffers(1, &fboMultiSampleID_);
      fboMultiSampleID_ = 0;
    }

    if (glIsRenderbuffer(renderBufferMultiSampleID_))
    {
      glDeleteRenderbuffers(1, &renderBufferMultiSampleID_);
      renderBufferMultiSampleID_ = 0;
    }
  }

  //  delete the renderbuffer
  if (glIsRenderbuffer(renderBufferID_))
  {
    glDeleteRenderbuffers(1, &renderBufferID_);
    renderBufferID_ = 0;
  }

  //  delete the fbo texture
  if (glIsTexture(textureID_))
  {
    glDeleteTextures(1, &textureID_);
    textureID_ = 0;
  }

  //  delete the fbo depth texture
  if (glIsTexture(depthTextureID_))
  {
    glDeleteTextures(1, &depthTextureID_);
    depthTextureID_ = 0;
  }

  glBindRenderbuffer(GL_RENDERBUFFER, 0);
  glBindFramebuffer(GL_FRAMEBUFFER, 0);

  glActiveTexture(GL_TEXTURE0);
  glBindTexture(GL_TEXTURE_2D, 0);
}