#include "OpenGLDriverInfo.h"
#include "UtilityFunctions.h"
#include <algorithm>

using namespace std;
using namespace OpenGLRenderingEngine;
using namespace Utils::UtilityFunctions;

namespace // anonymous namespace used instead of deprecated 'static' keyword used for cpp variable locality
{
  // const strings for GLSL shader versioning
  const string OPENGL_VERSION_FOR_120_SHADERS = "120";
  const string OPENGL_VERSION_FOR_330_SHADERS = "330";
  const string OPENGL_VERSION_FOR_400_SHADERS = "400";
  const string OPENGL_VERSION_FOR_420_SHADERS = "420";
  const string OPENGL_VERSION_FOR_430_SHADERS = "430";
  const string OPENGL_VERSION_FOR_440_SHADERS = "440";
  const string OPENGL_VERSION_FOR_450_SHADERS = "450";
  const string GLSL_LANGUAGE_MODE = "compatibility";
  // const floats for GLSL shader versioning
  const float MINIMUM_OPENGL_VERSION_FOR_120_SHADERS = 2.1f;
  const float MINIMUM_OPENGL_VERSION_FOR_330_SHADERS = 3.3f;
  const float MINIMUM_OPENGL_VERSION_FOR_400_SHADERS = 4.0f;
  const float MINIMUM_OPENGL_VERSION_FOR_420_SHADERS = 4.2f;
  const float MINIMUM_OPENGL_VERSION_FOR_430_SHADERS = 4.3f;
  const float MINIMUM_OPENGL_VERSION_FOR_440_SHADERS = 4.4f;
  const float MINIMUM_OPENGL_VERSION_FOR_450_SHADERS = 4.5f;
  // put space after vendor strings for fail-safe in-between string check
  const string VENDOR_NVIDIA    = "NVIDIA ";
  const string VENDOR_ATI       = "ATI ";
  const string VENDOR_AMD       = "AMD ";
  const string VENDOR_MICROSOFT = "MICROSOFT ";
  // no space after vendors needed below here
  const string VENDOR_INTEL     = "INTEL";
  const string VENDOR_MESA      = "BRIAN PAUL";
}

float OpenGLDriverInfo::getMinimumGLVersionForQualityRenderingAndShaders()
{
  return MINIMUM_OPENGL_VERSION_FOR_330_SHADERS; // could have been 2.1, but using 3.3 for excluding old drivers/hardware (mainly older Intel gfx cards), ie Geforce8+
}

string OpenGLDriverInfo::getMinimumGLSLVersionFor120Shaders()
{
  return OPENGL_VERSION_FOR_120_SHADERS;
}

string OpenGLDriverInfo::getMinimumGLSLVersionFor330Shaders()
{
  return OPENGL_VERSION_FOR_330_SHADERS;
}

string OpenGLDriverInfo::getMinimumGLSLVersionFor400Shaders()
{
  return OPENGL_VERSION_FOR_400_SHADERS;
}

string OpenGLDriverInfo::getMinimumGLSLVersionFor420Shaders()
{
  return OPENGL_VERSION_FOR_420_SHADERS;
}

string OpenGLDriverInfo::getMinimumGLSLVersionFor430Shaders()
{
  return OPENGL_VERSION_FOR_430_SHADERS;
}

string OpenGLDriverInfo::getMinimumGLSLVersionFor440Shaders()
{
  return OPENGL_VERSION_FOR_440_SHADERS;
}

string OpenGLDriverInfo::getMinimumGLSLVersionFor450Shaders()
{
  return OPENGL_VERSION_FOR_450_SHADERS;
}

string OpenGLDriverInfo::getGLSLLanguageMode()
{
  return GLSL_LANGUAGE_MODE;
}

OpenGLDriverInfo::OpenGLDriverInfo() noexcept
{
  getGLInfo();
}

void OpenGLDriverInfo::getGLInfo()
{
  char* str = nullptr;

  // get vendor string
  if ((str = (char*)glGetString(GL_VENDOR)) != nullptr)
  {
    vendor_ = StringAuxiliaryFunctions::trim(string(str));                  // check nullptr return value
  }

  // get renderer string
  if ((str = (char*)glGetString(GL_RENDERER)) != nullptr)
  {
    renderer_ = StringAuxiliaryFunctions::trim(string(str));                // check nullptr return value
  }

  // get version string
  if ((str = (char*)glGetString(GL_VERSION)) != nullptr)
  {
    version_ = StringAuxiliaryFunctions::trim(string(str));                 // check nullptr return value
  }

  // get shading language version string
  if ((str = (char*)glGetString(GL_SHADING_LANGUAGE_VERSION)) != nullptr)
  {
    shadingLanguageVersion_ = StringAuxiliaryFunctions::trim(string(str));  // check nullptr return value
  }

  // get all extensions as a string
  if ((str = (char*)glGetString(GL_EXTENSIONS)) != nullptr)
  {
    extensions_ = StringAuxiliaryFunctions::tokenize<set<string>>(StringAuxiliaryFunctions::trim(string(str))); // split extensions
  }

  string VendorUpperCase = StringAuxiliaryFunctions::toUpperCase(vendor_); // make sure the vendor is upper case before the search
  isNvidia_    = (string::npos  != VendorUpperCase.find(VENDOR_NVIDIA));
  isAMDATI_    = ((string::npos != VendorUpperCase.find(VENDOR_ATI)) || (string::npos != VendorUpperCase.find(VENDOR_AMD)));
  isIntel_     = (string::npos  != VendorUpperCase.find(VENDOR_INTEL));
  isMesa_      = (string::npos  != VendorUpperCase.find(VENDOR_MESA));
  isMicrosoft_ = (string::npos  != VendorUpperCase.find(VENDOR_MICROSOFT));

  const string versionSubString    = version_.substr(0, version_.find('.') + 2); // make sure to get the _version in format 'x.y' before converting to float as some vendors may pollute this string with extra information
  const float openGLVersion        = StringAuxiliaryFunctions::fromString<float>(versionSubString);
  use120Shaders_ = (openGLVersion >= MINIMUM_OPENGL_VERSION_FOR_120_SHADERS);
  use330Shaders_ = (openGLVersion >= MINIMUM_OPENGL_VERSION_FOR_330_SHADERS);
  use400Shaders_ = (openGLVersion >= MINIMUM_OPENGL_VERSION_FOR_400_SHADERS);
  use420Shaders_ = (openGLVersion >= MINIMUM_OPENGL_VERSION_FOR_420_SHADERS);
  use430Shaders_ = (openGLVersion >= MINIMUM_OPENGL_VERSION_FOR_430_SHADERS);
  use440Shaders_ = (openGLVersion >= MINIMUM_OPENGL_VERSION_FOR_440_SHADERS);
  use450Shaders_ = (openGLVersion >= MINIMUM_OPENGL_VERSION_FOR_450_SHADERS);

#ifdef _WINDOWS
  isVSynchSupported_ = isGLExtensionSupported("WGL_EXT_swap_control");
#else
  _isVSynchSupported = isGLExtensionSupported("GLX_SGI_swap_control");
#endif

  // get number of color bits
  glGetIntegerv(GL_RED_BITS,   &redBits_);
  glGetIntegerv(GL_GREEN_BITS, &greenBits_);
  glGetIntegerv(GL_BLUE_BITS,  &blueBits_);
  glGetIntegerv(GL_ALPHA_BITS, &alphaBits_);

  // get depth bits
  glGetIntegerv(GL_DEPTH_BITS, &depthBits_);

  // get stencil bits
  glGetIntegerv(GL_STENCIL_BITS, &stencilBits_);

  // get number of accumulator color bits
  glGetIntegerv(GL_ACCUM_RED_BITS,   &accumRedBits_);
  glGetIntegerv(GL_ACCUM_GREEN_BITS, &accumGreenBits_);
  glGetIntegerv(GL_ACCUM_BLUE_BITS,  &accumBlueBits_);
  glGetIntegerv(GL_ACCUM_ALPHA_BITS, &accumAlphaBits_);

  // get sample buffers
  glGetIntegerv(GL_SAMPLE_BUFFERS, &sampleBuffers_);

  // get Samples
  glGetIntegerv(GL_SAMPLES, &samples_);

  // get max number of lights allowed
  glGetIntegerv(GL_MAX_LIGHTS, &maxLights_);

  // get max texture resolution
  glGetIntegerv(GL_MAX_TEXTURE_SIZE, &maxTextureSize_);

  // get max texture buffer resolution
  glGetIntegerv(GL_MAX_TEXTURE_BUFFER_SIZE, &maxTextureBufferSize_);

  if (supports_GL_EXT_texture_filter_anisotropic())
  {
    // get max texture max anisotropy
    glGetIntegerv(GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT, &maxTextureMaxAnisotropy_);
  }

  // get max render buffer resolution
  glGetIntegerv(GL_MAX_RENDERBUFFER_SIZE, &maxRenderBufferSize_);

  // get max color attachments (for FBOs)
  glGetIntegerv(GL_MAX_COLOR_ATTACHMENTS, &maxColorAttachments_);

  // get max number of clipping planes
  glGetIntegerv(GL_MAX_CLIP_PLANES, &maxClipPlanes_);

  // get max modelView and projection matrix stacks
  glGetIntegerv(GL_MAX_MODELVIEW_STACK_DEPTH,  &maxModelViewStacks_);
  glGetIntegerv(GL_MAX_PROJECTION_STACK_DEPTH, &maxProjectionStacks_);
  glGetIntegerv(GL_MAX_ATTRIB_STACK_DEPTH,     &maxAttribStacks_);

  // get max texture stacks
  glGetIntegerv(GL_MAX_TEXTURE_STACK_DEPTH, &maxTextureStacks_);

  if (use330Shaders_)
  {
    // get max geometry output vertices
    glGetIntegerv(GL_MAX_GEOMETRY_OUTPUT_VERTICES, &maxGeometryOutputVertices_);
  }

  if (use420Shaders_)
  {
    // get max tessellation patch vertices
    glGetIntegerv(GL_MAX_PATCH_VERTICES, &maxTessellationPatchVertices_);

    // get max tessellation control output components
    glGetIntegerv(GL_MAX_TESS_CONTROL_OUTPUT_COMPONENTS, &maxTessellationControlOutputComponents_);

    // get max tessellation generation level
    glGetIntegerv(GL_MAX_TESS_GEN_LEVEL, &maxTessellationGenerationLevel_);
  }

  if (use330Shaders_)
  {
    // get max uniform buffer bindings
    glGetIntegerv(GL_MAX_UNIFORM_BUFFER_BINDINGS, &maxUniformBufferBindings_);

    // get max uniform buffer block size
    glGetIntegerv(GL_MAX_UNIFORM_BLOCK_SIZE, &maxUniformBlockSize_);

    // get max combined vertex uniform components
    glGetIntegerv(GL_MAX_COMBINED_VERTEX_UNIFORM_COMPONENTS, &maxCombinedVertexUniformComponents_);

    // get max combined geometry uniform components
    glGetIntegerv(GL_MAX_COMBINED_GEOMETRY_UNIFORM_COMPONENTS, &maxCombinedGeometryUniformComponents_);
  }

  if (use420Shaders_)
  {
    // get max combined tessellation control uniform components
    glGetIntegerv(GL_MAX_COMBINED_TESS_CONTROL_UNIFORM_COMPONENTS, &maxCombinedTessellationControlUniformComponents_);

    // get max combined tessellation evaluation uniform components
    glGetIntegerv(GL_MAX_COMBINED_TESS_EVALUATION_UNIFORM_COMPONENTS, &maxCombinedTessellationEvaluationUniformComponents_);
  }

  if (use330Shaders_)
  {
    // get max combined fragment uniform components
    glGetIntegerv(GL_MAX_COMBINED_FRAGMENT_UNIFORM_COMPONENTS, &maxCombinedFragmentUniformComponents_);
  }

  getGLMemoryInfo();
}

void OpenGLDriverInfo::getGLMemoryInfo()
{
  if (isNvidia_ && supports_GL_NVX_gpu_memory_info())
  {
    glGetIntegerv(GL_GPU_MEMORY_INFO_DEDICATED_VIDMEM_NVX,         &GPUMemoryInfoDedicatedVidmemNVX_       );
    glGetIntegerv(GL_GPU_MEMORY_INFO_TOTAL_AVAILABLE_MEMORY_NVX,   &GPUMemoryInfoTotalAvailableMemoryNVX_  );
    glGetIntegerv(GL_GPU_MEMORY_INFO_CURRENT_AVAILABLE_VIDMEM_NVX, &GPUMemoryInfoCurrentAvailableMemoryNVX_);
    glGetIntegerv(GL_GPU_MEMORY_INFO_EVICTION_COUNT_NVX,           &GPUMemoryInfoEvictionCountNVX_         );
    glGetIntegerv(GL_GPU_MEMORY_INFO_EVICTED_MEMORY_NVX,           &GPUMemoryInfoEvictedMemoryNVX_         );
  }
  else if (isAMDATI_ && supports_GL_ATI_meminfo())
  {
    glGetIntegerv(GL_VBO_FREE_MEMORY_ATI,          vboFreeMemoryATI_         );
    glGetIntegerv(GL_TEXTURE_FREE_MEMORY_ATI,      textureFreeMemoryATI_     );
    glGetIntegerv(GL_RENDERBUFFER_FREE_MEMORY_ATI, renderBufferFreeMemoryATI_);
  }
}

string OpenGLDriverInfo::getConciseGLDriverInfo() const
{
  ostringstream ss;
  ss << "Vendor: " << vendor_ << ", OpenGL renderer: " << renderer_ << ", version: " << version_ << ", GLSL: " << shadingLanguageVersion_;

  return ss.str();
}

void OpenGLDriverInfo::printGLInfo() const
{
  ostringstream ss;

  ss << endl; // blank line
  ss << "OpenGL Driver Info" << endl;
  ss << "==================" << endl;
  ss << "Vendor:   "         << vendor_   << endl;
  ss << "Renderer: "         << renderer_ << endl;
  ss << "Version:  "         << version_  << endl;
  ss << "Shading Language Version: " << shadingLanguageVersion_  << endl;

  ss << endl;
  ss << "isNvidia: " << StringAuxiliaryFunctions::toString<bool>(isNvidia_) << endl;
  ss << "isAMDATI: " << StringAuxiliaryFunctions::toString<bool>(isAMDATI_) << endl;
  ss << "isIntel:  "  << StringAuxiliaryFunctions::toString<bool>(isIntel_) << endl;
  ss << "isMesa:   " << StringAuxiliaryFunctions::toString<bool>(isMesa_)   << endl;

  ss << endl;
  ss << "120 GLSL Shaders: " << StringAuxiliaryFunctions::toString<bool>(use120Shaders_) << endl;
  ss << "330 GLSL Shaders: " << StringAuxiliaryFunctions::toString<bool>(use330Shaders_) << endl;
  ss << "400 GLSL Shaders: " << StringAuxiliaryFunctions::toString<bool>(use400Shaders_) << endl;
  ss << "420 GLSL Shaders: " << StringAuxiliaryFunctions::toString<bool>(use420Shaders_) << endl;
  ss << "430 GLSL Shaders: " << StringAuxiliaryFunctions::toString<bool>(use430Shaders_) << endl;
  ss << "440 GLSL Shaders: " << StringAuxiliaryFunctions::toString<bool>(use440Shaders_) << endl;
  ss << "450 GLSL Shaders: " << StringAuxiliaryFunctions::toString<bool>(use450Shaders_) << endl;

  ss << endl;
  ss << "GL_ARB_texture_rectangle:          " << StringAuxiliaryFunctions::toString<bool>(supports_GL_ARB_texture_rectangle())          << endl;
  ss << "GL_ARB_texture_buffer_object:      " << StringAuxiliaryFunctions::toString<bool>(supports_GL_ARB_texture_buffer_object())      << endl;
  ss << "GL_EXT_texture_filter_anisotropic: " << StringAuxiliaryFunctions::toString<bool>(supports_GL_EXT_texture_filter_anisotropic()) << endl;
  ss << "GL_EXT_framebuffer_object:         " << StringAuxiliaryFunctions::toString<bool>(supports_GL_EXT_framebuffer_object())         << endl;
  ss << "GL_EXT_packed_depth_stencil:       " << StringAuxiliaryFunctions::toString<bool>(supports_GL_EXT_packed_depth_stencil())       << endl;
  ss << "GL_EXT_gpu_shader4:                " << StringAuxiliaryFunctions::toString<bool>(supports_GL_EXT_gpu_shader4())                << endl;
  ss << "GL_ARB_geometry_shader4:           " << StringAuxiliaryFunctions::toString<bool>(supports_GL_ARB_geometry_shader4())           << endl;
  ss << "GL_ARB_tessellation_shader:        " << StringAuxiliaryFunctions::toString<bool>(supports_GL_ARB_tessellation_shader())        << endl;
  ss << "GL_ARB_compute_shader:             " << StringAuxiliaryFunctions::toString<bool>(supports_GL_ARB_compute_shader())             << endl;
  ss << "GL_ARB_gpu_shader5:                " << StringAuxiliaryFunctions::toString<bool>(supports_GL_ARB_gpu_shader5())                << endl;
  ss << "GL_ARB_gpu_shader_fp64:            " << StringAuxiliaryFunctions::toString<bool>(supports_GL_ARB_gpu_shader_fp64())            << endl;
  ss << "GL_ARB_vertex_type_2_10_10_10_rev: " << StringAuxiliaryFunctions::toString<bool>(supports_GL_ARB_vertex_type_2_10_10_10_rev()) << endl;
  ss << "GL_NVX_gpu_memory_info:            " << StringAuxiliaryFunctions::toString<bool>(supports_GL_NVX_gpu_memory_info())            << endl;
  ss << "GL_ATI_meminfo:                    " << StringAuxiliaryFunctions::toString<bool>(supports_GL_ATI_meminfo())                    << endl;

  ss << endl;
  ss << "Color Bits(R,G,B,A): (" << redBits_ << ", " << greenBits_ << ", " << blueBits_ << ", " << alphaBits_ << ")" << endl;
  ss << "Depth Bits:   " << depthBits_   << endl;
  ss << "Stencil Bits: " << stencilBits_ << endl;
  ss << "Accumulator Color Bits(R,G,B,A): (" << accumRedBits_ << ", " << accumGreenBits_ << ", " << accumBlueBits_ << ", " << accumAlphaBits_ << ")" << endl;

  ss << endl;
  ss << "Sample Buffers: " << sampleBuffers_ << endl;
  ss << "Samples:        " << samples_       << endl;

  ss << endl;
  ss << "Max Texture Size:                                        " << maxTextureSize_ << " x " << maxTextureSize_ << endl;
  ss << "Max Texture Buffer Size:                                 " << maxTextureBufferSize_    << endl;
  ss << "Max Texture Max Anisotropy:                              " << maxTextureMaxAnisotropy_ << endl;
  ss << "Max Render Buffer Size:                                  " << maxRenderBufferSize_     << endl;
  ss << "Max Color Attachments:                                   " << maxColorAttachments_     << endl;
  ss << "Max Lights:                                              " << maxLights_               << endl;
  ss << "Max Clip Planes:                                         " << maxClipPlanes_           << endl;
  ss << "Max ModelView Matrix Stacks:                             " << maxModelViewStacks_      << endl;
  ss << "Max Projection Matrix Stacks:                            " << maxProjectionStacks_     << endl;
  ss << "Max Attribute Stacks:                                    " << maxAttribStacks_         << endl;
  ss << "Max Texture Stacks:                                      " << maxTextureStacks_        << endl;
  if (use330Shaders_)
  {
    ss << "Max Geometry Output Vertices:                            " << maxGeometryOutputVertices_              << endl;
  }
  if (use420Shaders_)
  {
    ss << "Max Tessellation Patch Vertices:                         " << maxTessellationPatchVertices_           << endl;
    ss << "Max Tessellation Control Output Vertices:                " << maxTessellationControlOutputComponents_ << endl;
    ss << "Max Tessellation Generation Level:                       " << maxTessellationGenerationLevel_         << endl;
  }
  if (use330Shaders_)
  {
    ss << "Max Uniform Buffer Bindings:                             " << maxUniformBufferBindings_               << endl;
    ss << "Max Uniform Block Size:                                  " << maxUniformBlockSize_                    << endl;
    ss << "Max Combined Vertex Uniform Components:                  " << maxCombinedVertexUniformComponents_     << endl;
    ss << "Max Combined Geometry Uniform Components:                " << maxCombinedGeometryUniformComponents_   << endl;
  }
  if (use420Shaders_)
  {
    ss << "Max Combined Tessellation Control Uniform Components:    " << maxCombinedTessellationControlUniformComponents_    << endl;
    ss << "Max Combined Tessellation Evaluation Uniform Components: " << maxCombinedTessellationEvaluationUniformComponents_ << endl;
  }
  if (use330Shaders_)
  {
    ss << "Max Combined Fragment Uniform Components:                " << maxCombinedFragmentUniformComponents_               << endl;
  }

  const size_t totalNumberOfExtensions = extensions_.size();
  ss << endl;
  ss << "Total Number of extensions: "    << totalNumberOfExtensions << endl;
  ss << "===============================" << endl;
  for (const auto& currentExtension : extensions_)
  {
    ss << currentExtension << endl;
  }

  DebugConsole_consoleOutLine(ss.str());

  printGLMemoryInfo();
}

void OpenGLDriverInfo::printGLMemoryInfo() const
{
  ostringstream ss;

  ss << endl << "======================================================================" << endl;

  if (isNvidia_ && supports_GL_NVX_gpu_memory_info())
  {
    ss << "NVX GPU Dedicated Video Memory: "     << (GPUMemoryInfoDedicatedVidmemNVX_ >> 10)        << " Mb" << endl;
    ss << "NVX GPU Total Available Memory: "     << (GPUMemoryInfoTotalAvailableMemoryNVX_ >> 10)   << " Mb" << endl;
    ss << "NVX GPU Currently Available Memory: " << (GPUMemoryInfoCurrentAvailableMemoryNVX_ >> 10) << " Mb" << endl;
    ss << "NVX GPU Eviction Count: "             << (GPUMemoryInfoEvictionCountNVX_ >> 10)          << " Mb" << endl;
    ss << "NVX GPU Evicted Memory: "             << (GPUMemoryInfoEvictedMemoryNVX_ >> 10)          << " Mb" << endl;

    ss << "======================================================================" << endl;
  }
  else if (isAMDATI_ && supports_GL_ATI_meminfo())
  {
    ss << "VBO_FREE_MEMORY_ATI report:" << endl;
    ss << "  Total memory free in the pool: "            << (vboFreeMemoryATI_[0] >> 10) << " Mb" << endl;
    ss << "  Largest available free block in the pool: " << (vboFreeMemoryATI_[1] >> 10) << " Mb" << endl;
    ss << "  Total auxiliary memory free: "              << (vboFreeMemoryATI_[2] >> 10) << " Mb" << endl;
    ss << "  Largest auxiliary free block: "             << (vboFreeMemoryATI_[3] >> 10) << " Mb" << endl;
    ss << endl;

    ss << "TEXTURE_FREE_MEMORY_ATI report:" << endl;
    ss << "  Total memory free in the pool: "            << (textureFreeMemoryATI_[0] >> 10) << " Mb" << endl;
    ss << "  Largest available free block in the pool: " << (textureFreeMemoryATI_[1] >> 10) << " Mb" << endl;
    ss << "  Total auxiliary memory free: "              << (textureFreeMemoryATI_[2] >> 10) << " Mb" << endl;
    ss << "  Largest auxiliary free block: "             << (textureFreeMemoryATI_[3] >> 10) << " Mb" << endl;
    ss << endl;

    ss << "RENDERBUFFER_FREE_MEMORY_ATI report:" << endl;
    ss << "  Total memory free in the pool: "            << (renderBufferFreeMemoryATI_[0] >> 10) << " Mb" << endl;
    ss << "  Largest available free block in the pool: " << (renderBufferFreeMemoryATI_[1] >> 10) << " Mb" << endl;
    ss << "  Total auxiliary memory free: "              << (renderBufferFreeMemoryATI_[2] >> 10) << " Mb" << endl;
    ss << "  Largest auxiliary free block: "             << (renderBufferFreeMemoryATI_[3] >> 10) << " Mb" << endl;

    ss << "======================================================================" << endl;
  }

  DebugConsole_consoleOutLine(ss.str());
}