#include "OpenGLAssimpModelLoader.h"
#include "EnvironmentConfig.h"
#include "UtilityFunctions.h"
#include "cimport.h"
#include "postprocess.h"

using namespace std;
using namespace OpenGLRenderingEngine;
using namespace Utils::UtilityFunctions;

namespace // anonymous namespace used instead of deprecated 'static' keyword used for cpp variable locality
{
  const string ASSIMP_LOG_ERROR_STRING_1 = "failed";
  const string ASSIMP_LOG_ERROR_STRING_2 = "Found issues";
  const string MODEL_BIN_EXTENSION_NAME  = ".bin";
  const string MODEL_ZIP_EXTENSION_NAME  = ".zip";

  inline void color4ToFloat4(const aiColor4D& c, float f[4])
  {
    f[0] = c.r;
    f[1] = c.g;
    f[2] = c.b;
    f[3] = c.a;
  }

  inline void setFloat4(float f[4], float a, float b, float c, float d)
  {
    f[0] = a;
    f[1] = b;
    f[2] = c;
    f[3] = d;
  }

  inline void applyMaterial(const aiMaterial* material)
  {
    float c[4]      = { 0.0f };
    GLenum fillMode = 0;
    int return1 = 0;
    int return2 = 0;
    aiColor4D diffuse{};
    aiColor4D specular{};
    aiColor4D ambient{};
    aiColor4D emission{};
    float shininess = 0.0f, strength = 0.0f;
    int twoSided         = 0;
    int wireframe        = 0;
    unsigned int maximum = 0;

    setFloat4(c, 0.8f, 0.8f, 0.8f, 1.0f);
    if (AI_SUCCESS == aiGetMaterialColor(material, AI_MATKEY_COLOR_DIFFUSE, &diffuse))
    {
      color4ToFloat4(diffuse, c);
    }
    glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, c);

    setFloat4(c, 0.0f, 0.0f, 0.0f, 1.0f);
    if (AI_SUCCESS == aiGetMaterialColor(material, AI_MATKEY_COLOR_SPECULAR, &specular))
    {
      color4ToFloat4(specular, c);
    }
    glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, c);

    setFloat4(c, 0.2f, 0.2f, 0.2f, 1.0f);
    if (AI_SUCCESS == aiGetMaterialColor(material, AI_MATKEY_COLOR_AMBIENT, &ambient))
    {
      color4ToFloat4(ambient, c);
    }
    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, c);

    setFloat4(c, 0.0f, 0.0f, 0.0f, 1.0f);
    if (AI_SUCCESS == aiGetMaterialColor(material, AI_MATKEY_COLOR_EMISSIVE, &emission))
    {
      color4ToFloat4(emission, c);
    }
    glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, c);

    maximum = 1;
    return1 = aiGetMaterialFloatArray(material, AI_MATKEY_SHININESS, &shininess, &maximum);
    if (return1 == AI_SUCCESS)
    {
      maximum = 1;
      return2 = aiGetMaterialFloatArray(material, AI_MATKEY_SHININESS_STRENGTH, &strength, &maximum);
      if (return2 == AI_SUCCESS)
      {
        glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, shininess * strength);
      }
      else
      {
        glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, shininess);
      }
    }
    else
    {
      glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, 0.0f);
      setFloat4(c, 0.0f, 0.0f, 0.0f, 0.0f);
      glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, c);
    }

    maximum = 1;
    if (AI_SUCCESS == aiGetMaterialIntegerArray(material, AI_MATKEY_ENABLE_WIREFRAME, &wireframe, &maximum))
    {
      fillMode = wireframe ? GL_LINE : GL_FILL;
    }
    else
    {
      fillMode = GL_FILL;
    }
    glPolygonMode(GL_FRONT_AND_BACK, fillMode);

    maximum = 1;
    if ((AI_SUCCESS == aiGetMaterialIntegerArray(material, AI_MATKEY_TWOSIDED, &twoSided, &maximum)) && twoSided)
    {
      glDisable(GL_CULL_FACE);
    }
    else
    {
      glEnable(GL_CULL_FACE);
    }
  }

  void recursiveRender(const aiScene* scene, const aiNode* node)
  {
    aiMatrix4x4 m = node->mTransformation;

    // update transform
    aiTransposeMatrix4(&m);
    glPushMatrix();
    glMultMatrixf((float*)&m);

    // draw all meshes assigned to this node
    const aiMesh* mesh = nullptr;
    const aiFace* face = nullptr;
    GLenum faceMode = 0;
    GLuint index    = 0;
    for (unsigned int n = 0; n < node->mNumMeshes; ++n)
    {
      mesh = scene->mMeshes[node->mMeshes[n]];

      applyMaterial(scene->mMaterials[mesh->mMaterialIndex]);

      if (mesh->mNormals == nullptr)
      {
        glDisable(GL_LIGHTING);
      }
      else
      {
        glEnable(GL_LIGHTING);
      }

      for (unsigned int t = 0; t < mesh->mNumFaces; ++t)
      {
        face = &mesh->mFaces[t];
        switch (face->mNumIndices)
        {
        case 1:  faceMode = GL_POINTS;    break;
        case 2:  faceMode = GL_LINES;     break;
        case 3:  faceMode = GL_TRIANGLES; break;
        default: faceMode = GL_POLYGON;   break;
        }

        glBegin(faceMode);
        for (unsigned int i = 0; i < face->mNumIndices; ++i)
        {
          index = face->mIndices[i];
          if (mesh->mColors[0] != nullptr)
          {
            glColor4fv((GLfloat*)&mesh->mColors[0][index]);
          }
          if (mesh->mNormals != nullptr)
          {
            glNormal3fv(&mesh->mNormals[index].x);
          }
          glVertex3fv(&mesh->mVertices[index].x);
        }
        glEnd();
      }
    }

    // draw all children
    for (unsigned int n = 0; n < node->mNumChildren; ++n)
    {
      recursiveRender(scene, node->mChildren[n]);
    }

    glPopMatrix();
  }

  /**
  *  Reads the model loader descriptor file.
  */
  vector<string> readModelLoaderDescriptorFile(const string& fullpathWithFileName)
  {
    vector<string> allFileNames;
    ifstream in;
    in.open(fullpathWithFileName.c_str(), ios::in); // Read file
    if (!StdReadWriteFileFunctions::assure(in, fullpathWithFileName))
    {
      return allFileNames;
    }

    string line;
    while (getline(in, line)) // getline() removes '\n'
    {
      if (line.empty() || (line[0] == '#')) // '#' starts a comment line or empty, skipping
      {
        continue;
      }

      allFileNames.emplace_back(move(line));
    }
    in.close();

    return allFileNames;
  }

  /**
  *  Reads the log file.
  */
  bool readLogFileAndCheckForErrorsForModelDataWithTriangulationAndIndexing(const string& logFilename)
  {
    ifstream in;
    in.open(logFilename.c_str(), ios::in);
    if (!StdReadWriteFileFunctions::assure(in, logFilename))
    {
      return true;
    }

    string line;
    ostringstream ss;
    while (getline(in, line)) // getline() removes '\n'
    {
      ss << line << endl;     // must add end-of-line
    }
    in.close();

    string logFileString = ss.str();
    const bool error1 = (string::npos != logFileString.find(ASSIMP_LOG_ERROR_STRING_1));
    const bool error2 = (string::npos != logFileString.find(ASSIMP_LOG_ERROR_STRING_2));

    return (error1 || error2);
  }
}

OpenGLAssimpModelLoader::~OpenGLAssimpModelLoader() noexcept
{
  resetSceneDataStructures();
}

void OpenGLAssimpModelLoader::resetSceneDataStructures()
{
  totalNumberOfFaces_    = 0;
  totalNumberOfVertices_ = 0;
  sceneScaleFactor_      = 1.0f;

  scene_ = nullptr;
  sceneMin_.x    = sceneMin_.y    = sceneMin_.z    =  1e10f;
  sceneMax_.x    = sceneMax_.y    = sceneMax_.z    = -1e10f;
  sceneCenter_.x = sceneCenter_.y = sceneCenter_.z =   0.0f;

  int key = 0;
  for (const auto& mapEntry : allModelStorageVerticesData_)
  {
    key = mapEntry.first;
    const auto& positionColors  = allModelStorageColorsData_.find(key);
    if (positionColors != allModelStorageColorsData_.end())
    {
      allModelStorageColorsData_[key].clear();
    }
    const auto& positionNormals  = allModelStorageNormalsData_.find(key);
    if (positionNormals != allModelStorageNormalsData_.end())
    {
      allModelStorageNormalsData_[key].clear();
    }
    const auto& positionIndices  = allModelStorageIndicesData_.find(key);
    if (positionIndices != allModelStorageIndicesData_.end())
    {
      allModelStorageIndicesData_[key].clear();
    }
    allModelStorageVerticesData_[key].clear();
  }
}

void OpenGLAssimpModelLoader::load(const string& modelDirectory, const string& modelFileName, const string& modelLoaderDescriptorFileName, bool triangulate, bool generateSmoothNormals, bool joinIdenticalVertices)
{
  resetSceneDataStructures();

  if (modelLoaderDescriptorFileName.empty()) // single model to be loaded for the scene
  {
    loadModel(modelDirectory, modelFileName, triangulate, generateSmoothNormals, joinIdenticalVertices);
  }
  else
  {
    vector<string> allFileNames = readModelLoaderDescriptorFile(modelDirectory + modelLoaderDescriptorFileName);
    for (const auto& fileName : allFileNames)
    {
      DebugConsole_consoleOutLine("Now loading file: ", fileName, "\n");
      loadModel(modelDirectory, fileName, triangulate, generateSmoothNormals, false); // enforce indexing off with descriptor files to avoid rendering problems with indexing multiple model parts
    }
  }
}

void OpenGLAssimpModelLoader::recursiveRenderRoot() const
{
  recursiveRender(scene_, scene_->mRootNode);
}

void OpenGLAssimpModelLoader::getBoundingBoxForNode(const aiNode* node, aiMatrix4x4* trafo, aiVector3D& minimum, aiVector3D& maximum) const
{
  const aiMatrix4x4 prev = *trafo;

  aiMultiplyMatrix4(trafo, &node->mTransformation);

  for (unsigned int n = 0; n < node->mNumMeshes; ++n)
  {
    const aiMesh* mesh = scene_->mMeshes[node->mMeshes[n]];
    for (unsigned int t = 0; t < mesh->mNumVertices; ++t)
    {
      aiVector3D tmp = mesh->mVertices[t];
      aiTransformVecByMatrix4(&tmp, trafo);

      minimum.x = min<float>(minimum.x, tmp.x);
      minimum.y = min<float>(minimum.y, tmp.y);
      minimum.z = min<float>(minimum.z, tmp.z);

      maximum.x = max<float>(maximum.x, tmp.x);
      maximum.y = max<float>(maximum.y, tmp.y);
      maximum.z = max<float>(maximum.z, tmp.z);
    }
  }

  for (unsigned int n = 0; n < node->mNumChildren; ++n)
    getBoundingBoxForNode(node->mChildren[n], trafo, minimum, maximum);
  *trafo = prev;
}

void OpenGLAssimpModelLoader::getBoundingBox(aiVector3D& minimum, aiVector3D& maximum) const
{
  aiMatrix4x4 trafo{};
  aiIdentityMatrix4(&trafo);

  minimum.x = minimum.y = minimum.z =  1e10f;
  maximum.x = maximum.y = maximum.z = -1e10f;

  getBoundingBoxForNode(scene_->mRootNode, &trafo, minimum, maximum);
}

void OpenGLAssimpModelLoader::recursiveStoreModelData(const aiScene* scene, const aiNode* node)
{
  const aiMesh* mesh = nullptr;
  const aiFace* face = nullptr;
  GLuint index = 0;
  for (unsigned int n = 0; n < node->mNumMeshes; ++n)
  {
    mesh = scene->mMeshes[node->mMeshes[n]];
    totalNumberOfFaces_ += mesh->mNumFaces;

    for (unsigned int t = 0; t < mesh->mNumFaces; ++t)
    {
      face = &mesh->mFaces[t];
      totalNumberOfVertices_ += face->mNumIndices;

      for (unsigned int i = 0; i < face->mNumIndices; ++i)
      {
        index = face->mIndices[i];
        if (mesh->mColors[0] != nullptr)
        {
          storeColorDataInMap(face->mNumIndices, &mesh->mColors[0][index]);
        }
        if (mesh->mNormals != nullptr)
        {
          storeNormalDataInMap(face->mNumIndices, &mesh->mNormals[index]);
        }
        storeVertexDataInMap(face->mNumIndices, &mesh->mVertices[index]);
      }
    }
  }

  // draw all children
  for (unsigned int n = 0; n < node->mNumChildren; ++n)
    recursiveStoreModelData(scene, node->mChildren[n]);
}

void OpenGLAssimpModelLoader::storeColorDataInMap(int numberOfIndices, const aiColor4D* color)
{
  const auto& position = allModelStorageColorsData_.find(numberOfIndices);
  if (position == allModelStorageColorsData_.end())
  {
    allModelStorageColorsData_.emplace(numberOfIndices, vector<GLfloat>());
  }
  auto& container = allModelStorageColorsData_[numberOfIndices];
  container.emplace_back(color->r);
  container.emplace_back(color->g);
  container.emplace_back(color->b);
  container.emplace_back(color->a);
}

void OpenGLAssimpModelLoader::storeNormalDataInMap(int numberOfIndices, const aiVector3D* normal)
{
  const auto& position = allModelStorageNormalsData_.find(numberOfIndices);
  if (position == allModelStorageNormalsData_.end())
  {
    allModelStorageNormalsData_.emplace(numberOfIndices, vector<GLfloat>());
  }
  auto& container = allModelStorageNormalsData_[numberOfIndices];
  container.emplace_back(normal->x);
  container.emplace_back(normal->y);
  container.emplace_back(normal->z);
}

void OpenGLAssimpModelLoader::storeVertexDataInMap(int numberOfIndices, const aiVector3D* vertex)
{
  const auto& position = allModelStorageVerticesData_.find(numberOfIndices);
  if (position == allModelStorageVerticesData_.end())
  {
    allModelStorageVerticesData_.emplace(numberOfIndices, vector<GLfloat>());
  }
  auto& container = allModelStorageVerticesData_[numberOfIndices];
  container.emplace_back(vertex->x);
  container.emplace_back(vertex->y);
  container.emplace_back(vertex->z);
}

void OpenGLAssimpModelLoader::recursiveStoreModelDataWithTriangulationAndIndexing(const aiScene* scene, const aiNode* node)
{
  const int numberOfTriangleIndices = 3;
  const aiMesh* mesh = nullptr;

  for (unsigned int n = 0; n < node->mNumMeshes; ++n)
  {
    mesh = scene->mMeshes[node->mMeshes[n]];
    totalNumberOfFaces_ += mesh->mNumFaces;

    for (unsigned int i = 0; i < mesh->mNumVertices; ++i)
    {
      if (mesh->mColors[0] != nullptr)
      {
        storeColorDataInMap(numberOfTriangleIndices, &mesh->mColors[0][i]);
      }
      if (mesh->mNormals != nullptr)
      {
        storeNormalDataInMap(numberOfTriangleIndices, &mesh->mNormals[i]);
      }
      storeVertexDataInMap(numberOfTriangleIndices, &mesh->mVertices[i]);
    }

    for (unsigned int t = 0; t < mesh->mNumFaces; ++t)
    {
      storeIndexDataInMap(numberOfTriangleIndices, mesh->mFaces[t].mIndices);
    }

    totalNumberOfVertices_ += numberOfTriangleIndices * mesh->mNumFaces;
  }

  // draw all children
  for (unsigned int n = 0; n < node->mNumChildren; ++n)
  {
    recursiveStoreModelDataWithTriangulationAndIndexing(scene, node->mChildren[n]);
  }
}

void OpenGLAssimpModelLoader::storeIndexDataInMap(int numberOfIndices, const GLuint* index)
{
  const auto& position  = allModelStorageIndicesData_.find(numberOfIndices);
  if (position == allModelStorageIndicesData_.end())
  {
    allModelStorageIndicesData_.emplace(numberOfIndices, vector<GLuint>());
  }
  auto& container = allModelStorageIndicesData_[numberOfIndices];
  container.emplace_back(index[0]);
  container.emplace_back(index[1]);
  container.emplace_back(index[2]);
}

float OpenGLAssimpModelLoader::getSceneScaleFactor(aiVector3D& minimum, aiVector3D& maximum) const
{
  float scaleFactor = maximum.x - minimum.x;
  scaleFactor = max<float>(maximum.y - minimum.y, scaleFactor);
  scaleFactor = max<float>(maximum.z - minimum.z, scaleFactor);
  scaleFactor = 1.0f / scaleFactor;

  return scaleFactor;
}

void OpenGLAssimpModelLoader::reportModelPrimitiveUsage()
{
  GLenum faceMode = 0;
  for (const auto& mapEntry : allModelStorageVerticesData_)
  {
    int key                  = mapEntry.first;
    vector<GLfloat> vertices = mapEntry.second;
    switch (key)
    {
      case 1:
        faceMode = GL_POINTS;
        DebugConsole_consoleOutLine("Model has primitives of type GL_POINTS with ", vertices.size() / 3, " vertices.\nThe vertex stride is ", key, " (", ((vertices.size() / 3) / key), " points).\n");
        break;
      case 2:
        faceMode = GL_LINES;
        DebugConsole_consoleOutLine("Model has primitives of type GL_LINES with ", vertices.size() / 3, " vertices.\nThe vertex stride is ", key, " (", ((vertices.size() / 3) / key), " lines).\n");
        break;
      case 3:
        faceMode = GL_TRIANGLES;
        DebugConsole_consoleOutLine("Model has primitives of type GL_TRIANGLES with ", vertices.size() / 3, " vertices.\nThe vertex stride is ", key, " (", ((vertices.size() / 3) / key), " triangles).\n");
        break;
      default:
        faceMode = GL_POLYGON;
        DebugConsole_consoleOutLine("Model has primitives of type GL_POLYGON with ", vertices.size() / 3, " vertices.\nThe vertex stride is ", key, " (", ((vertices.size() / 3) / key), " polygons).\n");
        break;
    }
    const auto& positionNormals  = allModelStorageNormalsData_.find(key);
    if (positionNormals != allModelStorageNormalsData_.end())
    {
      vector<GLfloat> normals = allModelStorageNormalsData_[key];
      if (!normals.empty())
      {
        DebugConsole_consoleOutLine("Model has associated normals with ", normals.size() / 3, " normals.\nThe normal stride is 3 (", ((normals.size() / 3) / key), " normals per primitive).\n");
      }
    }
    const auto& positionColors  = allModelStorageColorsData_.find(key);
    if (positionColors != allModelStorageColorsData_.end())
    {
      vector<GLfloat> colors = allModelStorageColorsData_[key];
      if (!colors.empty())
      {
        DebugConsole_consoleOutLine("Model has associated colors with ", colors.size() / 4, " colors.\nThe color stride is 4 (", ((colors.size() / 4) / key), " colors per primitive).\n");
      }
    }
    const auto& positionIndices  = allModelStorageIndicesData_.find(key);
    if (positionIndices != allModelStorageIndicesData_.end())
    {
      vector<GLuint> indices = allModelStorageIndicesData_[key];
      if (!indices.empty())
      {
        DebugConsole_consoleOutLine("Model has associated indices with ", indices.size(), " indices.\n");
      }
    }
  }
}

void OpenGLAssimpModelLoader::loadModel(const string& modelDirectory, const string& modelFileName, bool triangulate, bool generateSmoothNormals, bool joinIdenticalVertices)
{
  if (triangulate && joinIdenticalVertices && loadFromBinaryFilesForModelDataWithTriangulationAndIndexing(modelDirectory, modelFileName))
  {
    getBoundingBoxForModelDataWithTriangulationAndIndexing();
  }
  else
  {
    aiLogStream stream{};
    // log messages to assimp_log.txt
    string logFilename = modelDirectory + modelFileName + string(".log");

#ifdef GPU_FRAMEWORK_DEBUG
    // get a handle to the predefined STDOUT log stream and attach
    // it to the logging system. It remains active for all further
    // calls to aiImportFile(Ex) and aiApplyPostProcessing.
    stream = aiGetPredefinedLogStream(aiDefaultLogStream_STDOUT, nullptr);
    aiAttachLogStream(&stream);
#endif // GPU_FRAMEWORK_DEBUG

    // same procedure, but this stream now writes the
    stream = aiGetPredefinedLogStream(aiDefaultLogStream_FILE, logFilename.c_str());
    aiAttachLogStream(&stream);

    if (loadAsset(modelDirectory, modelFileName, triangulate, generateSmoothNormals, joinIdenticalVertices))
    {
      DebugConsole_consoleOutLine("\nloadModel('", modelDirectory + modelFileName, "') initialized ok.\n");

      if (getenv("MODEL_IS_BROKEN"))
      {
        glFrontFace(GL_CCW);
      }

      if (triangulate && joinIdenticalVertices && !readLogFileAndCheckForErrorsForModelDataWithTriangulationAndIndexing(logFilename))
      {
        recursiveStoreModelDataWithTriangulationAndIndexing(scene_, scene_->mRootNode);
        writeToBinaryFilesForModelDataWithTriangulationAndIndexing(modelDirectory, modelFileName);
      }
      else
      {
        recursiveStoreModelData(scene_, scene_->mRootNode);
      }
    }
    else
    {
      DebugConsole_consoleOutLine("Error occurred with loadModel().\nNow aborting loading model '", modelDirectory + modelFileName, "'.");
    }

    // cleanup - calling 'aiReleaseImport' is important, as the library
    // keeps internal resources until the scene is freed again. Not 
    // doing so can cause severe resources leaking.
    aiReleaseImport(scene_);

    // We added a log stream to the library, it's our job to disable it
    // again. This will definitely release the last resources allocated
    // by Assimp.
    aiDetachAllLogStreams();
  }

  reportModelPrimitiveUsage();
}

bool OpenGLAssimpModelLoader::loadAsset(const string& modelDirectory, const string& modelFileName, bool triangulate, bool generateSmoothNormals, bool joinIdenticalVertices)
{
  // we are taking one of the post-processing presets to avoid spelling out 20+ single post-processing flags here
  unsigned int pFlags = aiProcess_ImproveCacheLocality | aiProcess_SplitLargeMeshes | aiProcess_OptimizeMeshes | aiProcess_ValidateDataStructure | aiProcess_FindInvalidData;
  if (triangulate)
  {
    pFlags |= aiProcess_Triangulate;
  }
  if (generateSmoothNormals)
  {
    pFlags |= aiProcess_GenSmoothNormals;
  }
  if (joinIdenticalVertices)
  {
    pFlags |= aiProcess_JoinIdenticalVertices;
  }

  string fullpathWithFileName = modelDirectory + modelFileName;
  scene_ = aiImportFile(fullpathWithFileName.c_str(), pFlags);
  if (scene_)
  {
    const aiVector3D previousSceneMin(sceneMin_);
    const aiVector3D previousSceneMax(sceneMax_);
    getBoundingBox(sceneMin_, sceneMax_);
    aiVector3D totalSceneMin(min<float>(previousSceneMin.x, sceneMin_.x), min<float>(previousSceneMin.y, sceneMin_.y), min<float>(previousSceneMin.z, sceneMin_.z));
    aiVector3D totalSceneMax(max<float>(previousSceneMax.x, sceneMax_.x), max<float>(previousSceneMax.y, sceneMax_.y), max<float>(previousSceneMax.z, sceneMax_.z));
    sceneCenter_.x = (totalSceneMin.x + totalSceneMax.x) / 2.0f;
    sceneCenter_.y = (totalSceneMin.y + totalSceneMax.y) / 2.0f;
    sceneCenter_.z = (totalSceneMin.z + totalSceneMax.z) / 2.0f;
    sceneScaleFactor_ = 10.0f * getSceneScaleFactor(totalSceneMin, totalSceneMax);

    return true;
  }

  return false;
}

bool OpenGLAssimpModelLoader::writeToBinaryFilesForModelDataWithTriangulationAndIndexing(const string& modelDirectory, const string& modelFileName)
{
  DebugConsole_consoleOutLine("\nNow writing model to binary files.\n");

  const auto& positionColors  = allModelStorageColorsData_.find(3);
  if (positionColors != allModelStorageColorsData_.end())
  {
    StdReadWriteFileFunctions::writeZipFile(modelDirectory + modelFileName + MODEL_ZIP_EXTENSION_NAME, modelFileName + "_colors"  + MODEL_BIN_EXTENSION_NAME, allModelStorageColorsData_[3].data(), allModelStorageColorsData_[3].size());
  }
  const auto& positionNormals  = allModelStorageNormalsData_.find(3);
  if (positionNormals != allModelStorageNormalsData_.end())
  {
    StdReadWriteFileFunctions::writeZipFile(modelDirectory + modelFileName + MODEL_ZIP_EXTENSION_NAME, modelFileName + "_normals" + MODEL_BIN_EXTENSION_NAME, allModelStorageNormalsData_[3].data(), allModelStorageNormalsData_[3].size());
  }
  const bool succeeded1     = StdReadWriteFileFunctions::writeZipFile(modelDirectory + modelFileName + MODEL_ZIP_EXTENSION_NAME, modelFileName + "_vertices" + MODEL_BIN_EXTENSION_NAME, allModelStorageVerticesData_[3].data(), allModelStorageVerticesData_[3].size());
  const bool succeeded2     = StdReadWriteFileFunctions::writeZipFile(modelDirectory + modelFileName + MODEL_ZIP_EXTENSION_NAME, modelFileName + "_indices"  + MODEL_BIN_EXTENSION_NAME, allModelStorageIndicesData_[3].data(), allModelStorageIndicesData_[3].size());
  const bool totalSucceeded = (succeeded1 && succeeded2);

  DebugConsole_consoleOutLine();

  return totalSucceeded;
}

bool OpenGLAssimpModelLoader::loadFromBinaryFilesForModelDataWithTriangulationAndIndexing(const string& modelDirectory, const string& modelFileName)
{
  DebugConsole_consoleOutLine("\nNow loading model from binary files.\n");

  const bool hasColorsData  = StdReadWriteFileFunctions::readZipFile(modelDirectory + modelFileName + MODEL_ZIP_EXTENSION_NAME, modelFileName + "_colors"   + MODEL_BIN_EXTENSION_NAME, allModelStorageColorsData_[3]);
  if (!hasColorsData) allModelStorageColorsData_.erase(3);
  const bool hasNormalsData = StdReadWriteFileFunctions::readZipFile(modelDirectory + modelFileName + MODEL_ZIP_EXTENSION_NAME, modelFileName + "_normals"  + MODEL_BIN_EXTENSION_NAME, allModelStorageNormalsData_[3]);
  if (!hasNormalsData) allModelStorageNormalsData_.erase(3);
  const bool succeeded1     = StdReadWriteFileFunctions::readZipFile(modelDirectory + modelFileName + MODEL_ZIP_EXTENSION_NAME, modelFileName + "_vertices" + MODEL_BIN_EXTENSION_NAME, allModelStorageVerticesData_[3]);
  const bool succeeded2     = StdReadWriteFileFunctions::readZipFile(modelDirectory + modelFileName + MODEL_ZIP_EXTENSION_NAME, modelFileName + "_indices"  + MODEL_BIN_EXTENSION_NAME, allModelStorageIndicesData_[3]);
  const bool totalSucceeded = (succeeded1 && succeeded2);

  if (totalSucceeded)
  {
    totalNumberOfFaces_    = int(allModelStorageIndicesData_[3].size()) / 3;
    totalNumberOfVertices_ = int(allModelStorageIndicesData_[3].size());
  }

  DebugConsole_consoleOutLine();

  return totalSucceeded;
}

void OpenGLAssimpModelLoader::getBoundingBoxForModelDataWithTriangulationAndIndexing()
{
  sceneMin_.x = sceneMin_.y = sceneMin_.z =  1e10f;
  sceneMax_.x = sceneMax_.y = sceneMax_.z = -1e10f;

  for (size_t i = 0; i < allModelStorageVerticesData_[3].size(); i += 3)
  {
    const aiVector3D vertex{ allModelStorageVerticesData_[3][i    ],
                             allModelStorageVerticesData_[3][i + 1],
                             allModelStorageVerticesData_[3][i + 2]
                           };
    if (vertex.x < sceneMin_.x) sceneMin_.x = vertex.x;
    if (vertex.x > sceneMax_.x) sceneMax_.x = vertex.x;

    if (vertex.y < sceneMin_.y) sceneMin_.y = vertex.y;
    if (vertex.y > sceneMax_.y) sceneMax_.y = vertex.y;

    if (vertex.z < sceneMin_.z) sceneMin_.z = vertex.z;
    if (vertex.z > sceneMax_.z) sceneMax_.z = vertex.z;
  }

  sceneCenter_.x = (sceneMin_.x + sceneMax_.x) / 2.0f;
  sceneCenter_.y = (sceneMin_.y + sceneMax_.y) / 2.0f;
  sceneCenter_.z = (sceneMin_.z + sceneMax_.z) / 2.0f;
  sceneScaleFactor_ = 10.0f * getSceneScaleFactor(sceneMin_, sceneMax_);
}