/*

Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
Released Under a Simplified BSD (FreeBSD) License
for academic, personal & non-commercial use.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the author and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.

A Commercial License is also available for commercial use with
special restrictions and obligations at a one-off fee. See links at:
1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

#pragma once

#ifndef __OpenGLUtilityFunctions_h
#define __OpenGLUtilityFunctions_h

#include "ModuleDLL.h"
#include "EnvironmentConfig.h"
#include "glew.h"
#include <string>
#include <cstdint>
#include <cstring> // for memcpy()
#include <memory>

#define GLAuxiliaryFunctions_checkGLError() \
          do { if (GPU_FRAMEWORK_GL_CONSOLE) OpenGLRenderingEngine::OpenGLUtilityFunctions::GLAuxiliaryFunctions::checkGLErrorImpl(__FILE__, __func__, __LINE__); } while (0)

/** @brief Namespace OpenGLRenderingEngine for the OpenGL rendering.
* @author Thanos Theo, 2018
* @version 14.0.0.0
*/
namespace OpenGLRenderingEngine
{
  /** @brief Namespace OpenGLUtilityFunctions for the OpenGL utility functions.
  * @author Thanos Theo, 2018
  * @version 14.0.0.0
  */
  namespace OpenGLUtilityFunctions
  {
    /** @brief This class contains only static CG & OpenGL related methods.
    * @author Thanos Theo, 2009-2018
    * @version 14.0.0.0
    */
    struct OPENGL_RENDERING_ENGINE_MODULE_API GLAuxiliaryFunctions final
    {
      /**
      *  Flips the given pixel data to adhere to OpenGL's bottom-top coordinate system.
      */
      template <typename T>
      static inline void flipPixels(std::size_t bytesPerPixel, std::size_t width, std::size_t height, T* __restrict pixelData)
      {
        const std::size_t stride = width * bytesPerPixel;
        auto rowData             = std::make_unique<T[]>((stride == 0) ? 1 : stride);
        T* __restrict row  = rowData.get();
        T* __restrict low  = pixelData;
        T* __restrict high = &pixelData[(height - 1) * stride];
        for (; low < high; low += stride, high -= stride)
        {
          std::memcpy(row,  low,  stride);
          std::memcpy(low,  high, stride);
          std::memcpy(high, row,  stride);
        }
      }

      /**
      *  Finds and returns the currently active texture unit.
      */
      static void findCurrentActiveTextureUnit(int textureValues[3]);

      /**
      *  Returns the current texture environment GL constant to a shader enum after polling the GL state.
      */
      static int currentTexEnvModeGLConstantToShaderEnum();

      /**
      *  Converts the texture environment GL constant to a shader enum.
      */
      static int convertTexEnvModeGLConstantToShaderEnum(int texEnvMode);

      /**
      *  Gets the current state of the given GL mode.
      */
      static int getCurrentGLState(GLenum mode);

      /**
      *  Set the VSync on/off state.
      */
      static void setVSynch(int enableVSynch, bool isVSyncSupported);

      /**
      *  Prepare GL high quality rendering (may have a minor speed-hit on older GPUs).
      */
      static void prepareHighQualityRendering(bool isNvidia);

      /**
      *  Prepare GL low quality rendering (may have a minor speed-up on older GPUs).
      */
      static void prepareLowQualityRendering(bool isNvidia);

      /**
      *  Create a fullscreen quad for fullscreen & FBO related effects.
      */
      static void createFullScreenQuad();

      /**
      *  Create a fullscreen quad for fullscreen & FBO related effects with a dummy VAO.
      */
      static void createFullScreenQuadWithDummyVAO(bool isNvidia, GLuint dummyVao);

      /**
      *  Pack given XYZ normal to UInt32 type (used for the GL_INT_2_10_10_10_REV normal packing conversion).
      */
      static uint32_t packNormalToUInt(float x, float y, float z);

      /**
      *  A simple OpenGL error checking routine. This compiles away
      *  to a no-op inline method if the GPU_FRAMEWORK_GL_CONSOLE preprocessor symbol
      *  is not defined during compilation.
      *
      *  - The first parameter is a GLenum.
      *
      *  - The second parameter (optional) is a string that can be
      *    used to indicate the location where the error check occurs.
      *
      *  - The third parameter (optional) determines the destination of the error
      *    message. It defaults to cout, but could also be a file.
      */
      static void checkGLErrorImpl(const char* file, const char* function, int line, GLenum errnum = glGetError());

      GLAuxiliaryFunctions()  = delete;
      ~GLAuxiliaryFunctions() = delete;
      GLAuxiliaryFunctions(const GLAuxiliaryFunctions&) = delete;
      GLAuxiliaryFunctions(GLAuxiliaryFunctions&&)      = delete;
      GLAuxiliaryFunctions& operator=(const GLAuxiliaryFunctions&) = delete;
      GLAuxiliaryFunctions& operator=(GLAuxiliaryFunctions&&)      = delete;
    };
  } // namespace OpenGLUtilityFunctions
} // namespace OpenGLRenderingEngine

#endif // __OpenGLUtilityFunctions_h