/*

Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
Released Under a Simplified BSD (FreeBSD) License
for academic, personal & non-commercial use.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the author and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.

A Commercial License is also available for commercial use with
special restrictions and obligations at a one-off fee. See links at:
1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

#pragma once

#ifndef __OpenGLShaderProgram_h
#define __OpenGLShaderProgram_h

#include "ModuleDLL.h"
#include "glew.h"
#include <unordered_map>
#include <string>

/** @brief Namespace OpenGLRenderingEngine for the OpenGL rendering.
* @author Thanos Theo, 2018
* @version 14.0.0.0
*/
namespace OpenGLRenderingEngine
{
  // forward declarations
  class OpenGLDriverInfo;
  class OpenGLShaderGLSLPreProcessorCommands;
  class OpenGLShaderCompileAndLink;

  /** @brief This abstract class encapsulates usage of a GLSL program. To be inherited from usage-specific sub-classes.
  * @author Thanos Theo, 2009-2018
  * @version 14.0.0.0
  */
  class OPENGL_RENDERING_ENGINE_MODULE_API OpenGLShaderProgram
  {
  public:

    /// integer uniform setter auxiliary function
    void setUniform1i(const std::string& name, GLint value);
    /// integer uniform setter auxiliary function
    void setUniform1iv(const std::string& name, GLsizei count, const GLint* values);
    /// integer uniform setter auxiliary function
    void setUniform2iv(const std::string& name, GLsizei count, const GLint* values);
    /// integer uniform setter auxiliary function
    void setUniform3iv(const std::string& name, GLsizei count, const GLint* values);
    /// integer uniform setter auxiliary function
    void setUniform4iv(const std::string& name, GLsizei count, const GLint* values);

    /// unsigned integer uniform setter auxiliary function
    void setUniform1ui(const std::string& name, GLuint value);
    /// unsigned integer uniform setter auxiliary function
    void setUniform1uiv(const std::string& name, GLsizei count, const GLuint* values);
    /// unsigned integer uniform setter auxiliary function
    void setUniform2uiv(const std::string& name, GLsizei count, const GLuint* values);
    /// unsigned integer uniform setter auxiliary function
    void setUniform3uiv(const std::string& name, GLsizei count, const GLuint* values);
    /// unsigned integer uniform setter auxiliary function
    void setUniform4uiv(const std::string& name, GLsizei count, const GLuint* values);

    /// 64bit integer (GL 4.0+ only) uniform setter auxiliary function
    void setUniform1i64NV(const std::string& name, GLint64EXT value);
    /// 64bit integer (GL 4.0+ only) uniform setter auxiliary function
    void setUniform1i64vNV(const std::string& name, GLsizei count, const GLint64EXT* values);
    /// 64bit integer (GL 4.0+ only) uniform setter auxiliary function
    void setUniform2i64vNV(const std::string& name, GLsizei count, const GLint64EXT* values);
    /// 64bit integer (GL 4.0+ only) uniform setter auxiliary function
    void setUniform3i64vNV(const std::string& name, GLsizei count, const GLint64EXT* values);
    /// 64bit integer (GL 4.0+ only) uniform setter auxiliary function
    void setUniform4i64vNV(const std::string& name, GLsizei count, const GLint64EXT* values);

    /// 64bit unsigned integer (GL 4.0+ only) uniform setter auxiliary function
    void setUniform1ui64NV(const std::string& name, GLuint64EXT value);
    /// 64bit unsigned integer (GL 4.0+ only) uniform setter auxiliary function
    void setUniform1ui64vNV(const std::string& name, GLsizei count, const GLuint64EXT* values);
    /// 64bit unsigned integer (GL 4.0+ only) uniform setter auxiliary function
    void setUniform2ui64vNV(const std::string& name, GLsizei count, const GLuint64EXT* values);
    /// 64bit unsigned integer (GL 4.0+ only) uniform setter auxiliary function
    void setUniform3ui64vNV(const std::string& name, GLsizei count, const GLuint64EXT* values);
    /// 64bit unsigned integer (GL 4.0+ only) uniform setter auxiliary function
    void setUniform4ui64vNV(const std::string& name, GLsizei count, const GLuint64EXT* values);

    /// float uniform setter auxiliary function
    void setUniform1f(const std::string& name, GLfloat value);
    /// float uniform setter auxiliary function
    void setUniform1fv(const std::string& name, GLsizei count, const GLfloat* values);
    /// float uniform setter auxiliary function
    void setUniform2fv(const std::string& name, GLsizei count, const GLfloat* values);
    /// float uniform setter auxiliary function
    void setUniform3fv(const std::string& name, GLsizei count, const GLfloat* values);
    /// float uniform setter auxiliary function
    void setUniform4fv(const std::string& name, GLsizei count, const GLfloat* values);

    /// double (GL 4.0+ only) uniform setter auxiliary function
    void setUniform1d(const std::string& name, GLdouble value);
    /// double (GL 4.0+ only) uniform setter auxiliary function
    void setUniform1dv(const std::string& name, GLsizei count, const GLdouble* values);
    /// double (GL 4.0+ only) uniform setter auxiliary function
    void setUniform2dv(const std::string& name, GLsizei count, const GLdouble* values);
    /// double (GL 4.0+ only) uniform setter auxiliary function
    void setUniform3dv(const std::string& name, GLsizei count, const GLdouble* values);
    /// double (GL 4.0+ only) uniform setter auxiliary function
    void setUniform4dv(const std::string& name, GLsizei count, const GLdouble* values);

    /// generic vertex integer attribute setter auxiliary function
    void setAttribute1i(const std::string& name, GLint value);
    /// generic vertex integer attribute setter auxiliary function
    void setAttribute1iv(const std::string& name, const GLint* values);
    /// generic vertex integer attribute setter auxiliary function
    void setAttribute2iv(const std::string& name, const GLint* values);
    /// generic vertex integer attribute setter auxiliary function
    void setAttribute3iv(const std::string& name, const GLint* values);
    /// generic vertex integer attribute setter auxiliary function
    void setAttribute4iv(const std::string& name, const GLint* values);

    /// generic vertex unsigned integer attribute setter auxiliary function
    void setAttribute1ui(const std::string& name, GLuint value);
    /// generic vertex integer attribute setter auxiliary function
    void setAttribute1uiv(const std::string& name, const GLuint* values);
    /// generic vertex integer attribute setter auxiliary function
    void setAttribute2uiv(const std::string& name, const GLuint* values);
    /// generic vertex integer attribute setter auxiliary function
    void setAttribute3uiv(const std::string& name, const GLuint* values);
    /// generic vertex integer attribute setter auxiliary function
    void setAttribute4uiv(const std::string& name, const GLuint* values);

    /// generic vertex 64bit integer (GL 4.0+ only) attribute setter auxiliary function
    void setAttributeL1i64NV(const std::string& name, GLint64EXT value);
    /// generic vertex 64bit integer (GL 4.0+ only) attribute setter auxiliary function
    void setAttributeL1i64vNV(const std::string& name, const GLint64EXT* values);
    /// generic vertex 64bit integer (GL 4.0+ only) attribute setter auxiliary function
    void setAttributeL2i64vNV(const std::string& name, const GLint64EXT* values);
    /// generic vertex 64bit integer (GL 4.0+ only) attribute setter auxiliary function
    void setAttributeL3i64vNV(const std::string& name, const GLint64EXT* values);
    /// generic vertex 64bit integer (GL 4.0+ only) attribute setter auxiliary function
    void setAttributeL4i64vNV(const std::string& name, const GLint64EXT* values);

    /// generic vertex 64bit unsigned integer (GL 4.0+ only) attribute setter auxiliary function
    void setAttributeL1ui64NV(const std::string& name, GLuint64EXT value);
    /// generic vertex 64bit unsigned integer (GL 4.0+ only) attribute setter auxiliary function
    void setAttributeL1ui64vNV(const std::string& name, const GLuint64EXT* values);
    /// generic vertex 64bit unsigned integer (GL 4.0+ only) attribute setter auxiliary function
    void setAttributeL2ui64vNV(const std::string& name, const GLuint64EXT* values);
    /// generic vertex 64bit unsigned integer (GL 4.0+ only) attribute setter auxiliary function
    void setAttributeL3ui64vNV(const std::string& name, const GLuint64EXT* values);
    /// generic vertex 64bit unsigned integer (GL 4.0+ only) attribute setter auxiliary function
    void setAttributeL4ui64vNV(const std::string& name, const GLuint64EXT* values);

    /// generic vertex unsigned integer packed attribute setter auxiliary function
    /// type parameter: type of packing used on the data. This parameter must be GL_INT_10_10_10_2 or GL_UNSIGNED_INT_10_10_10_2 to specify signed or unsigned data, respectively
    /// normalized parameter: if GL_TRUE, then the values are to be converted to floating point values by normalizing. Otherwise, they are converted directly to floating point values
    void setAttributeP1ui(const std::string& name, GLenum type, GLboolean normalized, GLuint value);
    /// generic vertex unsigned integer packed attribute setter auxiliary function
    /// type parameter: type of packing used on the data. This parameter must be GL_INT_10_10_10_2 or GL_UNSIGNED_INT_10_10_10_2 to specify signed or unsigned data, respectively
    /// normalized parameter: if GL_TRUE, then the values are to be converted to floating point values by normalizing. Otherwise, they are converted directly to floating point values
    void setAttributeP1uiv(const std::string& name, GLenum type, GLboolean normalized, const GLuint* values);
    /// generic vertex unsigned integer packed attribute setter auxiliary function
    /// type parameter: type of packing used on the data. This parameter must be GL_INT_10_10_10_2 or GL_UNSIGNED_INT_10_10_10_2 to specify signed or unsigned data, respectively
    /// normalized parameter: if GL_TRUE, then the values are to be converted to floating point values by normalizing. Otherwise, they are converted directly to floating point values
    void setAttributeP2uiv(const std::string& name, GLenum type, GLboolean normalized, const GLuint* values);
    /// generic vertex unsigned integer packed attribute setter auxiliary function
    /// type parameter: type of packing used on the data. This parameter must be GL_INT_10_10_10_2 or GL_UNSIGNED_INT_10_10_10_2 to specify signed or unsigned data, respectively
    /// normalized parameter: if GL_TRUE, then the values are to be converted to floating point values by normalizing. Otherwise, they are converted directly to floating point values
    void setAttributeP3uiv(const std::string& name, GLenum type, GLboolean normalized, const GLuint* values);
    /// generic vertex unsigned integer packed attribute setter auxiliary function
    /// type parameter: type of packing used on the data. This parameter must be GL_INT_10_10_10_2 or GL_UNSIGNED_INT_10_10_10_2 to specify signed or unsigned data, respectively
    /// normalized parameter: if GL_TRUE, then the values are to be converted to floating point values by normalizing. Otherwise, they are converted directly to floating point values
    void setAttributeP4uiv(const std::string& name, GLenum type, GLboolean normalized, const GLuint* values);

    /// generic vertex float attribute setter auxiliary function
    void setAttribute1f(const std::string& name, GLfloat value);
    /// generic vertex float attribute setter auxiliary function
    void setAttribute1fv(const std::string& name, const GLfloat* values);
    /// generic vertex float attribute setter auxiliary function
    void setAttribute2fv(const std::string& name, const GLfloat* values);
    /// generic vertex float attribute setter auxiliary function
    void setAttribute3fv(const std::string& name, const GLfloat* values);
    /// generic vertex float attribute setter auxiliary function
    void setAttribute4fv(const std::string& name, const GLfloat* values);

    /// generic vertex double (GL 4.0+ only) attribute setter auxiliary function
    void setAttribute1d(const std::string& name, GLdouble value);
    /// generic vertex double (GL 4.0+ only) attribute setter auxiliary function
    void setAttribute1dv(const std::string& name, const GLdouble* values);
    /// generic vertex double (GL 4.0+ only) attribute setter auxiliary function
    void setAttribute2dv(const std::string& name, const GLdouble* values);
    /// generic vertex double (GL 4.0+ only) attribute setter auxiliary function
    void setAttribute3dv(const std::string& name, const GLdouble* values);
    /// generic vertex double (GL 4.0+ only) attribute setter auxiliary function
    void setAttribute4dv(const std::string& name, const GLdouble* values);

    /// initialize shader program function to override
    virtual void initializeShaderProgram() = 0;
    /// enable shader program
    void enableShaderProgram() const;
    /// disable shader program
    void disableShaderProgram() const;

    OpenGLShaderProgram() = delete;
    OpenGLShaderProgram(const OpenGLShaderProgram&) = delete;
    OpenGLShaderProgram(OpenGLShaderProgram&&)      = delete;
    OpenGLShaderProgram& operator=(const OpenGLShaderProgram&) = delete;
    OpenGLShaderProgram& operator=(OpenGLShaderProgram&&)      = delete;

  protected:

    OpenGLDriverInfo* openGLDriverInfo_                                         = nullptr;
    OpenGLShaderGLSLPreProcessorCommands* openGLShaderGLSLPreProcessorCommands_ = nullptr;
    OpenGLShaderCompileAndLink* openGLShaderCompileAndLink_                     = nullptr;

    OpenGLShaderProgram(OpenGLDriverInfo* openGLDriverInfo, bool enableVertexProgramTwoSidedLighting = true) noexcept; // an abstract class has its constructor as protected
    virtual ~OpenGLShaderProgram() noexcept; // virtual destructor so as to be called from implementing sub-classes (only them can be deleted)

  private:

    GLuint shaderProgram_ = 0;
    std::unordered_map<std::string, GLint> allUniformLocationsMap_;
    std::unordered_map<std::string, GLint> allAttribLocationsMap_;

    /// create shader program
    void createGLShaderProgram();
    
    /// get uniform location
    GLint getUniformLocation(const std::string& name);
    
    /// get attribute location
    GLint getAttributeLocation(const std::string& name);

    /// release shader program
    void releaseGLShaderProgram() const;
  };
} // namespace OpenGLRenderingEngine

#endif // __OpenGLShaderProgram_h