/*

Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
Released Under a Simplified BSD (FreeBSD) License
for academic, personal & non-commercial use.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the author and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.

A Commercial License is also available for commercial use with
special restrictions and obligations at a one-off fee. See links at:
1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

#pragma once

#ifndef __OpenGLShaderObjects_h
#define __OpenGLShaderObjects_h

#include "ModuleDLL.h"
#include "glew.h"
#include "OpenGLAssetManager.h"
#include "UtilityFunctions.h"
#include <bitset>

/** @brief Namespace OpenGLRenderingEngine for the OpenGL rendering.
* @author Thanos Theo, 2018
* @version 14.0.0.0
*/
namespace OpenGLRenderingEngine
{
  /** @brief This class is holding all shader objects GL handles and type information.
  * @author Thanos Theo, 2009-2018
  * @version 14.0.0.0
  */
  class OPENGL_RENDERING_ENGINE_MODULE_API OpenGLShaderObjects final
  {
  public:

    /// get the shader type
    bool isUsingShaderType(int shaderTypeEnum) const { return shaderType_->test(Utils::UtilityFunctions::BitManipulationFunctions::getLowestBitPositionOfPowerOfTwoNumber(shaderTypeEnum)); }
    /// set the shader object by shader type
    void setShaderObject(GLuint shaderObject, int shaderTypeEnum) { shaderObjects_[Utils::UtilityFunctions::BitManipulationFunctions::getLowestBitPositionOfPowerOfTwoNumber(shaderTypeEnum)] = shaderObject; }
    /// get the shader object by shader type
    GLuint getShaderObject(int shaderTypeEnum) const { return shaderObjects_[Utils::UtilityFunctions::BitManipulationFunctions::getLowestBitPositionOfPowerOfTwoNumber(shaderTypeEnum)]; }
    /// attach shader objects to program (only if the attach shader objects state was already set to false)
    void attachShaderObjectsToProgram(GLuint shaderProgram);
    /// detach shader objects from program (only if the attach shader objects state was already set to true)
    void detachShaderObjectsFromProgram(GLuint shaderProgram);

    /// number of shader type programmable stages
    std::size_t numberOfShaderTypeProgrammableStages() const { return shaderType_->count(); }
    /// number of created shader objects
    std::size_t numberOfCreatedShaderObjects() const;
    /// get if any shader objects were created
    bool hasCreatedShaderObjects() const { return (numberOfCreatedShaderObjects() > 0); }
    /// get if the number of the shader type programmable stages equals the created shader objects
    bool isEqualNumberOfShaderTypeProgrammableStagesAndCreatedShaderObjects() const { return (numberOfCreatedShaderObjects() == numberOfShaderTypeProgrammableStages()); }

    explicit OpenGLShaderObjects(int shaderType) noexcept;
    ~OpenGLShaderObjects() noexcept;
    OpenGLShaderObjects() = delete;
    OpenGLShaderObjects(const OpenGLShaderObjects&) = delete;
    OpenGLShaderObjects(OpenGLShaderObjects&&)      = delete;
    OpenGLShaderObjects& operator=(const OpenGLShaderObjects&) = delete;
    OpenGLShaderObjects& operator=(OpenGLShaderObjects&&)      = delete;

  private:

    /// shader type flag storage variables
    std::bitset<OpenGLAssetManager::NUMBER_OF_TOTAL_SHADER_TYPES>* shaderType_ = nullptr;
    /// all shader objects GL handles
    GLuint shaderObjects_[OpenGLAssetManager::NUMBER_OF_TOTAL_SHADER_TYPES] = { 0 };
  };
} // namespace OpenGLRenderingEngine

#endif // __OpenGLShaderObjects_h