/*

Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
Released Under a Simplified BSD (FreeBSD) License
for academic, personal & non-commercial use.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the author and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.

A Commercial License is also available for commercial use with
special restrictions and obligations at a one-off fee. See links at:
1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

#pragma once

#ifndef __OpenGLPerlinNoise_h
#define __OpenGLPerlinNoise_h

#include "ModuleDLL.h"
#include "glew.h"
#include "Randomizers.h"
#include <vector>

/** @brief Namespace OpenGLRenderingEngine for the OpenGL rendering.
* @author Thanos Theo, 2018
* @version 14.0.0.0
*/
namespace OpenGLRenderingEngine
{
  /** @brief This class provides Perlin Noise functionality for GLSL Shaders usage through a 3D OpenGL texture.
  *
  *  Coherent Perlin Noise function over 1, 2 or 3 dimensions (original author Prof. Ken Perlin).
  *  Modifications by John Kessenich (GLSL Orange Book & setNoiseFrequency() support).
  *  N-CP conversion by Thanos Theo.
  *
  * @author Thanos Theo, 2009-2018
  * @version 14.0.0.0
  */
  class OPENGL_RENDERING_ENGINE_MODULE_API OpenGLPerlinNoise final
  {
  public:

    void initPerlinNoise(int frequency, std::vector<int>& p, std::vector<double>& g1, std::vector<std::vector<double>>& g2, std::vector<std::vector<double>>& g3);

    double perlinNoise1(double xIn, int frequency);
    double perlinNoise2(double xIn, double yIn, int frequency);
    double perlinNoise3(double xIn, double yIn, double zIn, int frequency);
    double perlinNoise1D(double xIn, double alpha, double beta, int n, int frequency);
    double perlinNoise2D(double xIn, double yIn, double alpha, double beta, int n, int frequency);
    double perlinNoise3D(double xIn, double yIn, double zIn, double alpha, double beta, int n, int frequency);

    void makePerlinNoise3DTexture();
    void initPerlinNoise3DTexture(GLenum textureUnit = 0);

    explicit OpenGLPerlinNoise(bool useSimplexNoise = false) noexcept;
    ~OpenGLPerlinNoise() noexcept;
    OpenGLPerlinNoise(const OpenGLPerlinNoise&) = delete;
    OpenGLPerlinNoise(OpenGLPerlinNoise&&)      = delete;
    OpenGLPerlinNoise& operator=(const OpenGLPerlinNoise&) = delete;
    OpenGLPerlinNoise& operator=(OpenGLPerlinNoise&&)      = delete;

  private:

    bool useSimplexNoise_          = false;
    GLuint perlinNoise3DTextureID_ = 0;
    Utils::Randomizers::RandomRNGWELL512 random_;
    std::vector<GLubyte> perlinNoise3DTextureData_;

    void makePerlinNoise3DTextureCalculation(int f, int inc, int frequency, double amp);
    void bindTexture(GLenum textureUnit = 0) const;
    void unbind(GLenum textureUnit) const;
  };
} // namespace OpenGLRenderingEngine

#endif // __OpenGLPerlinNoise_h