/*

Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
Released Under a Simplified BSD (FreeBSD) License
for academic, personal & non-commercial use.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the author and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.

A Commercial License is also available for commercial use with
special restrictions and obligations at a one-off fee. See links at:
1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

#pragma once

#ifndef __OpenGLFBORenderToTexture_h
#define __OpenGLFBORenderToTexture_h

#include "ModuleDLL.h"
#include "glew.h"
#include <string>
#include <utility>

/** @brief Namespace OpenGLRenderingEngine for the OpenGL rendering.
* @author Thanos Theo, 2018
* @version 14.0.0.0
*/
namespace OpenGLRenderingEngine
{
  // forward declaration
  class OpenGLDriverInfo;

  /** @brief This class provides Frame Buffer Object support using the GL_EXT_framebuffer_object OpenGL extension.\nIt also supports Frame Buffer Object multisampling via the GL_EXT_framebuffer_multisample & GL_EXT_framebuffer_blit extensions.
  * @author Thanos Theo, 2009-2018
  * @version 14.0.0.0
  */
  class OPENGL_RENDERING_ENGINE_MODULE_API OpenGLFrameBufferObject final
  {
  public:

    GLsizei getWidth()    const { return width_;     }
    GLsizei getHeight()   const { return height_;    }
    GLuint getTextureID() const { return textureID_; }

    bool getUseTexture()               const { return useTexture_;               }
    bool getDepthRenderBuffer()        const { return depthRenderBuffer_;        }
    bool getUseDepthTexture()          const { return useDepthTexture_;          }
    bool getDepthStencilRenderBuffer() const { return depthStencilRenderBuffer_; }
    GLint getTextureFormat1()          const { return textureFormat1_;           }
    GLenum getTextureFormat2()         const { return textureFormat2_;           }
    GLenum getTextureFormatType()      const { return textureFormatType_;        }
    bool getGenerateMipmap()           const { return generateMipmap_;           }
    std::pair<bool, GLint> getMultisampleFBO() const { return std::make_pair(multisampleFBO_, numberOfSamples_); }

    void setUseTexture(bool useTexture) { useTexture_ = useTexture; }
    void setDepthRenderBuffer(bool depthRenderBuffer);
    void setUseDepthTexture(bool useDepthTexture);
    void setDepthStencilRenderBuffer(bool depthStencilRenderBuffer);
    void setTextureFormat1(GLint textureFormat1) { textureFormat1_ = textureFormat1; }
    void setTextureFormat2(GLenum textureFormat2) { textureFormat2_ = textureFormat2; }
    void setTextureFormatType(GLenum textureFormatType) { textureFormatType_ = textureFormatType; }
    void setGenerateMipmap(bool generateMipmap) { generateMipmap_ = generateMipmap; }
    void setMultisampleFBO(bool multisampleFBO, GLint numberOfSamples);

    void initFrameBufferObjectResources(GLsizei width, GLsizei height, GLenum textureUnit = 0, GLenum depthTextureUnit = 1);
    void startRender() const;
    void finishRender() const;
    void enable() const;
    void disable() const;
    void bindTexture(GLenum textureUnit = 0) const;
    void bindDepthTexture(GLenum depthTextureUnit = 0) const;
    void unbind(GLenum textureUnit = 0) const;
    void renderTextureToFullScreenQuad(GLenum textureUnit, bool isNvidia, bool useDummyVAO = true) const;
    void renderDepthTextureToFullScreenQuad(GLenum depthTextureUnit, bool isNvidia, bool useDummyVAO = true) const;
    void disposeFrameBufferObjectResources();

    OpenGLFrameBufferObject(OpenGLDriverInfo* openGLDriverInfo,
                            bool useTexture = true, bool depthRenderBuffer = false, bool useDepthTexture = false, bool depthStencilRenderBuffer = false,
                            GLint textureFormat1 = GL_RGBA8, GLenum textureFormat2 = GL_RGBA, GLenum textureFormatType = GL_UNSIGNED_BYTE,
                            bool generateMipmap = false, bool multisampleFBO = false, GLint numberOfSamples = 4) noexcept;
    ~OpenGLFrameBufferObject() noexcept;
    OpenGLFrameBufferObject(const OpenGLFrameBufferObject&) = delete;
    OpenGLFrameBufferObject(OpenGLFrameBufferObject&&)      = delete;
    OpenGLFrameBufferObject& operator=(const OpenGLFrameBufferObject&) = delete;
    OpenGLFrameBufferObject& operator=(OpenGLFrameBufferObject&&)      = delete;

  private:

    OpenGLDriverInfo* openGLDriverInfo_ = nullptr;
    GLuint fboID_                       = 0;
    GLuint fboMultiSampleID_            = 0;
    GLuint renderBufferMultiSampleID_   = 0;
    GLuint renderBufferID_              = 0;
    GLsizei width_                      = 0;
    GLsizei height_                     = 0;
    GLuint textureID_                   = 0;
    GLuint depthTextureID_              = 0;
    bool depthRenderBuffer_             = false;
    bool useTexture_                    = false;
    bool useDepthTexture_               = false;
    bool depthStencilRenderBuffer_      = false;
    GLint textureFormat1_               = 0;
    GLenum textureFormat2_              = 0;
    GLenum textureFormatType_           = 0;
    bool generateMipmap_                = false;
    bool multisampleFBO_                = false;
    GLint numberOfSamples_              = 0;
    GLuint dummyVao_                    = 0;

    void initTextureResouces(GLenum textureUnit = 0);
    void initDepthTextureResouces(GLenum depthTextureUnit = 0);
    void initTextureParameters() const;
    void printFramebufferInfo() const;
    std::string getTextureParameters(GLuint id) const;
    std::string getRenderbufferParameters(GLuint id) const;
  };
} // namespace OpenGLRenderingEngine
#endif // __OpenGLFBORenderToTexture_h