/*

Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
Released Under a Simplified BSD (FreeBSD) License
for academic, personal & non-commercial use.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the author and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.

A Commercial License is also available for commercial use with
special restrictions and obligations at a one-off fee. See links at:
1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

#pragma once

#ifndef __OpenGLDriverInfo_h
#define __OpenGLDriverInfo_h

#include "ModuleDLL.h"
#include "glew.h"
#include <string>
#include <set>

/** @brief Namespace OpenGLRenderingEngine for the OpenGL rendering.
* @author Thanos Theo, 2018
* @version 14.0.0.0
*/
namespace OpenGLRenderingEngine
{
  /** @brief Gets GL vendor, version, supported extensions and other states using glGet* functions and store them in OpenGLDriverInfo class variables.\nTo get valid OpenGL infos, an OpenGL rendering context (RC) must be opened before calling OpenGLDriverInfo::getGLInfo(). Otherwise it returns false.
  * @author Thanos Theo, 2009-2018
  * @version 14.0.0.0
  */
  class OPENGL_RENDERING_ENGINE_MODULE_API OpenGLDriverInfo final
  {
  public:

    static float getMinimumGLVersionForQualityRenderingAndShaders();
    static std::string getMinimumGLSLVersionFor120Shaders();
    static std::string getMinimumGLSLVersionFor330Shaders();
    static std::string getMinimumGLSLVersionFor400Shaders();
    static std::string getMinimumGLSLVersionFor420Shaders();
    static std::string getMinimumGLSLVersionFor430Shaders();
    static std::string getMinimumGLSLVersionFor440Shaders();
    static std::string getMinimumGLSLVersionFor450Shaders();
    static std::string getGLSLLanguageMode();

    /// all getter functions inlined
    /// return-by-value instead return-by-reference so as to avoid changing original string
    std::string getVendor() const                 { return vendor_;                 }
    /// return-by-value instead return-by-reference so as to avoid changing original string
    std::string getRenderer() const               { return renderer_;               }
    /// return-by-value instead return-by-reference so as to avoid changing original string
    std::string getVersion() const                { return version_;                }
    /// return-by-value instead return-by-reference so as to avoid changing original string
    std::string getShadingLanguageVersion() const { return shadingLanguageVersion_; }
    /// return-by-value instead return-by-reference so as to avoid changing original string
    std::set<std::string> getExtensions() const   { return extensions_;             }
    bool isNvidia() const { return isNvidia_;       }
    bool isAMDATI() const    { return isAMDATI_;    }
    bool isIntel() const     { return isIntel_;     }
    bool isMesa() const      { return isMesa_;      }
    bool isMicrosoft() const { return isMicrosoft_; }
    bool supports120Shaders() const { return use120Shaders_; }
    bool supports330Shaders() const { return use330Shaders_; }
    bool supports400Shaders() const { return use400Shaders_; }
    bool supports420Shaders() const { return use420Shaders_; }
    bool supports430Shaders() const { return use430Shaders_; }
    bool supports440Shaders() const { return use440Shaders_; }
    bool supports450Shaders() const { return use450Shaders_; }
    bool isVSynchSupported() const  { return isVSynchSupported_; }
    GLint getRedBits() const                                { return redBits_;                                }
    GLint getGreenBits() const                              { return greenBits_;                              }
    GLint getBlueBits() const                               { return blueBits_;                               }
    GLint getAlphaBits() const                              { return alphaBits_;                              }
    GLint getDepthBits() const                              { return depthBits_;                              }
    GLint getStencilBits() const                            { return stencilBits_;                            }
    GLint getAccumRedBits() const                           { return accumRedBits_;                           }
    GLint getAccumGreenBits() const                         { return accumGreenBits_;                         }
    GLint getAccumBlueBits() const                          { return accumBlueBits_;                          }
    GLint getAccumAlphaBits() const                         { return accumAlphaBits_;                         }
    GLint getSampleBuffers() const                          { return sampleBuffers_;                          }
    GLint getMaxTextureSize() const                         { return maxTextureSize_;                         }
    GLint getMaxTextureBufferSize() const                   { return maxTextureBufferSize_;                   }
    GLint getMaxTextureMaxAnisotropy() const                { return maxTextureMaxAnisotropy_;                }
    GLint getMaxRenderBufferSize() const                    { return maxRenderBufferSize_;                    }
    GLint getMaxColorAttachments() const                    { return maxColorAttachments_;                    }
    GLint getMaxLights() const                              { return maxLights_;                              }
    GLint getMaxAttribStacks() const                        { return maxAttribStacks_;                        }
    GLint getMaxModelViewStacks() const                     { return maxModelViewStacks_;                     }
    GLint getMaxProjectionStacks() const                    { return maxProjectionStacks_;                    }
    GLint getMaxClipPlanes() const                          { return maxClipPlanes_;                          }
    GLint getMaxTextureStacks() const                       { return maxTextureStacks_;                       }
    GLint getMaxGeometryOutputVertices() const              { return maxGeometryOutputVertices_;              }
    GLint getMaxTessellationControlOutputComponents() const { return maxTessellationControlOutputComponents_; }
    GLint getMaxTessellationGenerationLevel() const         { return maxTessellationGenerationLevel_;         }
    GLint getGPUMemoryInfoDedicatedVidmemNVX() const        { return GPUMemoryInfoDedicatedVidmemNVX_;        }
    GLint getGPUMemoryInfoTotalAvailableMemoryNVX() const   { return GPUMemoryInfoTotalAvailableMemoryNVX_;   }
    GLint getGPUMemoryInfoCurrentAvailableMemoryNVX() const { return GPUMemoryInfoCurrentAvailableMemoryNVX_; }
    GLint getGPUMemoryInfoEvictionCountNVX() const          { return GPUMemoryInfoEvictionCountNVX_;          }
    GLint getGPUMemoryInfoEvictedMemoryNVX() const          { return GPUMemoryInfoEvictedMemoryNVX_;          }
    const GLint* getVBOFreeMemoryATI() const                { return vboFreeMemoryATI_;                       }
    const GLint* getTextureFreeMemoryATI() const            { return textureFreeMemoryATI_;                   }
    const GLint* getRenderBufferFreeMemoryATI() const       { return renderBufferFreeMemoryATI_;              }

    // convenience GLEW-to-bool functions of useful extensions
    bool supports_GL_ARB_texture_rectangle() const          { return (GLEW_ARB_texture_rectangle == GL_TRUE);          }
    bool supports_GL_ARB_texture_buffer_object() const      { return (GLEW_ARB_texture_buffer_object == GL_TRUE);      }
    bool supports_GL_EXT_texture_filter_anisotropic() const { return (GLEW_EXT_texture_filter_anisotropic == GL_TRUE); }
    bool supports_GL_EXT_framebuffer_object() const         { return (GLEW_EXT_framebuffer_object == GL_TRUE);         }
    bool supports_GL_EXT_framebuffer_multisample() const    { return (GLEW_EXT_framebuffer_multisample == GL_TRUE);    }
    bool supports_GL_EXT_framebuffer_blit() const           { return (GLEW_EXT_framebuffer_blit == GL_TRUE);           }
    bool supports_GL_EXT_packed_depth_stencil() const       { return (GLEW_EXT_packed_depth_stencil == GL_TRUE);       }
    bool supports_GL_EXT_gpu_shader4() const                { return (GLEW_EXT_gpu_shader4 == GL_TRUE);                }
    bool supports_GL_ARB_geometry_shader4() const           { return (GLEW_ARB_geometry_shader4 == GL_TRUE);           }
    bool supports_GL_ARB_tessellation_shader() const        { return (GLEW_ARB_tessellation_shader == GL_TRUE);        }
    bool supports_GL_ARB_compute_shader() const             { return (GLEW_ARB_compute_shader == GL_TRUE);             }
    bool supports_GL_ARB_gpu_shader5() const                { return (GLEW_ARB_gpu_shader5 == GL_TRUE);                }
    bool supports_GL_ARB_gpu_shader_fp64() const            { return (GLEW_ARB_gpu_shader_fp64 == GL_TRUE);            }
    bool supports_GL_ARB_vertex_type_2_10_10_10_rev() const { return (GLEW_ARB_vertex_type_2_10_10_10_rev == GL_TRUE); }
    bool supports_GL_NVX_gpu_memory_info() const            { return (GLEW_NVX_gpu_memory_info == GL_TRUE);            }
    bool supports_GL_ATI_meminfo() const                    { return (GLEW_ATI_meminfo == GL_TRUE);                    }

    /// extract GL memory info
    void getGLMemoryInfo();
    /// print GL info
    void printGLInfo() const;
    /// print GL memory info
    void printGLMemoryInfo() const;
    /// check if a GL extension is supported
    bool isGLExtensionSupported(const std::string& extension) const { return (extensions_.find(extension) != extensions_.end()); }
    /// get a concise GL driver info string
    std::string getConciseGLDriverInfo() const;

    OpenGLDriverInfo() noexcept;
    ~OpenGLDriverInfo() = default;
    OpenGLDriverInfo(const OpenGLDriverInfo&) = delete; // copy-constructor deleted
    OpenGLDriverInfo(OpenGLDriverInfo&&)      = delete; // move-constructor deleted
    OpenGLDriverInfo& operator=(const OpenGLDriverInfo&) = delete; //      assignment operator deleted
    OpenGLDriverInfo& operator=(OpenGLDriverInfo&&)      = delete; // move-assignment operator deleted

  private:

    std::string vendor_;
    std::string renderer_;
    std::string version_;
    std::string shadingLanguageVersion_;
    std::set<std::string> extensions_;
    bool isNvidia_                                            = false;
    bool isAMDATI_                                            = false;
    bool isIntel_                                             = false;
    bool isMesa_                                              = false;
    bool isMicrosoft_                                         = false;
    bool use120Shaders_                                       = false;
    bool use330Shaders_                                       = false;
    bool use400Shaders_                                       = false;
    bool use420Shaders_                                       = false;
    bool use430Shaders_                                       = false;
    bool use440Shaders_                                       = false;
    bool use450Shaders_                                       = false;
    bool isVSynchSupported_                                   = false;
    GLint redBits_                                            = 0;
    GLint greenBits_                                          = 0;
    GLint blueBits_                                           = 0;
    GLint alphaBits_                                          = 0;
    GLint depthBits_                                          = 0;
    GLint stencilBits_                                        = 0;
    GLint accumRedBits_                                       = 0;
    GLint accumGreenBits_                                     = 0;
    GLint accumBlueBits_                                      = 0;
    GLint accumAlphaBits_                                     = 0;
    GLint sampleBuffers_                                      = 0;
    GLint samples_                                            = 0;
    GLint maxTextureSize_                                     = 0;
    GLint maxTextureBufferSize_                               = 0;
    GLint maxTextureMaxAnisotropy_                            = 0;
    GLint maxRenderBufferSize_                                = 0;
    GLint maxColorAttachments_                                = 0;
    GLint maxLights_                                          = 0;
    GLint maxAttribStacks_                                    = 0;
    GLint maxModelViewStacks_                                 = 0;
    GLint maxProjectionStacks_                                = 0;
    GLint maxClipPlanes_                                      = 0;
    GLint maxTextureStacks_                                   = 0;
    GLint maxGeometryOutputVertices_                          = 0;
    GLint maxTessellationPatchVertices_                       = 0;
    GLint maxTessellationControlOutputComponents_             = 0;
    GLint maxTessellationGenerationLevel_                     = 0;
    GLint maxUniformBufferBindings_                           = 0;
    GLint maxUniformBlockSize_                                = 0;
    GLint maxCombinedVertexUniformComponents_                 = 0;
    GLint maxCombinedGeometryUniformComponents_               = 0;
    GLint maxCombinedTessellationControlUniformComponents_    = 0;
    GLint maxCombinedTessellationEvaluationUniformComponents_ = 0;
    GLint maxCombinedFragmentUniformComponents_               = 0;
    GLint GPUMemoryInfoDedicatedVidmemNVX_                    = 0;
    GLint GPUMemoryInfoTotalAvailableMemoryNVX_               = 0;
    GLint GPUMemoryInfoCurrentAvailableMemoryNVX_             = 0;
    GLint GPUMemoryInfoEvictionCountNVX_                      = 0;
    GLint GPUMemoryInfoEvictedMemoryNVX_                      = 0;
    GLint vboFreeMemoryATI_[4]                                = { 0 };
    GLint textureFreeMemoryATI_[4]                            = { 0 };
    GLint renderBufferFreeMemoryATI_[4]                       = { 0 };

    /// extract GL info
    void getGLInfo();
  };
} // namespace OpenGLRenderingEngine

#endif // __OpenGLDriverInfo_h