/*

Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
Released Under a Simplified BSD (FreeBSD) License
for academic, personal & non-commercial use.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the author and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.

A Commercial License is also available for commercial use with
special restrictions and obligations at a one-off fee. See links at:
1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

#pragma once

#ifndef __OpenGLAssimpModelLoader_h
#define __OpenGLAssimpModelLoader_h

#include "ModuleDLL.h"
#include "glew.h"
#include "scene.h"
#include <unordered_map>
#include <vector>
#include <string>

/** @brief Namespace OpenGLRenderingEngine for the OpenGL rendering.
* @author Thanos Theo, 2018
* @version 14.0.0.0
*/
namespace OpenGLRenderingEngine
{
  /** @brief This class encapsulates usage of an OpenGL Model Loader using the Assimp library.
  * @author Thanos Theo, 2009-2018
  * @version 14.0.0.0
  */
  class OPENGL_RENDERING_ENGINE_MODULE_API OpenGLAssimpModelLoader final
  {
  public:

    int getTotalNumberOfFaces() const { return totalNumberOfFaces_; }
    int getTotalNumberOfVertices() const { return totalNumberOfVertices_; }
    float getSceneScaleFactor() const { return sceneScaleFactor_; }
    float getSceneCenterX() const { return sceneCenter_.x; }
    float getSceneCenterY() const { return sceneCenter_.y; }
    float getSceneCenterZ() const { return sceneCenter_.z; }

    std::unordered_map<int, std::vector<GLfloat>>& getAllModelStorageColorsData()   { return allModelStorageColorsData_;   }
    std::unordered_map<int, std::vector<GLfloat>>& getAllModelStorageNormalsData()  { return allModelStorageNormalsData_;  }
    std::unordered_map<int, std::vector<GLfloat>>& getAllModelStorageVerticesData() { return allModelStorageVerticesData_; }
    std::unordered_map<int, std::vector<GLuint>>&  getAllModelStorageIndicesData()  { return allModelStorageIndicesData_;  }

    void load(const std::string& modelDirectory, const std::string& modelFileName, const std::string& modelLoaderDescriptorFileName, bool triangulate = true, bool generateSmoothNormals = true, bool joinIdenticalVertices = true);
    void recursiveRenderRoot() const;

    OpenGLAssimpModelLoader() = default;
    ~OpenGLAssimpModelLoader() noexcept;
    OpenGLAssimpModelLoader(const OpenGLAssimpModelLoader&) = delete;
    OpenGLAssimpModelLoader(OpenGLAssimpModelLoader&&)      = delete;
    OpenGLAssimpModelLoader& operator=(const OpenGLAssimpModelLoader&) = delete;
    OpenGLAssimpModelLoader& operator=(OpenGLAssimpModelLoader&&)      = delete;

  private:

    int totalNumberOfFaces_    = 0;
    int totalNumberOfVertices_ = 0;
    float sceneScaleFactor_    = 1.0f;

    std::unordered_map<int, std::vector<GLfloat>> allModelStorageColorsData_;
    std::unordered_map<int, std::vector<GLfloat>> allModelStorageNormalsData_;
    std::unordered_map<int, std::vector<GLfloat>> allModelStorageVerticesData_;
    std::unordered_map<int, std::vector<GLuint>>  allModelStorageIndicesData_;

    const aiScene* scene_ = nullptr;
    aiVector3D sceneMin_{};
    aiVector3D sceneMax_{};
    aiVector3D sceneCenter_{};

    void resetSceneDataStructures();
    void getBoundingBoxForNode(const aiNode* node, aiMatrix4x4* trafo, aiVector3D& minimum, aiVector3D& maximum) const;
    void getBoundingBox(aiVector3D& minimum, aiVector3D& maximum) const;
    void recursiveStoreModelData(const aiScene* scene, const aiNode* node);
    void storeColorDataInMap(int numberOfIndices, const aiColor4D* color);
    void storeNormalDataInMap(int numberOfIndices, const aiVector3D* normal);
    void storeVertexDataInMap(int numberOfIndices, const aiVector3D* vertex);
    void recursiveStoreModelDataWithTriangulationAndIndexing(const aiScene* scene, const aiNode* node);
    void storeIndexDataInMap(int numberOfIndices, const GLuint* index);
    float getSceneScaleFactor(aiVector3D& minimum, aiVector3D& maximum) const;
    void reportModelPrimitiveUsage();
    void loadModel(const std::string& modelDirectory, const std::string& modelFileName, bool triangulate, bool generateSmoothNormals, bool joinIdenticalVertices);
    bool loadAsset(const std::string& modelDirectory, const std::string& modelFileName, bool triangulate, bool generateSmoothNormals, bool joinIdenticalVertices);
    bool writeToBinaryFilesForModelDataWithTriangulationAndIndexing(const std::string& modelDirectory, const std::string& modelFileName);
    bool loadFromBinaryFilesForModelDataWithTriangulationAndIndexing(const std::string& modelDirectory, const std::string& modelFileName);
    void getBoundingBoxForModelDataWithTriangulationAndIndexing();
  };
} // namespace OpenGLRenderingEngine

#endif // __OpenGLAssimpModelLoader_h