/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  Surface Simple Phong Lighting Model GLSL code for Phong.
*  Using core functions from CommonFunctions GLSL library.
*  References:
*  1. en.wikipedia.org/wiki/Phong_reflection_model                   -> Phong Specular Lighting Model
*
*  Relevant function: applySimplePhongLightingModel()
*
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/

uniform int numberOfLights = 1;
uniform bool useColorMaterial = true;

// Machine float epsilon
const float FLT_EPSILON = 0.000001;

const vec3 EYE_POSITION_FOR_NON_PERSPECTIVE_CORRECTED_SPHERE_IMPOSTOR = vec3(0.0, 0.0, -1.0);

// Used for specular calculations
const float PHONG_SPECULAR_FACTOR = 3.5;

float applyPhongSpecularModel(in vec3 E, in vec3 N, in vec3 L, in float shininess, in float factor);
void applySimplePhongLightingModel(in vec3 position, in vec3 normal, in vec4 materialColor, in bool usePositionVector, inout vec4 color);

void applySimplePhongLightingModel(in vec3 position, in vec3 normal, in vec4 materialColor, in bool usePositionVector, inout vec4 color)
{
  int normalSign = (gl_FrontFacing) ? 1 : -1;
  vec3 E = (usePositionVector) ? normalize(-position) : normalize(-EYE_POSITION_FOR_NON_PERSPECTIVE_CORRECTED_SPHERE_IMPOSTOR);
  vec3 N = normalize(normalSign * normal);
  vec3 L = vec3(0.0);

  float lambertTerm   = 0.0;
  float specularPower = 0.0;

  gl_MaterialParameters glCurrentMaterial = (gl_FrontFacing) ? gl_FrontMaterial : gl_BackMaterial;

  // Don't use gl_FrontLightModelProduct.sceneColor as it won't work on Intel/AMD GPUs when the materialColor changes:
  // gl_FrontLightModelProduct is a uniform which does not track vertex changes
  if (useColorMaterial)
  {
    color += glCurrentMaterial.emission + materialColor * gl_LightModel.ambient;
  }
  else
  {
    color += glCurrentMaterial.emission + glCurrentMaterial.ambient * gl_LightModel.ambient;
  }

  for (int index = 0; index < numberOfLights; ++index)
  {
    L = normalize(gl_LightSource[index].position.xyz);

    // Don't use gl_FrontLightModelProduct.sceneColor as it won't work on Intel/AMD GPUs when the materialColor changes:
    // gl_FrontLightModelProduct is a uniform which does not track vertex changes
    if (useColorMaterial)
    {
      color += gl_LightSource[index].ambient * materialColor; // ambient light computation
    }
    else
    {
      color += gl_LightSource[index].ambient * glCurrentMaterial.ambient; // ambient light computation
    }

    lambertTerm = max(dot(N, L), FLT_EPSILON); // failsafe GLSL Shader value for diffuse light computation
    if (useColorMaterial)
    {
      color += gl_LightSource[index].diffuse * materialColor * lambertTerm;
    }
    else
    {
      color += gl_LightSource[index].diffuse * glCurrentMaterial.diffuse * lambertTerm;
    }

    specularPower = applyPhongSpecularModel(E, N, L, glCurrentMaterial.shininess, PHONG_SPECULAR_FACTOR); // specular light computation
    color += gl_LightSource[index].specular * glCurrentMaterial.specular * specularPower;
  }

  color = clamp(color, 0.0, 1.0);
}