/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  Surface Lighting Models GLSL code for Phong, Blinn-Phong, Gaussian, Toon, Gooch.
*  Using core functions from CommonFunctions GLSL library.
*  References:
*  1. en.wikipedia.org/wiki/Phong_reflection_model                   -> Phong Specular Lighting Model
*  2. en.wikipedia.org/wiki/Blinn%E2%80%93Phong_shading_model        -> Blinn-Phong Specular Lighting Model
*  3. en.wikipedia.org/wiki/Specular_highlight#Gaussian_distribution -> Gaussiang Specular Lighting Model
*  4. en.wikipedia.org/wiki/Cel_shading                              -> Toon Specular Lighting Model
*  5. www.cs.northwestern.edu/~ago820/SIG98/abstract.html            -> Gooch Specular Lighting Model
*  6. en.wikipedia.org/wiki/Oren%E2%80%93Nayar_reflectance_model     -> Oren-Nayar Diffuse Lighting Model
*  7. en.wikipedia.org/wiki/Reflection_mapping#Sphere_mapping        -> Reflection (Sphere) Mapping
*  8. en.wikipedia.org/wiki/Reflection_mapping                       -> Reflection Mapping
*
*  Relevant variables controlling the above: useSpecularDiffuseModel, useOrenNayarDiffuseModel & useFresnelFactorSchlickApproximationSpecularModel
*  Relevant functions: applyADSLightingModel(), applyNonPhotoRealisticLightingModel() & applyOrenNayarDiffuseModel()
*
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/

uniform int numberOfLights = 1;
uniform bool useOrenNayarDiffuseModel = false;
uniform bool useFresnelFactorSchlickApproximationSpecularModel = false;
uniform bool useColorMaterial = true;
uniform float extraGaussianSpecularFactor = 1.0;

// Machine float epsilon
const float FLT_EPSILON = 0.000001;

const float COS_OUTER_CONE_ANGLE = 0.8; // 36 degrees
const vec3 EYE_POSITION_FOR_NON_PERSPECTIVE_CORRECTED_SPHERE_IMPOSTOR = vec3(0.0, 0.0, -1.0);

// Used for specular calculations
const float PHONG_SPECULAR_FACTOR = 3.5;
const float BLINN_PHONG_SPECULAR_FACTOR = 1.0;
const float GAUSSIAN_SPECULAR_FACTOR = PHONG_SPECULAR_FACTOR;

// Silhouette color (default black)
const vec4 SILHOUETTE_COLOR = vec4(0.0, 0.0, 0.0, 1.0);
const float SILHOUETTE_MULTIPLICATION_FACTOR = 1.4;
const float SILHOUETTE_LOWER_BOUND_THRESHOLD = SILHOUETTE_MULTIPLICATION_FACTOR * 0.20;
const float SILHOUETTE_UPPER_BOUND_THRESHOLD = SILHOUETTE_MULTIPLICATION_FACTOR * 0.35;

float smootherstep(in float edge0, in float edge1, in float x);
vec4 applyOrenNayarDiffuseModel(in vec3 E, in vec3 N, in vec3 L, in vec4 diffuse);
float applyFresnelFactorSchlickApproximationSpecularModel(in vec3 E, in vec3 N, in vec3 L);
float applyPhongSpecularModel(in vec3 E, in vec3 N, in vec3 L, in float shininess, in float factor);
float applyBlinnPhongSpecularModel(in vec3 E, in vec3 N, in vec3 L, in float shininess, in float factor);
float applyGaussianSpecularModel(in vec3 E, in vec3 N, in vec3 L, in float shininess, in float factor);
vec4 applyToonColor(in float lambertTerm, in float specularPower, in float attenuation, in float thresholdLambertTerm, in float thresholdSpecularPower, in vec4 color);
vec4 applyGoochColor(in float lambertTerm, in vec4 color);
void applyADSLightingModel(in int useSpecularDiffuseModel, in vec3 position, in vec3 normal, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
void applyNonPhotoRealisticLightingModel(in int useNonPhotoRealisticModel, in vec3 position, in vec3 normal, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
void iterateAllLightsForNonPhotoRealisticLightingModel(in int useNonPhotoRealisticModel, in vec3 position, in vec3 N, in vec3 E, in vec4 materialColor, inout vec4 color);
#if GPU_SHADER4_COMPATIBILITY_CONDITION
  uint packFloat4ToUnsignedInteger(in vec4 values);
  vec4 unpackUnsignedIntegerToFloat4(in uint value);
  uint packNormalToUnsignedInteger(in vec3 normal);
  uint packNormalToUnsignedInteger(in vec3 normal, bool useFrontFacingForDeferredLighting);
  vec3 unpackUnsignedIntegerToNormal(in uint signedPackedNormal);
  vec3 unpackUnsignedIntegerToNormal(in uint signedPackedNormal, bool useFrontFacingForDeferredLighting);
  #if USE_DEFERRED_SHADING
    float getCurrentFragmentMaterialShininess();
    uint[4] getCurrentFragmentMaterialColors();
    void applyADSLightingModelDeferredShading(in int useSpecularDiffuseModel, in vec3 position, in vec3 normal, in uint[4] materialColors, in float shininess, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
    void applyNonPhotoRealisticLightingModelDeferredShading(in int useNonPhotoRealisticModel, in vec3 position, in vec3 normal, in uint[4] materialColors, in float shininess, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
    void iterateAllLightsForNonPhotoRealisticLightingModelDeferredShading(in int useNonPhotoRealisticModel, in vec3 position, in vec3 N, in vec3 E, in uint[4] materialColors, in float shininess, in vec4 materialColor, inout vec4 color);
  #endif
#endif

void applyADSLightingModel(in int useSpecularDiffuseModel, in vec3 position, in vec3 normal, in vec4 materialColor, in bool usePositionVector, inout vec4 color)
{
  int normalSign   = (gl_FrontFacing) ? 1 : -1;
  vec3 lightVector = vec3(0.0);
  vec3 E           = (usePositionVector) ? normalize(-position) : normalize(-EYE_POSITION_FOR_NON_PERSPECTIVE_CORRECTED_SPHERE_IMPOSTOR);
  vec3 N           = normalSign * normalize(normal);
  vec3 L           = vec3(0.0);

  float lambertTerm   = 0.0;
  float specularPower = 0.0;
  float attenuation   = 0.0;
  float lightLength   = 0.0;

  #if USE_SPOT_LIGHTS
    vec3 D = vec3(0.0);
    float cosCurrentAngle   = 0.0;
    float cosInnerConeAngle = 0.0;
    float spotWidth         = 0.0;
    float spotFactor        = 0.0;
  #endif

  gl_MaterialParameters glCurrentMaterial = (gl_FrontFacing) ? gl_FrontMaterial : gl_BackMaterial;

  if (useColorMaterial)
  {
    color += glCurrentMaterial.emission + materialColor * gl_LightModel.ambient;
  }
  else
  {
    color += glCurrentMaterial.emission + glCurrentMaterial.ambient * gl_LightModel.ambient;
  }

  for (int index = 0; index < numberOfLights; ++index)
  {
    if (gl_LightSource[index].position.w == 0.0) // directional rather than positional light
    {
      lightVector = vec3(gl_LightSource[index].position.xyz);
      attenuation = 1.0;
    }
    else // -position.xyz, positional rather than directional light
    {
      lightVector = vec3(gl_LightSource[index].position.xyz - position);
      lightLength = length(lightVector);
      attenuation = 1.0 / (gl_LightSource[index].constantAttenuation +
                           gl_LightSource[index].linearAttenuation * lightLength +
                           gl_LightSource[index].quadraticAttenuation * lightLength * lightLength);

      #if USE_SPOT_LIGHTS
        // used for spotlight lighting model
        D = normalize(gl_LightSource[index].spotDirection);
        cosCurrentAngle = dot(-L, D);
        cosInnerConeAngle = gl_LightSource[index].spotCosCutoff;
        spotWidth = (cosInnerConeAngle - COS_OUTER_CONE_ANGLE);
        spotFactor = (useOrenNayarDiffuseModel) ? smootherstep(cosInnerConeAngle - spotWidth, cosInnerConeAngle + spotWidth, cosCurrentAngle)
                                                : clamp((cosCurrentAngle - COS_OUTER_CONE_ANGLE) / spotWidth, 0.0, 1.0);
        spotFactor = pow(spotFactor, gl_LightSource[index].spotExponent);
        attenuation *= spotFactor;
      #endif
    }

    L = normalize(lightVector);
    lambertTerm = max(dot(N, L), FLT_EPSILON); // failsafe GLSL Shader value for diffuse light computation

    // Ambient light computation
    if (useColorMaterial)
    {
      color += gl_LightSource[index].ambient * materialColor * attenuation;
    }
    else
    {
      color += gl_LightSource[index].ambient * glCurrentMaterial.ambient * attenuation;
    }

    // Diffuse light computation
    if (useColorMaterial)
    {
      color += (useOrenNayarDiffuseModel ? applyOrenNayarDiffuseModel(-E, N, L, gl_LightSource[index].diffuse) : gl_LightSource[index].diffuse) * materialColor * lambertTerm * attenuation;
    }
    else
    {
      color += (useOrenNayarDiffuseModel ? applyOrenNayarDiffuseModel(-E, N, L, gl_LightSource[index].diffuse) : gl_LightSource[index].diffuse) * glCurrentMaterial.diffuse * lambertTerm * attenuation;
    }

    if (glCurrentMaterial.shininess > 0.0)
    {
      if (useSpecularDiffuseModel == 1)
      {
        specularPower = applyPhongSpecularModel(E, N, L, glCurrentMaterial.shininess, PHONG_SPECULAR_FACTOR);
      }
      else if (useSpecularDiffuseModel == 2)
      {
        specularPower = applyBlinnPhongSpecularModel(E, N, L, glCurrentMaterial.shininess, BLINN_PHONG_SPECULAR_FACTOR);
      }
      else // if (useSpecularDiffuseModel == 3)
      {
        specularPower = applyGaussianSpecularModel(E, N, L, glCurrentMaterial.shininess, extraGaussianSpecularFactor * GAUSSIAN_SPECULAR_FACTOR);
      }
    }
    else
    {
      specularPower = 0.0;
    }

    // Specular light computation & optional Fresnel Model
    color += gl_LightSource[index].specular * glCurrentMaterial.specular * specularPower * attenuation;
    if (useFresnelFactorSchlickApproximationSpecularModel)
    {
      color += applyFresnelFactorSchlickApproximationSpecularModel(E, N, L) * attenuation;
    }
  }

  color = clamp(color, 0.0, 1.0);
}

void applyNonPhotoRealisticLightingModel(in int useNonPhotoRealisticModel, in vec3 position, in vec3 normal, in vec4 materialColor, in bool usePositionVector, inout vec4 color)
{
  // Dot Product based for simple & fast silhouette rendering
  int normalSign = (gl_FrontFacing) ? 1 : -1;
  vec3 E = (usePositionVector) ? normalize(-position) : normalize(-EYE_POSITION_FOR_NON_PERSPECTIVE_CORRECTED_SPHERE_IMPOSTOR);
  vec3 N = normalSign * normalize(normal);
  float lambertTerm = max(dot(N, E), FLT_EPSILON); // failsafe GLSL Shader value for diffuse light computation
  if (useOrenNayarDiffuseModel)
  {
    float rateOfChange = fwidth(lambertTerm); // extra silhouette color based on fwidth()
    if (rateOfChange <= SILHOUETTE_LOWER_BOUND_THRESHOLD)
    {
      lambertTerm -= rateOfChange;
    }
  }

  if (lambertTerm < SILHOUETTE_LOWER_BOUND_THRESHOLD)
  {
    color = SILHOUETTE_COLOR;
  }
  else if ((lambertTerm >= SILHOUETTE_LOWER_BOUND_THRESHOLD) && (lambertTerm < SILHOUETTE_UPPER_BOUND_THRESHOLD))
  {
    iterateAllLightsForNonPhotoRealisticLightingModel(useNonPhotoRealisticModel, position, N, E, materialColor, color);
    color = mix(SILHOUETTE_COLOR, color, smootherstep(SILHOUETTE_LOWER_BOUND_THRESHOLD, SILHOUETTE_UPPER_BOUND_THRESHOLD, lambertTerm));
  }
  else
  {
    iterateAllLightsForNonPhotoRealisticLightingModel(useNonPhotoRealisticModel, position, N, E, materialColor, color);
  }
}

void iterateAllLightsForNonPhotoRealisticLightingModel(in int useNonPhotoRealisticModel, in vec3 position, in vec3 N, in vec3 E, in vec4 materialColor, inout vec4 color)
{
  vec3 lightVector = vec3(0.0);
  vec3 L           = vec3(0.0);

  float lambertTerm   = 0.0;
  float specularPower = 0.0;
  float attenuation   = 0.0;
  float lightLength   = 0.0;

  #if USE_SPOT_LIGHTS
    vec3 D = vec3(0.0);
    float cosCurrentAngle   = 0.0;
    float cosInnerConeAngle = 0.0;
    float spotWidth         = 0.0;
    float spotFactor        = 0.0;
  #endif

  gl_MaterialParameters glCurrentMaterial = (gl_FrontFacing) ? gl_FrontMaterial : gl_BackMaterial;

  if (useColorMaterial)
  {
    color += glCurrentMaterial.emission + materialColor * gl_LightModel.ambient;
  }
  else
  {
    color += glCurrentMaterial.emission + glCurrentMaterial.ambient * gl_LightModel.ambient;
  }

  vec4 nonPhotoRealisticSurfaceColor = vec4(0.0);

  for (int index = 0; index < numberOfLights; ++index)
  {
    if (gl_LightSource[index].position.w == 0.0) // directional rather than positional light
    {
      lightVector = vec3(gl_LightSource[index].position.xyz);
      attenuation = 1.0;
    }
    else // -position.xyz, positional rather than directional light
    {
      lightVector = vec3(gl_LightSource[index].position.xyz - position);
      lightLength = length(lightVector);
      attenuation = 1.0 / (gl_LightSource[index].constantAttenuation +
                           gl_LightSource[index].linearAttenuation * lightLength +
                           gl_LightSource[index].quadraticAttenuation * lightLength * lightLength);

      if (useNonPhotoRealisticModel == 4)
      {
        attenuation *= 2.0; // extra attenuation factor for Toon
      }

      #if USE_SPOT_LIGHTS
        // used for spotlight lighting model
        D = normalize(gl_LightSource[index].spotDirection);
        cosCurrentAngle = dot(-L, D);
        cosInnerConeAngle = gl_LightSource[index].spotCosCutoff;
        spotWidth = (cosInnerConeAngle - COS_OUTER_CONE_ANGLE);
        spotFactor = (useOrenNayarDiffuseModel) ? smootherstep(cosInnerConeAngle - spotWidth, cosInnerConeAngle + spotWidth, cosCurrentAngle)
                                                : clamp((cosCurrentAngle - COS_OUTER_CONE_ANGLE) / spotWidth, 0.0, 1.0);
        spotFactor = pow(spotFactor, gl_LightSource[index].spotExponent);
        attenuation *= spotFactor;
      #endif
    }

    L = normalize(lightVector);
    lambertTerm = max(dot(N, L), FLT_EPSILON); // failsafe GLSL Shader value for diffuse light computation

    specularPower = (glCurrentMaterial.shininess > 0.0) ? applyGaussianSpecularModel(E, N, L, glCurrentMaterial.shininess, extraGaussianSpecularFactor * GAUSSIAN_SPECULAR_FACTOR): 0.0; // Gaussian specular model is used for increased accuracy

    if (useColorMaterial)
    {
      if (useNonPhotoRealisticModel == 4) // Toon
      {
        nonPhotoRealisticSurfaceColor = applyToonColor(lambertTerm, specularPower, attenuation, 0.5, 0.2, materialColor);
      }
      else // if (useNonPhotoRealisticModel == 5) // for Gooch
      {
        nonPhotoRealisticSurfaceColor = applyGoochColor(lambertTerm, materialColor);
      }
    }
    else
    {
      if (useNonPhotoRealisticModel == 4) // Toon
      {
        nonPhotoRealisticSurfaceColor = applyToonColor(lambertTerm, specularPower, attenuation, 0.5, 0.2, glCurrentMaterial.diffuse);
      }
      else // if (useNonPhotoRealisticModel == 5) // for Gooch
      {
        nonPhotoRealisticSurfaceColor = applyGoochColor(lambertTerm, glCurrentMaterial.diffuse);
      }
    }

    // Ambient light computation
    color += gl_LightSource[index].ambient * nonPhotoRealisticSurfaceColor * attenuation;

    // Diffuse light computation
    color += (useOrenNayarDiffuseModel ? applyOrenNayarDiffuseModel(-E, N, L, gl_LightSource[index].diffuse) : gl_LightSource[index].diffuse) * nonPhotoRealisticSurfaceColor * lambertTerm * attenuation;

    // Specular light computation & optional Fresnel Model
    color += gl_LightSource[index].specular * glCurrentMaterial.specular * specularPower * attenuation;

    if (useFresnelFactorSchlickApproximationSpecularModel)
    {
      color += applyFresnelFactorSchlickApproximationSpecularModel(E, N, L) * attenuation;
    }
  }

  color = clamp(color, 0.0, 1.0);
}

#if GPU_SHADER4_COMPATIBILITY_CONDITION
  uint packFloat4ToUnsignedInteger(in vec4 values)
  {
    uvec4 uRGBA = uvec4(255.0 * values); // SIMD multiply commands in 4 channels
    return (uRGBA.r << 24) | (uRGBA.g << 16) | (uRGBA.b << 8) | uRGBA.a;
  }

  vec4 unpackUnsignedIntegerToFloat4(in uint value)
  {
    uvec4 uRGBA = uvec4(value >> 24, value >> 16, value >> 8, value);
    return vec4(uRGBA & 255U) / 255.0; // SIMD &-divide commands in 4 channels
  }

  uint packNormalToUnsignedInteger(in vec3 normal)
  {
    return packNormalToUnsignedInteger(normal, true);
  }

  uint packNormalToUnsignedInteger(in vec3 normal, bool useFrontFacingForDeferredLighting)
  {
    // packs the normal with 10bit accuracy and leaves 1bit for sign, normal has to be transformed to [-1,1] => [0,1]
  #if GPU_SHADER5_COMPATIBILITY_CONDITION
    vec3 nNormal           = fma(vec3(0.5), normalize(normal), vec3(0.5)); // fast fma usage
  #else
    vec3 nNormal           = 0.5 * normalize(normal) + 0.5;
  #endif
    uvec3 uNormal           = uvec3(1023.0 * nNormal); // SIMD multiply commands in 3 channels
    uint normalSignBit      = (!useFrontFacingForDeferredLighting || gl_FrontFacing) ? 0U : 1U;
    uint signedPackedNormal = (normalSignBit << 30) | (uNormal.x << 20) | (uNormal.y << 10) | uNormal.z;

    return signedPackedNormal;
  }

  vec3 unpackUnsignedIntegerToNormal(in uint signedPackedNormal)
  {
    return unpackUnsignedIntegerToNormal(signedPackedNormal, true);
  }

  vec3 unpackUnsignedIntegerToNormal(in uint signedPackedNormal, bool useFrontFacingForDeferredLighting)
  {
    // unpacks the normal with 10bit accuracy and leaves 1bit for sign
    uvec4 uNormal  = uvec4(signedPackedNormal >> 30, signedPackedNormal >> 20, signedPackedNormal >> 10, signedPackedNormal);
    int normalSign = (!useFrontFacingForDeferredLighting || (((int(uNormal.x) & int(1)) == 0))) ? 1 : -1;
    vec3 normal    = vec3(uNormal.yzw & 1023U) / 1023.0; // SIMD &-divide commands in 3 channels (normal stored in last 3 channels of uNormal above)

    // normal has to be transformed back to [0,1] => [-1,1]
  #if GPU_SHADER5_COMPATIBILITY_CONDITION
    return normalSign * fma(vec3(2.0), normal, vec3(-1.0)); // fast fma usage
  #else
    return normalSign * (2.0 * normal - 1.0);
  #endif
  }

  #if USE_DEFERRED_SHADING
    float getCurrentFragmentMaterialShininess()
    {
      gl_MaterialParameters glCurrentMaterial = (gl_FrontFacing) ? gl_FrontMaterial : gl_BackMaterial;

      return glCurrentMaterial.shininess;
    }

    uint[4] getCurrentFragmentMaterialColors()
    {
      gl_MaterialParameters glCurrentMaterial = (gl_FrontFacing) ? gl_FrontMaterial : gl_BackMaterial;

      uint[4] materialColors;
      materialColors[0] = packUnorm4x8(glCurrentMaterial.emission);
      materialColors[1] = packUnorm4x8(glCurrentMaterial.ambient);
      materialColors[2] = packUnorm4x8(glCurrentMaterial.diffuse);
      materialColors[3] = packUnorm4x8(glCurrentMaterial.specular);

      return materialColors;
    }

    void applyADSLightingModelDeferredShading(in int useSpecularDiffuseModel, in vec3 position, in vec3 normal, in uint[4] materialColors, in float shininess, in vec4 materialColor, in bool usePositionVector, inout vec4 color)
    {
      vec3 lightVector = vec3(0.0);
      vec3 E           = (usePositionVector) ? normalize(-position) : normalize(-EYE_POSITION_FOR_NON_PERSPECTIVE_CORRECTED_SPHERE_IMPOSTOR);
      vec3 N           = normal;
      vec3 L           = vec3(0.0);

      float lambertTerm   = 0.0;
      float specularPower = 0.0;
      float attenuation   = 0.0;
      float lightLength   = 0.0;

      #if USE_SPOT_LIGHTS
        vec3 D = vec3(0.0);
        float cosCurrentAngle   = 0.0;
        float cosInnerConeAngle = 0.0;
        float spotWidth         = 0.0;
        float spotFactor        = 0.0;
      #endif

      vec4 emission = unpackUnorm4x8(materialColors[0]);
      vec4 ambient  = unpackUnorm4x8(materialColors[1]);
      vec4 diffuse  = unpackUnorm4x8(materialColors[2]);
      vec4 specular = unpackUnorm4x8(materialColors[3]);

      if (useColorMaterial)
      {
        color += fma(materialColor, gl_LightModel.ambient, emission); // fast fma usage
      }
      else
      {
        color += fma(      ambient, gl_LightModel.ambient, emission); // fast fma usage
      }

      for (int index = 0; index < numberOfLights; ++index)
      {
        if (gl_LightSource[index].position.w == 0.0) // directional rather than positional light
        {
          lightVector = vec3(gl_LightSource[index].position.xyz);
          attenuation = 1.0;
        }
        else // -position.xyz, positional rather than directional light
        {
          lightVector = vec3(gl_LightSource[index].position.xyz - position);
          lightLength = length(lightVector);
          attenuation = 1.0 / (gl_LightSource[index].constantAttenuation +
                               gl_LightSource[index].linearAttenuation * lightLength +
                               gl_LightSource[index].quadraticAttenuation * lightLength * lightLength);

          #if USE_SPOT_LIGHTS
            // used for spotlight lighting model
            D = normalize(gl_LightSource[index].spotDirection);
            cosCurrentAngle = dot(-L, D);
            cosInnerConeAngle = gl_LightSource[index].spotCosCutoff;
            spotWidth = (cosInnerConeAngle - COS_OUTER_CONE_ANGLE);
            spotFactor = (useOrenNayarDiffuseModel) ? smootherstep(cosInnerConeAngle - spotWidth, cosInnerConeAngle + spotWidth, cosCurrentAngle)
                                                    : clamp((cosCurrentAngle - COS_OUTER_CONE_ANGLE) / spotWidth, 0.0, 1.0);
            spotFactor = pow(spotFactor, gl_LightSource[index].spotExponent);
            attenuation *= spotFactor;
          #endif
        }

        L = normalize(lightVector);
        lambertTerm = max(dot(N, L), FLT_EPSILON); // failsafe GLSL Shader value for diffuse light computation

        // Ambient light computation
        if (useColorMaterial)
        {
          color += gl_LightSource[index].ambient * materialColor * attenuation;
        }
        else
        {
          color += gl_LightSource[index].ambient * ambient * attenuation;
        }

        // Diffuse light computation
        if (useColorMaterial)
        {
          color += (useOrenNayarDiffuseModel ? applyOrenNayarDiffuseModel(-E, N, L, gl_LightSource[index].diffuse) : gl_LightSource[index].diffuse) * materialColor * lambertTerm * attenuation;
        }
        else
        {
          color += (useOrenNayarDiffuseModel ? applyOrenNayarDiffuseModel(-E, N, L, gl_LightSource[index].diffuse) : gl_LightSource[index].diffuse) * diffuse * lambertTerm * attenuation;
        }

        if (shininess > 0.0)
        {
          if (useSpecularDiffuseModel == 1)
          {
            specularPower = applyPhongSpecularModel(E, N, L, shininess, PHONG_SPECULAR_FACTOR);
          }
          else if (useSpecularDiffuseModel == 2)
          {
            specularPower = applyBlinnPhongSpecularModel(E, N, L, shininess, BLINN_PHONG_SPECULAR_FACTOR);
          }
          else // if (useSpecularDiffuseModel == 3)
          {
            specularPower = applyGaussianSpecularModel(E, N, L, shininess, extraGaussianSpecularFactor * GAUSSIAN_SPECULAR_FACTOR);
          }
        }
        else
        {
          specularPower = 0.0;
        }

        // Specular light computation & optional Fresnel Model
        color += gl_LightSource[index].specular * specular * specularPower * attenuation;

        if (useFresnelFactorSchlickApproximationSpecularModel)
        {
          color += applyFresnelFactorSchlickApproximationSpecularModel(E, N, L) * attenuation;
        }
      }

      color = clamp(color, 0.0, 1.0);
    }

    void applyNonPhotoRealisticLightingModelDeferredShading(in int useNonPhotoRealisticModel, in vec3 position, in vec3 normal, in uint[4] materialColors, in float shininess, in vec4 materialColor, in bool usePositionVector, inout vec4 color)
    {
      // Dot Product based for simple & fast silhouette rendering
      vec3 E = (usePositionVector) ? normalize(-position) : normalize(-EYE_POSITION_FOR_NON_PERSPECTIVE_CORRECTED_SPHERE_IMPOSTOR);
      vec3 N = normalize(normal); // re-normalize to avoid potential artifacts after stored for deferred shading as a uint
      float lambertTerm = max(dot(N, E), FLT_EPSILON); // failsafe GLSL Shader value for diffuse light computation
      if (useOrenNayarDiffuseModel)
      {
        float rateOfChange = fwidth(lambertTerm); // extra silhouette color based on fwidth()
        if (rateOfChange <= SILHOUETTE_LOWER_BOUND_THRESHOLD)
        {
          lambertTerm -= rateOfChange;
        }
      }

      if (lambertTerm < SILHOUETTE_LOWER_BOUND_THRESHOLD)
      {
        color = SILHOUETTE_COLOR;
      }
      else if ((lambertTerm >= SILHOUETTE_LOWER_BOUND_THRESHOLD) && (lambertTerm < SILHOUETTE_UPPER_BOUND_THRESHOLD))
      {
        iterateAllLightsForNonPhotoRealisticLightingModelDeferredShading(useNonPhotoRealisticModel, position, N, E, materialColors, shininess, materialColor, color);
        color = mix(SILHOUETTE_COLOR, color, smootherstep(SILHOUETTE_LOWER_BOUND_THRESHOLD, SILHOUETTE_UPPER_BOUND_THRESHOLD, lambertTerm));
      }
      else
      {
        iterateAllLightsForNonPhotoRealisticLightingModelDeferredShading(useNonPhotoRealisticModel, position, N, E, materialColors, shininess, materialColor, color);
      }
    }

    void iterateAllLightsForNonPhotoRealisticLightingModelDeferredShading(in int useNonPhotoRealisticModel, in vec3 position, in vec3 N, in vec3 E, in uint[4] materialColors, in float shininess, in vec4 materialColor, inout vec4 color)
    {
      vec3 lightVector = vec3(0.0);
      vec3 L           = vec3(0.0);

      float lambertTerm   = 0.0;
      float specularPower = 0.0;
      float attenuation   = 0.0;
      float lightLength   = 0.0;

      #if USE_SPOT_LIGHTS
        vec3 D = vec3(0.0);
        float cosCurrentAngle   = 0.0;
        float cosInnerConeAngle = 0.0;
        float spotWidth         = 0.0;
        float spotFactor        = 0.0;
      #endif

      vec4 emission = unpackUnorm4x8(materialColors[0]);
      vec4 ambient  = unpackUnorm4x8(materialColors[1]);
      vec4 diffuse  = unpackUnorm4x8(materialColors[2]);
      vec4 specular = unpackUnorm4x8(materialColors[3]);

      if (useColorMaterial)
      {
        color += fma(materialColor, gl_LightModel.ambient, emission); // fast fma usage
      }
      else
      {
        color += fma(      ambient, gl_LightModel.ambient, emission); // fast fma usage
      }

      vec4 nonPhotoRealisticSurfaceColor = vec4(0.0);

      for (int index = 0; index < numberOfLights; ++index)
      {
        if (gl_LightSource[index].position.w == 0.0) // directional rather than positional light
        {
          lightVector = vec3(gl_LightSource[index].position.xyz);
          attenuation = 1.0;
        }
        else // -position.xyz, positional rather than directional light
        {
          lightVector = vec3(gl_LightSource[index].position.xyz - position);
          lightLength = length(lightVector);
          attenuation = 1.0 / (gl_LightSource[index].constantAttenuation +
                               gl_LightSource[index].linearAttenuation * lightLength +
                               gl_LightSource[index].quadraticAttenuation * lightLength * lightLength);

          if (useNonPhotoRealisticModel == 4)
          {
            attenuation *= 2.0; // extra attenuation factor for Toon
          }

          #if USE_SPOT_LIGHTS
            // used for spotlight lighting model
            D = normalize(gl_LightSource[index].spotDirection);
            cosCurrentAngle = dot(-L, D);
            cosInnerConeAngle = gl_LightSource[index].spotCosCutoff;
            spotWidth = (cosInnerConeAngle - COS_OUTER_CONE_ANGLE);
            spotFactor = (useOrenNayarDiffuseModel) ? smootherstep(cosInnerConeAngle - spotWidth, cosInnerConeAngle + spotWidth, cosCurrentAngle)
                                                    : clamp((cosCurrentAngle - COS_OUTER_CONE_ANGLE) / spotWidth, 0.0, 1.0);
            spotFactor = pow(spotFactor, gl_LightSource[index].spotExponent);
            attenuation *= spotFactor;
          #endif
        }

        L = normalize(lightVector);
        lambertTerm = max(dot(N, L), FLT_EPSILON); // failsafe GLSL Shader value for diffuse light computation

        specularPower = (shininess > 0.0) ? applyGaussianSpecularModel(E, N, L, shininess, extraGaussianSpecularFactor * GAUSSIAN_SPECULAR_FACTOR): 0.0; // Gaussian specular model is used for increased accuracy

        if (useColorMaterial)
        {
          if (useNonPhotoRealisticModel == 4) // Toon
          {
            nonPhotoRealisticSurfaceColor = applyToonColor(lambertTerm, specularPower, attenuation, 0.5, 0.2, materialColor);
          }
          else // if (useNonPhotoRealisticModel == 5) // for Gooch
          {
            nonPhotoRealisticSurfaceColor = applyGoochColor(lambertTerm, materialColor);
          }
        }
        else
        {
          if (useNonPhotoRealisticModel == 4) // Toon
          {
            nonPhotoRealisticSurfaceColor = applyToonColor(lambertTerm, specularPower, attenuation, 0.5, 0.2, diffuse);
          }
          else // if (useNonPhotoRealisticModel == 5) // for Gooch
          {
            nonPhotoRealisticSurfaceColor = applyGoochColor(lambertTerm, diffuse);
          }
        }

        // Ambient light computation
        color += gl_LightSource[index].ambient * nonPhotoRealisticSurfaceColor * attenuation;

        // Diffuse light computation
        color += (useOrenNayarDiffuseModel ? applyOrenNayarDiffuseModel(-E, N, L, gl_LightSource[index].diffuse) : gl_LightSource[index].diffuse) * nonPhotoRealisticSurfaceColor * lambertTerm * attenuation;

        // Specular light computation & optional Fresnel Model
        color += gl_LightSource[index].specular * specular * specularPower * attenuation;
        if (useFresnelFactorSchlickApproximationSpecularModel)
        {
          color += applyFresnelFactorSchlickApproximationSpecularModel(E, N, L) * attenuation;
        }
      }

      color = clamp(color, 0.0, 1.0);
    }
  #endif
#endif