/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  Apply Surface Lighting LOD Models GLSL code for Phong, Blinn-Phong, Gaussian, Toon, Gooch.
*  Using core functions from CommonFunctions & SurfaceLightingModels GLSL libraries.
*  References:
*  1. en.wikipedia.org/wiki/Phong_reflection_model                   -> Phong Specular Lighting Model
*  2. en.wikipedia.org/wiki/Blinn%E2%80%93Phong_shading_model        -> Blinn-Phong Specular Lighting Model
*  3. en.wikipedia.org/wiki/Specular_highlight#Gaussian_distribution -> Gaussiang Specular Lighting Model
*  4. en.wikipedia.org/wiki/Cel_shading                              -> Toon Specular Lighting Model
*  5. www.cs.northwestern.edu/~ago820/SIG98/abstract.html            -> Gooch Specular Lighting Model
*  6. en.wikipedia.org/wiki/Oren%E2%80%93Nayar_reflectance_model     -> Oren-Nayar Diffuse Lighting Model
*  7. en.wikipedia.org/wiki/Reflection_mapping#Sphere_mapping        -> Reflection (Sphere) Mapping
*  8. en.wikipedia.org/wiki/Reflection_mapping                       -> Reflection Mapping
*
*  Relevant variables controlling the above: lightingModel & environmentMapping
*  Relevant functions: applyADSLightingModel() & applyNonPhotoRealisticLightingModel()
*
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/

#if GPU_GEOMETRY_SHADER4_COMPATIBILITY_CONDITION
  in vec4 FS_COLOR;
#endif
in vec3 FS_NORMAL;
in vec3 FS_EYE_VECTOR;

uniform bool enforceColor = false;
uniform bool blackOrWhite = true;
uniform int lightingModel = 2; // Blinn-Phong by default
uniform bool texturing = false;
uniform bool environmentMapping = false;
uniform bool fog = false;
uniform bool gammaCorrection = false;
uniform float opacity = 1.0;
uniform bool useGBuffer = false;

const vec4 BLACK_COLOR = vec4(0.0, 0.0, 0.0, 1.0);
const vec4 WHITE_COLOR = vec4(1.0, 1.0, 1.0, 1.0);

void applyADSLightingModel(in int useSpecularDiffuseModel, in vec3 position, in vec3 normal, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
void applyNonPhotoRealisticLightingModel(in int useNonPhotoRealisticModel, in vec3 position, in vec3 normal, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
void applyTexture(in vec2 texCoords, inout vec4 color);
void applyTextureWithEmbossFilter(in vec2 texCoords, inout vec4 color);
void applyJitteredEnvironmentMapping(in vec3 position, in vec3 normal, inout vec4 color);
void applyFastGammaCorrection(inout vec4 color);
void applyFog(inout vec4 color);
#if USE_DEFERRED_SHADING
  void renderToABuffer3DDeferredShading(in vec4 color, in vec3 normal);
#endif
#if USE_OIT
  void renderToABuffer3D(in vec4 color, in vec3 normal);
#endif
uint packFloat4ToUnsignedInteger(in vec4 values);
uint packNormalToUnsignedInteger(in vec3 normal, bool useFrontFacing);

void main()
{
  if (enforceColor)
  {
    vec4 color = (blackOrWhite) ? BLACK_COLOR : WHITE_COLOR;

    if (fog)
    {
      applyFog(color);
    }

    gl_FragColor = color;

    return;
  }

#if USE_DEFERRED_SHADING

  renderToABuffer3DDeferredShading(vec4(gl_Color.rgb, gl_Color.a * opacity), FS_NORMAL);

#else

  #if GPU_GEOMETRY_SHADER4_COMPATIBILITY_CONDITION
    vec4 color = FS_COLOR;
  #else
    vec4 color = vec4(0.0);
  #endif

  if (lightingModel > 3)
  {
    applyNonPhotoRealisticLightingModel(lightingModel, FS_EYE_VECTOR, FS_NORMAL, gl_Color, true, color);
    if (texturing)
    {
      if (environmentMapping)
      {
        applyJitteredEnvironmentMapping(FS_EYE_VECTOR, FS_NORMAL, color);
      }
      else
      {
        applyTextureWithEmbossFilter(gl_TexCoord[0].st, color);
      }
    }
  }
  else
  {
    applyADSLightingModel(lightingModel, FS_EYE_VECTOR, FS_NORMAL, gl_Color, true, color);
    if (texturing)
    {
      if (environmentMapping)
      {
        applyJitteredEnvironmentMapping(FS_EYE_VECTOR, FS_NORMAL, color);
      }
      else
      {
        applyTexture(gl_TexCoord[0].st, color);
      }
    }
  }

  if (fog)
  {
    applyFog(color);
  }

  if (gammaCorrection)
  {
    applyFastGammaCorrection(color);
  }

  #if USE_OIT
    renderToABuffer3D(vec4(color.rgb, gl_Color.a * opacity), FS_NORMAL);
  #else
    if (useGBuffer)
    {
      uint packedColor  = packFloat4ToUnsignedInteger(vec4(color.rgb, gl_Color.a * opacity));
      uint packedNormal = packNormalToUnsignedInteger(FS_NORMAL, false);
      gl_FragColor = vec4(uintBitsToFloat(packedColor), uintBitsToFloat(packedNormal), gl_FragCoord.z, 1.0);
    }
    else
    {
      gl_FragColor = vec4(color.rgb, gl_Color.a * opacity);
    }
  #endif

#endif
}