/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  Apply Point Sphere Tess Quad Model GLSL code.
*  Using core functions from CommonFunctions & SurfaceLightingModels GLSL libraries.
*
*  Relevant variables controlling the above: lightingModel & environmentMapping
*  Relevant functions: applyADSLightingModel() & applyNonPhotoRealisticLightingModel()
*
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/

#if GPU_SHADER_FP64_COMPATIBILITY_CONDITION
  #extension GL_ARB_gpu_shader_fp64: enable
#endif

uniform int lightingModel = 2; // Blinn-Phong by default
uniform bool pointAntialias = false;
uniform bool perspectiveCorrection = false;
uniform bool texturing = false;
uniform bool sphericalMapping = false;
uniform bool environmentMapping = false;
uniform bool fog = false;
uniform bool gammaCorrection = false;
uniform float opacity = 1.0;
uniform bool useGBuffer = false;

in float fsSphereRadius;
in float fsCircularFactor;
in vec4 fsSphereColor;
in vec2 fsMapping;
in vec3 fsCameraSpherePosition;

// Machine float epsilon
const float FLT_EPSILON = 0.000001;

float smootherstep(in float edge0, in float edge1, in float x);
void perspectiveSphereImpostor(in vec2 mapping, in float sphereRadius, in vec3 cameraSpherePosition, out vec3 cameraNormal, out vec3 cameraPosition);
void perspectiveSphereImpostorAntiAlias(in vec3 cameraNormal, in vec3 cameraPosition, inout float sphereAlpha);
void calculateDistanceMetricForSimpleSphereImpostor(in vec2 mapping, out float distanceMetric);
void simpleSphereImpostor(in vec2 mapping, in float distanceMetric, in float sphereRadius, in vec3 cameraSpherePosition, in bool multiplySphereRadius, out vec3 cameraNormal, out vec3 cameraPosition);
void applyPointSphereSimpleLighting(in float distanceMetric, in vec4 materialColor, inout vec4 color);
void applySimpleSphereImpostorAntiAlias(in vec2 mapping, in float circularFactor, inout float sphereAlpha);
void applyADSLightingModel(in int useSpecularDiffuseModel, in vec3 position, in vec3 normal, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
void applyNonPhotoRealisticLightingModel(in int useNonPhotoRealisticModel, in vec3 position, in vec3 normal, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
vec2 applySphericalCoordinates(in vec3 position, in vec3 normal);
vec2 applyImpostorSphereTextureCoords(in vec3 normal);
void applyTexture(in vec2 texCoords, inout vec4 color);
void applyTextureWithEmbossFilter(in vec2 texCoords, inout vec4 color);
void applyJitteredEnvironmentMapping(in vec3 position, in vec3 normal, inout vec4 color);
void applyFastGammaCorrection(inout vec4 color);
void applyFog(inout vec4 color);
float applyDepth(in vec3 position);
#if USE_OIT
  void renderToABuffer3D(in float depth, in vec4 color, in vec3 normal);
#endif
uint packFloat4ToUnsignedInteger(in vec4 values);
uint packNormalToUnsignedInteger(in vec3 normal, bool useFrontFacing);

void main()
{
  float distanceMetric = 0.0;
  vec3 cameraNormal    = vec3(0.0);
  vec3 cameraPosition  = vec3(0.0);
  if ((lightingModel != 0) && perspectiveCorrection)
  {
    perspectiveSphereImpostor(fsMapping, fsSphereRadius, fsCameraSpherePosition, cameraNormal, cameraPosition);
  }
  else
  {
    calculateDistanceMetricForSimpleSphereImpostor(fsMapping, distanceMetric);
    simpleSphereImpostor(fsMapping, distanceMetric, fsSphereRadius, fsCameraSpherePosition, true, cameraNormal, cameraPosition);
  }

  vec4 color = vec4(0.0);
  if (lightingModel == 0)
  {
    calculateDistanceMetricForSimpleSphereImpostor(fsMapping, distanceMetric); // re-run calculateDistanceMetricForSimpleSphereImpostor() due to a shader compiler issue
    applyPointSphereSimpleLighting(distanceMetric, fsSphereColor, color);
    if (texturing)
    {
      if (environmentMapping)
      {
        applyJitteredEnvironmentMapping(fsCameraSpherePosition, cameraNormal, color);
      }
      else
      {
        vec2 textureCoords = (sphericalMapping) ? applySphericalCoordinates(fsCameraSpherePosition, cameraNormal) : applyImpostorSphereTextureCoords(cameraNormal);
        applyTexture(textureCoords, color);
      }
    }
  }
  else if (lightingModel > 3)
  {
    applyNonPhotoRealisticLightingModel(lightingModel, cameraPosition, cameraNormal, fsSphereColor, perspectiveCorrection, color);
    if (texturing)
    {
      if (environmentMapping)
      {
        applyJitteredEnvironmentMapping(fsCameraSpherePosition, cameraNormal, color);
      }
      else
      {
        vec2 textureCoords = (sphericalMapping) ? applySphericalCoordinates(fsCameraSpherePosition, cameraNormal) : applyImpostorSphereTextureCoords(cameraNormal);
        applyTextureWithEmbossFilter(textureCoords, color);
      }
    }
  }
  else // between 1-3
  {
    applyADSLightingModel(lightingModel, cameraPosition, cameraNormal, fsSphereColor, perspectiveCorrection, color);
    if (texturing)
    {
      if (environmentMapping)
      {
        applyJitteredEnvironmentMapping(fsCameraSpherePosition, cameraNormal, color);
      }
      else
      {
        vec2 textureCoords = (sphericalMapping) ? applySphericalCoordinates(fsCameraSpherePosition, cameraNormal) : applyImpostorSphereTextureCoords(cameraNormal);
        applyTexture(textureCoords, color);
      }
    }
  }

  if (fog)
  {
    applyFog(color);
  }

  if (gammaCorrection)
  {
    applyFastGammaCorrection(color);
  }

  float sphereAlpha = fsSphereColor.a;
  if (pointAntialias)
  {
    if ((lightingModel != 0) && perspectiveCorrection)
    {
      perspectiveSphereImpostorAntiAlias(cameraPosition, cameraNormal, sphereAlpha);
    }
    else
    {
      applySimpleSphereImpostorAntiAlias(fsMapping, fsCircularFactor, sphereAlpha);
    }
  }

  float depth = applyDepth(cameraPosition); // set the depth based on the new cameraPosition

  #if USE_OIT
    renderToABuffer3D(depth, vec4(color.rgb, sphereAlpha * opacity), cameraNormal);
  #else
    if (useGBuffer)
    {
      uint packedColor       = packFloat4ToUnsignedInteger(vec4(color.rgb, sphereAlpha * opacity));
      uint packedNormal      = packNormalToUnsignedInteger(cameraNormal, false);
      gl_FragColor = vec4(uintBitsToFloat(packedColor), uintBitsToFloat(packedNormal), depth, 1.0);
      gl_FragDepth = depth;
    }
    else
    {
      gl_FragColor = vec4(color.rgb, sphereAlpha * opacity);
      gl_FragDepth = depth;
    }
  #endif
}

void perspectiveSphereImpostor(in vec2 mapping, in float sphereRadius, in vec3 cameraSpherePosition, out vec3 cameraNormal, out vec3 cameraPosition)
{
  #if GPU_SHADER_FP64_COMPATIBILITY_CONDITION
    dvec3 cameraPlanePosition = dvec3(mapping * sphereRadius, double(0.0)) + cameraSpherePosition;
    dvec3 rayDirection = normalize(cameraPlanePosition);

    double B = double(-2.0) * dot(rayDirection, cameraSpherePosition);
    double C = dot(cameraSpherePosition, cameraSpherePosition) - (sphereRadius * sphereRadius);

    double discriminant = (B * B) - (double(4.0) * C);
    if (discriminant < double(0.0))
    {
      discard;
    }

    double sqrtDiscriminant = sqrt(discriminant);
    dvec2 posNegT = double(0.5) * (-B + dvec2(sqrtDiscriminant, -sqrtDiscriminant));

    double intersectT = min(posNegT.x, posNegT.y);
  #else
    vec3 cameraPlanePosition = vec3(mapping * sphereRadius, 0.0) + cameraSpherePosition;
    vec3 rayDirection = normalize(cameraPlanePosition);

    float B = -2.0 * dot(rayDirection, cameraSpherePosition);
    float C = dot(cameraSpherePosition, cameraSpherePosition) - (sphereRadius * sphereRadius);

    float discriminant = (B * B) - (4.0 * C);
    if (discriminant < 0.0)
    {
      discard;
    }

    float sqrtDiscriminant = sqrt(discriminant);
    vec2 posNegT = (-B + vec2(sqrtDiscriminant, -sqrtDiscriminant)) / 2.0;

    float intersectT = min(posNegT.x, posNegT.y);
  #endif

  cameraPosition = vec3(intersectT * rayDirection);
  cameraNormal   = normalize(cameraPosition - cameraSpherePosition);
}

void perspectiveSphereImpostorAntiAlias(in vec3 cameraNormal, in vec3 cameraPosition, inout float sphereAlpha)
{
  int normalSign = (gl_FrontFacing) ? 1 : -1;
  vec3 E = normalize(-cameraPosition);
  vec3 N = normalize(normalSign * cameraNormal);
  float lambertTerm = max(dot(N, E), FLT_EPSILON); // failsafe GLSL Shader value for diffuse light computation

  if ((lambertTerm > 0.0001) && (lambertTerm < 0.35))
  {
    sphereAlpha *= smootherstep(0.0001, 0.35, lambertTerm);
  }
}