/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  Apply Point Sphere Tess LOD Model GLSL code.
*  Using core functions from CommonFunctions & SurfaceLightingModels GLSL libraries.
*
*  Relevant variables controlling the above: lightingModel & environmentMapping
*  Relevant functions: applyADSLightingModel() & applyNonPhotoRealisticLightingModel()
*
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/

uniform int lightingModel = 2; // Blinn-Phong by default
uniform bool texturing = false;
uniform bool environmentMapping = false;
uniform bool fog = false;
uniform bool gammaCorrection = false;
uniform float opacity = 1.0;
uniform bool useGBuffer = false;

in vec3 fsNormal;
in vec3 fsEyeVector;
in vec4 fsSphereColor;
in vec2 fsTextureCoords;

void applyADSLightingModel(in int useSpecularDiffuseModel, in vec3 position, in vec3 fsNormal, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
void applyNonPhotoRealisticLightingModel(in int useNonPhotoRealisticModel, in vec3 position, in vec3 fsNormal, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
void applyTexture(in vec2 texCoords, inout vec4 color);
void applyTextureWithEmbossFilter(in vec2 texCoords, inout vec4 color);
void applyJitteredEnvironmentMapping(in vec3 position, in vec3 normal, inout vec4 color);
void applyFastGammaCorrection(inout vec4 color);
void applyFog(inout vec4 color);
#if USE_OIT
  void renderToABuffer3D(in vec4 color, in vec3 normal);
#endif
uint packFloat4ToUnsignedInteger(in vec4 values);
uint packNormalToUnsignedInteger(in vec3 normal, bool useFrontFacing);

void main()
{
  vec4 color = vec4(0.0);

  if (lightingModel > 3)
  {
    applyNonPhotoRealisticLightingModel(lightingModel, fsEyeVector, fsNormal, fsSphereColor, true, color);
    if (texturing)
    {
      if (environmentMapping)
      {
        applyJitteredEnvironmentMapping(fsEyeVector, fsNormal, color);
      }
      else
      {
        applyTextureWithEmbossFilter(fsTextureCoords, color);
      }
    }
  }
  else
  {
    applyADSLightingModel(lightingModel, fsEyeVector, fsNormal, fsSphereColor, true, color);
    if (texturing)
    {
      if (environmentMapping)
      {
        applyJitteredEnvironmentMapping(fsEyeVector, fsNormal, color);
      }
      else
      {
        applyTexture(fsTextureCoords, color);
      }
    }
  }

  if (fog)
  {
    applyFog(color);
  }

  if (gammaCorrection)
  {
    applyFastGammaCorrection(color);
  }

  #if USE_OIT
    renderToABuffer3D(vec4(color.rgb, fsSphereColor.a * opacity), fsNormal);
  #else
    if (useGBuffer)
    {
      uint packedColor  = packFloat4ToUnsignedInteger(vec4(color.rgb, fsSphereColor.a * opacity));
      uint packedNormal = packNormalToUnsignedInteger(fsNormal, false);
      gl_FragColor = vec4(uintBitsToFloat(packedColor), uintBitsToFloat(packedNormal), gl_FragCoord.z, 1.0);
    }
    else
    {
      gl_FragColor = vec4(color.rgb, fsSphereColor.a * opacity);
    }
  #endif
}