/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  Apply Quad Sphere Model GLSL code.
*  Using core functions from CommonFunctions & SurfaceLightingModels GLSL libraries.
*
*  Relevant variables controlling the above: useActorColor, actorColor, lightingModel, perspectiveCorrection & windowRatio
*
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/

#extension GL_EXT_geometry_shader4 : enable

uniform int lightingModel = 2; // Blinn-Phong by default
uniform bool perspectiveCorrection = false;
uniform float windowRatio = 1.0;

GS_VARYING in float GS_VS_SPHERE_RADIUS[GS_PRIMITIVE_VERTICES_IN];
GS_VARYING in vec4 GS_VS_SPHERE_COLOR[GS_PRIMITIVE_VERTICES_IN];

GS_VARYING out float GS_FS_SPHERE_RADIUS;
GS_VARYING out float GS_FS_SPHERE_CIRCULAR_FACTOR;
GS_VARYING out vec4 GS_FS_SPHERE_COLOR;
GS_VARYING out vec2 GS_FS_MAPPING;
GS_VARYING out vec3 GS_FS_CAMERA_SPHERE_POSITION;

float getSphericalDistanceFromCenter(in float radius, in vec3 center);
bool discardClippedPoint(in vec4 center, float circularFactor);
void emitQuadVertex(in vec2 mapping, in vec4 eyeVector);

// warning, Intel GPU driver crashes with the GL GS CPU-side option calls, have to use the shader layout GL 3.3+ syntax
#if GPU_VENDOR_NOT_NVIDIA
  layout(points) in;
  layout(triangle_strip, max_vertices = 4) out;
#endif

void main()
{
  float radius = GS_VS_SPHERE_RADIUS[0];
  if (radius <= 0.0)
  {
    return; // do not emit any vertices if radius is too small
  }

  vec4 center = gl_in[0].gl_Position;  
  float circularFactor = getSphericalDistanceFromCenter(radius, center.xyz);
  if (discardClippedPoint(center, circularFactor))
  {
    return; // do not emit any vertices if point is clipped
  }

  // set common variables for all emitted vertices only once
  GS_FS_SPHERE_RADIUS          = radius;
  GS_FS_SPHERE_CIRCULAR_FACTOR = circularFactor;
  GS_FS_SPHERE_COLOR           = GS_VS_SPHERE_COLOR[0];
  vec4 eyeVector               = gl_ModelViewMatrix * center;
  GS_FS_CAMERA_SPHERE_POSITION = eyeVector.xyz;

  emitQuadVertex(vec2( 1.0,  1.0), eyeVector); // top-right
  emitQuadVertex(vec2(-1.0,  1.0), eyeVector); // top-left
  emitQuadVertex(vec2( 1.0, -1.0), eyeVector); // bottom-right
  emitQuadVertex(vec2(-1.0, -1.0), eyeVector); // bottom-left
  EndPrimitive();
}

void emitQuadVertex(in vec2 mapping, in vec4 eyeVector)
{
  GS_FS_MAPPING = ((lightingModel != 0) && perspectiveCorrection && (windowRatio > 1.0)) ? windowRatio * mapping : mapping;
  eyeVector.xy += GS_FS_SPHERE_RADIUS * GS_FS_MAPPING;
  gl_Position = gl_ProjectionMatrix * eyeVector;
  gl_ClipVertex = eyeVector;

  EmitVertex();
}