/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  Apply Point Sphere Model GLSL code.
*  Using core functions from CommonFunctions & SurfaceLightingModels GLSL libraries.
*
*  Relevant variables controlling the above: variablePointSphereRadius, inversePixelRadius, lightingModel, fog, gammaCorrection & opacity
*
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/

uniform bool variablePointSphereRadius = false;
uniform float inversePixelRadius = 1.0;
uniform int lightingModel = 2; // Blinn-Phong by default
uniform bool pointAntialias = false;
uniform bool texturing = false;
uniform bool sphericalMapping = false;
uniform bool environmentMapping = false;
uniform bool fog = false;
uniform bool gammaCorrection = false;
uniform float opacity = 1.0;
uniform bool useGBuffer = false;

FS_VARYING float sphereRadius;
FS_VARYING float sphereCircularFactor;
FS_VARYING vec4 sphereColor;
FS_VARYING vec2 sphereClipPosition;
FS_VARYING vec3 cameraSpherePosition;

void calculateDistanceMetricForSimpleSphereImpostor(in vec2 mapping, out float distanceMetric);
void simpleSphereImpostor(in vec2 mapping, in float distanceMetric, in float sphereRadius, in vec3 cameraSpherePosition, in bool multiplySphereRadius, out vec3 cameraNormal, out vec3 cameraPosition);
void applyPointSphereSimpleLighting(in float distanceMetric, in vec4 materialColor, inout vec4 color);
void applySimpleSphereImpostorAntiAlias(in vec2 mapping, in float circularFactor, inout float sphereAlpha);
#if APPLY_SURFACE_LIGHTING_MODELS
  void applyADSLightingModel(in int useSpecularDiffuseModel, in vec3 position, in vec3 normal, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
  void applyNonPhotoRealisticLightingModel(in int useNonPhotoRealisticModel, in vec3 position, in vec3 normal, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
#else
  void applySimplePhongLightingModel(in vec3 position, in vec3 normal, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
#endif
vec2 applySphericalCoordinates(in vec3 position, in vec3 normal);
vec2 applyImpostorSphereTextureCoords(in vec3 normal);
void applyTexture(in vec2 texCoords, inout vec4 color);
void applyTextureWithEmbossFilter(in vec2 texCoords, inout vec4 color);
void applyJitteredEnvironmentMapping(in vec3 position, in vec3 normal, inout vec4 color);
void applyFastGammaCorrection(inout vec4 color);
void applyFog(inout vec4 color);
float applyDepth(in vec3 position);
#if USE_OIT
  void renderToABuffer3D(in float depth, in vec4 color, in vec3 normal);
#endif
#if GPU_SHADER4_COMPATIBILITY_CONDITION
  uint packFloat4ToUnsignedInteger(in vec4 values);
  uint packNormalToUnsignedInteger(in vec3 normal, bool useFrontFacing);
#endif

void main()
{
  vec2 mapping = (variablePointSphereRadius) ? (gl_FragCoord.xy - sphereClipPosition) / sphereRadius : (gl_FragCoord.xy - sphereClipPosition) * inversePixelRadius;
  float distanceMetric = 0.0;
  vec3 cameraNormal    = vec3(0.0);
  vec3 cameraPosition  = vec3(0.0);
  calculateDistanceMetricForSimpleSphereImpostor(mapping, distanceMetric);
  simpleSphereImpostor(mapping, distanceMetric, sphereRadius, cameraSpherePosition, false, cameraNormal, cameraPosition);

  vec4 color = vec4(0.0);
  if (lightingModel == 0)
  {
    applyPointSphereSimpleLighting(distanceMetric, sphereColor, color);
    if (texturing)
    {
      if (environmentMapping)
      {
        applyJitteredEnvironmentMapping(cameraSpherePosition, cameraNormal, color);
      }
      else
      {
        vec2 textureCoords = (sphericalMapping) ? applySphericalCoordinates(cameraSpherePosition, cameraNormal) : applyImpostorSphereTextureCoords(cameraNormal);
        applyTexture(textureCoords, color);
      }
    }
  }
#if APPLY_SURFACE_LIGHTING_MODELS
  else if (lightingModel > 3)
  {
    applyNonPhotoRealisticLightingModel(lightingModel, cameraPosition, cameraNormal, sphereColor, false, color);
    if (texturing)
    {
      if (environmentMapping)
      {
        applyJitteredEnvironmentMapping(cameraSpherePosition, cameraNormal, color);
      }
      else
      {
        vec2 textureCoords = (sphericalMapping) ? applySphericalCoordinates(cameraSpherePosition, cameraNormal) : applyImpostorSphereTextureCoords(cameraNormal);
        applyTextureWithEmbossFilter(textureCoords, color);
      }
    }
  }
  else // between 1-3
  {
    applyADSLightingModel(lightingModel, cameraPosition, cameraNormal, sphereColor, false, color);
    if (texturing)
    {
      if (environmentMapping)
      {
        applyJitteredEnvironmentMapping(cameraSpherePosition, cameraNormal, color);
      }
      else
      {
        vec2 textureCoords = (sphericalMapping) ? applySphericalCoordinates(cameraSpherePosition, cameraNormal) : applyImpostorSphereTextureCoords(cameraNormal);
        applyTexture(textureCoords, color);
      }
    }
  }
#else
  else
  {
    applySimplePhongLightingModel(cameraPosition, cameraNormal, sphereColor, false, color);
    if (texturing)
    {
      if (environmentMapping)
      {
        applyJitteredEnvironmentMapping(cameraSpherePosition, cameraNormal, color);
      }
      else
      {
        vec2 textureCoords = (sphericalMapping) ? applySphericalCoordinates(cameraSpherePosition, cameraNormal) : applyImpostorSphereTextureCoords(cameraNormal);
        applyTexture(textureCoords, color);
      }
    }
  }
#endif

  if (fog)
  {
    applyFog(color);
  }

  if (gammaCorrection)
  {
    applyFastGammaCorrection(color);
  }

  float sphereAlpha = sphereColor.a;
  if (pointAntialias)
  {
    applySimpleSphereImpostorAntiAlias(mapping, sphereCircularFactor / 10.0, sphereAlpha); // divide by 10.0 for point sphere based model
  }

  float depth = applyDepth(cameraPosition); // set the depth based on the new cameraPosition

  #if USE_OIT
    renderToABuffer3D(depth, vec4(color.rgb, sphereAlpha * opacity), cameraNormal);
  #else
    #if GPU_SHADER4_COMPATIBILITY_CONDITION
      if (useGBuffer)
      {
        uint packedColor  = packFloat4ToUnsignedInteger(vec4(color.rgb, sphereAlpha * opacity));
        uint packedNormal = packNormalToUnsignedInteger(cameraNormal, false);
        gl_FragColor = vec4(uintBitsToFloat(packedColor), uintBitsToFloat(packedNormal), depth, 1.0);
        gl_FragDepth = depth;
      }
      else
      {
        gl_FragColor = vec4(color.rgb, sphereAlpha * opacity);
        gl_FragDepth = depth;
      }
    #else
      gl_FragColor = vec4(color.rgb, sphereAlpha * opacity);
      gl_FragDepth = depth;
    #endif
  #endif
}