/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  Apply Cylinder Model GLSL code.
*  Using core functions from CommonFunctions & SurfaceLightingModels GLSL libraries.
*
*  Relevant variables controlling the above: useActorColor, actorColor & lightingModel
*
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/

#extension GL_EXT_geometry_shader4 : enable

in float cylinderRadiusIn[GS_PRIMITIVE_VERTICES_IN];
in float cylinderHeightIn[GS_PRIMITIVE_VERTICES_IN];
in vec3  cylinderDirectionIn[GS_PRIMITIVE_VERTICES_IN];
in vec4  cylinderColorIn[GS_PRIMITIVE_VERTICES_IN];

out vec3  cylinderSurfacePoint;
out vec3  cylinderAxis;
out vec3  cylinderBase;
out vec3  cylinderEnd;
out vec3  cylinderU;
out vec3  cylinderV;
out float cylinderRadius;
out vec4  cylinderColor;

uniform int lightingModel = 2; // Blinn-Phong by default
uniform bool perspectiveCorrection = false;

float getSphericalDistanceFromCenter(in float radius, in vec3 center);
bool discardClippedPoint(in vec4 center, float circularFactor);
void emitTriangleStripVertex(in vec4 vertex);

// warning, Intel GPU driver crashes with the GL GS CPU-side option calls, have to use the shader layout GL 3.3+ syntax
#if GPU_VENDOR_NOT_NVIDIA
  layout(points) in;
  layout(triangle_strip, max_vertices = 6) out;
#endif

void main()
{
  float cylinderHeight = cylinderHeightIn[0];
  if (cylinderHeight <= 0.0)
  {
    return; // do not emit any vertices if radius is too small
  }

  vec3  cylinderCenter    = gl_in[0].gl_Position.xyz;
  vec3  cylinderDirection = cylinderDirectionIn[0];
  float circularFactor    = getSphericalDistanceFromCenter(cylinderHeight, cylinderCenter);
  if (discardClippedPoint(vec4(cylinderCenter, 1.0), circularFactor))
  {
    return; // do not emit any vertices if point is clipped  
  }

  cylinderColor  = cylinderColorIn[0];
  cylinderRadius = cylinderRadiusIn[0];

  // 2) Inverse Matrix Extraction (from https://www.opengl.org/discussion_boards/showthread.php/178484-Extracting-camera-position-from-a-ModelView-Matrix)
  // If you don't know if there is scaling in the matrix, you can invert the matrix then pull out the final column. Code:
  // mat4 viewModel = inverse(modelView);
  // vec3 cameraPos(viewModel[3]); // Might have to divide by w if you can't assume w == 1
  vec3 cameraDirection = normalize(vec3(gl_ModelViewMatrixInverse[3]) - cylinderCenter); // thus eyePoint.xyz can be vec3(gl_ModelViewMatrixInverse[3])
  float angle = dot(cameraDirection, cylinderDirection);
  vec3 cylinderHeightAndDirectionSigned = sign(angle) * cylinderHeight * cylinderDirection; // if (angle < 0.0) direction vector looks away, so flip, else direction vector already looks in my direction
  vec3 left = cylinderRadius * normalize(cross(cameraDirection, cylinderHeightAndDirectionSigned));
  vec3 up   = cylinderRadius * normalize(cross(left, cylinderHeightAndDirectionSigned));

  // transform to modelview coordinates
  cylinderAxis = normalize(gl_NormalMatrix * cylinderHeightAndDirectionSigned);
  cylinderU    = normalize(gl_NormalMatrix * up);
  cylinderV    = normalize(gl_NormalMatrix * left);

  vec4 base = gl_ModelViewMatrix * vec4(cylinderCenter - cylinderHeightAndDirectionSigned, 1.0);
  cylinderBase = base.xyz / base.w;

  vec4 topPosition = gl_ModelViewMatrix * (vec4(cylinderCenter + cylinderHeightAndDirectionSigned, 1.0));
  vec4 end = topPosition;
  cylinderEnd = end.xyz / end.w;
  
  vec4 xf0 = gl_ModelViewMatrix * vec4(cylinderCenter - cylinderHeightAndDirectionSigned + left - up, 1.0);
  vec4 xf2 = gl_ModelViewMatrix * vec4(cylinderCenter - cylinderHeightAndDirectionSigned - left - up, 1.0);
  vec4 xc0 = gl_ModelViewMatrix * vec4(cylinderCenter + cylinderHeightAndDirectionSigned + left - up, 1.0);
  vec4 xc1 = gl_ModelViewMatrix * vec4(cylinderCenter + cylinderHeightAndDirectionSigned + left + up, 1.0);
  vec4 xc2 = gl_ModelViewMatrix * vec4(cylinderCenter + cylinderHeightAndDirectionSigned - left - up, 1.0);
  vec4 xc3 = gl_ModelViewMatrix * vec4(cylinderCenter + cylinderHeightAndDirectionSigned - left + up, 1.0);

  emitTriangleStripVertex(xf0); // Triangle Strip Vertex 1
  emitTriangleStripVertex(xf2); // Triangle Strip Vertex 2
  emitTriangleStripVertex(xc0); // Triangle Strip Vertex 3
  emitTriangleStripVertex(xc2); // Triangle Strip Vertex 4
  emitTriangleStripVertex(xc1); // Triangle Strip Vertex 5
  emitTriangleStripVertex(xc3); // Triangle Strip Vertex 6
  EndPrimitive();
}

void emitTriangleStripVertex(in vec4 vertex)
{
  cylinderSurfacePoint = vertex.xyz / vertex.w;
  gl_Position = gl_ProjectionMatrix * vertex;
  gl_ClipVertex = vertex;
  EmitVertex();
}