/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  Apply Cylinder Model GLSL code.
*  Using core functions from CommonFunctions & SurfaceLightingModels GLSL libraries.
*
*  Relevant variables controlling the above: useActorColor, actorColor & lightingModel
*  Relevant functions: perspectiveCylinderImpostor()
*
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/

#if GPU_SHADER_FP64_COMPATIBILITY_CONDITION
  #extension GL_ARB_gpu_shader_fp64: enable
#endif

uniform int lightingModel = 2; // Blinn-Phong by default
uniform bool texturing = false;
uniform bool sphericalMapping = false;
uniform bool environmentMapping = false;
uniform bool fog = false;
uniform bool gammaCorrection = false;
uniform float opacity = 1.0;
uniform bool useGBuffer = false;

in vec3  cylinderSurfacePoint;
in vec3  cylinderAxis;
in vec3  cylinderBase;
in vec3  cylinderEnd;
in vec3  cylinderU;
in vec3  cylinderV;
in float cylinderRadius;
in vec4  cylinderColor;

void perspectiveCylinderImpostor(out vec3 cameraNormal, out vec3 cameraPosition);
#if APPLY_SURFACE_LIGHTING_MODELS
  void applyADSLightingModel(in int useSpecularDiffuseModel, in vec3 position, in vec3 normal, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
  void applyNonPhotoRealisticLightingModel(in int useNonPhotoRealisticModel, in vec3 position, in vec3 normal, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
#else
  void applySimplePhongLightingModel(in vec3 position, in vec3 normal, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
#endif
vec2 applySphericalCoordinates(in vec3 position, in vec3 normal);
vec2 applyImpostorSphereTextureCoords(in vec3 normal);
void applyTexture(in vec2 texCoords, inout vec4 color);
void applyTextureWithEmbossFilter(in vec2 texCoords, inout vec4 color);
void applyJitteredEnvironmentMapping(in vec3 position, in vec3 normal, inout vec4 color);
void applyFastGammaCorrection(inout vec4 color);
void applyFog(inout vec4 color);
float applyDepth(in vec3 position);
#if USE_OIT
  void renderToABuffer3D(in float depth, in vec4 color, in vec3 normal);
#endif
#if GPU_SHADER4_COMPATIBILITY_CONDITION
  uint packFloat4ToUnsignedInteger(in vec4 values);
  uint packNormalToUnsignedInteger(in vec3 normal, bool useFrontFacing);
#endif

void main()
{
  float distanceMetric = 0.0;
  vec3 cameraNormal    = vec3(0.0);
  vec3 cameraPosition  = vec3(0.0);
  perspectiveCylinderImpostor(cameraNormal, cameraPosition);

  vec4 color = vec4(0.0);
#if APPLY_SURFACE_LIGHTING_MODELS
  if (lightingModel > 3)
  {
    applyNonPhotoRealisticLightingModel(lightingModel, cameraPosition, cameraNormal, cylinderColor, true, color);
    if (texturing)
    {
      if (environmentMapping)
      {
        applyJitteredEnvironmentMapping(cylinderSurfacePoint, cameraNormal, color);
      }
      else
      {
        vec2 textureCoords = (sphericalMapping) ? applySphericalCoordinates(cylinderSurfacePoint, cameraNormal) : applyImpostorSphereTextureCoords(cameraNormal);
        applyTextureWithEmbossFilter(textureCoords, color);
      }
    }
  }
  else // between 1-3, 0 will do 'simple' lighting (without specular term)
  {
    applyADSLightingModel(lightingModel, cameraPosition, cameraNormal, cylinderColor, true, color);
    if (texturing)
    {
      if (environmentMapping)
      {
        applyJitteredEnvironmentMapping(cylinderSurfacePoint, cameraNormal, color);
      }
      else
      {
        vec2 textureCoords = (sphericalMapping) ? applySphericalCoordinates(cylinderSurfacePoint, cameraNormal) : applyImpostorSphereTextureCoords(cameraNormal);
        applyTexture(textureCoords, color);
      }
    }
  }
#else
  else
  {
    applySimplePhongLightingModel(cameraPosition, cameraNormal, cylinderColor, true, color);
    if (texturing)
    {
      if (environmentMapping)
      {
        applyJitteredEnvironmentMapping(cylinderSurfacePoint, cameraNormal, color);
      }
      else
      {
        vec2 textureCoords = (sphericalMapping) ? applySphericalCoordinates(cylinderSurfacePoint, cameraNormal) : applyImpostorSphereTextureCoords(cameraNormal);
        applyTexture(textureCoords, color);
      }
    }
  }
#endif

  if (fog)
  {
    applyFog(color);
  }

  if (gammaCorrection)
  {
    applyFastGammaCorrection(color);
  }

  float depth = applyDepth(cameraPosition); // set the depth based on the new cameraPosition
  float sphereAlpha = cylinderColor.a;

  #if USE_OIT
    renderToABuffer3D(depth, vec4(color.rgb, sphereAlpha * opacity), cameraNormal);
  #else
    #if GPU_SHADER4_COMPATIBILITY_CONDITION
      if (useGBuffer)
      {
        uint packedColor       = packFloat4ToUnsignedInteger(vec4(color.rgb, sphereAlpha * opacity));
        uint packedNormal      = packNormalToUnsignedInteger(cameraNormal, false);
        gl_FragColor = vec4(uintBitsToFloat(packedColor), uintBitsToFloat(packedNormal), depth, 1.0);
        gl_FragDepth = depth;
      }
      else
      {
        gl_FragColor = vec4(color.rgb, sphereAlpha * opacity);
        gl_FragDepth = depth;
      }
    #else
      gl_FragColor = vec4(color.rgb, sphereAlpha * opacity);
      gl_FragDepth = depth;
    #endif
  #endif
}

void perspectiveCylinderImpostor(out vec3 cameraNormal, out vec3 cameraPosition)
{
#if GPU_SHADER_FP64_COMPATIBILITY_CONDITION
  dvec3 rayTarget = cylinderSurfacePoint;
  dvec3 rayOrigin = dvec3(0.0);
  dvec3 rayDirection = mix(normalize(rayOrigin - rayTarget), dvec3(0.0, 0.0, 1.0), double(0.0));
  dmat3 basis = dmat3(cylinderU, cylinderV, cylinderAxis);
  dvec3 P = (rayTarget - double(0.5) * (cylinderBase + cylinderEnd)) * basis;

  // angle (cos) between cylinder cylinderAxis and rayDirection
  double angle = dot(cylinderAxis, rayDirection);
  double radiusSquared = cylinderRadius * cylinderRadius;

  // calculate distance to the cylinder from ray origin
  dvec3 D = dvec3(dot(cylinderU, rayDirection), dot(cylinderV, rayDirection), angle);
  double a0 = P.x * P.x + P.y * P.y - radiusSquared;
  double a1 = P.x * D.x + P.y * D.y;
  double a2 = D.x * D.x + D.y * D.y;
  // calculate a discriminant of the above quadratic equation
  double discriminant = a1 * a1 - a0 * a2;
  if (discriminant < double(0.0)) // outside of the cylinder
  {
    discard;
  }

  // point of intersection on cylinder surface
  double intersectT = (-a1 + sqrt(discriminant)) / a2;
  cameraPosition = vec3(rayTarget + intersectT * rayDirection);
  cameraNormal = vec3(normalize(cameraPosition - cylinderBase - cylinderAxis * dot(cameraPosition - cylinderBase, cylinderAxis)));

  // to calculate caps, simply check the angle between
  // the point of intersection - cylinder end vector
  // and a cap plane normal (which is the cylinder cylinderAxis)
  // if the angle < 0, the point is outside of cylinder

  // test front cap
  double capTest = dot((cameraPosition - cylinderBase), cylinderAxis);

  // flat
  if (capTest < double(0.0))
  {
    // ray-plane intersection
    double dNV = dot(-cylinderAxis, rayDirection);
    if (dNV < double(0.0))
    {
      discard;
    }

    double near = dot(-cylinderAxis, (cylinderBase)) / dNV;
    cameraPosition = vec3(rayDirection * near + rayOrigin);
    // within the cap cylinderRadius?
    if (dot(cameraPosition - cylinderBase, cameraPosition-cylinderBase) > radiusSquared)
    {
      discard;
    }

    cameraNormal = -cylinderAxis;
  }

  // test end cap
  capTest = dot((cameraPosition - cylinderEnd), cylinderAxis);

  // flat
  if (capTest > double(0.0))
  {
    // ray-plane intersection
    double dNV = dot(cylinderAxis, rayDirection);
    if (dNV < double(0.0))
    {
      discard;
    }

    double near = dot(cylinderAxis, cylinderEnd) / dNV;
    cameraPosition = vec3(rayDirection * near + rayOrigin);
    // within the cap cylinderRadius?
    if (dot(cameraPosition - cylinderEnd, cameraPosition-cylinderBase) > radiusSquared)
    {
      discard;
    }

    cameraNormal = cylinderAxis;
  }
#else
  vec3 rayTarget = cylinderSurfacePoint;
  vec3 rayOrigin = vec3(0.0);
  vec3 rayDirection = mix(normalize(rayOrigin - rayTarget), vec3(0.0, 0.0, 1.0), 0.0);
  mat3 basis = mat3(cylinderU, cylinderV, cylinderAxis);
  vec3 P = (rayTarget - 0.5 * (cylinderBase + cylinderEnd)) * basis;

  // angle (cos) between cylinder cylinderAxis and rayDirection
  float angle = dot(cylinderAxis, rayDirection);
  float radiusSquared = cylinderRadius * cylinderRadius;

  // calculate distance to the cylinder from ray origin
  vec3 D = vec3(dot(cylinderU, rayDirection), dot(cylinderV, rayDirection), angle);
  float a0 = P.x * P.x + P.y * P.y - radiusSquared;
  float a1 = P.x * D.x + P.y * D.y;
  float a2 = D.x * D.x + D.y * D.y;
  // calculate a discriminant of the above quadratic equation
  float discriminant = a1 * a1 - a0 * a2;
  if (discriminant < 0.0) // outside of the cylinder
  {
    discard;
  }

  // point of intersection on cylinder surface
  float intersectT = (-a1 + sqrt(discriminant)) / a2;
  cameraPosition = rayTarget + intersectT * rayDirection;
  cameraNormal = normalize(cameraPosition - cylinderBase - cylinderAxis * dot(cameraPosition - cylinderBase, cylinderAxis));

  // to calculate caps, simply check the angle between
  // the point of intersection - cylinder end vector
  // and a cap plane normal (which is the cylinder cylinderAxis)
  // if the angle < 0, the point is outside of cylinder

  // test front cap
  float capTest = dot((cameraPosition - cylinderBase), cylinderAxis);

  // flat
  if (capTest < 0.0)
  {
    // ray-plane intersection
    float dNV = dot(-cylinderAxis, rayDirection);
    if (dNV < 0.0)
    {
      discard;
    }

    float near = dot(-cylinderAxis, (cylinderBase)) / dNV;
    cameraPosition = rayDirection * near + rayOrigin;
    // within the cap cylinderRadius?
    if (dot(cameraPosition - cylinderBase, cameraPosition-cylinderBase) > radiusSquared)
    {
      discard;
    }

    cameraNormal = -cylinderAxis;
  }

  // test end cap
  capTest = dot((cameraPosition - cylinderEnd), cylinderAxis);

  // flat
  if (capTest > 0.0)
  {
    // ray-plane intersection
    float dNV = dot(cylinderAxis, rayDirection);
    if (dNV < 0.0)
    {
      discard;
    }

    float near = dot(cylinderAxis, cylinderEnd) / dNV;
    cameraPosition = rayDirection * near + rayOrigin;
    // within the cap cylinderRadius?
    if (dot(cameraPosition - cylinderEnd, cameraPosition-cylinderBase) > radiusSquared)
    {
      discard;
    }

    cameraNormal = cylinderAxis;
  }
#endif
}