/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  Apply Render A-Buffer-3D code.
*
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/

#extension GL_ARB_gpu_shader5 : enable
#if A_BUFFER_3D_USE_TEXTURES
  #extension GL_EXT_shader_image_load_store : enable
#else
  #extension GL_ARB_shader_storage_buffer_object : enable
#endif

struct ABuffer3D
{
  uint offset;
  uint color;
  uint normal;
  float depth;
#if USE_DEFERRED_SHADING
  uint[4] materialColors;
  float materialShininess;
  uint texCoords0;
#endif
};

// layout(pixel_center_integer) in vec4 gl_FragCoord; // whole number pixel offsets
layout(early_fragment_tests) in;
layout(binding = ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_ATOMIC_COUNTER) uniform atomic_uint aBuffer3DLinkedListAtomicCounterBuffer;
#if A_BUFFER_3D_USE_TEXTURES
  layout(r32ui,    binding = ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET) coherent volatile  uniform uimage2D     aBuffer3DLinkedListOffsetImage;
  layout(rgba32ui, binding = ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST)        coherent writeonly uniform uimageBuffer aBuffer3DLinkedListImage;
#else
  layout(std430, binding = ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET) coherent volatile buffer Offset { uint  aBuffer3DLinkedListOffsetBuffer[]; };
  layout(std430, binding = ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST)        coherent          buffer Data   { ABuffer3D aBuffer3DLinkedListBuffer[]; };
#endif

uniform int screenWidth = 512;
uniform int screenHeight = 512;
uniform int aBuffer3DSize = 32;

void renderToABuffer3D(in vec4 color, in vec3 normal);
void renderToABuffer3D(in float depth, in vec4 color, in vec3 normal);
#if GPU_SHADER4_COMPATIBILITY_CONDITION
  uint packNormalToUnsignedInteger(in vec3 normal);
  #if USE_DEFERRED_SHADING
    uint[4] getCurrentFragmentMaterialColors();
    float getCurrentFragmentMaterialShininess();
    void renderToABuffer3DDeferredShading(in vec4 color, in vec3 normal);
    void renderToABuffer3DDeferredShading(in float depth, in vec4 color, in vec3 normal);
  #endif
#endif

void renderToABuffer3D(in vec4 color, in vec3 normal)
{
  renderToABuffer3D(gl_FragCoord.z, color, normal);
}

void renderToABuffer3D(in float depth, in vec4 color, in vec3 normal)
{
  ivec2 coords = ivec2(gl_FragCoord.xy);
  if ((coords.x >= 0) && (coords.y >= 0) && (coords.x < screenWidth) && (coords.y < screenHeight))
  {
    // create fragment to be stored
    uint currentPackedColor  = packUnorm4x8(color);
    uint currentPackedNormal = packNormalToUnsignedInteger(normal);

    // retrieve next available offset in linked list buffer
    uint nextOffset = atomicCounterIncrement(aBuffer3DLinkedListAtomicCounterBuffer);

    // exchange previous offset with new one & write current fragment's data to next location in linked list image/buffer
  #if A_BUFFER_3D_USE_TEXTURES
    uint previousOffset = imageAtomicExchange(aBuffer3DLinkedListOffsetImage, coords, nextOffset);
    imageStore(aBuffer3DLinkedListImage, int(nextOffset), uvec4(previousOffset, currentPackedColor, currentPackedNormal, floatBitsToUint(depth)));
  #else
    uint previousOffset = atomicExchange(aBuffer3DLinkedListOffsetBuffer[coords.x + screenWidth * coords.y], nextOffset);
    aBuffer3DLinkedListBuffer[nextOffset].offset = previousOffset;
    aBuffer3DLinkedListBuffer[nextOffset].color  = currentPackedColor;
    aBuffer3DLinkedListBuffer[nextOffset].normal = currentPackedNormal;
    aBuffer3DLinkedListBuffer[nextOffset].depth  = depth;
  #endif
  }

  // discard fragment so nothing is written to the framebuffer
  discard;
}

#if USE_DEFERRED_SHADING
  void renderToABuffer3DDeferredShading(in vec4 color, in vec3 normal)
  {
    renderToABuffer3DDeferredShading(gl_FragCoord.z, color, normal);
  }

  void renderToABuffer3DDeferredShading(in float depth, in vec4 color, in vec3 normal)
  {
    ivec2 coords = ivec2(gl_FragCoord.xy);
    if ((coords.x >= 0) && (coords.y >= 0) && (coords.x < screenWidth) && (coords.y < screenHeight))
    {
      // create fragment to be stored
      uint currentPackedColor = packUnorm4x8(color);

      // retrieve next available offset in linked list buffer
      uint nextOffset = atomicCounterIncrement(aBuffer3DLinkedListAtomicCounterBuffer);

      // exchange previous offset with new one & write current fragment's data to next location in linked list image/buffer
      uint previousOffset = atomicExchange(aBuffer3DLinkedListOffsetBuffer[coords.x + screenWidth * coords.y], nextOffset);
      aBuffer3DLinkedListBuffer[nextOffset].offset            = previousOffset;
      aBuffer3DLinkedListBuffer[nextOffset].color             = currentPackedColor;
      aBuffer3DLinkedListBuffer[nextOffset].normal            = packNormalToUnsignedInteger(normal);
      aBuffer3DLinkedListBuffer[nextOffset].depth             = depth;
      aBuffer3DLinkedListBuffer[nextOffset].materialColors    = getCurrentFragmentMaterialColors();
      aBuffer3DLinkedListBuffer[nextOffset].materialShininess = getCurrentFragmentMaterialShininess();
      aBuffer3DLinkedListBuffer[nextOffset].texCoords0        = packHalf2x16(gl_TexCoord[0].st);
    }

    // discard fragment so nothing is written to the framebuffer
    discard;
  }
#endif