/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  Apply Display A-Buffer-3D code.
*
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/

#extension GL_ARB_gpu_shader5 : enable
#if A_BUFFER_3D_USE_TEXTURES
  #extension GL_EXT_shader_image_load_store : enable
#else
  #extension GL_ARB_shader_storage_buffer_object : enable
#endif

struct ABuffer3D
{
  uint offset;
  uint color;
  uint normal;
  float depth;
#if USE_DEFERRED_SHADING
  uint[4] materialColors;
  float materialShininess;
  uint texCoords0;
#endif
};

// layout(pixel_center_integer) in vec4 gl_FragCoord; // whole number pixel offsets
layout(early_fragment_tests) in;
layout(binding = ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_ATOMIC_COUNTER) uniform atomic_uint aBuffer3DLinkedListAtomicCounterBuffer;
#if A_BUFFER_3D_USE_TEXTURES
  layout(r32ui,    binding = ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET) coherent readonly volatile uniform uimage2D     aBuffer3DLinkedListOffsetImage;
  layout(rgba32ui, binding = ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST)        coherent readonly          uniform uimageBuffer aBuffer3DLinkedListImage;
#else
  layout(std430, binding = ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST_OFFSET) coherent volatile buffer Offset { uint  aBuffer3DLinkedListOffsetBuffer[]; };
  layout(std430, binding = ACTIVE_TEXTURE_UNIT_FOR_A_BUFFER_3D_LINKED_LIST)        coherent          buffer Data   { ABuffer3D aBuffer3DLinkedListBuffer[]; };
#endif

uniform int screenWidth = 512;
uniform int screenHeight = 512;
uniform bool sortFragments = true;
uniform bool resolveTranslucency = true;
uniform float zNear = 0.1;
uniform float zFar = 1000.0;
uniform float translucencyAbsorptionCoefficient = 10.0;
uniform vec4 backgroundColor = vec4(1.0); // white background color by default
uniform bool useGBuffer = false;
#if USE_DEFERRED_SHADING
  uniform float tanFieldOfView = 0.414214; // the tanFieldOfView default value of an angle of 45.0 degrees
  uniform int lightingModel = 2; // Blinn-Phong by default
  uniform bool texturing = false;
  uniform bool environmentMapping = false;
  uniform bool fog = false;
  uniform bool gammaCorrection = false;
#endif

const float MAX_FLOAT = 3.402823466e+38;

ABuffer3D minFragment;
ABuffer3D localFragmentArray[A_BUFFER_3D_MAX_SIZE]; // local memory array (probably in L1)
void processFragmentsOutput(in ivec2 coords);
vec4 resolveClosestFragment(in ivec2 coords);
int fillLocalFragmentArray(in ivec2 coords);
void insertionSortLocalFragmentArray(in int numberOfFragments);
vec4 resolveAlphaBlendedFragment(in ivec2 coords);
float applyLinearDepth(in float depthValue, in float zNear, in float zFar);
#if GPU_SHADER4_COMPATIBILITY_CONDITION
  vec3 unpackUnsignedIntegerToNormal(in uint signedPackedNormal);
  uint packFloat4ToUnsignedInteger(in vec4 values);
  #if USE_DEFERRED_SHADING
    vec3 reconstructEyePositionFromDepth(in float depthValue, in float zNear, in float zFar, in float tanFieldOfView, in int screenWidth, in int screenHeight);
    vec4 applyDeferredShading(in ABuffer3D fragment, in vec4 fragmentColor);
    void applyADSLightingModelDeferredShading(in int useSpecularDiffuseModel, in vec3 position, in vec3 normal, in uint[4] materialColors, in float materialShininess, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
    void applyNonPhotoRealisticLightingModelDeferredShading(in int useNonPhotoRealisticModel, in vec3 position, in vec3 normal, in uint[4] materialColors, in float materialShininess, in vec4 materialColor, in bool usePositionVector, inout vec4 color);
    void applyTexture(in vec2 texCoords, inout vec4 color);
    void applyTextureWithEmbossFilter(in vec2 texCoords, inout vec4 color);
    void applyJitteredEnvironmentMapping(in vec3 position, in vec3 normal, inout vec4 color);
    void applyFastGammaCorrection(inout vec4 color);
    void applyFog(in float depth, inout vec4 color);
  #endif
#endif

void main()
{
  ivec2 coords = ivec2(gl_FragCoord.xy);
  if ((coords.x >= 0) && (coords.y >= 0) && (coords.x < screenWidth) && (coords.y < screenHeight))
  {
    processFragmentsOutput(coords);
  }
  else
  {
    discard; // if no fragment, write nothing
  }
}

void processFragmentsOutput(in ivec2 coords)
{
  if (useGBuffer)
  {
    gl_FragColor = (sortFragments) ? vec4(uintBitsToFloat(packFloat4ToUnsignedInteger(resolveAlphaBlendedFragment(coords))), uintBitsToFloat(localFragmentArray[0].normal), localFragmentArray[0].depth, 1.0)
                                   : vec4(uintBitsToFloat(packFloat4ToUnsignedInteger(resolveClosestFragment(coords))),      uintBitsToFloat(minFragment.normal),           minFragment.depth,           1.0);
  }
  else
  {
    gl_FragColor = (sortFragments) ? resolveAlphaBlendedFragment(coords) : resolveClosestFragment(coords);
  }
}

vec4 resolveClosestFragment(in ivec2 coords)
{
  ABuffer3D currentFragment;
  minFragment.color  = 0U;
  minFragment.normal = 0U;
  minFragment.depth  = MAX_FLOAT;
#if USE_DEFERRED_SHADING
  minFragment.materialColors[0] = 0U;
  minFragment.materialColors[1] = 0U;
  minFragment.materialColors[2] = 0U;
  minFragment.materialColors[3] = 0U;
  minFragment.materialShininess = 0.0;
  minFragment.texCoords0 = 0U;
#endif
#if A_BUFFER_3D_USE_TEXTURES
  uint offset = imageLoad(aBuffer3DLinkedListOffsetImage, coords).x;
#else
  uint offset = aBuffer3DLinkedListOffsetBuffer[coords.x + screenWidth * coords.y];
#endif
  int numberOfFragments = 0;
  while ((offset != -1U) && (numberOfFragments < A_BUFFER_3D_MAX_SIZE))
  {
  #if A_BUFFER_3D_USE_TEXTURES
    uvec4 data = imageLoad(aBuffer3DLinkedListImage, int(offset));
    currentFragment.offset = data.x;
    currentFragment.color  = data.y;
    currentFragment.normal = data.z;
    currentFragment.depth  = uintBitsToFloat(data.w);
  #else
    currentFragment        = aBuffer3DLinkedListBuffer[offset];
  #endif

    offset = currentFragment.offset;
    numberOfFragments++;

    if (currentFragment.depth < minFragment.depth)
    {
      minFragment = currentFragment;
    }
  }

  if (numberOfFragments <= 0) // if no fragment, write nothing
  {
    discard;
  }

  vec4 color = unpackUnorm4x8(minFragment.color);
#if USE_DEFERRED_SHADING
  vec4 deferredLightingColor = applyDeferredShading(minFragment, color);
  color = vec4(deferredLightingColor.rgb, color.a);
#endif
  return clamp(vec4(color.rgb, 1.0), 0.0, 1.0);
}

int fillLocalFragmentArray(in ivec2 coords)
{
#if A_BUFFER_3D_USE_TEXTURES
  uint offset = imageLoad(aBuffer3DLinkedListOffsetImage, coords).x;
#else
  uint offset = aBuffer3DLinkedListOffsetBuffer[coords.x + screenWidth * coords.y];
#endif
  int numberOfFragments = 0;
  while ((offset != -1U) && (numberOfFragments < A_BUFFER_3D_MAX_SIZE))
  {
  #if A_BUFFER_3D_USE_TEXTURES
    uvec4 data = imageLoad(aBuffer3DLinkedListImage, int(offset));
    localFragmentArray[numberOfFragments].offset = data.x;
    localFragmentArray[numberOfFragments].color  = data.y;
    localFragmentArray[numberOfFragments].normal = data.z;
    localFragmentArray[numberOfFragments].depth  = uintBitsToFloat(data.w);
  #else
    localFragmentArray[numberOfFragments]        = aBuffer3DLinkedListBuffer[offset];
  #endif

    offset = localFragmentArray[numberOfFragments].offset;
    numberOfFragments++;
  }

  return numberOfFragments;
}

void insertionSortLocalFragmentArray(in int numberOfFragments)
{
  int k = 0;
  ABuffer3D temp;
  for (int z = 1; z < numberOfFragments; ++z)
  {
    temp = localFragmentArray[z];
    for (k = z - 1; k >= 0 && localFragmentArray[k].depth > temp.depth; --k)
    {
      localFragmentArray[k + 1] = localFragmentArray[k];
    }
    localFragmentArray[k + 1] = temp;
  }
}

vec4 resolveAlphaBlendedFragment(in ivec2 coords)
{
  // copy fragments in local memory array
  int numberOfFragments = fillLocalFragmentArray(coords);
  if (numberOfFragments <= 0) // if no fragment, write nothing
  {
    discard;
  }

  // sort fragments in local memory array
  insertionSortLocalFragmentArray(numberOfFragments);

  ABuffer3D currentFragment;
  vec4 sumColor = vec4(0.0);
  float thickness = (resolveTranslucency) ? applyLinearDepth(localFragmentArray[0].depth, zNear, zFar) / 2.0 : 0.0;
  vec4 color = vec4(0.0);
  for (int z = 0; z < numberOfFragments; ++z)
  {
    currentFragment = localFragmentArray[z];
    color = unpackUnorm4x8(currentFragment.color);

  #if USE_DEFERRED_SHADING
    vec4 deferredLightingColor = applyDeferredShading(currentFragment, color);
    color = vec4(deferredLightingColor.rgb, color.a);
  #endif

    if (resolveTranslucency)
    {
      if ((z & 1) == (numberOfFragments & 1))
      {
        thickness = (applyLinearDepth(localFragmentArray[z + 1].depth, zNear, zFar) - applyLinearDepth(currentFragment.depth, zNear, zFar)) / 2.0;
      }
      color.a = 1.0 - pow(1.0 - color.a, translucencyAbsorptionCoefficient * thickness);
    }

    color.rgb *= color.a;
    sumColor += color * (1.0 - sumColor.a);
    if (sumColor.a >= 1.0) break; // terminate if opacity > 1.0
  }

  sumColor += backgroundColor * (1.0 - sumColor.a); // blend with background color

  return clamp(sumColor, 0.0, 1.0);
}

#if USE_DEFERRED_SHADING
  vec4 applyDeferredShading(in ABuffer3D fragment, in vec4 fragmentColor)
  {
    vec3 position              = reconstructEyePositionFromDepth(fragment.depth, zNear, zFar, tanFieldOfView, screenWidth, screenHeight);
    vec3 normal                = unpackUnsignedIntegerToNormal(fragment.normal);
    vec2 texCoords0            = unpackHalf2x16(fragment.texCoords0);
    vec4 deferredLightingColor = vec4(0.0);

    if (lightingModel > 3)
    {
      applyNonPhotoRealisticLightingModelDeferredShading(lightingModel, position, normal, fragment.materialColors, fragment.materialShininess, fragmentColor, true, deferredLightingColor);
    }
    else
    {
      applyADSLightingModelDeferredShading(lightingModel, position, normal, fragment.materialColors, fragment.materialShininess, fragmentColor, true, deferredLightingColor);
    }

    if (texturing)
    {
      if (environmentMapping)
      {
        applyJitteredEnvironmentMapping(position, normal, fragmentColor);
      }
      else
      {
        applyTextureWithEmbossFilter(texCoords0, deferredLightingColor);
      }
    }

    if (fog)
    {
      // Below calculation substituting the original gl_FragCoord.z / gl_FragCoord.w passed in applyFog(inout vec4 color) in forward rendering.
      // See comments in applyFog(inout vec4 color) function implementation in CommunFunctions for more details.
      // Reminder: gl_FragCoord.w = 1.0 / clipPositionW, thus gl_FragCoord.z / gl_FragCoord.w = fragment.depth * clipPositionW.
      vec4 position4D = vec4(position, 1.0);
      float clipPositionW = dot(gl_ProjectionMatrixTranspose[3], position4D);
      applyFog(fragment.depth * clipPositionW, deferredLightingColor);
    }

    if (gammaCorrection)
    {
      applyFastGammaCorrection(deferredLightingColor);
    }

    return deferredLightingColor;
  }
#endif