/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  Illuminated Line Model GLSL code.
*  Using core functions from the CommonFunctions library.
*  References:
*  1. ieeexplore.ieee.org/xpl/login.jsp?tp=&arnumber=567777&url=http%3A%2F%2Fieeexplore.ieee.org%2Fxpls%2Fabs_all.jsp%3Farnumber%3D567777 -> Standard Illuminated Lines
*  2. www.scivis.ethz.ch/publications/pdf/2005/mallo2005illuminated.pdf                                                                   -> Advanced Illuminated Lines
*
*  Relevant variables controlling the above: tangent, useAdvancedDiffuseModel
*
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/

uniform int numberOfLights = 1;
uniform bool useAdvancedDiffuseModel = false;
uniform bool fog = false;
uniform bool gammaCorrection = false;
uniform float opacity = 1.0;

FS_VARYING vec3 tangent;
FS_VARYING vec3 eyeVec;

// Machine float epsilon
const float FLT_EPSILON = 0.000001;

// Used for diffuse & specular calculations
const float PI = 3.14159;
const float ADVANCED_DIFFUSE_FACTOR = 1.75;
const float PHONG_SPECULAR_FACTOR   = 3.5;

float findCylinderAngle(in vec3 E, in vec3 T, in vec3 L, in float lightProjTangent, in float oneMinusSquaredlightProjTangent);
float applyAdvancedIlluminatedLinesDiffuseFactor(float angle);
void applyIlluminatedLinesLighting(in vec3 E, in vec3 T, inout vec4 color, in int index);
void applyFastGammaCorrection(inout vec4 color);
void applyFog(inout vec4 color);

void main()
{
  vec4 color = vec4(0.0);

  vec3 E = normalize(-eyeVec);
  vec3 T = normalize(tangent);

  for (int index = 0; index < numberOfLights; ++index)
  {
    applyIlluminatedLinesLighting(E, T, color, index);
  }

  if (fog)
  {
    applyFog(color);
  }

  if (gammaCorrection)
  {
    applyFastGammaCorrection(color);
  }

  gl_FragColor = clamp(vec4(color.rgb, opacity), 0.0, 1.0);
}

float findCylinderAngle(in vec3 E, in vec3 T, in vec3 L, in float lightProjTangent, in float oneMinusSquaredlightProjTangent)
{
  float eyeProjTangent = abs(dot(E, T));
  float angle = (abs(dot(E, L)) -eyeProjTangent * lightProjTangent) / (sqrt(1.0 - eyeProjTangent * eyeProjTangent) * oneMinusSquaredlightProjTangent);
  return acos(clamp(angle, -1.0 + FLT_EPSILON, 1.0 - FLT_EPSILON)); // failsafe GLSL Shader value for finding cylinder angle computation
}

float applyAdvancedIlluminatedLinesDiffuseFactor(float angle)
{
  return ADVANCED_DIFFUSE_FACTOR * (sin(angle) + (PI - angle) * cos(angle)) / 4.0;
}

void applyIlluminatedLinesLighting(in vec3 E, in vec3 T, inout vec4 color, in int index)
{
  vec3 L = -normalize(gl_LightSource[index].position.xyz);
  vec3 halfVector = normalize(L + E);

  gl_MaterialParameters glCurrentMaterial = (gl_FrontFacing) ? gl_FrontMaterial : gl_BackMaterial;

  // Don't use gl_FrontLightModelProduct.sceneColor as it won't work on Intel/AMD GPUs when the gl_Color changes:
  // gl_FrontLightModelProduct is a uniform which does not track vertex changes
  // color += gl_FrontLightModelProduct.sceneColor + gl_LightSource[index].ambient * gl_Color;
  color += glCurrentMaterial.emission  + gl_LightSource[index].ambient * gl_Color; // ambient light computation

  float lightProjTangent = abs(dot(L, T));
  float oneMinusSquaredlightProjTangent = sqrt(1.0 - lightProjTangent * lightProjTangent);
  if (useAdvancedDiffuseModel)
  {
    oneMinusSquaredlightProjTangent *= applyAdvancedIlluminatedLinesDiffuseFactor(findCylinderAngle(E, T, L, lightProjTangent, oneMinusSquaredlightProjTangent));
  }
  vec4 diffuseIllumination = gl_Color * gl_LightSource[index].diffuse * oneMinusSquaredlightProjTangent; // diffuse light computation

  float halfVecProjTangent = abs(dot(halfVector, T));
  float specIllumination   = pow(sqrt(1.0 - halfVecProjTangent * halfVecProjTangent), glCurrentMaterial.shininess / PHONG_SPECULAR_FACTOR);
  vec4 specIlluminationColor = gl_LightSource[index].specular * glCurrentMaterial.specular * specIllumination; // specular light computation

  color += (diffuseIllumination + specIlluminationColor) * clamp(dot(E, L), 0.0, 1.0);
}