/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  Apply GBuffer Effects BlurXY GLSL code.
*
* @author Thanos Theo, Michael Kargas (for BlurXY effect), 2009-2018
* @version 14.0.0.0
*/

uniform sampler2D gbufferSampler;
uniform sampler2D sceneSampler;
uniform vec2 gbufferResolution = vec2(640, 480);
uniform int numberOfSamplesRange = 8;
uniform bool renderPlainScene  = true;
uniform bool renderBlurXYScene = false;
uniform bool renderWeightedBlurXYScene = false;
uniform bool blurX = true;
uniform vec3 sxValues = vec3(0.0);
uniform vec3 syValues = vec3(0.0);

const float PI        = 3.14159265359;
const float SQRT_2_PI = sqrt(2.0 * PI);
const float DEVIATION = 25.0;
const float DEVIATION_SQUARED_2X = 2.0 * DEVIATION * DEVIATION;

const float S  = 128.0;
const float E  = 131070.0;
const float FS = 1.0 / S;
const float FE = 1.0 / E;
const float FD = FS - FE;

vec4 unpackUnsignedIntegerToFloat4(in uint value);
vec4 applyRenderPlainScene();
vec4 applyRenderBlurXYScene();
vec4 applyRenderWeightedBlurXYScene();

void main()
{
  if (renderPlainScene)
  {
    gl_FragColor = applyRenderPlainScene();
  }
  else if (renderBlurXYScene)
  {
    gl_FragColor = applyRenderBlurXYScene();
  }
  else // if (renderWeightedBlurXYScene)
  {
    gl_FragColor = applyRenderWeightedBlurXYScene();
  }
}

vec4 applyRenderPlainScene()
{
  vec4 packedData    = texture2D(gbufferSampler, gl_TexCoord[0].st);
  vec4 unpackedColor = unpackUnsignedIntegerToFloat4(floatBitsToUint(packedData.x));

  return unpackedColor;
}

vec4 applyRenderBlurXYScene()
{
  vec2 inverseResolution = vec2(1.0 / gbufferResolution);
  vec4 sumColor = vec4(0.0);
  float totalWeigth = 0.0;
  if (blurX)
  {
    for (int x = -numberOfSamplesRange; x <= numberOfSamplesRange; ++x)
    {
      vec2 tc = gl_TexCoord[0].st + vec2(inverseResolution.x * x, 0.0);
      float weight = (1.0 / SQRT_2_PI * DEVIATION) * exp((-x * x) / DEVIATION_SQUARED_2X); // generic gaussian deviation
      sumColor += texture2D(sceneSampler, tc) * weight;
      totalWeigth += weight;
    }
  }
  else
  {
    for (int y = -numberOfSamplesRange; y <= numberOfSamplesRange; ++y)
    {
      vec2 tc = gl_TexCoord[0].st + vec2(0.0, inverseResolution.y * y);
      float weight = (1.0 / SQRT_2_PI * DEVIATION) * exp((-y * y) / DEVIATION_SQUARED_2X); // generic gaussian deviation
      sumColor += texture2D(sceneSampler, tc) * weight;
      totalWeigth += weight;
    }
  }

  return sumColor / totalWeigth;
}

vec4 applyRenderWeightedBlurXYScene()
{
  vec4 packedData = texture2D(gbufferSampler, gl_TexCoord[0].st);
  float factor = FS - FD * packedData.z;
  float SSAOPlus = 4.0 * texture2D(sceneSampler, gl_TexCoord[0].st).r;
  vec2 texCoord = gl_TexCoord[0].st;
  vec4 packedDataOffset = vec4(0.0);
  float weight = 0.0;
  float sum = 4.0;
  vec3 texCoordSteps = (blurX) ? sxValues : syValues;

  texCoord.s -= texCoordSteps.z;

  packedDataOffset = texture2D(gbufferSampler, texCoord);
  weight    = 1.0 - step(factor, abs(packedData.z - packedDataOffset.z));
  SSAOPlus += texture2D(sceneSampler, texCoord).r * weight;
  sum += weight;

  texCoord.s += texCoordSteps.x;

  packedDataOffset = texture2D(gbufferSampler, texCoord);
  weight    = 2.0 * (1.0 - step(factor, abs(packedData.z - packedDataOffset.z)));
  SSAOPlus += texture2D(sceneSampler, texCoord).r * weight;
  sum += weight;

  texCoord.s += texCoordSteps.x;

  packedDataOffset = texture2D(gbufferSampler, texCoord);
  weight    = 3.0 * (1.0 - step(factor, abs(packedData.z - packedDataOffset.z)));
  SSAOPlus += texture2D(sceneSampler, texCoord).r * weight;
  sum += weight;

  texCoord.s += texCoordSteps.y;

  packedDataOffset = texture2D(gbufferSampler, texCoord);
  weight    = 3.0 * (1.0 - step(factor, abs(packedData.z - packedDataOffset.z)));
  SSAOPlus += texture2D(sceneSampler, texCoord).r * weight;
  sum += weight;

  texCoord.s += texCoordSteps.x;

  packedDataOffset = texture2D(gbufferSampler, texCoord);
  weight    = 2.0 * (1.0 - step(factor, abs(packedData.z - packedDataOffset.z)));
  SSAOPlus += texture2D(sceneSampler, texCoord).r * weight;
  sum += weight;

  texCoord.s += texCoordSteps.x;

  packedDataOffset = texture2D(gbufferSampler, texCoord);
  weight    = 1.0 - step(factor, abs(packedData.z - packedDataOffset.z));
  SSAOPlus += texture2D(sceneSampler, texCoord).r * weight;
  sum += weight;

  return vec4(vec3(SSAOPlus / sum), 1.0);
}