/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  Apply GBuffer Effects GLSL code.
*
* @author Thanos Theo, Michael Kargas (for Depth-Of-Field effect), 2009-2018
* @version 14.0.0.0
*/

uniform sampler2D gbufferSampler;
uniform sampler2D sceneBlurredXYSampler;
uniform int gbufferEffectType = 1;
uniform vec2 gbufferResolution = vec2(640, 480);

// for Depth-Of-Field
uniform float zNear = 0.1;
uniform float zFar = 1000.0;
uniform float range = 10.0;
uniform float zFocus = 0.5;

// for Edge Enhancement
uniform vec2 edgeEnhancementSize = vec2(12.0 / 3.0);

// for HSSAO & HSSAO Plus
uniform sampler2D rotationTextureSampler;
uniform vec2 depthSamples[16];

// Machine float epsilon
const float FLT_EPSILON = 0.000001;

const float NDOTP2P_LOWER_LIMIT = 0.342;

vec4 unpackUnsignedIntegerToFloat4(in uint value);
vec3 unpackUnsignedIntegerToNormal(in uint signedPackedNormal, in bool useFrontFacingForDeferredLighting);
float applyLinearDepth(in float depthValue, in float zNear, in float zFar);
float smootherstep(in float edge0, in float edge1, in float x);
vec4 applyDepthOfField();
vec4 applyEdgeEnhancement();
vec4 applyHSSAO();

void main()
{
  if (gbufferEffectType == 1)
  {
    gl_FragColor = applyDepthOfField();
  }
  else if (gbufferEffectType == 2)
  {
    gl_FragColor = applyEdgeEnhancement();
  }
  else // if ((gbufferEffectType == 3) || (gbufferEffectType == 4))
  {
    gl_FragColor = applyHSSAO();
  }
}

vec4 applyDepthOfField()
{
  vec4 packedData  = texture2D(gbufferSampler, gl_TexCoord[0].st);
  float depthLinearValue = applyLinearDepth(packedData.z, zNear, zFar);
  float blurFactor = clamp(abs(depthLinearValue - zFocus) / range, 0.0, 1.0);

  #if DEPTH_OF_FIELD_DEBUG_MODE
    if (blurFactor <= 0.01)
    {
      return vec4(0.0, 0.0, 1.0, 1.0);
    }
    else if (blurFactor >= 0.495 && blurFactor <= 0.505)
    {
      return vec4(1.0, 0.0, 0.0, 1.0);
    }
    else if (blurFactor >= 0.99)
    {
      return vec4(0.0, 1.0, 0.0, 1.0);
    }
    else
    {
      return vec4(blurFactor);
    }
  #else
    vec4 unpackedColor = unpackUnsignedIntegerToFloat4(floatBitsToUint(packedData.x));
    vec4 colorBlurred  = texture2D(sceneBlurredXYSampler, gl_TexCoord[0].st);

    return vec4(mix(unpackedColor.xyz, colorBlurred.xyz, blurFactor), 1.0);
  #endif
}

vec4 applyEdgeEnhancement()
{
  // texture center and offset
  vec2 centerOffset = gl_TexCoord[0].st; // same as vec2(gl_FragCoord.xy / gbufferResolution);
  vec2 textureOffset = vec2((1.0 / gbufferResolution) * edgeEnhancementSize);
  
  // access in direction A
  vec4 textureCoords = vec4(centerOffset.x - textureOffset.x, centerOffset.y + textureOffset.y, 1.0, 1.0);
  vec4 packedDataA   = texture2D(gbufferSampler, textureCoords.xy);
  vec4 A = vec4(normalize(unpackUnsignedIntegerToNormal(floatBitsToUint(packedDataA.y), false)), -applyLinearDepth(packedDataA.z, zNear, zFar));
  
  // access in direction C
  textureCoords    = vec4(centerOffset.x + textureOffset.x, centerOffset.y + textureOffset.y, 1.0, 1.0);
  vec4 packedDataC = texture2D(gbufferSampler, textureCoords.xy);
  vec4 C = vec4(normalize(unpackUnsignedIntegerToNormal(floatBitsToUint(packedDataC.y), false)), -applyLinearDepth(packedDataC.z, zNear, zFar));
  
  // access in direction F
  textureCoords    =  vec4(centerOffset.x - textureOffset.x, centerOffset.y - textureOffset.y, 1.0, 1.0);
  vec4 packedDataF = texture2D(gbufferSampler, textureCoords.xy);
  vec4 F = vec4(normalize(unpackUnsignedIntegerToNormal(floatBitsToUint(packedDataF.y), false)), -applyLinearDepth(packedDataF.z, zNear, zFar));
  
  // access in direction H
  textureCoords    =  vec4(centerOffset.x + textureOffset.x, centerOffset.y - textureOffset.y, 1.0, 1.0);
  vec4 packedDataH = texture2D(gbufferSampler, textureCoords.xy);
  vec4 H = vec4(normalize(unpackUnsignedIntegerToNormal(floatBitsToUint(packedDataH.y), false)), -applyLinearDepth(packedDataH.z, zNear, zFar));

  // calculate discontinuities
  vec3 discontinuity = vec3(0.5 * (dot(A.xyz, H.xyz) + dot(C.xyz, F.xyz)),
                            (1.0 - 0.5 * abs(A.w - H.w)) * (1.0 - 0.5 * abs(C.w - F.w)),
                            0.0);
  discontinuity.xy = clamp(discontinuity.xy, 0.0, 1.0);
  discontinuity.z  = discontinuity.x * discontinuity.y;
  
  // output edge intensities to the edge map and combine with current color
  vec4 edgeEnhancementData = vec4(discontinuity, 1.0);
  vec4 packedData          = texture2D(gbufferSampler, gl_TexCoord[0].st);
  vec4 unpackedColor       = unpackUnsignedIntegerToFloat4(floatBitsToUint(packedData.x));

  return unpackedColor * vec4(vec3(smootherstep(0.0, 1.0, edgeEnhancementData.z)), 1.0);
}

vec4 applyHSSAO()
{
  vec4 packedData  = texture2D(gbufferSampler, gl_TexCoord[0].st);
  if (packedData.z < 1.0)
  {
    vec3 normal = normalize(unpackUnsignedIntegerToNormal(floatBitsToUint(packedData.y), false));
    vec4 position = gl_ProjectionMatrixInverse * (2.0 * vec4(gl_TexCoord[0].st, packedData.z, 1.0) - 1.0);
    position.xyz /= position.w;

    if (dot(normal, position.xyz) > 0.0) normal = -normal;

    float positionDistance = length(position.xyz);
    vec2 sr = normalize(texture2D(rotationTextureSampler, 2.0 * gl_TexCoord[1].st).rg - 1.0) / positionDistance;
    mat2x2 scaleRotationMatrix = mat2x2(sr.x, sr.y, -sr.y, sr.x);
    float SSAO = 0.0;
    vec2 texCoord = vec2(0.0);
    vec4 newPosition = vec4(0.0);
    vec3 p2p = vec3(0.0);
    float newDistancePosition = 0.0;
    float weight = 0.0;
    float NdotP2P = 0.0;
    for (int i = 0; i < 16; ++i)
    {
      texCoord = clamp(scaleRotationMatrix * depthSamples[i] + gl_TexCoord[0].st, 0.0, 0.999999);
      newPosition = gl_ProjectionMatrixInverse * (2.0 * vec4(texCoord, texture2D(gbufferSampler, texCoord).z, 1.0) - 1.0);
      newPosition.xyz /= newPosition.w;

      p2p = newPosition.xyz - position.xyz;
      newDistancePosition = dot(p2p, p2p);

      if (gbufferEffectType == 3) // for HSSAO (normal way)
      {
        SSAO += max(dot(normal, p2p), FLT_EPSILON) / (sqrt(newDistancePosition) * (1.0 + newDistancePosition)); // failsafe GLSL Shader value for dot product computation
      }
      else // if (gbufferEffectType == 4) // for HSSAO Plus weight-based approach
      {
        weight = 1.0 - 0.25 * newDistancePosition;
        if (weight > 0.0)
        {
          p2p /= sqrt(newDistancePosition);

          NdotP2P = dot(normal, p2p);
          if (NdotP2P > NDOTP2P_LOWER_LIMIT)
          {
            SSAO += weight * NdotP2P;
          }
        }
      }
    }

    return vec4(vec3(1.0 - SSAO * 0.0625), 1.0);
  }
  else
  {
    return vec4(vec3(1.0), 1.0);
  }
}