/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  Common Functions GLSL code.
*  References:
*  1. en.wikipedia.org/wiki/Phong_reflection_model                   -> Phong Specular Lighting Model
*  2. en.wikipedia.org/wiki/Blinn%E2%80%93Phong_shading_model        -> Blinn-Phong Specular Lighting Model
*  3. en.wikipedia.org/wiki/Specular_highlight#Gaussian_distribution -> Gaussiang Specular Lighting Model
*  4. en.wikipedia.org/wiki/Cel_shading                              -> Toon Specular Lighting Model
*  5. www.cs.northwestern.edu/~ago820/SIG98/abstract.html            -> Gooch Specular Lighting Model
*  6. en.wikipedia.org/wiki/Oren%E2%80%93Nayar_reflectance_model     -> Oren-Nayar Diffuse Lighting Model
*  7. en.wikipedia.org/wiki/Reflection_mapping#Sphere_mapping        -> Reflection (Sphere) Mapping
*  8. en.wikipedia.org/wiki/Reflection_mapping                       -> Reflection Mapping
*
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/

VS_VARYING vec4 clipPosition;
VS_VARYING vec2 scaledWindowSpacePosition;

uniform float tessellationLevel;
uniform bool useAdaptiveTessellation;
uniform bool useTrianglePositionAdaptiveTessellationMetric;
uniform bool useLightAdaptiveTessellationMetric;

// Machine float epsilon
const float FLT_EPSILON = 0.000001;

const float LAMBERT_TERM_DIVIDE_FACTOR = 4.0;
const float TRIANGLE_CLIP_POSITION_OFFSET = 0.5;
const float SHERICAL_DISTANCE_MIN_FACTOR  = 0.1;
const float SHERICAL_DISTANCE_MAX_FACTOR  = 2.0;

vec2 randSeed = vec2(0.0);
void initRand(in vec2 seed);
float rand1();
float rand1(in vec2 seed);
vec2 rand2();
vec2 rand2(in vec2 seed);
vec3 rand3();
vec3 rand3(in vec2 seed);
vec3 cosHemiRandom();
vec3 cosHemiRandom(in vec2 seed);
vec3 cosPowHemiRandom(in float k);
vec3 cosPowHemiRandom(in vec2 seed, float k);
vec2 applySphericalCoordinates(in vec3 position, in vec3 normal);
vec3 applyCubeCoordinates(in vec3 position, in vec3 normal);
float applyLambertTerm();
void applyTriangleClippingAndAdaptiveTessellationFactors();
float getSphericalDistanceFromCenter(in float radius, in vec3 center);

void initRand(in vec2 seed)
{
  randSeed = seed;
}

// This function returns uniformly distributed values in the range [0, 1]
float rand1()
{
  float dotProduct = dot(randSeed, vec2(12.9898,78.233));
  float x = fract(sin(dotProduct) * 43758.5453);
  randSeed.x = x;
  return x;
}

// This function returns uniformly distributed values in the range [0, 1]
float rand1(in vec2 seed)
{
  float dotProduct = dot(seed, vec2(12.9898,78.233));
  float x = fract(sin(dotProduct) * 43758.5453);
  return x;
}

// This function returns uniformly distributed values in the range [0, 1]
vec2 rand2()
{
  float dotProduct = dot(randSeed, vec2(12.9898,78.233));
  vec2 r = vec2(fract(sin(dotProduct) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453));
  randSeed = r;
  return r;
}

// This function returns uniformly distributed values in the range [0, 1]
vec2 rand2(in vec2 seed)
{
  float dotProduct = dot(seed, vec2(12.9898,78.233));
  vec2 r = vec2(fract(sin(dotProduct) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453));
  return r;
}

// This function returns uniformly distributed values in the range [0, 1]
vec3 rand3()
{
  float dotProduct = dot(randSeed, vec2(12.9898,78.233));
  vec3 r = vec3(fract(sin(dotProduct) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453), fract(sin(3.0 * dotProduct) * 43758.5453));
  randSeed = r.xy;
  return r;
}

// This function returns uniformly distributed values in the range [0, 1]
vec3 rand3(in vec2 seed)
{
  float dotProduct = dot(seed, vec2(12.9898,78.233));
  vec3 r = vec3(fract(sin(dotProduct) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453), fract(sin(3.0 * dotProduct) * 43758.5453));
  return r;
}

// Returns 3D unit vectors distributed according to a
// cosine distribution around the Z axis.
vec3 cosHemiRandom()
{
  vec2 e = rand2();

  // Jensen's method
  float sinTheta = sqrt(1.0 - e.x);
  float cosTheta = sqrt(e.x);
  float phi = 6.28318531 * e.y;

  // We could also use Malley's method (pbrt p.657), since they have the same cost:
  // r = sqrt(e.x);
  // t = 2 * pi * e.y;
  // x = cos(t) * r;
  // y = sin(t) * r;
  // z = sqrt(1.0 - x * x + y * y);

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

// Returns 3D unit vectors distributed according to a
// cosine distribution around the Z axis.
vec3 cosHemiRandom(in vec2 seed)
{
  vec2 e = rand2(seed);

  // Jensen's method
  float sinTheta = sqrt(1.0 - e.x);
  float cosTheta = sqrt(e.x);
  float phi = 6.28318531 * e.y;

  // We could also use Malley's method (pbrt p.657), since they have the same cost:
  // r = sqrt(e.x);
  // t = 2 * pi * e.y;
  // x = cos(t) * r;
  // y = sin(t) * r;
  // z = sqrt(1.0 - x * x + y * y);

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

// Returns 3D unit vectors distributed according to a cosine
// power distribution (pow(cos(theta), k) around the Z axis.
// The method comes from the G3D engine (http://g3d.sourceforge.net)
// The returned vector is closer to vec3(0.0, 0.0, 1.0) as k becomes big.
vec3 cosPowHemiRandom(in float k)
{
  vec2 e = rand2();

  float cosTheta = pow(e.x, 1.0 / (k + 1.0));
  float sinTheta = sqrt(1.0 - cosTheta * cosTheta);
  float phi = 6.28318531 * e.y;

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

// Returns 3D unit vectors distributed according to a cosine
// power distribution (pow(cos(theta), k) around the Z axis.
// The method comes from the G3D engine (http://g3d.sourceforge.net)
// The returned vector is closer to vec3(0.0, 0.0, 1.0) as k becomes big.
vec3 cosPowHemiRandom(in vec2 seed, float k)
{
  vec2 e = rand2(seed);

  float cosTheta = pow(e.x, 1.0 / (k + 1.0));
  float sinTheta = sqrt(1.0 - cosTheta * cosTheta);
  float phi = 6.28318531 * e.y;

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

vec2 applySphericalCoordinates(in vec3 position, in vec3 normal)
{
  vec3 r = applyCubeCoordinates(position, normal);
  // float m = 2.0 * sqrt(r.x * r.x + r.y * r.y + (r.z + 1.0) * (r.z + 1.0));
  float m = 2.0 * length(vec3(r.xy, r.z + 1.0));

  return vec2(r.xy) / m + 0.5;
}

vec3 applyCubeCoordinates(in vec3 position, in vec3 normal)
{
  return reflect(normalize(position), normalize(normal));
}

float applyLambertTerm()
{
  vec3 E = normalize(-vec3(gl_ModelViewMatrix * gl_Vertex));
  vec3 N = normalize(gl_NormalMatrix * gl_Normal);

  return (1.0 - abs(max(dot(N, E), FLT_EPSILON))) / LAMBERT_TERM_DIVIDE_FACTOR; // failsafe GLSL Shader value for diffuse light computation
}

void applyTriangleClippingAndAdaptiveTessellationFactors()
{
  clipPosition = gl_ModelViewProjectionMatrix * gl_Vertex;
  clipPosition.w += TRIANGLE_CLIP_POSITION_OFFSET;
  if (useAdaptiveTessellation)
  {
    scaledWindowSpacePosition = vec2(tessellationLevel);
    if (useTrianglePositionAdaptiveTessellationMetric)
    {
      vec2 ndcPosition = (clipPosition.xy / clipPosition.w);
      scaledWindowSpacePosition *= ndcPosition;
    }
    if (useLightAdaptiveTessellationMetric)
    {
      scaledWindowSpacePosition *= applyLambertTerm();
    }
  }
}

float getSphericalDistanceFromCenter(in float radius, in vec3 center)
{
  // get the extreme points of the circle
  vec4 mx = vec4(center - vec3(radius, 0.0, 0.0), 1.0);
  vec4 px = vec4(center + vec3(radius, 0.0, 0.0), 1.0);
  vec4 my = vec4(center - vec3(0.0, radius, 0.0), 1.0);
  vec4 py = vec4(center + vec3(0.0, radius, 0.0), 1.0);
  vec4 mz = vec4(center - vec3(0.0, 0.0, radius), 1.0);
  vec4 pz = vec4(center + vec3(0.0, 0.0, radius), 1.0);
  
  // get the extreme points in clip space
  mx = gl_ModelViewProjectionMatrix * mx;
  px = gl_ModelViewProjectionMatrix * px;
  my = gl_ModelViewProjectionMatrix * my;
  py = gl_ModelViewProjectionMatrix * py;
  mz = gl_ModelViewProjectionMatrix * mz;
  pz = gl_ModelViewProjectionMatrix * pz;
  
  // get the extreme points in NDC space
  mx.xy /= mx.w;
  px.xy /= px.w;
  my.xy /= my.w;
  py.xy /= py.w;
  mz.xy /= mz.w;
  pz.xy /= pz.w;
  
  // how much NDC do the extreme points subtend?
  float dx = distance(mx.xy, px.xy);
  float dy = distance(my.xy, py.xy);
  float dz = distance(mz.xy, pz.xy);
  
  return clamp(length(vec3(dx, dy, dz)), SHERICAL_DISTANCE_MIN_FACTOR, SHERICAL_DISTANCE_MAX_FACTOR) / 2.0; // implying a faster implementation of a sqrt(dx * dx + dy * dy + dz * dz) call
}