/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  Common Functions GLSL code.
*  References:
*  1. en.wikipedia.org/wiki/Phong_reflection_model                   -> Phong Specular Lighting Model
*  2. en.wikipedia.org/wiki/Blinn%E2%80%93Phong_shading_model        -> Blinn-Phong Specular Lighting Model
*  3. en.wikipedia.org/wiki/Specular_highlight#Gaussian_distribution -> Gaussiang Specular Lighting Model
*  4. en.wikipedia.org/wiki/Cel_shading                              -> Toon Specular Lighting Model
*  5. www.cs.northwestern.edu/~ago820/SIG98/abstract.html            -> Gooch Specular Lighting Model
*  6. en.wikipedia.org/wiki/Oren%E2%80%93Nayar_reflectance_model     -> Oren-Nayar Diffuse Lighting Model
*  7. en.wikipedia.org/wiki/Reflection_mapping#Sphere_mapping        -> Reflection (Sphere) Mapping
*  8. en.wikipedia.org/wiki/Reflection_mapping                       -> Reflection Mapping
*
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/

#extension GL_EXT_geometry_shader4 : enable

GS_VARYING in vec3 GS_VS_NORMAL[GS_PRIMITIVE_VERTICES_IN];
GS_VARYING in vec3 GS_VS_EYE_VECTOR[GS_PRIMITIVE_VERTICES_IN];

GS_VARYING out vec4 GS_FS_COLOR;
GS_VARYING out vec3 GS_FS_NORMAL;
GS_VARYING out vec3 GS_FS_EYE_VECTOR;

uniform float sceneScaleFactor = 1.0;
uniform float timer = 0.0;

const vec4 UNIT_NORMAL_COLOR_RED    = vec4(1.0, 0.0, 0.0, 1.0);
const vec4 UNIT_NORMAL_COLOR_BLACK  = vec4(0.0, 0.0, 0.0, 1.0);
const float UNIT_NORMAL_AXIS_OFFSET = 0.001;
const float TRIANGLE_CLIP_POSITION_OFFSET = 0.5;
const float SHERICAL_DISTANCE_MIN_FACTOR  = 0.1;
const float SHERICAL_DISTANCE_MAX_FACTOR  = 2.0;

vec2 randSeed = vec2(0.0);
void initRand(in vec2 seed);
float rand1();
float rand1(in vec2 seed);
vec2 rand2();
vec2 rand2(in vec2 seed);
vec3 rand3();
vec3 rand3(in vec2 seed);
vec3 cosHemiRandom();
vec3 cosHemiRandom(in vec2 seed);
vec3 cosPowHemiRandom(in float k);
vec3 cosPowHemiRandom(in vec2 seed, float k);
void applyTriangleGeometry(in bool useShrinkFactor, in float shrinkFactor, in bool useNoiseOffset, in bool useTexturing);
vec3 calculateTriangleCentroid(in bool useShrinkFactor, in bool useNoiseOffset);
void emitTriangleVertex(in vec3 triangleCentroid, in bool useShrinkFactor, in float shrinkFactor, in bool useNoiseOffset, in bool useTexturing, in int index);
void applyNormalsGeometry();
float createNormalTrianglePositionsAndAngle(inout vec4 vertex1, inout vec4 vertex2, inout vec4 vertex3, in int index);
void emitNormalTriangle(in vec4 vertex1, in vec4 vertex2, in vec4 vertex3);
float getSphericalDistanceFromCenter(in float radius, in vec3 center);
bool discardClippedPoint(in vec4 center, float circularFactor);
bool pointIsClipped(vec4 clipPosition);
bool triangleIsClipped(vec4 clipPositions[3]);
float positionIsClipped(vec4 clipPosition);

void initRand(in vec2 seed)
{
  randSeed = seed;
}

// This function returns uniformly distributed values in the range [0, 1]
float rand1()
{
  float dotProduct = dot(randSeed, vec2(12.9898,78.233));
  float x = fract(sin(dotProduct) * 43758.5453);
  randSeed.x = x;
  return x;
}

// This function returns uniformly distributed values in the range [0, 1]
float rand1(in vec2 seed)
{
  float dotProduct = dot(seed, vec2(12.9898,78.233));
  float x = fract(sin(dotProduct) * 43758.5453);
  return x;
}

// This function returns uniformly distributed values in the range [0, 1]
vec2 rand2()
{
  float dotProduct = dot(randSeed, vec2(12.9898,78.233));
  vec2 r = vec2(fract(sin(dotProduct) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453));
  randSeed = r;
  return r;
}

// This function returns uniformly distributed values in the range [0, 1]
vec2 rand2(in vec2 seed)
{
  float dotProduct = dot(seed, vec2(12.9898,78.233));
  vec2 r = vec2(fract(sin(dotProduct) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453));
  return r;
}

// This function returns uniformly distributed values in the range [0, 1]
vec3 rand3()
{
  float dotProduct = dot(randSeed, vec2(12.9898,78.233));
  vec3 r = vec3(fract(sin(dotProduct) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453), fract(sin(3.0 * dotProduct) * 43758.5453));
  randSeed = r.xy;
  return r;
}

// This function returns uniformly distributed values in the range [0, 1]
vec3 rand3(in vec2 seed)
{
  float dotProduct = dot(seed, vec2(12.9898,78.233));
  vec3 r = vec3(fract(sin(dotProduct) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453), fract(sin(3.0 * dotProduct) * 43758.5453));
  return r;
}

// Returns 3D unit vectors distributed according to a
// cosine distribution around the Z axis.
vec3 cosHemiRandom()
{
  vec2 e = rand2();

  // Jensen's method
  float sinTheta = sqrt(1.0 - e.x);
  float cosTheta = sqrt(e.x);
  float phi = 6.28318531 * e.y;

  // We could also use Malley's method (pbrt p.657), since they have the same cost:
  // r = sqrt(e.x);
  // t = 2 * pi * e.y;
  // x = cos(t) * r;
  // y = sin(t) * r;
  // z = sqrt(1.0 - x * x + y * y);

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

// Returns 3D unit vectors distributed according to a
// cosine distribution around the Z axis.
vec3 cosHemiRandom(in vec2 seed)
{
  vec2 e = rand2(seed);

  // Jensen's method
  float sinTheta = sqrt(1.0 - e.x);
  float cosTheta = sqrt(e.x);
  float phi = 6.28318531 * e.y;

  // We could also use Malley's method (pbrt p.657), since they have the same cost:
  // r = sqrt(e.x);
  // t = 2 * pi * e.y;
  // x = cos(t) * r;
  // y = sin(t) * r;
  // z = sqrt(1.0 - x * x + y * y);

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

// Returns 3D unit vectors distributed according to a cosine
// power distribution (pow(cos(theta), k) around the Z axis.
// The method comes from the G3D engine (http://g3d.sourceforge.net)
// The returned vector is closer to vec3(0.0, 0.0, 1.0) as k becomes big.
vec3 cosPowHemiRandom(in float k)
{
  vec2 e = rand2();

  float cosTheta = pow(e.x, 1.0 / (k + 1.0));
  float sinTheta = sqrt(1.0 - cosTheta * cosTheta);
  float phi = 6.28318531 * e.y;

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

// Returns 3D unit vectors distributed according to a cosine
// power distribution (pow(cos(theta), k) around the Z axis.
// The method comes from the G3D engine (http://g3d.sourceforge.net)
// The returned vector is closer to vec3(0.0, 0.0, 1.0) as k becomes big.
vec3 cosPowHemiRandom(in vec2 seed, float k)
{
  vec2 e = rand2(seed);

  float cosTheta = pow(e.x, 1.0 / (k + 1.0));
  float sinTheta = sqrt(1.0 - cosTheta * cosTheta);
  float phi = 6.28318531 * e.y;

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

void applyTriangleGeometry(in bool useShrinkFactor, in float shrinkFactor, in bool useNoiseOffset, in bool useTexturing)
{
  vec4 clipPositions[3];
  for (int i = 0; i < gl_in.length(); ++i)
  {
    clipPositions[i] = gl_ModelViewProjectionMatrix * gl_in[i].gl_Position;
    clipPositions[i].w += TRIANGLE_CLIP_POSITION_OFFSET;
  }
  if (triangleIsClipped(clipPositions)) // do not emit any vertices if triangle is clipped
  {
    return;
  }

  vec3 triangleCentroid = calculateTriangleCentroid(useShrinkFactor, useNoiseOffset);
  GS_FS_EYE_VECTOR           = vec3(0.0);
  GS_FS_NORMAL               = vec3(0.0);
  GS_FS_COLOR                = vec4(0.0);
  gl_TexCoord[0]             = vec4(0.0);
  gl_Position                = vec4(0.0);
  gl_ClipVertex              = vec4(0.0);
  gl_FrontColor              = vec4(0.0);
  gl_BackColor               = vec4(0.0);
  for (int i = 0; i < gl_in.length(); ++i)
  {
    GS_FS_EYE_VECTOR = GS_VS_EYE_VECTOR[i];
    GS_FS_NORMAL     = GS_VS_NORMAL[i];
    GS_FS_COLOR      = UNIT_NORMAL_COLOR_BLACK;

    emitTriangleVertex(triangleCentroid, useShrinkFactor, shrinkFactor, useNoiseOffset, useTexturing, i);
  }
  EndPrimitive();
}

vec3 calculateTriangleCentroid(in bool useShrinkFactor, in bool useNoiseOffset)
{
  // done so to avoid AMD compilation errors with gl_in[0], gl_in[1] & gl_in[2] when only a point is used for the sphere impostors
  vec3 triangleCentroid = (useShrinkFactor) ? (gl_in[0].gl_Position.xyz + gl_in[(gl_in.length() > 1) ? 1 : 0].gl_Position.xyz + gl_in[(gl_in.length() > 2) ? 2 : 0].gl_Position.xyz) / 3.0 : vec3(0.0);
  if (useNoiseOffset)
  {
    triangleCentroid += 5.0 * (1.0 + sin(timer)) * (rand1(triangleCentroid.xy) - 0.5) / sceneScaleFactor;
  }

  return triangleCentroid;
}

void emitTriangleVertex(in vec3 triangleCentroid, in bool useShrinkFactor, in float shrinkFactor, in bool useNoiseOffset, in bool useTexturing, in int index)
{
  float sinTimerShrinkFactor = ((1.0 - shrinkFactor) + 0.75 + 0.125 * (1.0 + sin(timer))) * shrinkFactor;
  vec4 vertex = (useShrinkFactor) ? vec4(triangleCentroid + sinTimerShrinkFactor * (gl_in[index].gl_Position.xyz - triangleCentroid), gl_in[index].gl_Position.w) : gl_in[index].gl_Position;
  gl_Position = gl_ModelViewProjectionMatrix * vertex;
  if (useNoiseOffset)
  {
    gl_Position.xyz += 0.2 * vec3(rand3(gl_in[index].gl_Position.xy) - 0.5) / sceneScaleFactor;
  }
  gl_ClipVertex = gl_ModelViewMatrix * vertex;
  gl_FrontColor = gl_in[index].gl_FrontColor;
  gl_BackColor  = gl_in[index].gl_BackColor;
  if (useTexturing)
  {
    gl_TexCoord[0] = gl_in[index].gl_TexCoord[0];
  }
  EmitVertex();
}

void applyNormalsGeometry()
{
  // Render unit normals
  float normalAngle = 0.0;
  vec4 vertex1    = vec4(0.0);
  vec4 vertex2    = vec4(0.0);
  vec4 vertex3    = vec4(0.0);
  GS_FS_EYE_VECTOR           = vec3(0.0);
  GS_FS_NORMAL               = vec3(0.0);
  gl_TexCoord[0]             = vec4(0.0);
  gl_Position                = vec4(0.0);
  gl_ClipVertex              = vec4(0.0);
  gl_FrontColor              = vec4(UNIT_NORMAL_COLOR_RED.xyz, 1.0);
  gl_BackColor               = vec4(UNIT_NORMAL_COLOR_RED.xyz, 1.0);
  for (int i = 0; i < gl_in.length(); ++i)
  {
    GS_FS_COLOR = UNIT_NORMAL_COLOR_RED;

    normalAngle = createNormalTrianglePositionsAndAngle(vertex1, vertex2, vertex3, i);
    if (normalAngle >= 0.0)
    {
      emitNormalTriangle(vertex1, vertex2, vertex3);
    }
    else
    {
      emitNormalTriangle(vertex3, vertex2, vertex1);
    }
  }
}

float createNormalTrianglePositionsAndAngle(inout vec4 vertex1, inout vec4 vertex2, inout vec4 vertex3, in int index)
{
  vertex1     = gl_in[index].gl_Position;
  vertex1.xy += UNIT_NORMAL_AXIS_OFFSET * rand2(vertex1.xy); // random small offset to compensate for AMD/ATI issues
  vertex2     = gl_in[index].gl_Position;
  vertex2.xy -= UNIT_NORMAL_AXIS_OFFSET * rand2(vertex2.xy); // random small offset to compensate for AMD/ATI issues
  float intensity = length(normalize(GS_VS_NORMAL[index])) / 10.0;
  vertex3         = vec4((1.0 + intensity) * gl_in[index].gl_Position.xyz, gl_in[index].gl_Position.w);
  vec3 normal     = cross(vertex2.xyz - vertex1.xyz, vertex3.xyz - vertex1.xyz);

  return dot(normalize(normal), normalize(GS_VS_EYE_VECTOR[index]));
}

void emitNormalTriangle(in vec4 vertex1, in vec4 vertex2, in vec4 vertex3)
{
  vec4 clipPosition1 = gl_ModelViewProjectionMatrix * vertex1;
  vec4 clipPosition2 = gl_ModelViewProjectionMatrix * vertex2;
  vec4 clipPosition3 = gl_ModelViewProjectionMatrix * vertex3;

  vec4 clipPositions[3];
  clipPositions[0] = clipPosition1;
  clipPositions[0].w += TRIANGLE_CLIP_POSITION_OFFSET;
  clipPositions[1] = clipPosition2;
  clipPositions[1].w += TRIANGLE_CLIP_POSITION_OFFSET;
  clipPositions[2] = clipPosition3;
  clipPositions[2].w += TRIANGLE_CLIP_POSITION_OFFSET;
  if (triangleIsClipped(clipPositions)) // do not emit any vertices if triangle is clipped
  {
    return;
  }

  gl_Position = clipPosition1;
  gl_ClipVertex = gl_ModelViewMatrix * vertex1;
  EmitVertex();

  gl_Position = clipPosition2;
  gl_ClipVertex = gl_ModelViewMatrix * vertex2;
  EmitVertex();

  gl_Position = clipPosition3;
  gl_ClipVertex = gl_ModelViewMatrix * vertex3;
  EmitVertex();

  EndPrimitive();
}

float getSphericalDistanceFromCenter(in float radius, in vec3 center)
{
  // get the extreme points of the circle
  vec4 mx = vec4(center - vec3(radius, 0.0, 0.0), 1.0);
  vec4 px = vec4(center + vec3(radius, 0.0, 0.0), 1.0);
  vec4 my = vec4(center - vec3(0.0, radius, 0.0), 1.0);
  vec4 py = vec4(center + vec3(0.0, radius, 0.0), 1.0);
  vec4 mz = vec4(center - vec3(0.0, 0.0, radius), 1.0);
  vec4 pz = vec4(center + vec3(0.0, 0.0, radius), 1.0);

  // get the extreme points in clip space
  mx = gl_ModelViewProjectionMatrix * mx;
  px = gl_ModelViewProjectionMatrix * px;
  my = gl_ModelViewProjectionMatrix * my;
  py = gl_ModelViewProjectionMatrix * py;
  mz = gl_ModelViewProjectionMatrix * mz;
  pz = gl_ModelViewProjectionMatrix * pz;
 
  // get the extreme points in NDC space
  mx.xy /= mx.w;
  px.xy /= px.w;
  my.xy /= my.w;
  py.xy /= py.w;
  mz.xy /= mz.w;
  pz.xy /= pz.w;

  // how much NDC do the extreme points subtend?
  float dx = distance(mx.xy, px.xy);
  float dy = distance(my.xy, py.xy);
  float dz = distance(mz.xy, pz.xy);

  return clamp(length(vec3(dx, dy, dz)), SHERICAL_DISTANCE_MIN_FACTOR, SHERICAL_DISTANCE_MAX_FACTOR) / 2.0; // implying a faster implementation of a sqrt(dx * dx + dy * dy + dz * dz) call
}

bool discardClippedPoint(in vec4 center, float circularFactor)
{
  vec4 clipPosition = gl_ModelViewProjectionMatrix * center; // get the point in clip space
  clipPosition.xyz /= clipPosition.w; // get the point in NDC space
  clipPosition.w = 1.0 + circularFactor;

  return pointIsClipped(clipPosition);
}

bool pointIsClipped(vec4 clipPosition)
{
  return (positionIsClipped(clipPosition) == 1.0);
}

bool triangleIsClipped(vec4 clipPositions[3])
{
  return (min(min(positionIsClipped(clipPositions[0]), positionIsClipped(clipPositions[1])), positionIsClipped(clipPositions[2])) == 1.0);
}

float positionIsClipped(vec4 clipPosition)
{
  // test whether the position is entirely inside the view frustum
  return (-clipPosition.w <= clipPosition.x && clipPosition.x <= clipPosition.w
       && -clipPosition.w <= clipPosition.y && clipPosition.y <= clipPosition.w
       && -clipPosition.w <= clipPosition.z && clipPosition.z <= clipPosition.w)
       ? 0.0 : 1.0;
}