/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

/**
*
*  Common Functions GLSL code.
*  References:
*  1. en.wikipedia.org/wiki/Phong_reflection_model                   -> Phong Specular Lighting Model
*  2. en.wikipedia.org/wiki/Blinn%E2%80%93Phong_shading_model        -> Blinn-Phong Specular Lighting Model
*  3. en.wikipedia.org/wiki/Specular_highlight#Gaussian_distribution -> Gaussiang Specular Lighting Model
*  4. en.wikipedia.org/wiki/Cel_shading                              -> Toon Specular Lighting Model
*  5. www.cs.northwestern.edu/~ago820/SIG98/abstract.html            -> Gooch Specular Lighting Model
*  6. en.wikipedia.org/wiki/Oren%E2%80%93Nayar_reflectance_model     -> Oren-Nayar Diffuse Lighting Model
*  7. en.wikipedia.org/wiki/Reflection_mapping#Sphere_mapping        -> Reflection (Sphere) Mapping
*  8. en.wikipedia.org/wiki/Reflection_mapping                       -> Reflection Mapping
*
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/

uniform sampler2D texture;
uniform int texturingMode = 0;
uniform int filterTexturingMode = 2;
uniform float px = 1500; // default texture size width  if not passed explicitly for emboss filter on texture
uniform float py =  500; // default texture size height if not passed explicitly for emboss filter on texture
uniform bool useOrenNayarDiffuseModel = false;

// Machine float epsilon
const float FLT_EPSILON = 0.000001;

// Oren-Nayar diffuse model constant values
const float P = 0.458105 * 2.0;
const float SIGMA = 1.057673 * 2.0;
const float Eo = 0.998868 * 1.85;
const float SIGMA_SQUARED = SIGMA * SIGMA;
const float A = 1.0 - 0.5 * (SIGMA_SQUARED) / (SIGMA_SQUARED + 0.33);
const float B = 0.45 * (SIGMA_SQUARED) / (SIGMA_SQUARED + 0.09);
const float R = 2.0 * P / SIGMA * Eo;

// Fresnel (Schlick's approximation) specular model constant values
const float REFRACTION_INDEX_RATIO = 1.2; // for water
const float R0 = ((1.0 - REFRACTION_INDEX_RATIO) * (1.0 - REFRACTION_INDEX_RATIO)) / ((1.0 + REFRACTION_INDEX_RATIO) * (1.0 + REFRACTION_INDEX_RATIO));
const float REFRACTION_DIVISOR_FACTOR = 2.0;

// Gooch shading model constant values
const vec4 WARM_GOOCH_COLOR = vec4(0.2, 0.2, 0.0, 1.0);
const vec4 COOL_GOOCH_COLOR = vec4(0.0, 0.0, 0.2, 1.0);
const float DIFFUSE_WARM_GOOCH_COLOR = 0.89;
const float DIFFUSE_COOL_GOOCH_COLOR = 0.89;

// Gamma correction constant values
const float GAMMA = 2.0;
const float INVERSE_GAMMA = 1.0 / GAMMA;

// Emboss & Fog constants
const float K = 0.85;
const float LOG2 = 1.442695;

// Simple sphere constant values
const float SIMPLE_SPHERE_ALPHA_FACTOR = 60.0;
const float PI = 3.14159265;
const float PI_TWO = 2.0 * PI;

// Erosion constants
const float EROSION_MIN = 0.25;
const float EROSION_MAX = 0.75;

// Jitter Environment Mapping
const int NUMBER_OF_JITTER_SAMPLING_STEPS = 16;

vec2 randSeed = vec2(0.0);
void initRand(in vec2 seed);
float rand1();
float rand1(in vec2 seed);
vec2 rand2();
vec2 rand2(in vec2 seed);
vec3 rand3();
vec3 rand3(in vec2 seed);
vec3 cosHemiRandom();
vec3 cosHemiRandom(in vec2 seed);
vec3 cosPowHemiRandom(in float k);
vec3 cosPowHemiRandom(in vec2 seed, float k);
float smootherstep(in float edge0, in float edge1, in float x);
vec2 smootherstep(in vec2 edge0, in vec2 edge1, in vec2 x);
vec3 smootherstep(in vec3 edge0, in vec3 edge1, in vec3 x);
vec4 smootherstep(in vec4 edge0, in vec4 edge1, in vec4 x);
void applyOldStyleTransparency();
vec4 applyOrenNayarDiffuseModel(in vec3 E, in vec3 N, in vec3 L, in vec4 diffuse);
float applyFresnelFactorSchlickApproximationSpecularModel(in vec3 E, in vec3 N, in vec3 L);
float applyPhongSpecularModel(in vec3 E, in vec3 N, in vec3 L, in float shininess, in float factor);
float applyHalfVectorNxH(in vec3 E, in vec3 N, in vec3 L);
float applyBlinnPhongSpecularModel(in vec3 E, in vec3 N, in vec3 L, in float shininess, in float factor);
float applyGaussianSpecularModel(in vec3 E, in vec3 N, in vec3 L, in float shininess, in float factor);
vec4 applyToonColor(in float lambertTerm, in float specularPower, in float attenuation, in float thresholdLambertTerm, in float thresholdSpecularPower, in vec4 color);
vec4 applyGoochColor(in float lambertTerm, in vec4 color);
vec2 applySphericalCoordinates(in vec3 position, in vec3 normal);
vec3 applyCubeCoordinates(in vec3 position, in vec3 normal);
vec4 smootherFilterTexture(in vec2 texCoords);
vec4 cubic(in float v);
vec4 bicubicFilterTexture(in vec2 texCoords);
vec4 applyFilterTexture(in vec2 texCoords);
void applyTexture(in vec2 texCoords, inout vec4 color);
void applyTextureWithEmbossFilter(in vec2 texCoords, inout vec4 color);
void applyJitteredEnvironmentMapping(in vec3 position, in vec3 normal, inout vec4 color);
vec4 applyCombineTextureWithColor(in vec2 texCoords, inout vec4 color);
void applyGammaCorrection(inout vec4 color);
void applyFastGammaCorrection(inout vec4 color);
void applyFog(inout vec4 color);
void applyFog(in float depth, inout vec4 color);
void calculateDistanceMetricForSimpleSphereImpostor(in vec2 mapping, out float distanceMetric);
void simpleSphereImpostor(in vec2 mapping, in float distanceMetric, in float sphereRadius, in vec3 cameraSpherePosition, in bool multiplySphereRadius, out vec3 cameraNormal, out vec3 cameraPosition);
void applyPointSphereSimpleLighting(in float distanceMetric, in vec4 materialColor, inout vec4 color);
void applySimpleSphereImpostorAntiAlias(in vec2 mapping, in float circularFactor, inout float sphereAlpha);
vec2 applyImpostorSphereTextureCoords(in vec3 normal);
float applyDepth(in vec3 position);
float applyLinearDepth(in float depthValue, in float zNear, in float zFar);
vec3 reconstructEyePositionFromDepth(in float depthValue, in float zNear, in float zFar, in float tanFieldOfView, in int screenWidth, in int screenHeight);
void applyErosion(in vec4 noisevec);

void initRand(in vec2 seed)
{
  randSeed = seed;
}

// This function returns uniformly distributed values in the range [0, 1]
float rand1()
{
  float dotProduct = dot(randSeed, vec2(12.9898,78.233));
  float x = fract(sin(dotProduct) * 43758.5453);
  randSeed.x = x;
  return x;
}

// This function returns uniformly distributed values in the range [0, 1]
float rand1(in vec2 seed)
{
  float dotProduct = dot(seed, vec2(12.9898,78.233));
  float x = fract(sin(dotProduct) * 43758.5453);
  return x;
}

// This function returns uniformly distributed values in the range [0, 1]
vec2 rand2()
{
  float dotProduct = dot(randSeed, vec2(12.9898,78.233));
  vec2 r = vec2(fract(sin(dotProduct) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453));
  randSeed = r;
  return r;
}

// This function returns uniformly distributed values in the range [0, 1]
vec2 rand2(in vec2 seed)
{
  float dotProduct = dot(seed, vec2(12.9898,78.233));
  vec2 r = vec2(fract(sin(dotProduct) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453));
  return r;
}

// This function returns uniformly distributed values in the range [0, 1]
vec3 rand3()
{
  float dotProduct = dot(randSeed, vec2(12.9898,78.233));
  vec3 r = vec3(fract(sin(dotProduct) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453), fract(sin(3.0 * dotProduct) * 43758.5453));
  randSeed = r.xy;
  return r;
}

// This function returns uniformly distributed values in the range [0, 1]
vec3 rand3(in vec2 seed)
{
  float dotProduct = dot(seed, vec2(12.9898,78.233));
  vec3 r = vec3(fract(sin(dotProduct) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453), fract(sin(3.0 * dotProduct) * 43758.5453));
  return r;
}

// Returns 3D unit vectors distributed according to a
// cosine distribution around the Z axis.
vec3 cosHemiRandom()
{
  vec2 e = rand2();

  // Jensen's method
  float sinTheta = sqrt(1.0 - e.x);
  float cosTheta = sqrt(e.x);
  float phi = 6.28318531 * e.y;

  // We could also use Malley's method (pbrt p.657), since they have the same cost:
  // r = sqrt(e.x);
  // t = 2 * pi * e.y;
  // x = cos(t) * r;
  // y = sin(t) * r;
  // z = sqrt(1.0 - x * x + y * y);

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

// Returns 3D unit vectors distributed according to a
// cosine distribution around the Z axis.
vec3 cosHemiRandom(in vec2 seed)
{
  vec2 e = rand2(seed);

  // Jensen's method
  float sinTheta = sqrt(1.0 - e.x);
  float cosTheta = sqrt(e.x);
  float phi = 6.28318531 * e.y;

  // We could also use Malley's method (pbrt p.657), since they have the same cost:
  // r = sqrt(e.x);
  // t = 2 * pi * e.y;
  // x = cos(t) * r;
  // y = sin(t) * r;
  // z = sqrt(1.0 - x * x + y * y);

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

// Returns 3D unit vectors distributed according to a cosine
// power distribution (pow(cos(theta), k) around the Z axis.
// The method comes from the G3D engine (http://g3d.sourceforge.net)
// The returned vector is closer to vec3(0.0, 0.0, 1.0) as k becomes big.
vec3 cosPowHemiRandom(in float k)
{
  vec2 e = rand2();

  float cosTheta = pow(e.x, 1.0 / (k + 1.0));
  float sinTheta = sqrt(1.0 - cosTheta * cosTheta);
  float phi = 6.28318531 * e.y;

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

// Returns 3D unit vectors distributed according to a cosine
// power distribution (pow(cos(theta), k) around the Z axis.
// The method comes from the G3D engine (http://g3d.sourceforge.net)
// The returned vector is closer to vec3(0.0, 0.0, 1.0) as k becomes big.
vec3 cosPowHemiRandom(in vec2 seed, float k)
{
  vec2 e = rand2(seed);

  float cosTheta = pow(e.x, 1.0 / (k + 1.0));
  float sinTheta = sqrt(1.0 - cosTheta * cosTheta);
  float phi = 6.28318531 * e.y;

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

// Scale, and clamp x to 0...1 (first line) range & evaluate polynomial (second line)
#define smootherstepDefinition(edge0, edge1, x)             \
        x = clamp((x - edge0) / (edge1 - edge0), 0.0, 1.0); \
        return x * x * x * (x * (x * 6.0 - 15.0) + 10.0);

// Prof. Ken Perlin suggests an improved version of the smoothstep function which has zero 1st and 2nd order derivatives at t=0 and t=1
float smootherstep(in float edge0, in float edge1, in float x)
{
  smootherstepDefinition(edge0, edge1, x);
}

// vec2 variaty of the smootherstep() function
vec2 smootherstep(in vec2 edge0, in vec2 edge1, in vec2 x)
{
  smootherstepDefinition(edge0, edge1, x);
}

// vec3 variaty of the smootherstep() function
vec3 smootherstep(in vec3 edge0, in vec3 edge1, in vec3 x)
{
  smootherstepDefinition(edge0, edge1, x);
}

// vec4 variaty of the smootherstep() function
vec4 smootherstep(in vec4 edge0, in vec4 edge1, in vec4 x)
{
  smootherstepDefinition(edge0, edge1, x);
}

void applyOldStyleTransparency()
{
  if (mod(gl_FragCoord.x, 2.0) == mod(gl_FragCoord.y, 2.0))
  {
    discard; // screen-door transparency effect
  }
}

vec4 applyOrenNayarDiffuseModel(in vec3 E, in vec3 N, in vec3 L, in vec4 diffuse)
{
  float nDotL = dot(N, L);
  float nDotE = dot(N, E);

  float sinTr = length(cross(E, N));
  float cosTr = clamp(nDotE, 0.0001, 1.0);
  float sinTi = length(cross(L, N));
  float cosTi = clamp(nDotL, 0.0001, 1.0);
  float tanTi = sinTi / cosTi;
  float tanTr = sinTr / cosTr;

  vec3 Ep = normalize(E - nDotE * N);
  vec3 Lp = normalize(L - nDotL * N);

  return R * cosTi * (A + B * max(0.0, dot(Ep, Lp)) * max(sinTr, sinTi) * min(tanTi, tanTr)) * diffuse;
}

float applyFresnelFactorSchlickApproximationSpecularModel(in vec3 E, in vec3 N, in vec3 L)
{
  // optimize away a pow(t1, 5.0) call
  // original call: float fresnelFactorSchlickApproximation = pow(1.0 - applyHalfVectorNxH(E, N, L), 5.0);
  float t1 = 1.0 - applyHalfVectorNxH(E, N, L);
  float t1_2 = t1 * t1;
  float t1_4 = t1_2 * t1_2;
  float fresnelFactorSchlickApproximation = t1 * t1_4;

  return mix(R0, 1.0, fresnelFactorSchlickApproximation) / REFRACTION_DIVISOR_FACTOR;
}

float applyPhongSpecularModel(in vec3 E, in vec3 N, in vec3 L, in float shininess, in float factor)
{
  vec3 R = reflect(-L, N);
  float ExR = max(dot(E, R), FLT_EPSILON); // failsafe GLSL Shader value for specular light computation

  return pow(ExR, shininess / factor);
}

float applyHalfVectorNxH(in vec3 E, in vec3 N, in vec3 L)
{
  vec3 H = normalize(L + E);

  return max(dot(N, H), FLT_EPSILON); // failsafe GLSL Shader value for specular light computation
}

float applyBlinnPhongSpecularModel(in vec3 E, in vec3 N, in vec3 L, in float shininess, in float factor)
{
  float NxH = applyHalfVectorNxH(E, N, L);

  return pow(NxH, shininess / factor);
}

float applyGaussianSpecularModel(in vec3 E, in vec3 N, in vec3 L, in float shininess, in float factor)
{
  float NxH = applyHalfVectorNxH(E, N, L);
  float angleH = acos(clamp(NxH, -1.0 + FLT_EPSILON, 1.0 - FLT_EPSILON)); // failsafe GLSL Shader value for specular light computation
  float exponent = angleH * shininess / factor;

  return exp(-exponent * exponent);
}

vec4 applyToonColor(in float lambertTerm, in float specularPower, in float attenuation, in float thresholdLambertTerm, in float thresholdSpecularPower, in vec4 color)
{
  vec4 toonColor = color;
  if ((lambertTerm * attenuation) < thresholdLambertTerm)
  {
    toonColor.rgb *= 0.5; // darkens original color
  }
  else if (useOrenNayarDiffuseModel)
  {
    toonColor.rgb *= (lambertTerm * attenuation); // better diffuse for toon
  }

  if ((specularPower * attenuation) < thresholdSpecularPower)
  {
    toonColor.rgb *= 1.1; // brightens original color
  }

  return toonColor;
}

vec4 applyGoochColor(in float lambertTerm, in vec4 color)
{
  vec4 goochColor = color;
  vec4 kcool = min(COOL_GOOCH_COLOR + DIFFUSE_COOL_GOOCH_COLOR * goochColor, 1.0);
  vec4 kwarm = min(WARM_GOOCH_COLOR + DIFFUSE_WARM_GOOCH_COLOR * goochColor, 1.0);

  return mix(kcool, kwarm, useOrenNayarDiffuseModel ? smootherstep(0.0, 1.0, lambertTerm) : lambertTerm); // better color mix for gooch
}

vec2 applySphericalCoordinates(in vec3 position, in vec3 normal)
{
  vec3 r = applyCubeCoordinates(position, normal);
  // float m = 2.0 * sqrt(r.x * r.x + r.y * r.y + (r.z + 1.0) * (r.z + 1.0));
  float m = 2.0 * length(vec3(r.xy, r.z + 1.0));

  return vec2(r.xy) / m + 0.5;
}

vec3 applyCubeCoordinates(in vec3 position, in vec3 normal)
{
  return reflect(normalize(position), normalize(normal));
}

vec4 smootherFilterTexture(in vec2 texCoords)
{
  vec2 textureDimensions = textureSize(texture, 0); // GL3.0+ function  
  vec2 texCoordsScaled   = texCoords * textureDimensions + 0.5;
  vec2 texCoordsInt      = floor(texCoordsScaled);
  vec2 texCoordsFrac     = smootherstep(vec2(0.0), vec2(1.0), fract(texCoordsScaled));
  vec2 smootherTexCoords = (texCoordsInt + texCoordsFrac - 0.5) / textureDimensions;

  return texture2D(texture, smootherTexCoords);
}

vec4 cubic(in float v)
{
  vec4 n = vec4(1.0, 2.0, 3.0, 4.0) - v;
  vec4 s = n * n * n;
  float x = s.x;
  float y = s.y - 4.0 * s.x;
  float z = s.z - 4.0 * s.y + 6.0 * s.x;
  float w = 6.0 - x - y - z;

  return vec4(x, y, z, w) / 6.0;
}

vec4 bicubicFilterTexture(in vec2 texCoords)
{
  vec2 textureDimensions = textureSize(texture, 0);
  texCoords = texCoords * textureDimensions - 0.5; // shift the coordinate from [0,1] to [-0.5, textureDimensions - 0.5]

  vec2 fxy = fract(texCoords);
  texCoords -= fxy;

  vec4 xcubic = cubic(fxy.x);
  vec4 ycubic = cubic(fxy.y);

  vec4 c = texCoords.sstt + vec2(-0.5, 1.5).xyxy;
  vec4 s = vec4(xcubic.xz + xcubic.yw, ycubic.xz + ycubic.yw);
  vec4 offset = c + vec4(xcubic.yw, ycubic.yw) / s;
  offset /= textureDimensions.xxyy;

  vec4 sample0 = texture2D(texture, offset.xz);
  vec4 sample1 = texture2D(texture, offset.yz);
  vec4 sample2 = texture2D(texture, offset.xw);
  vec4 sample3 = texture2D(texture, offset.yw);

  float sx = s.x / (s.x + s.y);
  float sy = s.z / (s.z + s.w);

  return mix(mix(sample3, sample2, sx), mix(sample1, sample0, sx), sy);
}

vec4 applyFilterTexture(in vec2 texCoords)
{
  if (filterTexturingMode == 0) // standard hardware bilinear filter
  {
    return texture2D(texture, texCoords);
  }
  else if (filterTexturingMode == 1) // smoother texture filter
  {
    return smootherFilterTexture(texCoords);
  }
  else // if (filterTexturingMode == 2) // bicubic texture filter
  {
    return bicubicFilterTexture(texCoords);
  }
}

void applyTexture(in vec2 texCoords, inout vec4 color)
{
  vec4 textureColor = applyFilterTexture(texCoords);
  if (texturingMode == 0) // emulate GL_MODULATE (also used for environment mapping)
  {
    color *= textureColor;
  }
  else if (texturingMode == 1) // emulate GL_DECAL
  {
    color = vec4(mix(color.rgb, textureColor.rgb, textureColor.a), color.a);
  }
  else if (texturingMode == 2) // emulate GL_BLEND
  {
    color = vec4(mix(color.rgb, gl_TextureEnvColor[0].rgb, textureColor.rgb), color.a * textureColor.a); // assume texture unit 0
  }
  else if (texturingMode == 3) // emulate GL_REPLACE
  {
    color = textureColor;
  }
  else if (texturingMode == 4) // emulate GL_ADD
  {
    color.rgb += textureColor.rgb;
    color.a   *= textureColor.a;
    color = clamp(color, 0.0, 1.0);
  }
  else // if (texturingMode == 5) // emulate GL_COMBINE
  {
    color = clamp(textureColor * color, 0.0, 1.0);
  }
}

void applyTextureWithEmbossFilter(in vec2 texCoords, inout vec4 color)
{
  vec2 textureCoords = texCoords;

  // center
  vec4 neighborCenter = applyFilterTexture(texCoords);
  vec2 neighborFragments = 4.0 * (1.0 / vec2(px, py)); // multiply by 4.0 to make the emboss filter more visible

  // one right
  textureCoords.s = textureCoords.s + neighborFragments.x;
  vec4 neighborRight = applyFilterTexture(texCoords);

  // one down
  textureCoords.t = textureCoords.t - neighborFragments.y;
  vec4 neighborRightDown = applyFilterTexture(texCoords);

  // one left
  textureCoords.s = textureCoords.s - neighborFragments.x;
  vec4 neighborDown = applyFilterTexture(texCoords);

  // two up
  textureCoords.t = textureCoords.t + 2.0 * neighborFragments.y;
  vec4 neighborUp = -applyFilterTexture(texCoords);

  // one left
  textureCoords.s = textureCoords.s - neighborFragments.x;
  vec4 neighborUpLeft = -applyFilterTexture(texCoords);

  // one down
  textureCoords.t = textureCoords.t - neighborFragments.y;
  vec4 neighborLeft = -applyFilterTexture(texCoords);

  vec4 meanColor = (neighborRight + neighborRightDown + neighborDown + neighborUp + neighborUpLeft + neighborLeft) / 9.0;
  vec3 gradient = vec3(0.3 + (meanColor.r + meanColor.g + meanColor.b + meanColor.a) / 4.0);
  vec3 gradientRGB = mix(neighborCenter.rgb, gradient, K);

  if (neighborCenter.a > 0.0)
  {
    color.rgb = mix(color.rgb, gradientRGB, neighborCenter.a);
  }
}

void applyJitteredEnvironmentMapping(in vec3 position, in vec3 normal, inout vec4 color)
{
  vec3 N = normalize(normal);
  vec3 jitteredNormal    = vec3(0.0);
  vec2 jitteredTexCoords = vec2(0.0);
  vec4 jitteredColorSum  = vec4(0.0);
  for (int i = 0; i < NUMBER_OF_JITTER_SAMPLING_STEPS; ++i)
  {
    jitteredNormal = normalize(N + 0.1 * rand3(float(i) * N.xy));
    jitteredTexCoords = applySphericalCoordinates(position, jitteredNormal);
    jitteredColorSum += applyCombineTextureWithColor(jitteredTexCoords, color);
  }
  color = jitteredColorSum / float(NUMBER_OF_JITTER_SAMPLING_STEPS);
}

vec4 applyCombineTextureWithColor(in vec2 texCoords, inout vec4 color)
{
  vec4 textureColor = applyFilterTexture(texCoords);
  return clamp(textureColor * color, 0.0, 1.0); // emulate GL_COMBINE
}

void applyGammaCorrection(inout vec4 color)
{
  vec4 gamma = vec4(vec3(INVERSE_GAMMA), 1.0);
  color = pow(color, gamma);
}

void applyFastGammaCorrection(inout vec4 color)
{
  color = vec4(sqrt(color.rgb), color.a); // 2.0 gamma correction
}

// Note that for fogging we need the clipPositionZ &  NOT the ndcDepth normalized to window coordinates (see applyDepth(in vec3 position)) for it to mimick the fixed functionality, which is what gl_FragCoord.z / gl_FragCoord.w does.
// Below is the detailed explanation:
// gl_FragCoord is an input variable that contains the window relative coordinate (x, y, z, 1/w) values for the fragment.
// In VS we do: gl_Position = gl_ModelViewProjectionMatrix * gl_Vertex & the last thing the VS stage does is a divide by w, or gl_Position.w.
// So, we could say that: gl_FragCoord.xyz = gl_Position.xyz / gl_Position.w & gl_FragCoord.w = 1.0 / gl_Position.w.
// Dividing gl_FragCoord.xyz / gl_FragCoord.w yields the original gl_Position.xyz, whose gl_Position.z (ie clipPositionZ) we need for fog depth
// Multiply gl_Position.xyz by the gl_ProjectionMatrixInverse should give you back the eye-space position, ie gl_ModelViewMatrix * gl_Vertex.
// Finally, multiply gl_Position.xyz by the gl_ModelViewProjectionMatrixInverse should give you world/model space position, ie gl_Vertex.
// Therefore gl_FragCoord.z / gl_FragCoord.w is not the camera-space (or world-space) distance to the camera. Nor is it the camera-space planar distance to the camera.
// But it is a linear transform of the camera-space depth. We could use it as a way to compare two depth values together, if they came from the same projection matrix and so forth.
void applyFog(inout vec4 color)
{
  applyFog(gl_FragCoord.z / gl_FragCoord.w, color);
}

// Note that for fogging we need the clipPositionZ &  NOT the ndcDepth normalized to window coordinates (see applyDepth(in vec3 position)) for it to mimick the fixed functionality.
void applyFog(in float depth, inout vec4 color)
{
  // per-pixel exponential fog
  float fogFactor = exp2(-gl_Fog.density * gl_Fog.density * depth * depth * LOG2);
  fogFactor = clamp(fogFactor, 0.0, 1.0);

  color = mix(gl_Fog.color, color, fogFactor);
}

void applyErosion(in vec4 noisevec)
{
  float intensity = (noisevec[0] + noisevec[1] + noisevec[2] + noisevec[3]) / 2.0;
  if ((intensity < EROSION_MIN) || (intensity > EROSION_MAX))
  {
    discard;
  }
}

void calculateDistanceMetricForSimpleSphereImpostor(in vec2 mapping, out float distanceMetric)
{
  float lengthSquared = dot(mapping, mapping);
  if (lengthSquared > 1.0)
  {
    discard;
  }

  float oneMinusLengthSquared = 1.0 - lengthSquared;
  distanceMetric = sqrt(oneMinusLengthSquared);
}

void simpleSphereImpostor(in vec2 mapping, in float distanceMetric, in float sphereRadius, in vec3 cameraSpherePosition, in bool multiplySphereRadius, out vec3 cameraNormal, out vec3 cameraPosition)
{
  cameraNormal = normalize(vec3(mapping, distanceMetric));
  cameraPosition = ((multiplySphereRadius) ? (cameraNormal * sphereRadius) : (cameraNormal / sphereRadius)) + cameraSpherePosition;
}

void applyPointSphereSimpleLighting(in float distanceMetric, in vec4 materialColor, inout vec4 color)
{
  color = clamp(vec4(materialColor.rgb * distanceMetric + vec3(0.5) * distanceMetric * distanceMetric * distanceMetric, 1.0), 0.0, 1.0);
}

void applySimpleSphereImpostorAntiAlias(in vec2 mapping, in float circularFactor, inout float sphereAlpha)
{
  float lengthSquared = dot(mapping, mapping);
  if (lengthSquared > 0.75)
  {
    float alphaFactor = clamp(SIMPLE_SPHERE_ALPHA_FACTOR * circularFactor * (1.0 - lengthSquared), 0.0, 1.0);
    sphereAlpha *= smootherstep(0.0, 1.0, 2.0 * alphaFactor);
  }
}

vec2 applyImpostorSphereTextureCoords(in vec3 normal)
{
  normal = normalize(mat3(gl_ModelViewMatrixInverse) * normal); // gl_NormalMatrixInverse does not exist in GLSL specs, is upper left 3x3 part of gl_ModelViewMatrixInverse
  float s = atan(normal.z, normal.x);
  float t = -2.0 * asin(normal.y);

  return 0.5 - vec2(s, t) / PI_TWO;
}

float applyDepth(in vec3 position)
{
  vec4 position4D = vec4(position, 1.0);
  float clipPositionZ = dot(gl_ProjectionMatrixTranspose[2], position4D);
  float clipPositionW = dot(gl_ProjectionMatrixTranspose[3], position4D);
  float ndcDepth = clipPositionZ / clipPositionW;

  return (gl_DepthRange.diff * ndcDepth + gl_DepthRange.near + gl_DepthRange.far) / 2.0;
}

float applyLinearDepth(in float depthValue, in float zNear, in float zFar)
{
  // float depthNormalized = 2.0 * depthValue - 1.0; // // depthNormalized should be in the range [-1, 1], depthValue is in the range [0, 1]
  // return 2.0 * zNear * zFar / (zFar + zNear - depthNormalized * (zFar - zNear));

  return (zNear * zFar) / (zFar - depthValue * (zFar - zNear)); // one-line solution of the above from GPU Computing Gems book by combining the equations above
}

vec3 reconstructEyePositionFromDepth(in float depthValue, in float zNear, in float zFar, in float tanFieldOfView, in int screenWidth, in int screenHeight)
{
  float top    =  zNear * tanFieldOfView; // original calculation conducted on the CPU side is: tan(fieldOfView * 0.5 * PI / 180.0)
  float right  =  top * float(screenWidth) / float(screenHeight);
  // float bottom = -top;
  // float left   = -right;

  // Reconstructed ndc-space position
  // vec2 ndcPosition = 2.0 * gl_FragCoord.xy / vec2(screenWidth, screenHeight) - 1.0; // SIMD *-divide commands in 2 channels
  vec2 ndcPosition = 2.0 * gl_TexCoord[0].st - 1.0;

  // Reconstructed eye-space position
  vec3 eyePosition = vec3(0.0, 0.0, -applyLinearDepth(depthValue, zNear, zFar));

  // full equatios for eyePosition.xy
  // eyePosition.x = ((-ndcPosition.x * eyePosition.z) * (right - left) / (2.0 * zNear) - eyePosition.z * (right + left) / (2.0 * zNear));
  // eyePosition.y = ((-ndcPosition.y * eyePosition.z) * (top - bottom) / (2.0 * zNear) - eyePosition.z * (top + bottom) / (2.0 * zNear));
  // the equations above simplify to the equations below
  // eyePosition.x = (-ndcPosition.x * eyePosition.z) * right / zNear;
  // eyePosition.y = (-ndcPosition.y * eyePosition.z) * top   / zNear;

  eyePosition.xy = (-ndcPosition * eyePosition.z) * vec2(right, top) / zNear; // SIMD *-divide commands in 2 channels

  return eyePosition;
}